/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securityhub.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the signals involved in an Amazon GuardDuty Extended Threat Detection attack sequence. An
 * attack sequence is a type of threat detected by GuardDuty. GuardDuty generates an attack sequence finding when
 * multiple events, or signals, align to a potentially suspicious activity. When GuardDuty and Security Hub are
 * integrated, GuardDuty sends attack sequence findings to Security Hub.
 * </p>
 * <p>
 * A signal can be an API activity or a finding that GuardDuty uses to detect an attack sequence finding.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Signal implements SdkPojo, Serializable, ToCopyableBuilder<Signal.Builder, Signal> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(Signal::type)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(Signal::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Title")
            .getter(getter(Signal::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Title").build()).build();

    private static final SdkField<String> PRODUCT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProductArn").getter(getter(Signal::productArn)).setter(setter(Builder::productArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProductArn").build()).build();

    private static final SdkField<List<String>> RESOURCE_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ResourceIds")
            .getter(getter(Signal::resourceIds))
            .setter(setter(Builder::resourceIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Indicator>> SIGNAL_INDICATORS_FIELD = SdkField
            .<List<Indicator>> builder(MarshallingType.LIST)
            .memberName("SignalIndicators")
            .getter(getter(Signal::signalIndicators))
            .setter(setter(Builder::signalIndicators))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SignalIndicators").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Indicator> builder(MarshallingType.SDK_POJO)
                                            .constructor(Indicator::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Signal::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<Long> CREATED_AT_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("CreatedAt")
            .getter(getter(Signal::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedAt").build()).build();

    private static final SdkField<Long> UPDATED_AT_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("UpdatedAt")
            .getter(getter(Signal::updatedAt)).setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdatedAt").build()).build();

    private static final SdkField<Long> FIRST_SEEN_AT_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("FirstSeenAt").getter(getter(Signal::firstSeenAt)).setter(setter(Builder::firstSeenAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FirstSeenAt").build()).build();

    private static final SdkField<Long> LAST_SEEN_AT_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("LastSeenAt").getter(getter(Signal::lastSeenAt)).setter(setter(Builder::lastSeenAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastSeenAt").build()).build();

    private static final SdkField<Double> SEVERITY_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("Severity").getter(getter(Signal::severity)).setter(setter(Builder::severity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Severity").build()).build();

    private static final SdkField<Integer> COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Count")
            .getter(getter(Signal::count)).setter(setter(Builder::count))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Count").build()).build();

    private static final SdkField<List<String>> ACTOR_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ActorIds")
            .getter(getter(Signal::actorIds))
            .setter(setter(Builder::actorIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActorIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> ENDPOINT_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("EndpointIds")
            .getter(getter(Signal::endpointIds))
            .setter(setter(Builder::endpointIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD, ID_FIELD,
            TITLE_FIELD, PRODUCT_ARN_FIELD, RESOURCE_IDS_FIELD, SIGNAL_INDICATORS_FIELD, NAME_FIELD, CREATED_AT_FIELD,
            UPDATED_AT_FIELD, FIRST_SEEN_AT_FIELD, LAST_SEEN_AT_FIELD, SEVERITY_FIELD, COUNT_FIELD, ACTOR_IDS_FIELD,
            ENDPOINT_IDS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String type;

    private final String id;

    private final String title;

    private final String productArn;

    private final List<String> resourceIds;

    private final List<Indicator> signalIndicators;

    private final String name;

    private final Long createdAt;

    private final Long updatedAt;

    private final Long firstSeenAt;

    private final Long lastSeenAt;

    private final Double severity;

    private final Integer count;

    private final List<String> actorIds;

    private final List<String> endpointIds;

    private Signal(BuilderImpl builder) {
        this.type = builder.type;
        this.id = builder.id;
        this.title = builder.title;
        this.productArn = builder.productArn;
        this.resourceIds = builder.resourceIds;
        this.signalIndicators = builder.signalIndicators;
        this.name = builder.name;
        this.createdAt = builder.createdAt;
        this.updatedAt = builder.updatedAt;
        this.firstSeenAt = builder.firstSeenAt;
        this.lastSeenAt = builder.lastSeenAt;
        this.severity = builder.severity;
        this.count = builder.count;
        this.actorIds = builder.actorIds;
        this.endpointIds = builder.endpointIds;
    }

    /**
     * <p>
     * The type of the signal used to identify an attack sequence.
     * </p>
     * <p>
     * Signals can be GuardDuty findings or activities observed in data sources that GuardDuty monitors. For more
     * information, see <a href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_data-sources.html">GuardDuty
     * foundational data sources</a> in the <i>Amazon GuardDuty User Guide</i>.
     * </p>
     * <p>
     * A signal type can be one of the following values. Here are the related descriptions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>FINDING</code> - Individually generated GuardDuty finding.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CLOUD_TRAIL</code> - Activity observed from CloudTrail logs
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>S3_DATA_EVENTS</code> - Activity observed from CloudTrail data events for Amazon Simple Storage Service
     * (S3). Activities associated with this type will show up only when you have enabled GuardDuty S3 Protection
     * feature in your account. For more information about S3 Protection and the steps to enable it, see <a
     * href="https://docs.aws.amazon.com/guardduty/latest/ug/s3-protection.html">S3 Protection</a> in the <i>Amazon
     * GuardDuty User Guide</i>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The type of the signal used to identify an attack sequence. </p>
     *         <p>
     *         Signals can be GuardDuty findings or activities observed in data sources that GuardDuty monitors. For
     *         more information, see <a
     *         href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_data-sources.html">GuardDuty foundational
     *         data sources</a> in the <i>Amazon GuardDuty User Guide</i>.
     *         </p>
     *         <p>
     *         A signal type can be one of the following values. Here are the related descriptions:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>FINDING</code> - Individually generated GuardDuty finding.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CLOUD_TRAIL</code> - Activity observed from CloudTrail logs
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>S3_DATA_EVENTS</code> - Activity observed from CloudTrail data events for Amazon Simple Storage
     *         Service (S3). Activities associated with this type will show up only when you have enabled GuardDuty S3
     *         Protection feature in your account. For more information about S3 Protection and the steps to enable it,
     *         see <a href="https://docs.aws.amazon.com/guardduty/latest/ug/s3-protection.html">S3 Protection</a> in the
     *         <i>Amazon GuardDuty User Guide</i>.
     *         </p>
     *         </li>
     */
    public final String type() {
        return type;
    }

    /**
     * <p>
     * The identifier of the signal.
     * </p>
     * 
     * @return The identifier of the signal.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The description of the GuardDuty finding.
     * </p>
     * 
     * @return The description of the GuardDuty finding.
     */
    public final String title() {
        return title;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the product that generated the signal.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the product that generated the signal.
     */
    public final String productArn() {
        return productArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceIds() {
        return resourceIds != null && !(resourceIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The ARN or ID of the Amazon Web Services resource associated with the signal.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceIds} method.
     * </p>
     * 
     * @return The ARN or ID of the Amazon Web Services resource associated with the signal.
     */
    public final List<String> resourceIds() {
        return resourceIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the SignalIndicators property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSignalIndicators() {
        return signalIndicators != null && !(signalIndicators instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Contains information about the indicators associated with the signals in this attack sequence finding. The values
     * for <code>SignalIndicators</code> are a subset of the values for <a
     * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_Sequence.html">SequenceIndicators</a>, but the
     * values for these fields don't always match 1:1.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSignalIndicators} method.
     * </p>
     * 
     * @return Contains information about the indicators associated with the signals in this attack sequence finding.
     *         The values for <code>SignalIndicators</code> are a subset of the values for <a
     *         href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_Sequence.html">SequenceIndicators</a>,
     *         but the values for these fields don't always match 1:1.
     */
    public final List<Indicator> signalIndicators() {
        return signalIndicators;
    }

    /**
     * <p>
     * The name of the GuardDuty signal. For example, when signal type is <code>FINDING</code>, the signal name is the
     * name of the finding.
     * </p>
     * 
     * @return The name of the GuardDuty signal. For example, when signal type is <code>FINDING</code>, the signal name
     *         is the name of the finding.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The timestamp when the first finding or activity related to this signal was observed.
     * </p>
     * 
     * @return The timestamp when the first finding or activity related to this signal was observed.
     */
    public final Long createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The timestamp when this signal was last observed.
     * </p>
     * 
     * @return The timestamp when this signal was last observed.
     */
    public final Long updatedAt() {
        return updatedAt;
    }

    /**
     * <p>
     * The timestamp when the first finding or activity related to this signal was observed.
     * </p>
     * 
     * @return The timestamp when the first finding or activity related to this signal was observed.
     */
    public final Long firstSeenAt() {
        return firstSeenAt;
    }

    /**
     * <p>
     * The timestamp when the last finding or activity related to this signal was observed.
     * </p>
     * 
     * @return The timestamp when the last finding or activity related to this signal was observed.
     */
    public final Long lastSeenAt() {
        return lastSeenAt;
    }

    /**
     * <p>
     * The severity associated with the signal. For more information about severity, see <a
     * href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_findings-severity.html">Severity levels for
     * GuardDuty findings</a> in the <i>Amazon GuardDuty User Guide</i>.
     * </p>
     * 
     * @return The severity associated with the signal. For more information about severity, see <a
     *         href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_findings-severity.html">Severity levels
     *         for GuardDuty findings</a> in the <i>Amazon GuardDuty User Guide</i>.
     */
    public final Double severity() {
        return severity;
    }

    /**
     * <p>
     * The number of times this signal was observed.
     * </p>
     * 
     * @return The number of times this signal was observed.
     */
    public final Integer count() {
        return count;
    }

    /**
     * For responses, this returns true if the service returned a value for the ActorIds property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasActorIds() {
        return actorIds != null && !(actorIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IDs of the threat actors involved in the signal.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasActorIds} method.
     * </p>
     * 
     * @return The IDs of the threat actors involved in the signal.
     */
    public final List<String> actorIds() {
        return actorIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the EndpointIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEndpointIds() {
        return endpointIds != null && !(endpointIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Information about the endpoint IDs associated with this signal.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEndpointIds} method.
     * </p>
     * 
     * @return Information about the endpoint IDs associated with this signal.
     */
    public final List<String> endpointIds() {
        return endpointIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(type());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(productArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceIds() ? resourceIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSignalIndicators() ? signalIndicators() : null);
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(firstSeenAt());
        hashCode = 31 * hashCode + Objects.hashCode(lastSeenAt());
        hashCode = 31 * hashCode + Objects.hashCode(severity());
        hashCode = 31 * hashCode + Objects.hashCode(count());
        hashCode = 31 * hashCode + Objects.hashCode(hasActorIds() ? actorIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEndpointIds() ? endpointIds() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Signal)) {
            return false;
        }
        Signal other = (Signal) obj;
        return Objects.equals(type(), other.type()) && Objects.equals(id(), other.id()) && Objects.equals(title(), other.title())
                && Objects.equals(productArn(), other.productArn()) && hasResourceIds() == other.hasResourceIds()
                && Objects.equals(resourceIds(), other.resourceIds()) && hasSignalIndicators() == other.hasSignalIndicators()
                && Objects.equals(signalIndicators(), other.signalIndicators()) && Objects.equals(name(), other.name())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(updatedAt(), other.updatedAt())
                && Objects.equals(firstSeenAt(), other.firstSeenAt()) && Objects.equals(lastSeenAt(), other.lastSeenAt())
                && Objects.equals(severity(), other.severity()) && Objects.equals(count(), other.count())
                && hasActorIds() == other.hasActorIds() && Objects.equals(actorIds(), other.actorIds())
                && hasEndpointIds() == other.hasEndpointIds() && Objects.equals(endpointIds(), other.endpointIds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Signal").add("Type", type()).add("Id", id()).add("Title", title())
                .add("ProductArn", productArn()).add("ResourceIds", hasResourceIds() ? resourceIds() : null)
                .add("SignalIndicators", hasSignalIndicators() ? signalIndicators() : null).add("Name", name())
                .add("CreatedAt", createdAt()).add("UpdatedAt", updatedAt()).add("FirstSeenAt", firstSeenAt())
                .add("LastSeenAt", lastSeenAt()).add("Severity", severity()).add("Count", count())
                .add("ActorIds", hasActorIds() ? actorIds() : null).add("EndpointIds", hasEndpointIds() ? endpointIds() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Type":
            return Optional.ofNullable(clazz.cast(type()));
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Title":
            return Optional.ofNullable(clazz.cast(title()));
        case "ProductArn":
            return Optional.ofNullable(clazz.cast(productArn()));
        case "ResourceIds":
            return Optional.ofNullable(clazz.cast(resourceIds()));
        case "SignalIndicators":
            return Optional.ofNullable(clazz.cast(signalIndicators()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "CreatedAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "UpdatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        case "FirstSeenAt":
            return Optional.ofNullable(clazz.cast(firstSeenAt()));
        case "LastSeenAt":
            return Optional.ofNullable(clazz.cast(lastSeenAt()));
        case "Severity":
            return Optional.ofNullable(clazz.cast(severity()));
        case "Count":
            return Optional.ofNullable(clazz.cast(count()));
        case "ActorIds":
            return Optional.ofNullable(clazz.cast(actorIds()));
        case "EndpointIds":
            return Optional.ofNullable(clazz.cast(endpointIds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Type", TYPE_FIELD);
        map.put("Id", ID_FIELD);
        map.put("Title", TITLE_FIELD);
        map.put("ProductArn", PRODUCT_ARN_FIELD);
        map.put("ResourceIds", RESOURCE_IDS_FIELD);
        map.put("SignalIndicators", SIGNAL_INDICATORS_FIELD);
        map.put("Name", NAME_FIELD);
        map.put("CreatedAt", CREATED_AT_FIELD);
        map.put("UpdatedAt", UPDATED_AT_FIELD);
        map.put("FirstSeenAt", FIRST_SEEN_AT_FIELD);
        map.put("LastSeenAt", LAST_SEEN_AT_FIELD);
        map.put("Severity", SEVERITY_FIELD);
        map.put("Count", COUNT_FIELD);
        map.put("ActorIds", ACTOR_IDS_FIELD);
        map.put("EndpointIds", ENDPOINT_IDS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Signal, T> g) {
        return obj -> g.apply((Signal) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Signal> {
        /**
         * <p>
         * The type of the signal used to identify an attack sequence.
         * </p>
         * <p>
         * Signals can be GuardDuty findings or activities observed in data sources that GuardDuty monitors. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_data-sources.html">GuardDuty foundational
         * data sources</a> in the <i>Amazon GuardDuty User Guide</i>.
         * </p>
         * <p>
         * A signal type can be one of the following values. Here are the related descriptions:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>FINDING</code> - Individually generated GuardDuty finding.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CLOUD_TRAIL</code> - Activity observed from CloudTrail logs
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>S3_DATA_EVENTS</code> - Activity observed from CloudTrail data events for Amazon Simple Storage Service
         * (S3). Activities associated with this type will show up only when you have enabled GuardDuty S3 Protection
         * feature in your account. For more information about S3 Protection and the steps to enable it, see <a
         * href="https://docs.aws.amazon.com/guardduty/latest/ug/s3-protection.html">S3 Protection</a> in the <i>Amazon
         * GuardDuty User Guide</i>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of the signal used to identify an attack sequence. </p>
         *        <p>
         *        Signals can be GuardDuty findings or activities observed in data sources that GuardDuty monitors. For
         *        more information, see <a
         *        href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_data-sources.html">GuardDuty
         *        foundational data sources</a> in the <i>Amazon GuardDuty User Guide</i>.
         *        </p>
         *        <p>
         *        A signal type can be one of the following values. Here are the related descriptions:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>FINDING</code> - Individually generated GuardDuty finding.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CLOUD_TRAIL</code> - Activity observed from CloudTrail logs
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>S3_DATA_EVENTS</code> - Activity observed from CloudTrail data events for Amazon Simple Storage
         *        Service (S3). Activities associated with this type will show up only when you have enabled GuardDuty
         *        S3 Protection feature in your account. For more information about S3 Protection and the steps to
         *        enable it, see <a href="https://docs.aws.amazon.com/guardduty/latest/ug/s3-protection.html">S3
         *        Protection</a> in the <i>Amazon GuardDuty User Guide</i>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder type(String type);

        /**
         * <p>
         * The identifier of the signal.
         * </p>
         * 
         * @param id
         *        The identifier of the signal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The description of the GuardDuty finding.
         * </p>
         * 
         * @param title
         *        The description of the GuardDuty finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the product that generated the signal.
         * </p>
         * 
         * @param productArn
         *        The Amazon Resource Name (ARN) of the product that generated the signal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder productArn(String productArn);

        /**
         * <p>
         * The ARN or ID of the Amazon Web Services resource associated with the signal.
         * </p>
         * 
         * @param resourceIds
         *        The ARN or ID of the Amazon Web Services resource associated with the signal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceIds(Collection<String> resourceIds);

        /**
         * <p>
         * The ARN or ID of the Amazon Web Services resource associated with the signal.
         * </p>
         * 
         * @param resourceIds
         *        The ARN or ID of the Amazon Web Services resource associated with the signal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceIds(String... resourceIds);

        /**
         * <p>
         * Contains information about the indicators associated with the signals in this attack sequence finding. The
         * values for <code>SignalIndicators</code> are a subset of the values for <a
         * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_Sequence.html">SequenceIndicators</a>, but
         * the values for these fields don't always match 1:1.
         * </p>
         * 
         * @param signalIndicators
         *        Contains information about the indicators associated with the signals in this attack sequence finding.
         *        The values for <code>SignalIndicators</code> are a subset of the values for <a
         *        href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_Sequence.html"
         *        >SequenceIndicators</a>, but the values for these fields don't always match 1:1.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signalIndicators(Collection<Indicator> signalIndicators);

        /**
         * <p>
         * Contains information about the indicators associated with the signals in this attack sequence finding. The
         * values for <code>SignalIndicators</code> are a subset of the values for <a
         * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_Sequence.html">SequenceIndicators</a>, but
         * the values for these fields don't always match 1:1.
         * </p>
         * 
         * @param signalIndicators
         *        Contains information about the indicators associated with the signals in this attack sequence finding.
         *        The values for <code>SignalIndicators</code> are a subset of the values for <a
         *        href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_Sequence.html"
         *        >SequenceIndicators</a>, but the values for these fields don't always match 1:1.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signalIndicators(Indicator... signalIndicators);

        /**
         * <p>
         * Contains information about the indicators associated with the signals in this attack sequence finding. The
         * values for <code>SignalIndicators</code> are a subset of the values for <a
         * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_Sequence.html">SequenceIndicators</a>, but
         * the values for these fields don't always match 1:1.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.securityhub.model.Indicator.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.securityhub.model.Indicator#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.securityhub.model.Indicator.Builder#build()} is called immediately and
         * its result is passed to {@link #signalIndicators(List<Indicator>)}.
         * 
         * @param signalIndicators
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.securityhub.model.Indicator.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #signalIndicators(java.util.Collection<Indicator>)
         */
        Builder signalIndicators(Consumer<Indicator.Builder>... signalIndicators);

        /**
         * <p>
         * The name of the GuardDuty signal. For example, when signal type is <code>FINDING</code>, the signal name is
         * the name of the finding.
         * </p>
         * 
         * @param name
         *        The name of the GuardDuty signal. For example, when signal type is <code>FINDING</code>, the signal
         *        name is the name of the finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The timestamp when the first finding or activity related to this signal was observed.
         * </p>
         * 
         * @param createdAt
         *        The timestamp when the first finding or activity related to this signal was observed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Long createdAt);

        /**
         * <p>
         * The timestamp when this signal was last observed.
         * </p>
         * 
         * @param updatedAt
         *        The timestamp when this signal was last observed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Long updatedAt);

        /**
         * <p>
         * The timestamp when the first finding or activity related to this signal was observed.
         * </p>
         * 
         * @param firstSeenAt
         *        The timestamp when the first finding or activity related to this signal was observed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder firstSeenAt(Long firstSeenAt);

        /**
         * <p>
         * The timestamp when the last finding or activity related to this signal was observed.
         * </p>
         * 
         * @param lastSeenAt
         *        The timestamp when the last finding or activity related to this signal was observed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastSeenAt(Long lastSeenAt);

        /**
         * <p>
         * The severity associated with the signal. For more information about severity, see <a
         * href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_findings-severity.html">Severity levels for
         * GuardDuty findings</a> in the <i>Amazon GuardDuty User Guide</i>.
         * </p>
         * 
         * @param severity
         *        The severity associated with the signal. For more information about severity, see <a
         *        href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_findings-severity.html">Severity
         *        levels for GuardDuty findings</a> in the <i>Amazon GuardDuty User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder severity(Double severity);

        /**
         * <p>
         * The number of times this signal was observed.
         * </p>
         * 
         * @param count
         *        The number of times this signal was observed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder count(Integer count);

        /**
         * <p>
         * The IDs of the threat actors involved in the signal.
         * </p>
         * 
         * @param actorIds
         *        The IDs of the threat actors involved in the signal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actorIds(Collection<String> actorIds);

        /**
         * <p>
         * The IDs of the threat actors involved in the signal.
         * </p>
         * 
         * @param actorIds
         *        The IDs of the threat actors involved in the signal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actorIds(String... actorIds);

        /**
         * <p>
         * Information about the endpoint IDs associated with this signal.
         * </p>
         * 
         * @param endpointIds
         *        Information about the endpoint IDs associated with this signal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointIds(Collection<String> endpointIds);

        /**
         * <p>
         * Information about the endpoint IDs associated with this signal.
         * </p>
         * 
         * @param endpointIds
         *        Information about the endpoint IDs associated with this signal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointIds(String... endpointIds);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String id;

        private String title;

        private String productArn;

        private List<String> resourceIds = DefaultSdkAutoConstructList.getInstance();

        private List<Indicator> signalIndicators = DefaultSdkAutoConstructList.getInstance();

        private String name;

        private Long createdAt;

        private Long updatedAt;

        private Long firstSeenAt;

        private Long lastSeenAt;

        private Double severity;

        private Integer count;

        private List<String> actorIds = DefaultSdkAutoConstructList.getInstance();

        private List<String> endpointIds = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Signal model) {
            type(model.type);
            id(model.id);
            title(model.title);
            productArn(model.productArn);
            resourceIds(model.resourceIds);
            signalIndicators(model.signalIndicators);
            name(model.name);
            createdAt(model.createdAt);
            updatedAt(model.updatedAt);
            firstSeenAt(model.firstSeenAt);
            lastSeenAt(model.lastSeenAt);
            severity(model.severity);
            count(model.count);
            actorIds(model.actorIds);
            endpointIds(model.endpointIds);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getTitle() {
            return title;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final String getProductArn() {
            return productArn;
        }

        public final void setProductArn(String productArn) {
            this.productArn = productArn;
        }

        @Override
        public final Builder productArn(String productArn) {
            this.productArn = productArn;
            return this;
        }

        public final Collection<String> getResourceIds() {
            if (resourceIds instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceIds;
        }

        public final void setResourceIds(Collection<String> resourceIds) {
            this.resourceIds = NonEmptyStringListCopier.copy(resourceIds);
        }

        @Override
        public final Builder resourceIds(Collection<String> resourceIds) {
            this.resourceIds = NonEmptyStringListCopier.copy(resourceIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceIds(String... resourceIds) {
            resourceIds(Arrays.asList(resourceIds));
            return this;
        }

        public final List<Indicator.Builder> getSignalIndicators() {
            List<Indicator.Builder> result = IndicatorsListCopier.copyToBuilder(this.signalIndicators);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSignalIndicators(Collection<Indicator.BuilderImpl> signalIndicators) {
            this.signalIndicators = IndicatorsListCopier.copyFromBuilder(signalIndicators);
        }

        @Override
        public final Builder signalIndicators(Collection<Indicator> signalIndicators) {
            this.signalIndicators = IndicatorsListCopier.copy(signalIndicators);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder signalIndicators(Indicator... signalIndicators) {
            signalIndicators(Arrays.asList(signalIndicators));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder signalIndicators(Consumer<Indicator.Builder>... signalIndicators) {
            signalIndicators(Stream.of(signalIndicators).map(c -> Indicator.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Long getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Long createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Long createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Long getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Long updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        public final Builder updatedAt(Long updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        public final Long getFirstSeenAt() {
            return firstSeenAt;
        }

        public final void setFirstSeenAt(Long firstSeenAt) {
            this.firstSeenAt = firstSeenAt;
        }

        @Override
        public final Builder firstSeenAt(Long firstSeenAt) {
            this.firstSeenAt = firstSeenAt;
            return this;
        }

        public final Long getLastSeenAt() {
            return lastSeenAt;
        }

        public final void setLastSeenAt(Long lastSeenAt) {
            this.lastSeenAt = lastSeenAt;
        }

        @Override
        public final Builder lastSeenAt(Long lastSeenAt) {
            this.lastSeenAt = lastSeenAt;
            return this;
        }

        public final Double getSeverity() {
            return severity;
        }

        public final void setSeverity(Double severity) {
            this.severity = severity;
        }

        @Override
        public final Builder severity(Double severity) {
            this.severity = severity;
            return this;
        }

        public final Integer getCount() {
            return count;
        }

        public final void setCount(Integer count) {
            this.count = count;
        }

        @Override
        public final Builder count(Integer count) {
            this.count = count;
            return this;
        }

        public final Collection<String> getActorIds() {
            if (actorIds instanceof SdkAutoConstructList) {
                return null;
            }
            return actorIds;
        }

        public final void setActorIds(Collection<String> actorIds) {
            this.actorIds = NonEmptyStringListCopier.copy(actorIds);
        }

        @Override
        public final Builder actorIds(Collection<String> actorIds) {
            this.actorIds = NonEmptyStringListCopier.copy(actorIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actorIds(String... actorIds) {
            actorIds(Arrays.asList(actorIds));
            return this;
        }

        public final Collection<String> getEndpointIds() {
            if (endpointIds instanceof SdkAutoConstructList) {
                return null;
            }
            return endpointIds;
        }

        public final void setEndpointIds(Collection<String> endpointIds) {
            this.endpointIds = NonEmptyStringListCopier.copy(endpointIds);
        }

        @Override
        public final Builder endpointIds(Collection<String> endpointIds) {
            this.endpointIds = NonEmptyStringListCopier.copy(endpointIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder endpointIds(String... endpointIds) {
            endpointIds(Arrays.asList(endpointIds));
            return this;
        }

        @Override
        public Signal build() {
            return new Signal(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
