/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securityhub.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about an Amazon GuardDuty Extended Threat Detection attack sequence finding. GuardDuty generates
 * an attack sequence finding when multiple events align to a potentially suspicious activity. To receive GuardDuty
 * attack sequence findings in Security Hub, you must have GuardDuty enabled. For more information, see <a
 * href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty-extended-threat-detection.html">GuardDuty Extended
 * Threat Detection </a> in the <i>Amazon GuardDuty User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Sequence implements SdkPojo, Serializable, ToCopyableBuilder<Sequence.Builder, Sequence> {
    private static final SdkField<String> UID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Uid")
            .getter(getter(Sequence::uid)).setter(setter(Builder::uid))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Uid").build()).build();

    private static final SdkField<List<Actor>> ACTORS_FIELD = SdkField
            .<List<Actor>> builder(MarshallingType.LIST)
            .memberName("Actors")
            .getter(getter(Sequence::actors))
            .setter(setter(Builder::actors))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Actors").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Actor> builder(MarshallingType.SDK_POJO)
                                            .constructor(Actor::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<NetworkEndpoint>> ENDPOINTS_FIELD = SdkField
            .<List<NetworkEndpoint>> builder(MarshallingType.LIST)
            .memberName("Endpoints")
            .getter(getter(Sequence::endpoints))
            .setter(setter(Builder::endpoints))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Endpoints").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<NetworkEndpoint> builder(MarshallingType.SDK_POJO)
                                            .constructor(NetworkEndpoint::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Signal>> SIGNALS_FIELD = SdkField
            .<List<Signal>> builder(MarshallingType.LIST)
            .memberName("Signals")
            .getter(getter(Sequence::signals))
            .setter(setter(Builder::signals))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Signals").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Signal> builder(MarshallingType.SDK_POJO)
                                            .constructor(Signal::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Indicator>> SEQUENCE_INDICATORS_FIELD = SdkField
            .<List<Indicator>> builder(MarshallingType.LIST)
            .memberName("SequenceIndicators")
            .getter(getter(Sequence::sequenceIndicators))
            .setter(setter(Builder::sequenceIndicators))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SequenceIndicators").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Indicator> builder(MarshallingType.SDK_POJO)
                                            .constructor(Indicator::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(UID_FIELD, ACTORS_FIELD,
            ENDPOINTS_FIELD, SIGNALS_FIELD, SEQUENCE_INDICATORS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String uid;

    private final List<Actor> actors;

    private final List<NetworkEndpoint> endpoints;

    private final List<Signal> signals;

    private final List<Indicator> sequenceIndicators;

    private Sequence(BuilderImpl builder) {
        this.uid = builder.uid;
        this.actors = builder.actors;
        this.endpoints = builder.endpoints;
        this.signals = builder.signals;
        this.sequenceIndicators = builder.sequenceIndicators;
    }

    /**
     * <p>
     * Unique identifier of the attack sequence.
     * </p>
     * 
     * @return Unique identifier of the attack sequence.
     */
    public final String uid() {
        return uid;
    }

    /**
     * For responses, this returns true if the service returned a value for the Actors property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasActors() {
        return actors != null && !(actors instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Provides information about the actors involved in the attack sequence.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasActors} method.
     * </p>
     * 
     * @return Provides information about the actors involved in the attack sequence.
     */
    public final List<Actor> actors() {
        return actors;
    }

    /**
     * For responses, this returns true if the service returned a value for the Endpoints property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasEndpoints() {
        return endpoints != null && !(endpoints instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Contains information about the network endpoints that were used in the attack sequence.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEndpoints} method.
     * </p>
     * 
     * @return Contains information about the network endpoints that were used in the attack sequence.
     */
    public final List<NetworkEndpoint> endpoints() {
        return endpoints;
    }

    /**
     * For responses, this returns true if the service returned a value for the Signals property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSignals() {
        return signals != null && !(signals instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Contains information about the signals involved in the attack sequence.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSignals} method.
     * </p>
     * 
     * @return Contains information about the signals involved in the attack sequence.
     */
    public final List<Signal> signals() {
        return signals;
    }

    /**
     * For responses, this returns true if the service returned a value for the SequenceIndicators property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSequenceIndicators() {
        return sequenceIndicators != null && !(sequenceIndicators instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Contains information about the indicators observed in the attack sequence. The values for <a
     * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_Signal.html">SignalIndicators</a> are a subset
     * of the values for <code>SequenceIndicators</code>, but the values for these fields don't always match 1:1.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSequenceIndicators} method.
     * </p>
     * 
     * @return Contains information about the indicators observed in the attack sequence. The values for <a
     *         href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_Signal.html">SignalIndicators</a> are
     *         a subset of the values for <code>SequenceIndicators</code>, but the values for these fields don't always
     *         match 1:1.
     */
    public final List<Indicator> sequenceIndicators() {
        return sequenceIndicators;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(uid());
        hashCode = 31 * hashCode + Objects.hashCode(hasActors() ? actors() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEndpoints() ? endpoints() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSignals() ? signals() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSequenceIndicators() ? sequenceIndicators() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Sequence)) {
            return false;
        }
        Sequence other = (Sequence) obj;
        return Objects.equals(uid(), other.uid()) && hasActors() == other.hasActors() && Objects.equals(actors(), other.actors())
                && hasEndpoints() == other.hasEndpoints() && Objects.equals(endpoints(), other.endpoints())
                && hasSignals() == other.hasSignals() && Objects.equals(signals(), other.signals())
                && hasSequenceIndicators() == other.hasSequenceIndicators()
                && Objects.equals(sequenceIndicators(), other.sequenceIndicators());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Sequence").add("Uid", uid()).add("Actors", hasActors() ? actors() : null)
                .add("Endpoints", hasEndpoints() ? endpoints() : null).add("Signals", hasSignals() ? signals() : null)
                .add("SequenceIndicators", hasSequenceIndicators() ? sequenceIndicators() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Uid":
            return Optional.ofNullable(clazz.cast(uid()));
        case "Actors":
            return Optional.ofNullable(clazz.cast(actors()));
        case "Endpoints":
            return Optional.ofNullable(clazz.cast(endpoints()));
        case "Signals":
            return Optional.ofNullable(clazz.cast(signals()));
        case "SequenceIndicators":
            return Optional.ofNullable(clazz.cast(sequenceIndicators()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Uid", UID_FIELD);
        map.put("Actors", ACTORS_FIELD);
        map.put("Endpoints", ENDPOINTS_FIELD);
        map.put("Signals", SIGNALS_FIELD);
        map.put("SequenceIndicators", SEQUENCE_INDICATORS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Sequence, T> g) {
        return obj -> g.apply((Sequence) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Sequence> {
        /**
         * <p>
         * Unique identifier of the attack sequence.
         * </p>
         * 
         * @param uid
         *        Unique identifier of the attack sequence.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder uid(String uid);

        /**
         * <p>
         * Provides information about the actors involved in the attack sequence.
         * </p>
         * 
         * @param actors
         *        Provides information about the actors involved in the attack sequence.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actors(Collection<Actor> actors);

        /**
         * <p>
         * Provides information about the actors involved in the attack sequence.
         * </p>
         * 
         * @param actors
         *        Provides information about the actors involved in the attack sequence.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actors(Actor... actors);

        /**
         * <p>
         * Provides information about the actors involved in the attack sequence.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.securityhub.model.Actor.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.securityhub.model.Actor#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.securityhub.model.Actor.Builder#build()} is called immediately and its
         * result is passed to {@link #actors(List<Actor>)}.
         * 
         * @param actors
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.securityhub.model.Actor.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #actors(java.util.Collection<Actor>)
         */
        Builder actors(Consumer<Actor.Builder>... actors);

        /**
         * <p>
         * Contains information about the network endpoints that were used in the attack sequence.
         * </p>
         * 
         * @param endpoints
         *        Contains information about the network endpoints that were used in the attack sequence.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpoints(Collection<NetworkEndpoint> endpoints);

        /**
         * <p>
         * Contains information about the network endpoints that were used in the attack sequence.
         * </p>
         * 
         * @param endpoints
         *        Contains information about the network endpoints that were used in the attack sequence.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpoints(NetworkEndpoint... endpoints);

        /**
         * <p>
         * Contains information about the network endpoints that were used in the attack sequence.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.securityhub.model.NetworkEndpoint.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.securityhub.model.NetworkEndpoint#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.securityhub.model.NetworkEndpoint.Builder#build()} is called
         * immediately and its result is passed to {@link #endpoints(List<NetworkEndpoint>)}.
         * 
         * @param endpoints
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.securityhub.model.NetworkEndpoint.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #endpoints(java.util.Collection<NetworkEndpoint>)
         */
        Builder endpoints(Consumer<NetworkEndpoint.Builder>... endpoints);

        /**
         * <p>
         * Contains information about the signals involved in the attack sequence.
         * </p>
         * 
         * @param signals
         *        Contains information about the signals involved in the attack sequence.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signals(Collection<Signal> signals);

        /**
         * <p>
         * Contains information about the signals involved in the attack sequence.
         * </p>
         * 
         * @param signals
         *        Contains information about the signals involved in the attack sequence.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signals(Signal... signals);

        /**
         * <p>
         * Contains information about the signals involved in the attack sequence.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.securityhub.model.Signal.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.securityhub.model.Signal#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.securityhub.model.Signal.Builder#build()} is called immediately and
         * its result is passed to {@link #signals(List<Signal>)}.
         * 
         * @param signals
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.securityhub.model.Signal.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #signals(java.util.Collection<Signal>)
         */
        Builder signals(Consumer<Signal.Builder>... signals);

        /**
         * <p>
         * Contains information about the indicators observed in the attack sequence. The values for <a
         * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_Signal.html">SignalIndicators</a> are a
         * subset of the values for <code>SequenceIndicators</code>, but the values for these fields don't always match
         * 1:1.
         * </p>
         * 
         * @param sequenceIndicators
         *        Contains information about the indicators observed in the attack sequence. The values for <a
         *        href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_Signal.html">SignalIndicators</a>
         *        are a subset of the values for <code>SequenceIndicators</code>, but the values for these fields don't
         *        always match 1:1.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sequenceIndicators(Collection<Indicator> sequenceIndicators);

        /**
         * <p>
         * Contains information about the indicators observed in the attack sequence. The values for <a
         * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_Signal.html">SignalIndicators</a> are a
         * subset of the values for <code>SequenceIndicators</code>, but the values for these fields don't always match
         * 1:1.
         * </p>
         * 
         * @param sequenceIndicators
         *        Contains information about the indicators observed in the attack sequence. The values for <a
         *        href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_Signal.html">SignalIndicators</a>
         *        are a subset of the values for <code>SequenceIndicators</code>, but the values for these fields don't
         *        always match 1:1.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sequenceIndicators(Indicator... sequenceIndicators);

        /**
         * <p>
         * Contains information about the indicators observed in the attack sequence. The values for <a
         * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_Signal.html">SignalIndicators</a> are a
         * subset of the values for <code>SequenceIndicators</code>, but the values for these fields don't always match
         * 1:1.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.securityhub.model.Indicator.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.securityhub.model.Indicator#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.securityhub.model.Indicator.Builder#build()} is called immediately and
         * its result is passed to {@link #sequenceIndicators(List<Indicator>)}.
         * 
         * @param sequenceIndicators
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.securityhub.model.Indicator.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sequenceIndicators(java.util.Collection<Indicator>)
         */
        Builder sequenceIndicators(Consumer<Indicator.Builder>... sequenceIndicators);
    }

    static final class BuilderImpl implements Builder {
        private String uid;

        private List<Actor> actors = DefaultSdkAutoConstructList.getInstance();

        private List<NetworkEndpoint> endpoints = DefaultSdkAutoConstructList.getInstance();

        private List<Signal> signals = DefaultSdkAutoConstructList.getInstance();

        private List<Indicator> sequenceIndicators = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Sequence model) {
            uid(model.uid);
            actors(model.actors);
            endpoints(model.endpoints);
            signals(model.signals);
            sequenceIndicators(model.sequenceIndicators);
        }

        public final String getUid() {
            return uid;
        }

        public final void setUid(String uid) {
            this.uid = uid;
        }

        @Override
        public final Builder uid(String uid) {
            this.uid = uid;
            return this;
        }

        public final List<Actor.Builder> getActors() {
            List<Actor.Builder> result = ActorsListCopier.copyToBuilder(this.actors);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setActors(Collection<Actor.BuilderImpl> actors) {
            this.actors = ActorsListCopier.copyFromBuilder(actors);
        }

        @Override
        public final Builder actors(Collection<Actor> actors) {
            this.actors = ActorsListCopier.copy(actors);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actors(Actor... actors) {
            actors(Arrays.asList(actors));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actors(Consumer<Actor.Builder>... actors) {
            actors(Stream.of(actors).map(c -> Actor.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<NetworkEndpoint.Builder> getEndpoints() {
            List<NetworkEndpoint.Builder> result = NetworkEndpointsListCopier.copyToBuilder(this.endpoints);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEndpoints(Collection<NetworkEndpoint.BuilderImpl> endpoints) {
            this.endpoints = NetworkEndpointsListCopier.copyFromBuilder(endpoints);
        }

        @Override
        public final Builder endpoints(Collection<NetworkEndpoint> endpoints) {
            this.endpoints = NetworkEndpointsListCopier.copy(endpoints);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder endpoints(NetworkEndpoint... endpoints) {
            endpoints(Arrays.asList(endpoints));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder endpoints(Consumer<NetworkEndpoint.Builder>... endpoints) {
            endpoints(Stream.of(endpoints).map(c -> NetworkEndpoint.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<Signal.Builder> getSignals() {
            List<Signal.Builder> result = SignalsListCopier.copyToBuilder(this.signals);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSignals(Collection<Signal.BuilderImpl> signals) {
            this.signals = SignalsListCopier.copyFromBuilder(signals);
        }

        @Override
        public final Builder signals(Collection<Signal> signals) {
            this.signals = SignalsListCopier.copy(signals);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder signals(Signal... signals) {
            signals(Arrays.asList(signals));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder signals(Consumer<Signal.Builder>... signals) {
            signals(Stream.of(signals).map(c -> Signal.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<Indicator.Builder> getSequenceIndicators() {
            List<Indicator.Builder> result = IndicatorsListCopier.copyToBuilder(this.sequenceIndicators);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSequenceIndicators(Collection<Indicator.BuilderImpl> sequenceIndicators) {
            this.sequenceIndicators = IndicatorsListCopier.copyFromBuilder(sequenceIndicators);
        }

        @Override
        public final Builder sequenceIndicators(Collection<Indicator> sequenceIndicators) {
            this.sequenceIndicators = IndicatorsListCopier.copy(sequenceIndicators);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sequenceIndicators(Indicator... sequenceIndicators) {
            sequenceIndicators(Arrays.asList(sequenceIndicators));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sequenceIndicators(Consumer<Indicator.Builder>... sequenceIndicators) {
            sequenceIndicators(Stream.of(sequenceIndicators).map(c -> Indicator.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public Sequence build() {
            return new Sequence(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
