/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securityhub.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the authenticated session used by the threat actor identified in an Amazon GuardDuty
 * Extended Threat Detection attack sequence. GuardDuty generates an attack sequence finding when multiple events align
 * to a potentially suspicious activity. To receive GuardDuty attack sequence findings in Security Hub, you must have
 * GuardDuty enabled. For more information, see <a
 * href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty-extended-threat-detection.html">GuardDuty Extended
 * Threat Detection </a> in the <i>Amazon GuardDuty User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ActorSession implements SdkPojo, Serializable, ToCopyableBuilder<ActorSession.Builder, ActorSession> {
    private static final SdkField<String> UID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Uid")
            .getter(getter(ActorSession::uid)).setter(setter(Builder::uid))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Uid").build()).build();

    private static final SdkField<String> MFA_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MfaStatus").getter(getter(ActorSession::mfaStatusAsString)).setter(setter(Builder::mfaStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MfaStatus").build()).build();

    private static final SdkField<Long> CREATED_TIME_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("CreatedTime").getter(getter(ActorSession::createdTime)).setter(setter(Builder::createdTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTime").build()).build();

    private static final SdkField<String> ISSUER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Issuer")
            .getter(getter(ActorSession::issuer)).setter(setter(Builder::issuer))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Issuer").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(UID_FIELD, MFA_STATUS_FIELD,
            CREATED_TIME_FIELD, ISSUER_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String uid;

    private final String mfaStatus;

    private final Long createdTime;

    private final String issuer;

    private ActorSession(BuilderImpl builder) {
        this.uid = builder.uid;
        this.mfaStatus = builder.mfaStatus;
        this.createdTime = builder.createdTime;
        this.issuer = builder.issuer;
    }

    /**
     * <p>
     * Unique identifier of the session.
     * </p>
     * 
     * @return Unique identifier of the session.
     */
    public final String uid() {
        return uid;
    }

    /**
     * <p>
     * Indicates whether multi-factor authentication (MFA) was used for authentication during the session.
     * </p>
     * <p>
     * In CloudTrail, you can find this value as <code>userIdentity.sessionContext.attributes.mfaAuthenticated</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mfaStatus} will
     * return {@link ActorSessionMfaStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #mfaStatusAsString}.
     * </p>
     * 
     * @return Indicates whether multi-factor authentication (MFA) was used for authentication during the session.</p>
     *         <p>
     *         In CloudTrail, you can find this value as
     *         <code>userIdentity.sessionContext.attributes.mfaAuthenticated</code>.
     * @see ActorSessionMfaStatus
     */
    public final ActorSessionMfaStatus mfaStatus() {
        return ActorSessionMfaStatus.fromValue(mfaStatus);
    }

    /**
     * <p>
     * Indicates whether multi-factor authentication (MFA) was used for authentication during the session.
     * </p>
     * <p>
     * In CloudTrail, you can find this value as <code>userIdentity.sessionContext.attributes.mfaAuthenticated</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mfaStatus} will
     * return {@link ActorSessionMfaStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #mfaStatusAsString}.
     * </p>
     * 
     * @return Indicates whether multi-factor authentication (MFA) was used for authentication during the session.</p>
     *         <p>
     *         In CloudTrail, you can find this value as
     *         <code>userIdentity.sessionContext.attributes.mfaAuthenticated</code>.
     * @see ActorSessionMfaStatus
     */
    public final String mfaStatusAsString() {
        return mfaStatus;
    }

    /**
     * <p>
     * The timestamp for when the session was created.
     * </p>
     * <p>
     * In CloudTrail, you can find this value as <code>userIdentity.sessionContext.attributes.creationDate</code>.
     * </p>
     * 
     * @return The timestamp for when the session was created. </p>
     *         <p>
     *         In CloudTrail, you can find this value as
     *         <code>userIdentity.sessionContext.attributes.creationDate</code>.
     */
    public final Long createdTime() {
        return createdTime;
    }

    /**
     * <p>
     * The issuer of the session.
     * </p>
     * <p>
     * In CloudTrail, you can find this value as <code>userIdentity.sessionContext.sessionIssuer.arn</code>.
     * </p>
     * 
     * @return The issuer of the session. </p>
     *         <p>
     *         In CloudTrail, you can find this value as <code>userIdentity.sessionContext.sessionIssuer.arn</code>.
     */
    public final String issuer() {
        return issuer;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(uid());
        hashCode = 31 * hashCode + Objects.hashCode(mfaStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createdTime());
        hashCode = 31 * hashCode + Objects.hashCode(issuer());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ActorSession)) {
            return false;
        }
        ActorSession other = (ActorSession) obj;
        return Objects.equals(uid(), other.uid()) && Objects.equals(mfaStatusAsString(), other.mfaStatusAsString())
                && Objects.equals(createdTime(), other.createdTime()) && Objects.equals(issuer(), other.issuer());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ActorSession").add("Uid", uid()).add("MfaStatus", mfaStatusAsString())
                .add("CreatedTime", createdTime()).add("Issuer", issuer()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Uid":
            return Optional.ofNullable(clazz.cast(uid()));
        case "MfaStatus":
            return Optional.ofNullable(clazz.cast(mfaStatusAsString()));
        case "CreatedTime":
            return Optional.ofNullable(clazz.cast(createdTime()));
        case "Issuer":
            return Optional.ofNullable(clazz.cast(issuer()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Uid", UID_FIELD);
        map.put("MfaStatus", MFA_STATUS_FIELD);
        map.put("CreatedTime", CREATED_TIME_FIELD);
        map.put("Issuer", ISSUER_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ActorSession, T> g) {
        return obj -> g.apply((ActorSession) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ActorSession> {
        /**
         * <p>
         * Unique identifier of the session.
         * </p>
         * 
         * @param uid
         *        Unique identifier of the session.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder uid(String uid);

        /**
         * <p>
         * Indicates whether multi-factor authentication (MFA) was used for authentication during the session.
         * </p>
         * <p>
         * In CloudTrail, you can find this value as
         * <code>userIdentity.sessionContext.attributes.mfaAuthenticated</code>.
         * </p>
         * 
         * @param mfaStatus
         *        Indicates whether multi-factor authentication (MFA) was used for authentication during the
         *        session.</p>
         *        <p>
         *        In CloudTrail, you can find this value as
         *        <code>userIdentity.sessionContext.attributes.mfaAuthenticated</code>.
         * @see ActorSessionMfaStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ActorSessionMfaStatus
         */
        Builder mfaStatus(String mfaStatus);

        /**
         * <p>
         * Indicates whether multi-factor authentication (MFA) was used for authentication during the session.
         * </p>
         * <p>
         * In CloudTrail, you can find this value as
         * <code>userIdentity.sessionContext.attributes.mfaAuthenticated</code>.
         * </p>
         * 
         * @param mfaStatus
         *        Indicates whether multi-factor authentication (MFA) was used for authentication during the
         *        session.</p>
         *        <p>
         *        In CloudTrail, you can find this value as
         *        <code>userIdentity.sessionContext.attributes.mfaAuthenticated</code>.
         * @see ActorSessionMfaStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ActorSessionMfaStatus
         */
        Builder mfaStatus(ActorSessionMfaStatus mfaStatus);

        /**
         * <p>
         * The timestamp for when the session was created.
         * </p>
         * <p>
         * In CloudTrail, you can find this value as <code>userIdentity.sessionContext.attributes.creationDate</code>.
         * </p>
         * 
         * @param createdTime
         *        The timestamp for when the session was created. </p>
         *        <p>
         *        In CloudTrail, you can find this value as
         *        <code>userIdentity.sessionContext.attributes.creationDate</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTime(Long createdTime);

        /**
         * <p>
         * The issuer of the session.
         * </p>
         * <p>
         * In CloudTrail, you can find this value as <code>userIdentity.sessionContext.sessionIssuer.arn</code>.
         * </p>
         * 
         * @param issuer
         *        The issuer of the session. </p>
         *        <p>
         *        In CloudTrail, you can find this value as <code>userIdentity.sessionContext.sessionIssuer.arn</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder issuer(String issuer);
    }

    static final class BuilderImpl implements Builder {
        private String uid;

        private String mfaStatus;

        private Long createdTime;

        private String issuer;

        private BuilderImpl() {
        }

        private BuilderImpl(ActorSession model) {
            uid(model.uid);
            mfaStatus(model.mfaStatus);
            createdTime(model.createdTime);
            issuer(model.issuer);
        }

        public final String getUid() {
            return uid;
        }

        public final void setUid(String uid) {
            this.uid = uid;
        }

        @Override
        public final Builder uid(String uid) {
            this.uid = uid;
            return this;
        }

        public final String getMfaStatus() {
            return mfaStatus;
        }

        public final void setMfaStatus(String mfaStatus) {
            this.mfaStatus = mfaStatus;
        }

        @Override
        public final Builder mfaStatus(String mfaStatus) {
            this.mfaStatus = mfaStatus;
            return this;
        }

        @Override
        public final Builder mfaStatus(ActorSessionMfaStatus mfaStatus) {
            this.mfaStatus(mfaStatus == null ? null : mfaStatus.toString());
            return this;
        }

        public final Long getCreatedTime() {
            return createdTime;
        }

        public final void setCreatedTime(Long createdTime) {
            this.createdTime = createdTime;
        }

        @Override
        public final Builder createdTime(Long createdTime) {
            this.createdTime = createdTime;
            return this;
        }

        public final String getIssuer() {
            return issuer;
        }

        public final void setIssuer(String issuer) {
            this.issuer = issuer;
        }

        @Override
        public final Builder issuer(String issuer) {
            this.issuer = issuer;
            return this;
        }

        @Override
        public ActorSession build() {
            return new ActorSession(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
