/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.secretsmanager.model;

import java.io.Serializable;
import java.nio.ByteBuffer;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A structure that contains the secret value and other details for a secret.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SecretValueEntry implements SdkPojo, Serializable,
        ToCopyableBuilder<SecretValueEntry.Builder, SecretValueEntry> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("ARN")
            .getter(getter(SecretValueEntry::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ARN").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(SecretValueEntry::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> VERSION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VersionId").getter(getter(SecretValueEntry::versionId)).setter(setter(Builder::versionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VersionId").build()).build();

    private static final SdkField<SdkBytes> SECRET_BINARY_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("SecretBinary").getter(getter(SecretValueEntry::secretBinary)).setter(setter(Builder::secretBinary))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretBinary").build()).build();

    private static final SdkField<String> SECRET_STRING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecretString").getter(getter(SecretValueEntry::secretString)).setter(setter(Builder::secretString))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretString").build()).build();

    private static final SdkField<List<String>> VERSION_STAGES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("VersionStages")
            .getter(getter(SecretValueEntry::versionStages))
            .setter(setter(Builder::versionStages))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VersionStages").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> CREATED_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedDate").getter(getter(SecretValueEntry::createdDate)).setter(setter(Builder::createdDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedDate").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD, NAME_FIELD,
            VERSION_ID_FIELD, SECRET_BINARY_FIELD, SECRET_STRING_FIELD, VERSION_STAGES_FIELD, CREATED_DATE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final String name;

    private final String versionId;

    private final SdkBytes secretBinary;

    private final String secretString;

    private final List<String> versionStages;

    private final Instant createdDate;

    private SecretValueEntry(BuilderImpl builder) {
        this.arn = builder.arn;
        this.name = builder.name;
        this.versionId = builder.versionId;
        this.secretBinary = builder.secretBinary;
        this.secretString = builder.secretString;
        this.versionStages = builder.versionStages;
        this.createdDate = builder.createdDate;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the secret.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the secret.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The friendly name of the secret.
     * </p>
     * 
     * @return The friendly name of the secret.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The unique version identifier of this version of the secret.
     * </p>
     * 
     * @return The unique version identifier of this version of the secret.
     */
    public final String versionId() {
        return versionId;
    }

    /**
     * <p>
     * The decrypted secret value, if the secret value was originally provided as binary data in the form of a byte
     * array. The parameter represents the binary data as a <a
     * href="https://tools.ietf.org/html/rfc4648#section-4">base64-encoded</a> string.
     * </p>
     * 
     * @return The decrypted secret value, if the secret value was originally provided as binary data in the form of a
     *         byte array. The parameter represents the binary data as a <a
     *         href="https://tools.ietf.org/html/rfc4648#section-4">base64-encoded</a> string.
     */
    public final SdkBytes secretBinary() {
        return secretBinary;
    }

    /**
     * <p>
     * The decrypted secret value, if the secret value was originally provided as a string or through the Secrets
     * Manager console.
     * </p>
     * 
     * @return The decrypted secret value, if the secret value was originally provided as a string or through the
     *         Secrets Manager console.
     */
    public final String secretString() {
        return secretString;
    }

    /**
     * For responses, this returns true if the service returned a value for the VersionStages property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasVersionStages() {
        return versionStages != null && !(versionStages instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of all of the staging labels currently attached to this version of the secret.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVersionStages} method.
     * </p>
     * 
     * @return A list of all of the staging labels currently attached to this version of the secret.
     */
    public final List<String> versionStages() {
        return versionStages;
    }

    /**
     * <p>
     * The date the secret was created.
     * </p>
     * 
     * @return The date the secret was created.
     */
    public final Instant createdDate() {
        return createdDate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(versionId());
        hashCode = 31 * hashCode + Objects.hashCode(secretBinary());
        hashCode = 31 * hashCode + Objects.hashCode(secretString());
        hashCode = 31 * hashCode + Objects.hashCode(hasVersionStages() ? versionStages() : null);
        hashCode = 31 * hashCode + Objects.hashCode(createdDate());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SecretValueEntry)) {
            return false;
        }
        SecretValueEntry other = (SecretValueEntry) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(name(), other.name())
                && Objects.equals(versionId(), other.versionId()) && Objects.equals(secretBinary(), other.secretBinary())
                && Objects.equals(secretString(), other.secretString()) && hasVersionStages() == other.hasVersionStages()
                && Objects.equals(versionStages(), other.versionStages()) && Objects.equals(createdDate(), other.createdDate());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SecretValueEntry").add("ARN", arn()).add("Name", name()).add("VersionId", versionId())
                .add("SecretBinary", secretBinary() == null ? null : "*** Sensitive Data Redacted ***")
                .add("SecretString", secretString() == null ? null : "*** Sensitive Data Redacted ***")
                .add("VersionStages", hasVersionStages() ? versionStages() : null).add("CreatedDate", createdDate()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ARN":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "VersionId":
            return Optional.ofNullable(clazz.cast(versionId()));
        case "SecretBinary":
            return Optional.ofNullable(clazz.cast(secretBinary()));
        case "SecretString":
            return Optional.ofNullable(clazz.cast(secretString()));
        case "VersionStages":
            return Optional.ofNullable(clazz.cast(versionStages()));
        case "CreatedDate":
            return Optional.ofNullable(clazz.cast(createdDate()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ARN", ARN_FIELD);
        map.put("Name", NAME_FIELD);
        map.put("VersionId", VERSION_ID_FIELD);
        map.put("SecretBinary", SECRET_BINARY_FIELD);
        map.put("SecretString", SECRET_STRING_FIELD);
        map.put("VersionStages", VERSION_STAGES_FIELD);
        map.put("CreatedDate", CREATED_DATE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SecretValueEntry, T> g) {
        return obj -> g.apply((SecretValueEntry) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SecretValueEntry> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the secret.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the secret.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The friendly name of the secret.
         * </p>
         * 
         * @param name
         *        The friendly name of the secret.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The unique version identifier of this version of the secret.
         * </p>
         * 
         * @param versionId
         *        The unique version identifier of this version of the secret.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionId(String versionId);

        /**
         * <p>
         * The decrypted secret value, if the secret value was originally provided as binary data in the form of a byte
         * array. The parameter represents the binary data as a <a
         * href="https://tools.ietf.org/html/rfc4648#section-4">base64-encoded</a> string.
         * </p>
         * 
         * @param secretBinary
         *        The decrypted secret value, if the secret value was originally provided as binary data in the form of
         *        a byte array. The parameter represents the binary data as a <a
         *        href="https://tools.ietf.org/html/rfc4648#section-4">base64-encoded</a> string.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretBinary(SdkBytes secretBinary);

        /**
         * <p>
         * The decrypted secret value, if the secret value was originally provided as a string or through the Secrets
         * Manager console.
         * </p>
         * 
         * @param secretString
         *        The decrypted secret value, if the secret value was originally provided as a string or through the
         *        Secrets Manager console.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretString(String secretString);

        /**
         * <p>
         * A list of all of the staging labels currently attached to this version of the secret.
         * </p>
         * 
         * @param versionStages
         *        A list of all of the staging labels currently attached to this version of the secret.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionStages(Collection<String> versionStages);

        /**
         * <p>
         * A list of all of the staging labels currently attached to this version of the secret.
         * </p>
         * 
         * @param versionStages
         *        A list of all of the staging labels currently attached to this version of the secret.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionStages(String... versionStages);

        /**
         * <p>
         * The date the secret was created.
         * </p>
         * 
         * @param createdDate
         *        The date the secret was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdDate(Instant createdDate);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private String name;

        private String versionId;

        private SdkBytes secretBinary;

        private String secretString;

        private List<String> versionStages = DefaultSdkAutoConstructList.getInstance();

        private Instant createdDate;

        private BuilderImpl() {
        }

        private BuilderImpl(SecretValueEntry model) {
            arn(model.arn);
            name(model.name);
            versionId(model.versionId);
            secretBinary(model.secretBinary);
            secretString(model.secretString);
            versionStages(model.versionStages);
            createdDate(model.createdDate);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getVersionId() {
            return versionId;
        }

        public final void setVersionId(String versionId) {
            this.versionId = versionId;
        }

        @Override
        public final Builder versionId(String versionId) {
            this.versionId = versionId;
            return this;
        }

        public final ByteBuffer getSecretBinary() {
            return secretBinary == null ? null : secretBinary.asByteBuffer();
        }

        public final void setSecretBinary(ByteBuffer secretBinary) {
            secretBinary(secretBinary == null ? null : SdkBytes.fromByteBuffer(secretBinary));
        }

        @Override
        public final Builder secretBinary(SdkBytes secretBinary) {
            this.secretBinary = secretBinary;
            return this;
        }

        public final String getSecretString() {
            return secretString;
        }

        public final void setSecretString(String secretString) {
            this.secretString = secretString;
        }

        @Override
        public final Builder secretString(String secretString) {
            this.secretString = secretString;
            return this;
        }

        public final Collection<String> getVersionStages() {
            if (versionStages instanceof SdkAutoConstructList) {
                return null;
            }
            return versionStages;
        }

        public final void setVersionStages(Collection<String> versionStages) {
            this.versionStages = SecretVersionStagesTypeCopier.copy(versionStages);
        }

        @Override
        public final Builder versionStages(Collection<String> versionStages) {
            this.versionStages = SecretVersionStagesTypeCopier.copy(versionStages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder versionStages(String... versionStages) {
            versionStages(Arrays.asList(versionStages));
            return this;
        }

        public final Instant getCreatedDate() {
            return createdDate;
        }

        public final void setCreatedDate(Instant createdDate) {
            this.createdDate = createdDate;
        }

        @Override
        public final Builder createdDate(Instant createdDate) {
            this.createdDate = createdDate;
            return this;
        }

        @Override
        public SecretValueEntry build() {
            return new SecretValueEntry(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
