/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.scheduler;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.scheduler.model.ConflictException;
import software.amazon.awssdk.services.scheduler.model.CreateScheduleGroupRequest;
import software.amazon.awssdk.services.scheduler.model.CreateScheduleGroupResponse;
import software.amazon.awssdk.services.scheduler.model.CreateScheduleRequest;
import software.amazon.awssdk.services.scheduler.model.CreateScheduleResponse;
import software.amazon.awssdk.services.scheduler.model.DeleteScheduleGroupRequest;
import software.amazon.awssdk.services.scheduler.model.DeleteScheduleGroupResponse;
import software.amazon.awssdk.services.scheduler.model.DeleteScheduleRequest;
import software.amazon.awssdk.services.scheduler.model.DeleteScheduleResponse;
import software.amazon.awssdk.services.scheduler.model.GetScheduleGroupRequest;
import software.amazon.awssdk.services.scheduler.model.GetScheduleGroupResponse;
import software.amazon.awssdk.services.scheduler.model.GetScheduleRequest;
import software.amazon.awssdk.services.scheduler.model.GetScheduleResponse;
import software.amazon.awssdk.services.scheduler.model.InternalServerException;
import software.amazon.awssdk.services.scheduler.model.ListScheduleGroupsRequest;
import software.amazon.awssdk.services.scheduler.model.ListScheduleGroupsResponse;
import software.amazon.awssdk.services.scheduler.model.ListSchedulesRequest;
import software.amazon.awssdk.services.scheduler.model.ListSchedulesResponse;
import software.amazon.awssdk.services.scheduler.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.scheduler.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.scheduler.model.ResourceNotFoundException;
import software.amazon.awssdk.services.scheduler.model.SchedulerException;
import software.amazon.awssdk.services.scheduler.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.scheduler.model.TagResourceRequest;
import software.amazon.awssdk.services.scheduler.model.TagResourceResponse;
import software.amazon.awssdk.services.scheduler.model.ThrottlingException;
import software.amazon.awssdk.services.scheduler.model.UntagResourceRequest;
import software.amazon.awssdk.services.scheduler.model.UntagResourceResponse;
import software.amazon.awssdk.services.scheduler.model.UpdateScheduleRequest;
import software.amazon.awssdk.services.scheduler.model.UpdateScheduleResponse;
import software.amazon.awssdk.services.scheduler.model.ValidationException;
import software.amazon.awssdk.services.scheduler.paginators.ListScheduleGroupsIterable;
import software.amazon.awssdk.services.scheduler.paginators.ListSchedulesIterable;

/**
 * Service client for accessing Amazon EventBridge Scheduler. This can be created using the static {@link #builder()}
 * method.
 *
 * <p>
 * Amazon EventBridge Scheduler is a serverless scheduler that allows you to create, run, and manage tasks from one
 * central, managed service. EventBridge Scheduler delivers your tasks reliably, with built-in mechanisms that adjust
 * your schedules based on the availability of downstream targets. The following reference lists the available API
 * actions, and data types for EventBridge Scheduler.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface SchedulerClient extends AwsClient {
    String SERVICE_NAME = "scheduler";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "scheduler";

    /**
     * <p>
     * Creates the specified schedule.
     * </p>
     *
     * @param createScheduleRequest
     * @return Result of the CreateSchedule operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeds a service quota.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.CreateSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/CreateSchedule" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateScheduleResponse createSchedule(CreateScheduleRequest createScheduleRequest)
            throws ServiceQuotaExceededException, ValidationException, InternalServerException, ConflictException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, SchedulerException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates the specified schedule.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateScheduleRequest.Builder} avoiding the need to
     * create one manually via {@link CreateScheduleRequest#builder()}
     * </p>
     *
     * @param createScheduleRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.scheduler.model.CreateScheduleRequest.Builder} to create a request.
     * @return Result of the CreateSchedule operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeds a service quota.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.CreateSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/CreateSchedule" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateScheduleResponse createSchedule(Consumer<CreateScheduleRequest.Builder> createScheduleRequest)
            throws ServiceQuotaExceededException, ValidationException, InternalServerException, ConflictException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, SchedulerException {
        return createSchedule(CreateScheduleRequest.builder().applyMutation(createScheduleRequest).build());
    }

    /**
     * <p>
     * Creates the specified schedule group.
     * </p>
     *
     * @param createScheduleGroupRequest
     * @return Result of the CreateScheduleGroup operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeds a service quota.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.CreateScheduleGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/CreateScheduleGroup" target="_top">AWS
     *      API Documentation</a>
     */
    default CreateScheduleGroupResponse createScheduleGroup(CreateScheduleGroupRequest createScheduleGroupRequest)
            throws ServiceQuotaExceededException, ValidationException, InternalServerException, ConflictException,
            ThrottlingException, AwsServiceException, SdkClientException, SchedulerException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates the specified schedule group.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateScheduleGroupRequest.Builder} avoiding the
     * need to create one manually via {@link CreateScheduleGroupRequest#builder()}
     * </p>
     *
     * @param createScheduleGroupRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.scheduler.model.CreateScheduleGroupRequest.Builder} to create a
     *        request.
     * @return Result of the CreateScheduleGroup operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeds a service quota.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.CreateScheduleGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/CreateScheduleGroup" target="_top">AWS
     *      API Documentation</a>
     */
    default CreateScheduleGroupResponse createScheduleGroup(
            Consumer<CreateScheduleGroupRequest.Builder> createScheduleGroupRequest) throws ServiceQuotaExceededException,
            ValidationException, InternalServerException, ConflictException, ThrottlingException, AwsServiceException,
            SdkClientException, SchedulerException {
        return createScheduleGroup(CreateScheduleGroupRequest.builder().applyMutation(createScheduleGroupRequest).build());
    }

    /**
     * <p>
     * Deletes the specified schedule.
     * </p>
     *
     * @param deleteScheduleRequest
     * @return Result of the DeleteSchedule operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.DeleteSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/DeleteSchedule" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteScheduleResponse deleteSchedule(DeleteScheduleRequest deleteScheduleRequest) throws ValidationException,
            InternalServerException, ConflictException, ResourceNotFoundException, ThrottlingException, AwsServiceException,
            SdkClientException, SchedulerException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the specified schedule.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteScheduleRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteScheduleRequest#builder()}
     * </p>
     *
     * @param deleteScheduleRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.scheduler.model.DeleteScheduleRequest.Builder} to create a request.
     * @return Result of the DeleteSchedule operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.DeleteSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/DeleteSchedule" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteScheduleResponse deleteSchedule(Consumer<DeleteScheduleRequest.Builder> deleteScheduleRequest)
            throws ValidationException, InternalServerException, ConflictException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, SchedulerException {
        return deleteSchedule(DeleteScheduleRequest.builder().applyMutation(deleteScheduleRequest).build());
    }

    /**
     * <p>
     * Deletes the specified schedule group. Deleting a schedule group results in EventBridge Scheduler deleting all
     * schedules associated with the group. When you delete a group, it remains in a <code>DELETING</code> state until
     * all of its associated schedules are deleted. Schedules associated with the group that are set to run while the
     * schedule group is in the process of being deleted might continue to invoke their targets until the schedule group
     * and its associated schedules are deleted.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent.
     * </p>
     * </note>
     *
     * @param deleteScheduleGroupRequest
     * @return Result of the DeleteScheduleGroup operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.DeleteScheduleGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/DeleteScheduleGroup" target="_top">AWS
     *      API Documentation</a>
     */
    default DeleteScheduleGroupResponse deleteScheduleGroup(DeleteScheduleGroupRequest deleteScheduleGroupRequest)
            throws ValidationException, InternalServerException, ConflictException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, SchedulerException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the specified schedule group. Deleting a schedule group results in EventBridge Scheduler deleting all
     * schedules associated with the group. When you delete a group, it remains in a <code>DELETING</code> state until
     * all of its associated schedules are deleted. Schedules associated with the group that are set to run while the
     * schedule group is in the process of being deleted might continue to invoke their targets until the schedule group
     * and its associated schedules are deleted.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteScheduleGroupRequest.Builder} avoiding the
     * need to create one manually via {@link DeleteScheduleGroupRequest#builder()}
     * </p>
     *
     * @param deleteScheduleGroupRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.scheduler.model.DeleteScheduleGroupRequest.Builder} to create a
     *        request.
     * @return Result of the DeleteScheduleGroup operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.DeleteScheduleGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/DeleteScheduleGroup" target="_top">AWS
     *      API Documentation</a>
     */
    default DeleteScheduleGroupResponse deleteScheduleGroup(
            Consumer<DeleteScheduleGroupRequest.Builder> deleteScheduleGroupRequest) throws ValidationException,
            InternalServerException, ConflictException, ResourceNotFoundException, ThrottlingException, AwsServiceException,
            SdkClientException, SchedulerException {
        return deleteScheduleGroup(DeleteScheduleGroupRequest.builder().applyMutation(deleteScheduleGroupRequest).build());
    }

    /**
     * <p>
     * Retrieves the specified schedule.
     * </p>
     *
     * @param getScheduleRequest
     * @return Result of the GetSchedule operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.GetSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/GetSchedule" target="_top">AWS API
     *      Documentation</a>
     */
    default GetScheduleResponse getSchedule(GetScheduleRequest getScheduleRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException,
            SchedulerException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the specified schedule.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetScheduleRequest.Builder} avoiding the need to
     * create one manually via {@link GetScheduleRequest#builder()}
     * </p>
     *
     * @param getScheduleRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.scheduler.model.GetScheduleRequest.Builder} to create a request.
     * @return Result of the GetSchedule operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.GetSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/GetSchedule" target="_top">AWS API
     *      Documentation</a>
     */
    default GetScheduleResponse getSchedule(Consumer<GetScheduleRequest.Builder> getScheduleRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException,
            SchedulerException {
        return getSchedule(GetScheduleRequest.builder().applyMutation(getScheduleRequest).build());
    }

    /**
     * <p>
     * Retrieves the specified schedule group.
     * </p>
     *
     * @param getScheduleGroupRequest
     * @return Result of the GetScheduleGroup operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.GetScheduleGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/GetScheduleGroup" target="_top">AWS
     *      API Documentation</a>
     */
    default GetScheduleGroupResponse getScheduleGroup(GetScheduleGroupRequest getScheduleGroupRequest)
            throws ValidationException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, SchedulerException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the specified schedule group.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetScheduleGroupRequest.Builder} avoiding the need
     * to create one manually via {@link GetScheduleGroupRequest#builder()}
     * </p>
     *
     * @param getScheduleGroupRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.scheduler.model.GetScheduleGroupRequest.Builder} to create a
     *        request.
     * @return Result of the GetScheduleGroup operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.GetScheduleGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/GetScheduleGroup" target="_top">AWS
     *      API Documentation</a>
     */
    default GetScheduleGroupResponse getScheduleGroup(Consumer<GetScheduleGroupRequest.Builder> getScheduleGroupRequest)
            throws ValidationException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, SchedulerException {
        return getScheduleGroup(GetScheduleGroupRequest.builder().applyMutation(getScheduleGroupRequest).build());
    }

    /**
     * <p>
     * Returns a paginated list of your schedule groups.
     * </p>
     *
     * @param listScheduleGroupsRequest
     * @return Result of the ListScheduleGroups operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.ListScheduleGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/ListScheduleGroups" target="_top">AWS
     *      API Documentation</a>
     */
    default ListScheduleGroupsResponse listScheduleGroups(ListScheduleGroupsRequest listScheduleGroupsRequest)
            throws ValidationException, InternalServerException, ThrottlingException, AwsServiceException, SdkClientException,
            SchedulerException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a paginated list of your schedule groups.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListScheduleGroupsRequest.Builder} avoiding the
     * need to create one manually via {@link ListScheduleGroupsRequest#builder()}
     * </p>
     *
     * @param listScheduleGroupsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.scheduler.model.ListScheduleGroupsRequest.Builder} to create a
     *        request.
     * @return Result of the ListScheduleGroups operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.ListScheduleGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/ListScheduleGroups" target="_top">AWS
     *      API Documentation</a>
     */
    default ListScheduleGroupsResponse listScheduleGroups(Consumer<ListScheduleGroupsRequest.Builder> listScheduleGroupsRequest)
            throws ValidationException, InternalServerException, ThrottlingException, AwsServiceException, SdkClientException,
            SchedulerException {
        return listScheduleGroups(ListScheduleGroupsRequest.builder().applyMutation(listScheduleGroupsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listScheduleGroups(software.amazon.awssdk.services.scheduler.model.ListScheduleGroupsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.scheduler.paginators.ListScheduleGroupsIterable responses = client.listScheduleGroupsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.scheduler.paginators.ListScheduleGroupsIterable responses = client
     *             .listScheduleGroupsPaginator(request);
     *     for (software.amazon.awssdk.services.scheduler.model.ListScheduleGroupsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.scheduler.paginators.ListScheduleGroupsIterable responses = client.listScheduleGroupsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listScheduleGroups(software.amazon.awssdk.services.scheduler.model.ListScheduleGroupsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listScheduleGroupsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.ListScheduleGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/ListScheduleGroups" target="_top">AWS
     *      API Documentation</a>
     */
    default ListScheduleGroupsIterable listScheduleGroupsPaginator(ListScheduleGroupsRequest listScheduleGroupsRequest)
            throws ValidationException, InternalServerException, ThrottlingException, AwsServiceException, SdkClientException,
            SchedulerException {
        return new ListScheduleGroupsIterable(this, listScheduleGroupsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listScheduleGroups(software.amazon.awssdk.services.scheduler.model.ListScheduleGroupsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.scheduler.paginators.ListScheduleGroupsIterable responses = client.listScheduleGroupsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.scheduler.paginators.ListScheduleGroupsIterable responses = client
     *             .listScheduleGroupsPaginator(request);
     *     for (software.amazon.awssdk.services.scheduler.model.ListScheduleGroupsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.scheduler.paginators.ListScheduleGroupsIterable responses = client.listScheduleGroupsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listScheduleGroups(software.amazon.awssdk.services.scheduler.model.ListScheduleGroupsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListScheduleGroupsRequest.Builder} avoiding the
     * need to create one manually via {@link ListScheduleGroupsRequest#builder()}
     * </p>
     *
     * @param listScheduleGroupsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.scheduler.model.ListScheduleGroupsRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.ListScheduleGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/ListScheduleGroups" target="_top">AWS
     *      API Documentation</a>
     */
    default ListScheduleGroupsIterable listScheduleGroupsPaginator(
            Consumer<ListScheduleGroupsRequest.Builder> listScheduleGroupsRequest) throws ValidationException,
            InternalServerException, ThrottlingException, AwsServiceException, SdkClientException, SchedulerException {
        return listScheduleGroupsPaginator(ListScheduleGroupsRequest.builder().applyMutation(listScheduleGroupsRequest).build());
    }

    /**
     * <p>
     * Returns a paginated list of your EventBridge Scheduler schedules.
     * </p>
     *
     * @param listSchedulesRequest
     * @return Result of the ListSchedules operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.ListSchedules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/ListSchedules" target="_top">AWS API
     *      Documentation</a>
     */
    default ListSchedulesResponse listSchedules(ListSchedulesRequest listSchedulesRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException,
            SchedulerException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a paginated list of your EventBridge Scheduler schedules.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSchedulesRequest.Builder} avoiding the need to
     * create one manually via {@link ListSchedulesRequest#builder()}
     * </p>
     *
     * @param listSchedulesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.scheduler.model.ListSchedulesRequest.Builder} to create a request.
     * @return Result of the ListSchedules operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.ListSchedules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/ListSchedules" target="_top">AWS API
     *      Documentation</a>
     */
    default ListSchedulesResponse listSchedules(Consumer<ListSchedulesRequest.Builder> listSchedulesRequest)
            throws ValidationException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, SchedulerException {
        return listSchedules(ListSchedulesRequest.builder().applyMutation(listSchedulesRequest).build());
    }

    /**
     * <p>
     * This is a variant of {@link #listSchedules(software.amazon.awssdk.services.scheduler.model.ListSchedulesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.scheduler.paginators.ListSchedulesIterable responses = client.listSchedulesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.scheduler.paginators.ListSchedulesIterable responses = client.listSchedulesPaginator(request);
     *     for (software.amazon.awssdk.services.scheduler.model.ListSchedulesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.scheduler.paginators.ListSchedulesIterable responses = client.listSchedulesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSchedules(software.amazon.awssdk.services.scheduler.model.ListSchedulesRequest)} operation.</b>
     * </p>
     *
     * @param listSchedulesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.ListSchedules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/ListSchedules" target="_top">AWS API
     *      Documentation</a>
     */
    default ListSchedulesIterable listSchedulesPaginator(ListSchedulesRequest listSchedulesRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException,
            SchedulerException {
        return new ListSchedulesIterable(this, listSchedulesRequest);
    }

    /**
     * <p>
     * This is a variant of {@link #listSchedules(software.amazon.awssdk.services.scheduler.model.ListSchedulesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.scheduler.paginators.ListSchedulesIterable responses = client.listSchedulesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.scheduler.paginators.ListSchedulesIterable responses = client.listSchedulesPaginator(request);
     *     for (software.amazon.awssdk.services.scheduler.model.ListSchedulesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.scheduler.paginators.ListSchedulesIterable responses = client.listSchedulesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSchedules(software.amazon.awssdk.services.scheduler.model.ListSchedulesRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSchedulesRequest.Builder} avoiding the need to
     * create one manually via {@link ListSchedulesRequest#builder()}
     * </p>
     *
     * @param listSchedulesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.scheduler.model.ListSchedulesRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.ListSchedules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/ListSchedules" target="_top">AWS API
     *      Documentation</a>
     */
    default ListSchedulesIterable listSchedulesPaginator(Consumer<ListSchedulesRequest.Builder> listSchedulesRequest)
            throws ValidationException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, SchedulerException {
        return listSchedulesPaginator(ListSchedulesRequest.builder().applyMutation(listSchedulesRequest).build());
    }

    /**
     * <p>
     * Lists the tags associated with the Scheduler resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, SchedulerException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the tags associated with the Scheduler resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.scheduler.model.ListTagsForResourceRequest.Builder} to create a
     *        request.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException,
            SchedulerException {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified EventBridge Scheduler resource. You can only assign
     * tags to schedule groups.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ValidationException,
            InternalServerException, ConflictException, ResourceNotFoundException, ThrottlingException, AwsServiceException,
            SdkClientException, SchedulerException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified EventBridge Scheduler resource. You can only assign
     * tags to schedule groups.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.scheduler.model.TagResourceRequest.Builder} to create a request.
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest) throws ValidationException,
            InternalServerException, ConflictException, ResourceNotFoundException, ThrottlingException, AwsServiceException,
            SdkClientException, SchedulerException {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes one or more tags from the specified EventBridge Scheduler schedule group.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            InternalServerException, ConflictException, ResourceNotFoundException, ThrottlingException, AwsServiceException,
            SdkClientException, SchedulerException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes one or more tags from the specified EventBridge Scheduler schedule group.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.scheduler.model.UntagResourceRequest.Builder} to create a request.
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest)
            throws ValidationException, InternalServerException, ConflictException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, SchedulerException {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Updates the specified schedule. When you call <code>UpdateSchedule</code>, EventBridge Scheduler uses all values,
     * including empty values, specified in the request and overrides the existing schedule. This is by design. This
     * means that if you do not set an optional field in your request, that field will be set to its system-default
     * value after the update.
     * </p>
     * <p>
     * Before calling this operation, we recommend that you call the <code>GetSchedule</code> API operation and make a
     * note of all optional parameters for your <code>UpdateSchedule</code> call.
     * </p>
     *
     * @param updateScheduleRequest
     * @return Result of the UpdateSchedule operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.UpdateSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/UpdateSchedule" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateScheduleResponse updateSchedule(UpdateScheduleRequest updateScheduleRequest) throws ValidationException,
            InternalServerException, ConflictException, ResourceNotFoundException, ThrottlingException, AwsServiceException,
            SdkClientException, SchedulerException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates the specified schedule. When you call <code>UpdateSchedule</code>, EventBridge Scheduler uses all values,
     * including empty values, specified in the request and overrides the existing schedule. This is by design. This
     * means that if you do not set an optional field in your request, that field will be set to its system-default
     * value after the update.
     * </p>
     * <p>
     * Before calling this operation, we recommend that you call the <code>GetSchedule</code> API operation and make a
     * note of all optional parameters for your <code>UpdateSchedule</code> call.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateScheduleRequest.Builder} avoiding the need to
     * create one manually via {@link UpdateScheduleRequest#builder()}
     * </p>
     *
     * @param updateScheduleRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.scheduler.model.UpdateScheduleRequest.Builder} to create a request.
     * @return Result of the UpdateSchedule operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error encountered while processing the request.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ResourceNotFoundException
     *         The request references a resource which does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SchedulerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SchedulerClient.UpdateSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/scheduler-2021-06-30/UpdateSchedule" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateScheduleResponse updateSchedule(Consumer<UpdateScheduleRequest.Builder> updateScheduleRequest)
            throws ValidationException, InternalServerException, ConflictException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, SchedulerException {
        return updateSchedule(UpdateScheduleRequest.builder().applyMutation(updateScheduleRequest).build());
    }

    /**
     * Create a {@link SchedulerClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static SchedulerClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link SchedulerClient}.
     */
    static SchedulerClientBuilder builder() {
        return new DefaultSchedulerClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default SchedulerServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
