/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3vectors;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.s3vectors.internal.S3VectorsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.s3vectors.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.s3vectors.model.AccessDeniedException;
import software.amazon.awssdk.services.s3vectors.model.ConflictException;
import software.amazon.awssdk.services.s3vectors.model.CreateIndexRequest;
import software.amazon.awssdk.services.s3vectors.model.CreateIndexResponse;
import software.amazon.awssdk.services.s3vectors.model.CreateVectorBucketRequest;
import software.amazon.awssdk.services.s3vectors.model.CreateVectorBucketResponse;
import software.amazon.awssdk.services.s3vectors.model.DeleteIndexRequest;
import software.amazon.awssdk.services.s3vectors.model.DeleteIndexResponse;
import software.amazon.awssdk.services.s3vectors.model.DeleteVectorBucketPolicyRequest;
import software.amazon.awssdk.services.s3vectors.model.DeleteVectorBucketPolicyResponse;
import software.amazon.awssdk.services.s3vectors.model.DeleteVectorBucketRequest;
import software.amazon.awssdk.services.s3vectors.model.DeleteVectorBucketResponse;
import software.amazon.awssdk.services.s3vectors.model.DeleteVectorsRequest;
import software.amazon.awssdk.services.s3vectors.model.DeleteVectorsResponse;
import software.amazon.awssdk.services.s3vectors.model.GetIndexRequest;
import software.amazon.awssdk.services.s3vectors.model.GetIndexResponse;
import software.amazon.awssdk.services.s3vectors.model.GetVectorBucketPolicyRequest;
import software.amazon.awssdk.services.s3vectors.model.GetVectorBucketPolicyResponse;
import software.amazon.awssdk.services.s3vectors.model.GetVectorBucketRequest;
import software.amazon.awssdk.services.s3vectors.model.GetVectorBucketResponse;
import software.amazon.awssdk.services.s3vectors.model.GetVectorsRequest;
import software.amazon.awssdk.services.s3vectors.model.GetVectorsResponse;
import software.amazon.awssdk.services.s3vectors.model.InternalServerException;
import software.amazon.awssdk.services.s3vectors.model.KmsDisabledException;
import software.amazon.awssdk.services.s3vectors.model.KmsInvalidKeyUsageException;
import software.amazon.awssdk.services.s3vectors.model.KmsInvalidStateException;
import software.amazon.awssdk.services.s3vectors.model.KmsNotFoundException;
import software.amazon.awssdk.services.s3vectors.model.ListIndexesRequest;
import software.amazon.awssdk.services.s3vectors.model.ListIndexesResponse;
import software.amazon.awssdk.services.s3vectors.model.ListVectorBucketsRequest;
import software.amazon.awssdk.services.s3vectors.model.ListVectorBucketsResponse;
import software.amazon.awssdk.services.s3vectors.model.ListVectorsRequest;
import software.amazon.awssdk.services.s3vectors.model.ListVectorsResponse;
import software.amazon.awssdk.services.s3vectors.model.NotFoundException;
import software.amazon.awssdk.services.s3vectors.model.PutVectorBucketPolicyRequest;
import software.amazon.awssdk.services.s3vectors.model.PutVectorBucketPolicyResponse;
import software.amazon.awssdk.services.s3vectors.model.PutVectorsRequest;
import software.amazon.awssdk.services.s3vectors.model.PutVectorsResponse;
import software.amazon.awssdk.services.s3vectors.model.QueryVectorsRequest;
import software.amazon.awssdk.services.s3vectors.model.QueryVectorsResponse;
import software.amazon.awssdk.services.s3vectors.model.S3VectorsException;
import software.amazon.awssdk.services.s3vectors.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.s3vectors.model.ServiceUnavailableException;
import software.amazon.awssdk.services.s3vectors.model.TooManyRequestsException;
import software.amazon.awssdk.services.s3vectors.model.ValidationException;
import software.amazon.awssdk.services.s3vectors.transform.CreateIndexRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.CreateVectorBucketRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.DeleteIndexRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.DeleteVectorBucketPolicyRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.DeleteVectorBucketRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.DeleteVectorsRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.GetIndexRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.GetVectorBucketPolicyRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.GetVectorBucketRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.GetVectorsRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.ListIndexesRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.ListVectorBucketsRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.ListVectorsRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.PutVectorBucketPolicyRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.PutVectorsRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.QueryVectorsRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link S3VectorsClient}.
 *
 * @see S3VectorsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultS3VectorsClient implements S3VectorsClient {
    private static final Logger log = Logger.loggerFor(DefaultS3VectorsClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultS3VectorsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "S3Vectors" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Creates a vector index within a vector bucket. To specify the vector bucket, you must use either the vector
     * bucket name or the vector bucket Amazon Resource Name (ARN).
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:CreateIndex</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param createIndexRequest
     * @return Result of the CreateIndex operation returned by the service.
     * @throws ValidationException
     *         The requested action isn't valid.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Wait briefly and retry your request. If it continues to fail, increase your
     *         waiting time between retries.
     * @throws TooManyRequestsException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request failed due to an internal server error.
     * @throws AccessDeniedException
     *         Access denied.
     * @throws ConflictException
     *         The request failed because a vector bucket name or a vector index name already exists. Vector bucket
     *         names must be unique within your Amazon Web Services account for each Amazon Web Services Region. Vector
     *         index names must be unique within your vector bucket. Choose a different vector bucket name or vector
     *         index name, and try again.
     * @throws NotFoundException
     *         The request was rejected because the specified resource can't be found.
     * @throws ServiceQuotaExceededException
     *         Your request exceeds a service quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3VectorsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3VectorsClient.CreateIndex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/CreateIndex" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateIndexResponse createIndex(CreateIndexRequest createIndexRequest) throws ValidationException,
            ServiceUnavailableException, TooManyRequestsException, InternalServerException, AccessDeniedException,
            ConflictException, NotFoundException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            S3VectorsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateIndexResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateIndexResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "KmsInvalidKeyUsageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
            case "KmsInvalidStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
            case "KmsNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "KmsDisabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsDisabledException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createIndexRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createIndexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateIndex");

            return clientHandler.execute(new ClientExecutionParams<CreateIndexRequest, CreateIndexResponse>()
                    .withOperationName("CreateIndex").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createIndexRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateIndexRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Creates a vector bucket in the Amazon Web Services Region that you want your bucket to be in.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:CreateVectorBucket</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param createVectorBucketRequest
     * @return Result of the CreateVectorBucket operation returned by the service.
     * @throws ValidationException
     *         The requested action isn't valid.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Wait briefly and retry your request. If it continues to fail, increase your
     *         waiting time between retries.
     * @throws TooManyRequestsException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request failed due to an internal server error.
     * @throws AccessDeniedException
     *         Access denied.
     * @throws ConflictException
     *         The request failed because a vector bucket name or a vector index name already exists. Vector bucket
     *         names must be unique within your Amazon Web Services account for each Amazon Web Services Region. Vector
     *         index names must be unique within your vector bucket. Choose a different vector bucket name or vector
     *         index name, and try again.
     * @throws ServiceQuotaExceededException
     *         Your request exceeds a service quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3VectorsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3VectorsClient.CreateVectorBucket
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/CreateVectorBucket" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateVectorBucketResponse createVectorBucket(CreateVectorBucketRequest createVectorBucketRequest)
            throws ValidationException, ServiceUnavailableException, TooManyRequestsException, InternalServerException,
            AccessDeniedException, ConflictException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            S3VectorsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateVectorBucketResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateVectorBucketResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "KmsInvalidKeyUsageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
            case "KmsInvalidStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
            case "KmsNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "KmsDisabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsDisabledException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createVectorBucketRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVectorBucketRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVectorBucket");

            return clientHandler.execute(new ClientExecutionParams<CreateVectorBucketRequest, CreateVectorBucketResponse>()
                    .withOperationName("CreateVectorBucket").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createVectorBucketRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateVectorBucketRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Deletes a vector index. To specify the vector index, you can either use both the vector bucket name and vector
     * index name, or use the vector index Amazon Resource Name (ARN).
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:DeleteIndex</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param deleteIndexRequest
     * @return Result of the DeleteIndex operation returned by the service.
     * @throws ValidationException
     *         The requested action isn't valid.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Wait briefly and retry your request. If it continues to fail, increase your
     *         waiting time between retries.
     * @throws TooManyRequestsException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request failed due to an internal server error.
     * @throws AccessDeniedException
     *         Access denied.
     * @throws ServiceQuotaExceededException
     *         Your request exceeds a service quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3VectorsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3VectorsClient.DeleteIndex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/DeleteIndex" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteIndexResponse deleteIndex(DeleteIndexRequest deleteIndexRequest) throws ValidationException,
            ServiceUnavailableException, TooManyRequestsException, InternalServerException, AccessDeniedException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, S3VectorsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteIndexResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteIndexResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "KmsInvalidKeyUsageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
            case "KmsInvalidStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
            case "KmsNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "KmsDisabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsDisabledException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteIndexRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteIndexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteIndex");

            return clientHandler.execute(new ClientExecutionParams<DeleteIndexRequest, DeleteIndexResponse>()
                    .withOperationName("DeleteIndex").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteIndexRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteIndexRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Deletes a vector bucket. All vector indexes in the vector bucket must be deleted before the vector bucket can be
     * deleted. To perform this operation, you must use either the vector bucket name or the vector bucket Amazon
     * Resource Name (ARN).
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:DeleteVectorBucket</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param deleteVectorBucketRequest
     * @return Result of the DeleteVectorBucket operation returned by the service.
     * @throws ValidationException
     *         The requested action isn't valid.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Wait briefly and retry your request. If it continues to fail, increase your
     *         waiting time between retries.
     * @throws TooManyRequestsException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request failed due to an internal server error.
     * @throws AccessDeniedException
     *         Access denied.
     * @throws ConflictException
     *         The request failed because a vector bucket name or a vector index name already exists. Vector bucket
     *         names must be unique within your Amazon Web Services account for each Amazon Web Services Region. Vector
     *         index names must be unique within your vector bucket. Choose a different vector bucket name or vector
     *         index name, and try again.
     * @throws ServiceQuotaExceededException
     *         Your request exceeds a service quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3VectorsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3VectorsClient.DeleteVectorBucket
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/DeleteVectorBucket" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteVectorBucketResponse deleteVectorBucket(DeleteVectorBucketRequest deleteVectorBucketRequest)
            throws ValidationException, ServiceUnavailableException, TooManyRequestsException, InternalServerException,
            AccessDeniedException, ConflictException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            S3VectorsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteVectorBucketResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteVectorBucketResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "KmsInvalidKeyUsageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
            case "KmsInvalidStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
            case "KmsNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "KmsDisabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsDisabledException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteVectorBucketRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVectorBucketRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVectorBucket");

            return clientHandler.execute(new ClientExecutionParams<DeleteVectorBucketRequest, DeleteVectorBucketResponse>()
                    .withOperationName("DeleteVectorBucket").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteVectorBucketRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteVectorBucketRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Deletes a vector bucket policy. To specify the bucket, you must use either the vector bucket name or the vector
     * bucket Amazon Resource Name (ARN).
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:DeleteVectorBucketPolicy</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param deleteVectorBucketPolicyRequest
     * @return Result of the DeleteVectorBucketPolicy operation returned by the service.
     * @throws ValidationException
     *         The requested action isn't valid.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Wait briefly and retry your request. If it continues to fail, increase your
     *         waiting time between retries.
     * @throws TooManyRequestsException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request failed due to an internal server error.
     * @throws AccessDeniedException
     *         Access denied.
     * @throws NotFoundException
     *         The request was rejected because the specified resource can't be found.
     * @throws ServiceQuotaExceededException
     *         Your request exceeds a service quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3VectorsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3VectorsClient.DeleteVectorBucketPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/DeleteVectorBucketPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteVectorBucketPolicyResponse deleteVectorBucketPolicy(
            DeleteVectorBucketPolicyRequest deleteVectorBucketPolicyRequest) throws ValidationException,
            ServiceUnavailableException, TooManyRequestsException, InternalServerException, AccessDeniedException,
            NotFoundException, ServiceQuotaExceededException, AwsServiceException, SdkClientException, S3VectorsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteVectorBucketPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteVectorBucketPolicyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "KmsInvalidKeyUsageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
            case "KmsInvalidStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
            case "KmsNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "KmsDisabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsDisabledException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteVectorBucketPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVectorBucketPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVectorBucketPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteVectorBucketPolicyRequest, DeleteVectorBucketPolicyResponse>()
                            .withOperationName("DeleteVectorBucketPolicy").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteVectorBucketPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteVectorBucketPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Deletes one or more vectors in a vector index. To specify the vector index, you can either use both the vector
     * bucket name and vector index name, or use the vector index Amazon Resource Name (ARN).
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:DeleteVectors</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param deleteVectorsRequest
     * @return Result of the DeleteVectors operation returned by the service.
     * @throws ValidationException
     *         The requested action isn't valid.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Wait briefly and retry your request. If it continues to fail, increase your
     *         waiting time between retries.
     * @throws TooManyRequestsException
     *         The request was denied due to request throttling.
     * @throws KmsInvalidKeyUsageException
     *         The request was rejected for one of the following reasons: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The <code>KeyUsage</code> value of the KMS key is incompatible with the API operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The encryption algorithm or signing algorithm specified for the operation is incompatible with the type
     *         of key material in the KMS key (<code>KeySpec</code>).
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html#API_Encrypt_Errors"
     *         >InvalidKeyUsageException</a> in the <i>Amazon Web Services Key Management Service API Reference</i>.
     * @throws InternalServerException
     *         The request failed due to an internal server error.
     * @throws KmsInvalidStateException
     *         The key state of the KMS key isn't compatible with the operation.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html#API_Encrypt_Errors"
     *         >KMSInvalidStateException</a> in the <i>Amazon Web Services Key Management Service API Reference</i>.
     * @throws AccessDeniedException
     *         Access denied.
     * @throws KmsNotFoundException
     *         The KMS key can't be found.
     * @throws NotFoundException
     *         The request was rejected because the specified resource can't be found.
     * @throws ServiceQuotaExceededException
     *         Your request exceeds a service quota.
     * @throws KmsDisabledException
     *         The specified Amazon Web Services KMS key isn't enabled.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3VectorsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3VectorsClient.DeleteVectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/DeleteVectors" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteVectorsResponse deleteVectors(DeleteVectorsRequest deleteVectorsRequest) throws ValidationException,
            ServiceUnavailableException, TooManyRequestsException, KmsInvalidKeyUsageException, InternalServerException,
            KmsInvalidStateException, AccessDeniedException, KmsNotFoundException, NotFoundException,
            ServiceQuotaExceededException, KmsDisabledException, AwsServiceException, SdkClientException, S3VectorsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteVectorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteVectorsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "KmsInvalidKeyUsageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
            case "KmsInvalidStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
            case "KmsNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "KmsDisabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsDisabledException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteVectorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVectors");

            return clientHandler.execute(new ClientExecutionParams<DeleteVectorsRequest, DeleteVectorsResponse>()
                    .withOperationName("DeleteVectors").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteVectorsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteVectorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Returns vector index attributes. To specify the vector index, you can either use both the vector bucket name and
     * the vector index name, or use the vector index Amazon Resource Name (ARN).
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:GetIndex</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param getIndexRequest
     * @return Result of the GetIndex operation returned by the service.
     * @throws ValidationException
     *         The requested action isn't valid.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Wait briefly and retry your request. If it continues to fail, increase your
     *         waiting time between retries.
     * @throws TooManyRequestsException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request failed due to an internal server error.
     * @throws AccessDeniedException
     *         Access denied.
     * @throws NotFoundException
     *         The request was rejected because the specified resource can't be found.
     * @throws ServiceQuotaExceededException
     *         Your request exceeds a service quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3VectorsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3VectorsClient.GetIndex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/GetIndex" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetIndexResponse getIndex(GetIndexRequest getIndexRequest) throws ValidationException, ServiceUnavailableException,
            TooManyRequestsException, InternalServerException, AccessDeniedException, NotFoundException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, S3VectorsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetIndexResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetIndexResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "KmsInvalidKeyUsageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
            case "KmsInvalidStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
            case "KmsNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "KmsDisabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsDisabledException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getIndexRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getIndexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetIndex");

            return clientHandler.execute(new ClientExecutionParams<GetIndexRequest, GetIndexResponse>()
                    .withOperationName("GetIndex").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getIndexRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetIndexRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Returns vector bucket attributes. To specify the bucket, you must use either the vector bucket name or the vector
     * bucket Amazon Resource Name (ARN).
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:GetVectorBucket</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param getVectorBucketRequest
     * @return Result of the GetVectorBucket operation returned by the service.
     * @throws ValidationException
     *         The requested action isn't valid.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Wait briefly and retry your request. If it continues to fail, increase your
     *         waiting time between retries.
     * @throws TooManyRequestsException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request failed due to an internal server error.
     * @throws AccessDeniedException
     *         Access denied.
     * @throws NotFoundException
     *         The request was rejected because the specified resource can't be found.
     * @throws ServiceQuotaExceededException
     *         Your request exceeds a service quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3VectorsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3VectorsClient.GetVectorBucket
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/GetVectorBucket" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetVectorBucketResponse getVectorBucket(GetVectorBucketRequest getVectorBucketRequest) throws ValidationException,
            ServiceUnavailableException, TooManyRequestsException, InternalServerException, AccessDeniedException,
            NotFoundException, ServiceQuotaExceededException, AwsServiceException, SdkClientException, S3VectorsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetVectorBucketResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetVectorBucketResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "KmsInvalidKeyUsageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
            case "KmsInvalidStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
            case "KmsNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "KmsDisabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsDisabledException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getVectorBucketRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getVectorBucketRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetVectorBucket");

            return clientHandler.execute(new ClientExecutionParams<GetVectorBucketRequest, GetVectorBucketResponse>()
                    .withOperationName("GetVectorBucket").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getVectorBucketRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetVectorBucketRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Gets details about a vector bucket policy. To specify the bucket, you must use either the vector bucket name or
     * the vector bucket Amazon Resource Name (ARN).
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:GetVectorBucketPolicy</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param getVectorBucketPolicyRequest
     * @return Result of the GetVectorBucketPolicy operation returned by the service.
     * @throws ValidationException
     *         The requested action isn't valid.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Wait briefly and retry your request. If it continues to fail, increase your
     *         waiting time between retries.
     * @throws TooManyRequestsException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request failed due to an internal server error.
     * @throws AccessDeniedException
     *         Access denied.
     * @throws NotFoundException
     *         The request was rejected because the specified resource can't be found.
     * @throws ServiceQuotaExceededException
     *         Your request exceeds a service quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3VectorsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3VectorsClient.GetVectorBucketPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/GetVectorBucketPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetVectorBucketPolicyResponse getVectorBucketPolicy(GetVectorBucketPolicyRequest getVectorBucketPolicyRequest)
            throws ValidationException, ServiceUnavailableException, TooManyRequestsException, InternalServerException,
            AccessDeniedException, NotFoundException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            S3VectorsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetVectorBucketPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetVectorBucketPolicyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "KmsInvalidKeyUsageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
            case "KmsInvalidStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
            case "KmsNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "KmsDisabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsDisabledException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getVectorBucketPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getVectorBucketPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetVectorBucketPolicy");

            return clientHandler.execute(new ClientExecutionParams<GetVectorBucketPolicyRequest, GetVectorBucketPolicyResponse>()
                    .withOperationName("GetVectorBucketPolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getVectorBucketPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetVectorBucketPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Returns vector attributes. To specify the vector index, you can either use both the vector bucket name and the
     * vector index name, or use the vector index Amazon Resource Name (ARN).
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:GetVectors</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param getVectorsRequest
     * @return Result of the GetVectors operation returned by the service.
     * @throws ValidationException
     *         The requested action isn't valid.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Wait briefly and retry your request. If it continues to fail, increase your
     *         waiting time between retries.
     * @throws TooManyRequestsException
     *         The request was denied due to request throttling.
     * @throws KmsInvalidKeyUsageException
     *         The request was rejected for one of the following reasons: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The <code>KeyUsage</code> value of the KMS key is incompatible with the API operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The encryption algorithm or signing algorithm specified for the operation is incompatible with the type
     *         of key material in the KMS key (<code>KeySpec</code>).
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html#API_Encrypt_Errors"
     *         >InvalidKeyUsageException</a> in the <i>Amazon Web Services Key Management Service API Reference</i>.
     * @throws InternalServerException
     *         The request failed due to an internal server error.
     * @throws KmsInvalidStateException
     *         The key state of the KMS key isn't compatible with the operation.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html#API_Encrypt_Errors"
     *         >KMSInvalidStateException</a> in the <i>Amazon Web Services Key Management Service API Reference</i>.
     * @throws AccessDeniedException
     *         Access denied.
     * @throws KmsNotFoundException
     *         The KMS key can't be found.
     * @throws NotFoundException
     *         The request was rejected because the specified resource can't be found.
     * @throws ServiceQuotaExceededException
     *         Your request exceeds a service quota.
     * @throws KmsDisabledException
     *         The specified Amazon Web Services KMS key isn't enabled.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3VectorsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3VectorsClient.GetVectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/GetVectors" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetVectorsResponse getVectors(GetVectorsRequest getVectorsRequest) throws ValidationException,
            ServiceUnavailableException, TooManyRequestsException, KmsInvalidKeyUsageException, InternalServerException,
            KmsInvalidStateException, AccessDeniedException, KmsNotFoundException, NotFoundException,
            ServiceQuotaExceededException, KmsDisabledException, AwsServiceException, SdkClientException, S3VectorsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetVectorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetVectorsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "KmsInvalidKeyUsageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
            case "KmsInvalidStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
            case "KmsNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "KmsDisabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsDisabledException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getVectorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getVectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetVectors");

            return clientHandler.execute(new ClientExecutionParams<GetVectorsRequest, GetVectorsResponse>()
                    .withOperationName("GetVectors").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getVectorsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetVectorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Returns a list of all the vector indexes within the specified vector bucket. To specify the bucket, you must use
     * either the vector bucket name or the vector bucket Amazon Resource Name (ARN).
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:ListIndexes</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param listIndexesRequest
     * @return Result of the ListIndexes operation returned by the service.
     * @throws ValidationException
     *         The requested action isn't valid.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Wait briefly and retry your request. If it continues to fail, increase your
     *         waiting time between retries.
     * @throws TooManyRequestsException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request failed due to an internal server error.
     * @throws AccessDeniedException
     *         Access denied.
     * @throws NotFoundException
     *         The request was rejected because the specified resource can't be found.
     * @throws ServiceQuotaExceededException
     *         Your request exceeds a service quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3VectorsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3VectorsClient.ListIndexes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/ListIndexes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListIndexesResponse listIndexes(ListIndexesRequest listIndexesRequest) throws ValidationException,
            ServiceUnavailableException, TooManyRequestsException, InternalServerException, AccessDeniedException,
            NotFoundException, ServiceQuotaExceededException, AwsServiceException, SdkClientException, S3VectorsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListIndexesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListIndexesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "KmsInvalidKeyUsageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
            case "KmsInvalidStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
            case "KmsNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "KmsDisabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsDisabledException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listIndexesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listIndexesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListIndexes");

            return clientHandler.execute(new ClientExecutionParams<ListIndexesRequest, ListIndexesResponse>()
                    .withOperationName("ListIndexes").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listIndexesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListIndexesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Returns a list of all the vector buckets that are owned by the authenticated sender of the request.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:ListVectorBuckets</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param listVectorBucketsRequest
     * @return Result of the ListVectorBuckets operation returned by the service.
     * @throws ValidationException
     *         The requested action isn't valid.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Wait briefly and retry your request. If it continues to fail, increase your
     *         waiting time between retries.
     * @throws TooManyRequestsException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request failed due to an internal server error.
     * @throws AccessDeniedException
     *         Access denied.
     * @throws ServiceQuotaExceededException
     *         Your request exceeds a service quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3VectorsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3VectorsClient.ListVectorBuckets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/ListVectorBuckets" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListVectorBucketsResponse listVectorBuckets(ListVectorBucketsRequest listVectorBucketsRequest)
            throws ValidationException, ServiceUnavailableException, TooManyRequestsException, InternalServerException,
            AccessDeniedException, ServiceQuotaExceededException, AwsServiceException, SdkClientException, S3VectorsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListVectorBucketsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListVectorBucketsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "KmsInvalidKeyUsageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
            case "KmsInvalidStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
            case "KmsNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "KmsDisabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsDisabledException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listVectorBucketsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVectorBucketsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVectorBuckets");

            return clientHandler.execute(new ClientExecutionParams<ListVectorBucketsRequest, ListVectorBucketsResponse>()
                    .withOperationName("ListVectorBuckets").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listVectorBucketsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListVectorBucketsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * List vectors in the specified vector index. To specify the vector index, you can either use both the vector
     * bucket name and the vector index name, or use the vector index Amazon Resource Name (ARN).
     * </p>
     * <p>
     * <code>ListVectors</code> operations proceed sequentially; however, for faster performance on a large number of
     * vectors in a vector index, applications can request a parallel <code>ListVectors</code> operation by providing
     * the <code>segmentCount</code> and <code>segmentIndex</code> parameters.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:ListVectors</code> permission to use this operation. Additional permissions are
     * required based on the request parameters you specify:
     * </p>
     * <ul>
     * <li>
     * <p>
     * With only <code>s3vectors:ListVectors</code> permission, you can list vector keys when <code>returnData</code>
     * and <code>returnMetadata</code> are both set to false or not specified..
     * </p>
     * </li>
     * <li>
     * <p>
     * If you set <code>returnData</code> or <code>returnMetadata</code> to true, you must have both
     * <code>s3vectors:ListVectors</code> and <code>s3vectors:GetVectors</code> permissions. The request fails with a
     * <code>403 Forbidden</code> error if you request vector data or metadata without the
     * <code>s3vectors:GetVectors</code> permission.
     * </p>
     * </li>
     * </ul>
     * </dd>
     * </dl>
     * </p>
     *
     * @param listVectorsRequest
     * @return Result of the ListVectors operation returned by the service.
     * @throws ValidationException
     *         The requested action isn't valid.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Wait briefly and retry your request. If it continues to fail, increase your
     *         waiting time between retries.
     * @throws TooManyRequestsException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request failed due to an internal server error.
     * @throws AccessDeniedException
     *         Access denied.
     * @throws NotFoundException
     *         The request was rejected because the specified resource can't be found.
     * @throws ServiceQuotaExceededException
     *         Your request exceeds a service quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3VectorsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3VectorsClient.ListVectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/ListVectors" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListVectorsResponse listVectors(ListVectorsRequest listVectorsRequest) throws ValidationException,
            ServiceUnavailableException, TooManyRequestsException, InternalServerException, AccessDeniedException,
            NotFoundException, ServiceQuotaExceededException, AwsServiceException, SdkClientException, S3VectorsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListVectorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListVectorsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "KmsInvalidKeyUsageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
            case "KmsInvalidStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
            case "KmsNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "KmsDisabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsDisabledException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listVectorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVectors");

            return clientHandler.execute(new ClientExecutionParams<ListVectorsRequest, ListVectorsResponse>()
                    .withOperationName("ListVectors").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listVectorsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListVectorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Creates a bucket policy for a vector bucket. To specify the bucket, you must use either the vector bucket name or
     * the vector bucket Amazon Resource Name (ARN).
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:PutVectorBucketPolicy</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param putVectorBucketPolicyRequest
     * @return Result of the PutVectorBucketPolicy operation returned by the service.
     * @throws ValidationException
     *         The requested action isn't valid.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Wait briefly and retry your request. If it continues to fail, increase your
     *         waiting time between retries.
     * @throws TooManyRequestsException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request failed due to an internal server error.
     * @throws AccessDeniedException
     *         Access denied.
     * @throws NotFoundException
     *         The request was rejected because the specified resource can't be found.
     * @throws ServiceQuotaExceededException
     *         Your request exceeds a service quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3VectorsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3VectorsClient.PutVectorBucketPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/PutVectorBucketPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutVectorBucketPolicyResponse putVectorBucketPolicy(PutVectorBucketPolicyRequest putVectorBucketPolicyRequest)
            throws ValidationException, ServiceUnavailableException, TooManyRequestsException, InternalServerException,
            AccessDeniedException, NotFoundException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            S3VectorsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutVectorBucketPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutVectorBucketPolicyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "KmsInvalidKeyUsageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
            case "KmsInvalidStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
            case "KmsNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "KmsDisabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsDisabledException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putVectorBucketPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putVectorBucketPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutVectorBucketPolicy");

            return clientHandler.execute(new ClientExecutionParams<PutVectorBucketPolicyRequest, PutVectorBucketPolicyResponse>()
                    .withOperationName("PutVectorBucketPolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putVectorBucketPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutVectorBucketPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Adds one or more vectors to a vector index. To specify the vector index, you can either use both the vector
     * bucket name and the vector index name, or use the vector index Amazon Resource Name (ARN).
     * </p>
     * <p>
     * For more information about limits, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-vectors-limitations.html">Limitations and
     * restrictions</a> in the <i>Amazon S3 User Guide</i>.
     * </p>
     * <note>
     * <p>
     * When inserting vector data into your vector index, you must provide the vector data as <code>float32</code>
     * (32-bit floating point) values. If you pass higher-precision values to an Amazon Web Services SDK, S3 Vectors
     * converts the values to 32-bit floating point before storing them, and <code>GetVectors</code>,
     * <code>ListVectors</code>, and <code>QueryVectors</code> operations return the float32 values. Different Amazon
     * Web Services SDKs may have different default numeric types, so ensure your vectors are properly formatted as
     * <code>float32</code> values regardless of which SDK you're using. For example, in Python, use
     * <code>numpy.float32</code> or explicitly cast your values.
     * </p>
     * </note>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:PutVectors</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param putVectorsRequest
     * @return Result of the PutVectors operation returned by the service.
     * @throws ValidationException
     *         The requested action isn't valid.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Wait briefly and retry your request. If it continues to fail, increase your
     *         waiting time between retries.
     * @throws TooManyRequestsException
     *         The request was denied due to request throttling.
     * @throws KmsInvalidKeyUsageException
     *         The request was rejected for one of the following reasons: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The <code>KeyUsage</code> value of the KMS key is incompatible with the API operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The encryption algorithm or signing algorithm specified for the operation is incompatible with the type
     *         of key material in the KMS key (<code>KeySpec</code>).
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html#API_Encrypt_Errors"
     *         >InvalidKeyUsageException</a> in the <i>Amazon Web Services Key Management Service API Reference</i>.
     * @throws InternalServerException
     *         The request failed due to an internal server error.
     * @throws KmsInvalidStateException
     *         The key state of the KMS key isn't compatible with the operation.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html#API_Encrypt_Errors"
     *         >KMSInvalidStateException</a> in the <i>Amazon Web Services Key Management Service API Reference</i>.
     * @throws AccessDeniedException
     *         Access denied.
     * @throws KmsNotFoundException
     *         The KMS key can't be found.
     * @throws NotFoundException
     *         The request was rejected because the specified resource can't be found.
     * @throws ServiceQuotaExceededException
     *         Your request exceeds a service quota.
     * @throws KmsDisabledException
     *         The specified Amazon Web Services KMS key isn't enabled.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3VectorsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3VectorsClient.PutVectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/PutVectors" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutVectorsResponse putVectors(PutVectorsRequest putVectorsRequest) throws ValidationException,
            ServiceUnavailableException, TooManyRequestsException, KmsInvalidKeyUsageException, InternalServerException,
            KmsInvalidStateException, AccessDeniedException, KmsNotFoundException, NotFoundException,
            ServiceQuotaExceededException, KmsDisabledException, AwsServiceException, SdkClientException, S3VectorsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutVectorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutVectorsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "KmsInvalidKeyUsageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
            case "KmsInvalidStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
            case "KmsNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "KmsDisabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsDisabledException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putVectorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putVectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutVectors");

            return clientHandler.execute(new ClientExecutionParams<PutVectorsRequest, PutVectorsResponse>()
                    .withOperationName("PutVectors").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(putVectorsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutVectorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Performs an approximate nearest neighbor search query in a vector index using a query vector. By default, it
     * returns the keys of approximate nearest neighbors. You can optionally include the computed distance (between the
     * query vector and each vector in the response), the vector data, and metadata of each vector in the response.
     * </p>
     * <p>
     * To specify the vector index, you can either use both the vector bucket name and the vector index name, or use the
     * vector index Amazon Resource Name (ARN).
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:QueryVectors</code> permission to use this operation. Additional permissions
     * are required based on the request parameters you specify:
     * </p>
     * <ul>
     * <li>
     * <p>
     * With only <code>s3vectors:QueryVectors</code> permission, you can retrieve vector keys of approximate nearest
     * neighbors and computed distances between these vectors. This permission is sufficient only when you don't set any
     * metadata filters and don't request vector data or metadata (by keeping the <code>returnMetadata</code> parameter
     * set to <code>false</code> or not specified).
     * </p>
     * </li>
     * <li>
     * <p>
     * If you specify a metadata filter or set <code>returnMetadata</code> to true, you must have both
     * <code>s3vectors:QueryVectors</code> and <code>s3vectors:GetVectors</code> permissions. The request fails with a
     * <code>403 Forbidden error</code> if you request metadata filtering, vector data, or metadata without the
     * <code>s3vectors:GetVectors</code> permission.
     * </p>
     * </li>
     * </ul>
     * </dd>
     * </dl>
     * </p>
     *
     * @param queryVectorsRequest
     * @return Result of the QueryVectors operation returned by the service.
     * @throws ValidationException
     *         The requested action isn't valid.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Wait briefly and retry your request. If it continues to fail, increase your
     *         waiting time between retries.
     * @throws TooManyRequestsException
     *         The request was denied due to request throttling.
     * @throws KmsInvalidKeyUsageException
     *         The request was rejected for one of the following reasons: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The <code>KeyUsage</code> value of the KMS key is incompatible with the API operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The encryption algorithm or signing algorithm specified for the operation is incompatible with the type
     *         of key material in the KMS key (<code>KeySpec</code>).
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html#API_Encrypt_Errors"
     *         >InvalidKeyUsageException</a> in the <i>Amazon Web Services Key Management Service API Reference</i>.
     * @throws InternalServerException
     *         The request failed due to an internal server error.
     * @throws KmsInvalidStateException
     *         The key state of the KMS key isn't compatible with the operation.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html#API_Encrypt_Errors"
     *         >KMSInvalidStateException</a> in the <i>Amazon Web Services Key Management Service API Reference</i>.
     * @throws AccessDeniedException
     *         Access denied.
     * @throws KmsNotFoundException
     *         The KMS key can't be found.
     * @throws NotFoundException
     *         The request was rejected because the specified resource can't be found.
     * @throws ServiceQuotaExceededException
     *         Your request exceeds a service quota.
     * @throws KmsDisabledException
     *         The specified Amazon Web Services KMS key isn't enabled.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3VectorsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3VectorsClient.QueryVectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/QueryVectors" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public QueryVectorsResponse queryVectors(QueryVectorsRequest queryVectorsRequest) throws ValidationException,
            ServiceUnavailableException, TooManyRequestsException, KmsInvalidKeyUsageException, InternalServerException,
            KmsInvalidStateException, AccessDeniedException, KmsNotFoundException, NotFoundException,
            ServiceQuotaExceededException, KmsDisabledException, AwsServiceException, SdkClientException, S3VectorsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<QueryVectorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                QueryVectorsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "KmsInvalidKeyUsageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
            case "KmsInvalidStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
            case "KmsNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "KmsDisabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsDisabledException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(queryVectorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, queryVectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "QueryVectors");

            return clientHandler.execute(new ClientExecutionParams<QueryVectorsRequest, QueryVectorsResponse>()
                    .withOperationName("QueryVectors").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(queryVectorsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new QueryVectorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        S3VectorsServiceClientConfigurationBuilder serviceConfigBuilder = new S3VectorsServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(S3VectorsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    @Override
    public final S3VectorsServiceClientConfiguration serviceClientConfiguration() {
        return new S3VectorsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
