/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.services.s3.internal.endpoints;

import java.net.URI;
import java.util.Optional;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.arns.Arn;
import software.amazon.awssdk.http.SdkHttpRequest;
import software.amazon.awssdk.regions.PartitionMetadata;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.s3.S3Configuration;
import software.amazon.awssdk.services.s3.internal.ConfiguredS3SdkHttpRequest;
import software.amazon.awssdk.services.s3.internal.endpoints.S3EndpointResolver;
import software.amazon.awssdk.services.s3.internal.endpoints.S3EndpointResolverContext;
import software.amazon.awssdk.services.s3.internal.endpoints.S3EndpointUtils;
import software.amazon.awssdk.services.s3.internal.resource.S3AccessPointBuilder;
import software.amazon.awssdk.services.s3.internal.resource.S3AccessPointResource;
import software.amazon.awssdk.services.s3.internal.resource.S3ArnConverter;
import software.amazon.awssdk.services.s3.internal.resource.S3ObjectLambdaEndpointBuilder;
import software.amazon.awssdk.services.s3.internal.resource.S3ObjectLambdaResource;
import software.amazon.awssdk.services.s3.internal.resource.S3OutpostAccessPointBuilder;
import software.amazon.awssdk.services.s3.internal.resource.S3OutpostResource;
import software.amazon.awssdk.services.s3.internal.resource.S3Resource;
import software.amazon.awssdk.utils.Validate;
import software.amazon.awssdk.utils.http.SdkHttpUtils;

@SdkInternalApi
public final class S3AccessPointEndpointResolver
implements S3EndpointResolver {
    private static final String S3_CONFIG_ERROR_MESSAGE = "An access point ARN cannot be passed as a bucket parameter to an S3 operation if the S3 client has been configured with %s";
    private static final String S3_OUTPOSTS_NAME = "s3-outposts";
    private static final String S3_OBJECT_LAMBDA_NAME = "s3-object-lambda";

    private S3AccessPointEndpointResolver() {
    }

    public static S3AccessPointEndpointResolver create() {
        return new S3AccessPointEndpointResolver();
    }

    @Override
    public ConfiguredS3SdkHttpRequest applyEndpointConfiguration(S3EndpointResolverContext context) {
        S3Resource s3Resource = S3ArnConverter.create().convertArn(Arn.fromString((String)this.getBucketName(context)));
        S3AccessPointResource s3EndpointResource = (S3AccessPointResource)Validate.isInstanceOf(S3AccessPointResource.class, (Object)s3Resource, (String)"An ARN was passed as a bucket parameter to an S3 operation, however it does not appear to be a valid S3 access point ARN.", (Object[])new Object[0]);
        PartitionMetadata clientPartitionMetadata = PartitionMetadata.of((Region)context.region());
        this.validateConfiguration(context, s3EndpointResource);
        URI accessPointUri = this.getUriForAccessPointResource(context, clientPartitionMetadata, s3EndpointResource);
        String path = this.buildPath(accessPointUri, context);
        SdkHttpRequest httpRequest = (SdkHttpRequest)((SdkHttpRequest.Builder)context.request().toBuilder()).protocol(accessPointUri.getScheme()).host(accessPointUri.getHost()).port(Integer.valueOf(accessPointUri.getPort())).encodedPath(path).build();
        Region signingRegionModification = s3EndpointResource.region().map(Region::of).orElse(null);
        String signingServiceModification = s3EndpointResource.parentS3Resource().flatMap(S3AccessPointEndpointResolver::resolveSigningService).orElse(null);
        return ConfiguredS3SdkHttpRequest.builder().sdkHttpRequest(httpRequest).signingRegionModification(signingRegionModification).signingServiceModification(signingServiceModification).build();
    }

    private String buildPath(URI accessPointUri, S3EndpointResolverContext context) {
        String key = context.originalRequest().getValueForField("Key", String.class).orElse(null);
        StringBuilder pathBuilder = new StringBuilder();
        if (accessPointUri.getPath() != null) {
            pathBuilder.append(accessPointUri.getPath());
        }
        if (key != null) {
            if (pathBuilder.length() > 0) {
                pathBuilder.append('/');
            }
            pathBuilder.append(SdkHttpUtils.urlEncodeIgnoreSlashes((String)key));
        }
        return pathBuilder.length() > 0 ? pathBuilder.toString() : null;
    }

    private void validateConfiguration(S3EndpointResolverContext context, S3AccessPointResource s3Resource) {
        S3Configuration serviceConfig = context.serviceConfiguration();
        Validate.isFalse((boolean)S3EndpointUtils.isAccelerateEnabled(serviceConfig), (String)S3_CONFIG_ERROR_MESSAGE, (Object[])new Object[]{"accelerate mode enabled."});
        Validate.isFalse((boolean)S3EndpointUtils.isPathStyleAccessEnabled(serviceConfig), (String)S3_CONFIG_ERROR_MESSAGE, (Object[])new Object[]{"path style addressing enabled."});
        Validate.isTrue((boolean)s3Resource.accountId().isPresent(), (String)"An S3 access point ARN must have an account ID", (Object[])new Object[0]);
        Region clientRegion = context.region();
        if (s3Resource.region().isPresent()) {
            this.validateRegion(s3Resource, serviceConfig, clientRegion, context.fipsEnabled());
        } else {
            this.validateGlobalConfiguration(context);
        }
        this.validatePartition(s3Resource, clientRegion);
    }

    private void validatePartition(S3AccessPointResource s3Resource, Region clientRegion) {
        String clientPartition = PartitionMetadata.of((Region)clientRegion).id();
        Validate.isFalse((boolean)this.illegalPartitionConfiguration(s3Resource, clientPartition), (String)"The partition field of the ARN being passed as a bucket parameter to an S3 operation does not match the partition the S3 client has been configured with. Provided partition: '%s'; client partition: '%s'.", (Object[])new Object[]{s3Resource.partition().orElse(""), clientPartition});
    }

    private boolean illegalPartitionConfiguration(S3Resource s3Resource, String clientPartition) {
        return clientPartition == null || clientPartition.isEmpty() || !s3Resource.partition().isPresent() || !clientPartition.equals(s3Resource.partition().get());
    }

    private void validateRegion(S3AccessPointResource s3Resource, S3Configuration serviceConfig, Region clientRegion, boolean fipsEnabledOnClient) {
        String arnRegion = s3Resource.region().get();
        Validate.isFalse((boolean)S3EndpointUtils.isFipsRegion(arnRegion), (String)("Invalid ARN, FIPS region is not allowed in ARN. Provided arn region: '" + arnRegion + "'."), (Object[])new Object[0]);
        boolean fipsEnabled = fipsEnabledOnClient || S3EndpointUtils.isFipsRegion(clientRegion.id());
        Validate.isFalse((fipsEnabled && this.clientRegionDiffersFromArnRegion(clientRegion, arnRegion) ? 1 : 0) != 0, (String)String.format("The region field of the ARN being passed as a bucket parameter to an S3 operation does not match the region the client was configured with. Cross region access not allowed for fips region in client or arn. Provided region: '%s'; client region:'%s'.", arnRegion, clientRegion), (Object[])new Object[0]);
        Validate.isFalse((!S3EndpointUtils.isArnRegionEnabled(serviceConfig) && this.clientRegionDiffersFromArnRegion(clientRegion, arnRegion) ? 1 : 0) != 0, (String)"The region field of the ARN being passed as a bucket parameter to an S3 operation does not match the region the client was configured with. To enable this behavior and prevent this exception set 'useArnRegionEnabled' to true in the configuration when building the S3 client. Provided region: '%s'; client region: '%s'.", (Object[])new Object[]{arnRegion, clientRegion});
    }

    private boolean clientRegionDiffersFromArnRegion(Region clientRegion, String arnRegion) {
        return !S3EndpointUtils.removeFipsIfNeeded(clientRegion.id()).equals(arnRegion);
    }

    private void validateGlobalConfiguration(S3EndpointResolverContext context) {
        S3Configuration serviceConfiguration = context.serviceConfiguration();
        Validate.isTrue((boolean)serviceConfiguration.multiRegionEnabled(), (String)"An Access Point ARN without a region value was passed as a bucket parameter but multi-region is disabled. Check client configuration, environment variables and system configuration for multi-region disable configurations.", (Object[])new Object[0]);
        Validate.isFalse((boolean)S3EndpointUtils.isDualstackEnabled(serviceConfiguration), (String)S3_CONFIG_ERROR_MESSAGE, (Object[])new Object[]{"dualstack, if the ARN contains no region."});
        Validate.isFalse((boolean)S3AccessPointEndpointResolver.isFipsEnabled(context), (String)S3_CONFIG_ERROR_MESSAGE, (Object[])new Object[]{"a FIPS enabled region, if the ARN contains no region."});
    }

    private String getBucketName(S3EndpointResolverContext context) {
        return (String)context.originalRequest().getValueForField("Bucket", String.class).orElseThrow(() -> new IllegalArgumentException("Bucket name cannot be empty when parsing access points."));
    }

    private URI getUriForAccessPointResource(S3EndpointResolverContext context, PartitionMetadata clientPartitionMetadata, S3AccessPointResource s3EndpointResource) {
        if (this.isOutpostAccessPoint(s3EndpointResource)) {
            return this.getOutpostAccessPointUri(context, clientPartitionMetadata, s3EndpointResource);
        }
        if (this.isObjectLambdaAccessPoint(s3EndpointResource)) {
            return this.getObjectLambdaAccessPointUri(context, clientPartitionMetadata, s3EndpointResource);
        }
        return S3AccessPointBuilder.create().endpointOverride(context.endpointOverride()).accessPointName(s3EndpointResource.accessPointName()).accountId(s3EndpointResource.accountId().get()).fipsEnabled(S3AccessPointEndpointResolver.isFipsEnabled(context)).region(s3EndpointResource.region().orElse(null)).protocol(context.request().protocol()).domain(clientPartitionMetadata.dnsSuffix()).dualstackEnabled(S3EndpointUtils.isDualstackEnabled(context.serviceConfiguration())).toUri();
    }

    private boolean isOutpostAccessPoint(S3AccessPointResource s3EndpointResource) {
        return s3EndpointResource.parentS3Resource().filter(r -> r instanceof S3OutpostResource).isPresent();
    }

    private boolean isObjectLambdaAccessPoint(S3AccessPointResource s3EndpointResource) {
        return s3EndpointResource.parentS3Resource().filter(r -> r instanceof S3ObjectLambdaResource).isPresent();
    }

    private URI getOutpostAccessPointUri(S3EndpointResolverContext context, PartitionMetadata clientPartitionMetadata, S3AccessPointResource s3EndpointResource) {
        if (S3EndpointUtils.isDualstackEnabled(context.serviceConfiguration())) {
            throw new IllegalArgumentException("An Outpost Access Point ARN cannot be passed as a bucket parameter to an S3 operation if the S3 client has been configured with dualstack");
        }
        if (S3AccessPointEndpointResolver.isFipsEnabled(context)) {
            throw new IllegalArgumentException("An access point ARN cannot be passed as a bucket parameter to an S3 operation if the S3 client has been configured with a FIPS enabled region.");
        }
        S3OutpostResource parentResource = (S3OutpostResource)s3EndpointResource.parentS3Resource().get();
        return S3OutpostAccessPointBuilder.create().endpointOverride(context.endpointOverride()).accountId(s3EndpointResource.accountId().get()).outpostId(parentResource.outpostId()).region(s3EndpointResource.region().get()).accessPointName(s3EndpointResource.accessPointName()).protocol(context.request().protocol()).domain(clientPartitionMetadata.dnsSuffix()).toUri();
    }

    private URI getObjectLambdaAccessPointUri(S3EndpointResolverContext context, PartitionMetadata clientPartitionMetadata, S3AccessPointResource s3EndpointResource) {
        if (S3EndpointUtils.isDualstackEnabled(context.serviceConfiguration())) {
            throw new IllegalArgumentException("An Object Lambda Access Point ARN cannot be passed as a bucket parameter to an S3 operation if the S3 client has been configured with dualstack.");
        }
        return S3ObjectLambdaEndpointBuilder.create().endpointOverride(context.endpointOverride()).accountId(s3EndpointResource.accountId().get()).region(s3EndpointResource.region().get()).accessPointName(s3EndpointResource.accessPointName()).protocol(context.request().protocol()).fipsEnabled(S3AccessPointEndpointResolver.isFipsEnabled(context)).dualstackEnabled(S3EndpointUtils.isDualstackEnabled(context.serviceConfiguration())).domain(clientPartitionMetadata.dnsSuffix()).toUri();
    }

    private static Optional<String> resolveSigningService(S3Resource resource) {
        if (resource instanceof S3OutpostResource) {
            return Optional.of(S3_OUTPOSTS_NAME);
        }
        if (resource instanceof S3ObjectLambdaResource) {
            return Optional.of(S3_OBJECT_LAMBDA_NAME);
        }
        return Optional.empty();
    }

    private static boolean isFipsEnabled(S3EndpointResolverContext context) {
        return context.fipsEnabled() || S3EndpointUtils.isFipsRegion(context.region().toString());
    }
}

