/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.services.s3.internal;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.List;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.arns.Arn;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.http.SdkHttpRequest;
import software.amazon.awssdk.regions.PartitionMetadata;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.regions.RegionMetadata;
import software.amazon.awssdk.services.s3.S3Configuration;
import software.amazon.awssdk.services.s3.internal.BucketUtils;
import software.amazon.awssdk.services.s3.internal.resource.S3AccessPointBuilder;
import software.amazon.awssdk.services.s3.internal.resource.S3AccessPointResource;
import software.amazon.awssdk.services.s3.internal.resource.S3ArnConverter;
import software.amazon.awssdk.services.s3.internal.resource.S3Resource;
import software.amazon.awssdk.services.s3.internal.resource.S3ResourceType;
import software.amazon.awssdk.services.s3.model.CreateBucketRequest;
import software.amazon.awssdk.services.s3.model.DeleteBucketRequest;
import software.amazon.awssdk.services.s3.model.ListBucketsRequest;
import software.amazon.awssdk.utils.FunctionalUtils;
import software.amazon.awssdk.utils.Validate;

@SdkInternalApi
public final class S3EndpointUtils {
    private static final List<Class<?>> ACCELERATE_DISABLED_OPERATIONS = Arrays.asList(ListBucketsRequest.class, CreateBucketRequest.class, DeleteBucketRequest.class);

    private S3EndpointUtils() {
    }

    public static SdkHttpRequest applyEndpointConfiguration(SdkHttpRequest request, SdkRequest originalRequest, Region region, S3Configuration serviceConfiguration, boolean endpointOverridden) {
        String bucketName = originalRequest.getValueForField("Bucket", String.class).orElse(null);
        String key = originalRequest.getValueForField("Key", String.class).orElse(null);
        if (bucketName != null && S3EndpointUtils.isArn(bucketName)) {
            return S3EndpointUtils.applyEndpointConfigurationForAccessPointArn(request, region, endpointOverridden, serviceConfiguration, bucketName, key);
        }
        SdkHttpRequest.Builder mutableRequest = (SdkHttpRequest.Builder)request.toBuilder();
        URI endpoint = S3EndpointUtils.resolveEndpoint(request, originalRequest, region, serviceConfiguration);
        mutableRequest.uri(endpoint);
        if ((serviceConfiguration == null || !serviceConfiguration.pathStyleAccessEnabled()) && bucketName != null && BucketUtils.isVirtualAddressingCompatibleBucketName(bucketName, false)) {
            S3EndpointUtils.changeToDnsEndpoint(mutableRequest, bucketName);
        }
        return (SdkHttpRequest)mutableRequest.build();
    }

    private static SdkHttpRequest applyEndpointConfigurationForAccessPointArn(SdkHttpRequest request, Region region, boolean endpointOverridden, S3Configuration serviceConfiguration, String bucketName, String key) {
        Arn resourceArn = Arn.fromString((String)bucketName);
        S3Resource s3Resource = S3ArnConverter.create().convertArn(resourceArn);
        if (S3ResourceType.fromValue(s3Resource.type()) != S3ResourceType.ACCESS_POINT) {
            throw new IllegalArgumentException("An ARN was passed as a bucket parameter to an S3 operation, however it does not appear to be a valid S3 access point ARN.");
        }
        String arnRegion = (String)resourceArn.region().orElseThrow(() -> new IllegalArgumentException("An S3 access point ARN must have a region"));
        if (S3EndpointUtils.isFipsRegion(region.toString())) {
            throw new IllegalArgumentException("An access point ARN cannot be passed as a bucket parameter to an S3 operation if the S3 client has been configured with a FIPS enabled region.");
        }
        if (serviceConfiguration != null && serviceConfiguration.accelerateModeEnabled()) {
            throw new IllegalArgumentException("An access point ARN cannot be passed as a bucket parameter to an S3 operation if the S3 client has been configured with accelerate mode enabled.");
        }
        if (serviceConfiguration != null && serviceConfiguration.pathStyleAccessEnabled()) {
            throw new IllegalArgumentException("An access point ARN cannot be passed as a bucket parameter to an S3 operation if the S3 client has been configured with path style addressing enabled.");
        }
        if (endpointOverridden) {
            throw new IllegalArgumentException("An access point ARN cannot be passed as a bucket parameter to an S3 operation if the S3 client has been configured with an endpoint override.");
        }
        if (!(serviceConfiguration != null && serviceConfiguration.useArnRegionEnabled() || region.id().equals(arnRegion))) {
            throw new IllegalArgumentException(String.format("The region field of the ARN being passed as a bucket parameter to an S3 operation does not match the region the client was configured with. To enable this behavior and prevent this exception set 'useArnRegionEnabled' to true in the configuration when building the S3 client. Provided region: '%s'; client region: '%s'.", arnRegion, region));
        }
        PartitionMetadata clientPartitionMetadata = PartitionMetadata.of((Region)region);
        String clientPartition = clientPartitionMetadata.id();
        if (clientPartition == null || clientPartition.isEmpty() || !s3Resource.partition().isPresent() || !clientPartition.equals(s3Resource.partition().get())) {
            throw new IllegalArgumentException(String.format("The partition field of the ARN being passed as a bucket parameter to an S3 operation does not match the partition the S3 client has been configured with. Provided partition: '%s'; client partition: '%s'.", s3Resource.partition().orElse(""), clientPartition));
        }
        S3AccessPointResource s3EndpointResource = (S3AccessPointResource)Validate.isInstanceOf(S3AccessPointResource.class, (Object)s3Resource, (String)"An ARN was passed as a bucket parameter to an S3 operation, however it does not appear to be a valid S3 access point ARN.", (Object[])new Object[0]);
        boolean dualstackEnabled = serviceConfiguration != null && serviceConfiguration.dualstackEnabled();
        URI accessPointUri = S3AccessPointBuilder.create().accessPointName(s3EndpointResource.accessPointName()).accountId(s3EndpointResource.accountId().orElseThrow(() -> new IllegalArgumentException("An S3 access point ARN must have an account ID"))).region(arnRegion).protocol(request.protocol()).domain(clientPartitionMetadata.dnsSuffix()).dualstackEnabled(dualstackEnabled).toUri();
        return (SdkHttpRequest)((SdkHttpRequest.Builder)request.toBuilder()).protocol(accessPointUri.getScheme()).host(accessPointUri.getHost()).port(Integer.valueOf(accessPointUri.getPort())).encodedPath(key).build();
    }

    private static URI resolveEndpoint(SdkHttpRequest request, SdkRequest originalRequest, Region region, S3Configuration serviceConfiguration) {
        String protocol = request.protocol();
        RegionMetadata regionMetadata = RegionMetadata.of((Region)region);
        if (S3EndpointUtils.isAccelerateEnabled(serviceConfiguration) && S3EndpointUtils.isAccelerateSupported(originalRequest)) {
            return S3EndpointUtils.accelerateEndpoint(serviceConfiguration, regionMetadata, protocol);
        }
        if (serviceConfiguration != null && serviceConfiguration.dualstackEnabled()) {
            return S3EndpointUtils.dualstackEndpoint(regionMetadata, protocol);
        }
        return (URI)FunctionalUtils.invokeSafely(() -> new URI(request.protocol(), null, request.host(), request.port(), null, null, null));
    }

    private static void changeToDnsEndpoint(SdkHttpRequest.Builder mutableRequest, String bucketName) {
        if (mutableRequest.host().startsWith("s3")) {
            String newHost = mutableRequest.host().replaceFirst("s3", bucketName + ".s3");
            String newPath = mutableRequest.encodedPath().replaceFirst("/" + bucketName, "");
            mutableRequest.host(newHost).encodedPath(newPath);
        }
    }

    private static URI dualstackEndpoint(RegionMetadata metadata, String protocol) {
        String serviceEndpoint = String.format("%s.%s.%s.%s", "s3", "dualstack", metadata.id(), metadata.domain());
        return S3EndpointUtils.toUri(protocol, serviceEndpoint);
    }

    private static boolean isAccelerateEnabled(S3Configuration serviceConfiguration) {
        return serviceConfiguration != null && serviceConfiguration.accelerateModeEnabled();
    }

    private static boolean isAccelerateSupported(SdkRequest originalRequest) {
        return !ACCELERATE_DISABLED_OPERATIONS.contains(originalRequest.getClass());
    }

    private static URI accelerateEndpoint(S3Configuration serviceConfiguration, RegionMetadata metadata, String protocol) {
        if (serviceConfiguration.dualstackEnabled()) {
            return S3EndpointUtils.toUri(protocol, "s3-accelerate.dualstack." + metadata.domain());
        }
        return S3EndpointUtils.toUri(protocol, "s3-accelerate." + metadata.domain());
    }

    private static URI toUri(String protocol, String endpoint) {
        try {
            return new URI(String.format("%s://%s", protocol, endpoint));
        }
        catch (URISyntaxException e) {
            throw new IllegalArgumentException(e);
        }
    }

    private static boolean isArn(String s) {
        return s.startsWith("arn:");
    }

    private static boolean isFipsRegion(String region) {
        return region.startsWith("fips-") || region.endsWith("-fips");
    }
}

