/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.transfer.s3.internal;

import java.io.IOException;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.SdkTestInternalApi;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.services.s3.model.PutObjectRequest;
import software.amazon.awssdk.transfer.s3.CompletedDirectoryUpload;
import software.amazon.awssdk.transfer.s3.CompletedFileUpload;
import software.amazon.awssdk.transfer.s3.DirectoryUpload;
import software.amazon.awssdk.transfer.s3.FailedFileUpload;
import software.amazon.awssdk.transfer.s3.FileUpload;
import software.amazon.awssdk.transfer.s3.S3TransferManager;
import software.amazon.awssdk.transfer.s3.UploadDirectoryRequest;
import software.amazon.awssdk.transfer.s3.UploadFileRequest;
import software.amazon.awssdk.transfer.s3.internal.DefaultDirectoryUpload;
import software.amazon.awssdk.transfer.s3.internal.TransferConfigurationOption;
import software.amazon.awssdk.transfer.s3.internal.TransferManagerConfiguration;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Logger;
import software.amazon.awssdk.utils.StringUtils;
import software.amazon.awssdk.utils.Validate;

@SdkInternalApi
public class UploadDirectoryHelper {
    private static final Logger log = Logger.loggerFor(S3TransferManager.class);
    private final TransferManagerConfiguration transferConfiguration;
    private final Function<UploadFileRequest, FileUpload> uploadFunction;
    private final FileSystem fileSystem;

    public UploadDirectoryHelper(TransferManagerConfiguration transferConfiguration, Function<UploadFileRequest, FileUpload> uploadFunction) {
        this.transferConfiguration = transferConfiguration;
        this.uploadFunction = uploadFunction;
        this.fileSystem = FileSystems.getDefault();
    }

    @SdkTestInternalApi
    UploadDirectoryHelper(TransferManagerConfiguration transferConfiguration, Function<UploadFileRequest, FileUpload> uploadFunction, FileSystem fileSystem) {
        this.transferConfiguration = transferConfiguration;
        this.uploadFunction = uploadFunction;
        this.fileSystem = fileSystem;
    }

    public DirectoryUpload uploadDirectory(UploadDirectoryRequest uploadDirectoryRequest) {
        CompletableFuture<CompletedDirectoryUpload> returnFuture = new CompletableFuture<CompletedDirectoryUpload>();
        CompletableFuture.runAsync(() -> this.doUploadDirectory(returnFuture, uploadDirectoryRequest), this.transferConfiguration.option(TransferConfigurationOption.EXECUTOR)).whenComplete((r, t) -> {
            if (t != null) {
                returnFuture.completeExceptionally((Throwable)t);
            }
        });
        return new DefaultDirectoryUpload(returnFuture);
    }

    private void doUploadDirectory(CompletableFuture<CompletedDirectoryUpload> returnFuture, UploadDirectoryRequest uploadDirectoryRequest) {
        List<CompletableFuture> futures;
        Path directory = uploadDirectoryRequest.sourceDirectory();
        this.validateDirectory(uploadDirectoryRequest);
        ConcurrentLinkedQueue failedFileUploads = new ConcurrentLinkedQueue();
        try (Stream<Path> entries = this.listFiles(directory, uploadDirectoryRequest);){
            futures = entries.map(path -> {
                CompletableFuture<CompletedFileUpload> future = this.uploadSingleFile(uploadDirectoryRequest, failedFileUploads, (Path)path);
                CompletableFutureUtils.forwardExceptionTo((CompletableFuture)returnFuture, future);
                return future;
            }).collect(Collectors.toList());
        }
        CompletableFuture.allOf(futures.toArray(new CompletableFuture[0])).whenComplete((r, t) -> returnFuture.complete(CompletedDirectoryUpload.builder().failedTransfers(failedFileUploads).build()));
    }

    private void validateDirectory(UploadDirectoryRequest uploadDirectoryRequest) {
        Path directory = uploadDirectoryRequest.sourceDirectory();
        Validate.isTrue((boolean)Files.exists(directory, new LinkOption[0]), (String)"The source directory provided (%s) does not exist", (Object[])new Object[]{directory});
        boolean followSymbolicLinks = this.transferConfiguration.resolveUploadDirectoryFollowSymbolicLinks(uploadDirectoryRequest);
        if (followSymbolicLinks) {
            Validate.isTrue((boolean)Files.isDirectory(directory, new LinkOption[0]), (String)"The source directory provided (%s) is not a directory", (Object[])new Object[]{directory});
        } else {
            Validate.isTrue((boolean)Files.isDirectory(directory, LinkOption.NOFOLLOW_LINKS), (String)"The source directory provided (%s) is not a directory", (Object[])new Object[]{directory});
        }
    }

    private CompletableFuture<CompletedFileUpload> uploadSingleFile(UploadDirectoryRequest uploadDirectoryRequest, Collection<FailedFileUpload> failedFileUploads, Path path) {
        int nameCount = uploadDirectoryRequest.sourceDirectory().getNameCount();
        UploadFileRequest uploadFileRequest = this.constructUploadRequest(uploadDirectoryRequest, nameCount, path);
        log.debug(() -> String.format("Sending upload request (%s) for path (%s)", uploadFileRequest, path));
        CompletableFuture<CompletedFileUpload> future = this.uploadFunction.apply(uploadFileRequest).completionFuture();
        future.whenComplete((r, t) -> {
            if (t != null) {
                failedFileUploads.add((FailedFileUpload)FailedFileUpload.builder().exception((Throwable)t).request(uploadFileRequest).build());
            }
        });
        return future;
    }

    private Stream<Path> listFiles(Path directory, UploadDirectoryRequest request) {
        try {
            boolean recursive = this.transferConfiguration.resolveUploadDirectoryRecursive(request);
            boolean followSymbolicLinks = this.transferConfiguration.resolveUploadDirectoryFollowSymbolicLinks(request);
            if (!recursive) {
                return Files.list(directory).filter(p -> this.isRegularFile((Path)p, followSymbolicLinks));
            }
            int maxDepth = this.transferConfiguration.resolveUploadDirectoryMaxDepth(request);
            if (followSymbolicLinks) {
                return Files.walk(directory, maxDepth, FileVisitOption.FOLLOW_LINKS).filter(path -> this.isRegularFile((Path)path, true));
            }
            return Files.walk(directory, maxDepth, new FileVisitOption[0]).filter(path -> this.isRegularFile((Path)path, false));
        }
        catch (IOException e) {
            throw SdkClientException.create((String)("Failed to list files within the provided directory: " + directory), (Throwable)e);
        }
    }

    private boolean isRegularFile(Path path, boolean followSymlinks) {
        if (followSymlinks) {
            return Files.isRegularFile(path, new LinkOption[0]);
        }
        return Files.isRegularFile(path, LinkOption.NOFOLLOW_LINKS);
    }

    private static String normalizePrefix(String prefix, String delimiter) {
        if (StringUtils.isEmpty((CharSequence)prefix)) {
            return "";
        }
        return prefix.endsWith(delimiter) ? prefix : prefix + delimiter;
    }

    private String getRelativePathName(int directoryNameCount, Path path, String delimiter) {
        String relativePathName = path.subpath(directoryNameCount, path.getNameCount()).toString();
        String separator = this.fileSystem.getSeparator();
        if (delimiter.equals(separator)) {
            return relativePathName;
        }
        return relativePathName.replace(separator, delimiter);
    }

    private UploadFileRequest constructUploadRequest(UploadDirectoryRequest uploadDirectoryRequest, int directoryNameCount, Path path) {
        String delimiter = uploadDirectoryRequest.delimiter().filter(s -> !s.isEmpty()).orElse("/");
        String prefix = uploadDirectoryRequest.prefix().map(s -> UploadDirectoryHelper.normalizePrefix(s, delimiter)).orElse("");
        String relativePathName = this.getRelativePathName(directoryNameCount, path, delimiter);
        String key = prefix + relativePathName;
        PutObjectRequest putObjectRequest = (PutObjectRequest)PutObjectRequest.builder().bucket(uploadDirectoryRequest.bucket()).key(key).build();
        return (UploadFileRequest)UploadFileRequest.builder().source(path).putObjectRequest(putObjectRequest).build();
    }
}

