/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rum.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A RUM app monitor collects telemetry data from your application and sends that data to RUM. The data includes
 * performance and reliability information such as page load time, client-side errors, and user behavior.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AppMonitor implements SdkPojo, Serializable, ToCopyableBuilder<AppMonitor.Builder, AppMonitor> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(AppMonitor::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DOMAIN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Domain")
            .getter(getter(AppMonitor::domain)).setter(setter(Builder::domain))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Domain").build()).build();

    private static final SdkField<List<String>> DOMAIN_LIST_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DomainList")
            .getter(getter(AppMonitor::domainList))
            .setter(setter(Builder::domainList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(AppMonitor::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> CREATED_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Created")
            .getter(getter(AppMonitor::created)).setter(setter(Builder::created))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Created").build()).build();

    private static final SdkField<String> LAST_MODIFIED_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LastModified").getter(getter(AppMonitor::lastModified)).setter(setter(Builder::lastModified))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModified").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Tags")
            .getter(getter(AppMonitor::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(AppMonitor::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<AppMonitorConfiguration> APP_MONITOR_CONFIGURATION_FIELD = SdkField
            .<AppMonitorConfiguration> builder(MarshallingType.SDK_POJO).memberName("AppMonitorConfiguration")
            .getter(getter(AppMonitor::appMonitorConfiguration)).setter(setter(Builder::appMonitorConfiguration))
            .constructor(AppMonitorConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AppMonitorConfiguration").build())
            .build();

    private static final SdkField<DataStorage> DATA_STORAGE_FIELD = SdkField.<DataStorage> builder(MarshallingType.SDK_POJO)
            .memberName("DataStorage").getter(getter(AppMonitor::dataStorage)).setter(setter(Builder::dataStorage))
            .constructor(DataStorage::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataStorage").build()).build();

    private static final SdkField<CustomEvents> CUSTOM_EVENTS_FIELD = SdkField.<CustomEvents> builder(MarshallingType.SDK_POJO)
            .memberName("CustomEvents").getter(getter(AppMonitor::customEvents)).setter(setter(Builder::customEvents))
            .constructor(CustomEvents::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomEvents").build()).build();

    private static final SdkField<DeobfuscationConfiguration> DEOBFUSCATION_CONFIGURATION_FIELD = SdkField
            .<DeobfuscationConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("DeobfuscationConfiguration")
            .getter(getter(AppMonitor::deobfuscationConfiguration))
            .setter(setter(Builder::deobfuscationConfiguration))
            .constructor(DeobfuscationConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeobfuscationConfiguration").build())
            .build();

    private static final SdkField<String> PLATFORM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Platform").getter(getter(AppMonitor::platformAsString)).setter(setter(Builder::platform))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Platform").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, DOMAIN_FIELD,
            DOMAIN_LIST_FIELD, ID_FIELD, CREATED_FIELD, LAST_MODIFIED_FIELD, TAGS_FIELD, STATE_FIELD,
            APP_MONITOR_CONFIGURATION_FIELD, DATA_STORAGE_FIELD, CUSTOM_EVENTS_FIELD, DEOBFUSCATION_CONFIGURATION_FIELD,
            PLATFORM_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String domain;

    private final List<String> domainList;

    private final String id;

    private final String created;

    private final String lastModified;

    private final Map<String, String> tags;

    private final String state;

    private final AppMonitorConfiguration appMonitorConfiguration;

    private final DataStorage dataStorage;

    private final CustomEvents customEvents;

    private final DeobfuscationConfiguration deobfuscationConfiguration;

    private final String platform;

    private AppMonitor(BuilderImpl builder) {
        this.name = builder.name;
        this.domain = builder.domain;
        this.domainList = builder.domainList;
        this.id = builder.id;
        this.created = builder.created;
        this.lastModified = builder.lastModified;
        this.tags = builder.tags;
        this.state = builder.state;
        this.appMonitorConfiguration = builder.appMonitorConfiguration;
        this.dataStorage = builder.dataStorage;
        this.customEvents = builder.customEvents;
        this.deobfuscationConfiguration = builder.deobfuscationConfiguration;
        this.platform = builder.platform;
    }

    /**
     * <p>
     * The name of the app monitor.
     * </p>
     * 
     * @return The name of the app monitor.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The top-level internet domain name for which your application has administrative authority.
     * </p>
     * 
     * @return The top-level internet domain name for which your application has administrative authority.
     */
    public final String domain() {
        return domain;
    }

    /**
     * For responses, this returns true if the service returned a value for the DomainList property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDomainList() {
        return domainList != null && !(domainList instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List the domain names for which your application has administrative authority.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDomainList} method.
     * </p>
     * 
     * @return List the domain names for which your application has administrative authority.
     */
    public final List<String> domainList() {
        return domainList;
    }

    /**
     * <p>
     * The unique ID of this app monitor.
     * </p>
     * 
     * @return The unique ID of this app monitor.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The date and time that this app monitor was created.
     * </p>
     * 
     * @return The date and time that this app monitor was created.
     */
    public final String created() {
        return created;
    }

    /**
     * <p>
     * The date and time of the most recent changes to this app monitor's configuration.
     * </p>
     * 
     * @return The date and time of the most recent changes to this app monitor's configuration.
     */
    public final String lastModified() {
        return lastModified;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The list of tag keys and values associated with this app monitor.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The list of tag keys and values associated with this app monitor.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The current state of the app monitor.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link StateEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the app monitor.
     * @see StateEnum
     */
    public final StateEnum state() {
        return StateEnum.fromValue(state);
    }

    /**
     * <p>
     * The current state of the app monitor.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link StateEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the app monitor.
     * @see StateEnum
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * A structure that contains much of the configuration data for the app monitor.
     * </p>
     * 
     * @return A structure that contains much of the configuration data for the app monitor.
     */
    public final AppMonitorConfiguration appMonitorConfiguration() {
        return appMonitorConfiguration;
    }

    /**
     * <p>
     * A structure that contains information about whether this app monitor stores a copy of the telemetry data that RUM
     * collects using CloudWatch Logs.
     * </p>
     * 
     * @return A structure that contains information about whether this app monitor stores a copy of the telemetry data
     *         that RUM collects using CloudWatch Logs.
     */
    public final DataStorage dataStorage() {
        return dataStorage;
    }

    /**
     * <p>
     * Specifies whether this app monitor allows the web client to define and send custom events.
     * </p>
     * <p>
     * For more information about custom events, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-custom-events.html">Send
     * custom events</a>.
     * </p>
     * 
     * @return Specifies whether this app monitor allows the web client to define and send custom events.</p>
     *         <p>
     *         For more information about custom events, see <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-custom-events.html"
     *         >Send custom events</a>.
     */
    public final CustomEvents customEvents() {
        return customEvents;
    }

    /**
     * <p>
     * A structure that contains the configuration for how an app monitor can deobfuscate stack traces.
     * </p>
     * 
     * @return A structure that contains the configuration for how an app monitor can deobfuscate stack traces.
     */
    public final DeobfuscationConfiguration deobfuscationConfiguration() {
        return deobfuscationConfiguration;
    }

    /**
     * <p>
     * The platform type for this app monitor. Valid values are <code>Web</code> for web applications ,
     * <code>Android</code> for Android applications, and <code>iOS</code> for IOS applications.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #platform} will
     * return {@link AppMonitorPlatform#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #platformAsString}.
     * </p>
     * 
     * @return The platform type for this app monitor. Valid values are <code>Web</code> for web applications ,
     *         <code>Android</code> for Android applications, and <code>iOS</code> for IOS applications.
     * @see AppMonitorPlatform
     */
    public final AppMonitorPlatform platform() {
        return AppMonitorPlatform.fromValue(platform);
    }

    /**
     * <p>
     * The platform type for this app monitor. Valid values are <code>Web</code> for web applications ,
     * <code>Android</code> for Android applications, and <code>iOS</code> for IOS applications.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #platform} will
     * return {@link AppMonitorPlatform#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #platformAsString}.
     * </p>
     * 
     * @return The platform type for this app monitor. Valid values are <code>Web</code> for web applications ,
     *         <code>Android</code> for Android applications, and <code>iOS</code> for IOS applications.
     * @see AppMonitorPlatform
     */
    public final String platformAsString() {
        return platform;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(domain());
        hashCode = 31 * hashCode + Objects.hashCode(hasDomainList() ? domainList() : null);
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(created());
        hashCode = 31 * hashCode + Objects.hashCode(lastModified());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(appMonitorConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(dataStorage());
        hashCode = 31 * hashCode + Objects.hashCode(customEvents());
        hashCode = 31 * hashCode + Objects.hashCode(deobfuscationConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(platformAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AppMonitor)) {
            return false;
        }
        AppMonitor other = (AppMonitor) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(domain(), other.domain())
                && hasDomainList() == other.hasDomainList() && Objects.equals(domainList(), other.domainList())
                && Objects.equals(id(), other.id()) && Objects.equals(created(), other.created())
                && Objects.equals(lastModified(), other.lastModified()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(appMonitorConfiguration(), other.appMonitorConfiguration())
                && Objects.equals(dataStorage(), other.dataStorage()) && Objects.equals(customEvents(), other.customEvents())
                && Objects.equals(deobfuscationConfiguration(), other.deobfuscationConfiguration())
                && Objects.equals(platformAsString(), other.platformAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AppMonitor").add("Name", name()).add("Domain", domain())
                .add("DomainList", hasDomainList() ? domainList() : null).add("Id", id()).add("Created", created())
                .add("LastModified", lastModified()).add("Tags", hasTags() ? tags() : null).add("State", stateAsString())
                .add("AppMonitorConfiguration", appMonitorConfiguration()).add("DataStorage", dataStorage())
                .add("CustomEvents", customEvents()).add("DeobfuscationConfiguration", deobfuscationConfiguration())
                .add("Platform", platformAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Domain":
            return Optional.ofNullable(clazz.cast(domain()));
        case "DomainList":
            return Optional.ofNullable(clazz.cast(domainList()));
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Created":
            return Optional.ofNullable(clazz.cast(created()));
        case "LastModified":
            return Optional.ofNullable(clazz.cast(lastModified()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "AppMonitorConfiguration":
            return Optional.ofNullable(clazz.cast(appMonitorConfiguration()));
        case "DataStorage":
            return Optional.ofNullable(clazz.cast(dataStorage()));
        case "CustomEvents":
            return Optional.ofNullable(clazz.cast(customEvents()));
        case "DeobfuscationConfiguration":
            return Optional.ofNullable(clazz.cast(deobfuscationConfiguration()));
        case "Platform":
            return Optional.ofNullable(clazz.cast(platformAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("Domain", DOMAIN_FIELD);
        map.put("DomainList", DOMAIN_LIST_FIELD);
        map.put("Id", ID_FIELD);
        map.put("Created", CREATED_FIELD);
        map.put("LastModified", LAST_MODIFIED_FIELD);
        map.put("Tags", TAGS_FIELD);
        map.put("State", STATE_FIELD);
        map.put("AppMonitorConfiguration", APP_MONITOR_CONFIGURATION_FIELD);
        map.put("DataStorage", DATA_STORAGE_FIELD);
        map.put("CustomEvents", CUSTOM_EVENTS_FIELD);
        map.put("DeobfuscationConfiguration", DEOBFUSCATION_CONFIGURATION_FIELD);
        map.put("Platform", PLATFORM_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AppMonitor, T> g) {
        return obj -> g.apply((AppMonitor) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AppMonitor> {
        /**
         * <p>
         * The name of the app monitor.
         * </p>
         * 
         * @param name
         *        The name of the app monitor.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The top-level internet domain name for which your application has administrative authority.
         * </p>
         * 
         * @param domain
         *        The top-level internet domain name for which your application has administrative authority.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domain(String domain);

        /**
         * <p>
         * List the domain names for which your application has administrative authority.
         * </p>
         * 
         * @param domainList
         *        List the domain names for which your application has administrative authority.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainList(Collection<String> domainList);

        /**
         * <p>
         * List the domain names for which your application has administrative authority.
         * </p>
         * 
         * @param domainList
         *        List the domain names for which your application has administrative authority.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainList(String... domainList);

        /**
         * <p>
         * The unique ID of this app monitor.
         * </p>
         * 
         * @param id
         *        The unique ID of this app monitor.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The date and time that this app monitor was created.
         * </p>
         * 
         * @param created
         *        The date and time that this app monitor was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder created(String created);

        /**
         * <p>
         * The date and time of the most recent changes to this app monitor's configuration.
         * </p>
         * 
         * @param lastModified
         *        The date and time of the most recent changes to this app monitor's configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModified(String lastModified);

        /**
         * <p>
         * The list of tag keys and values associated with this app monitor.
         * </p>
         * 
         * @param tags
         *        The list of tag keys and values associated with this app monitor.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The current state of the app monitor.
         * </p>
         * 
         * @param state
         *        The current state of the app monitor.
         * @see StateEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StateEnum
         */
        Builder state(String state);

        /**
         * <p>
         * The current state of the app monitor.
         * </p>
         * 
         * @param state
         *        The current state of the app monitor.
         * @see StateEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StateEnum
         */
        Builder state(StateEnum state);

        /**
         * <p>
         * A structure that contains much of the configuration data for the app monitor.
         * </p>
         * 
         * @param appMonitorConfiguration
         *        A structure that contains much of the configuration data for the app monitor.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appMonitorConfiguration(AppMonitorConfiguration appMonitorConfiguration);

        /**
         * <p>
         * A structure that contains much of the configuration data for the app monitor.
         * </p>
         * This is a convenience method that creates an instance of the {@link AppMonitorConfiguration.Builder} avoiding
         * the need to create one manually via {@link AppMonitorConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AppMonitorConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #appMonitorConfiguration(AppMonitorConfiguration)}.
         * 
         * @param appMonitorConfiguration
         *        a consumer that will call methods on {@link AppMonitorConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #appMonitorConfiguration(AppMonitorConfiguration)
         */
        default Builder appMonitorConfiguration(Consumer<AppMonitorConfiguration.Builder> appMonitorConfiguration) {
            return appMonitorConfiguration(AppMonitorConfiguration.builder().applyMutation(appMonitorConfiguration).build());
        }

        /**
         * <p>
         * A structure that contains information about whether this app monitor stores a copy of the telemetry data that
         * RUM collects using CloudWatch Logs.
         * </p>
         * 
         * @param dataStorage
         *        A structure that contains information about whether this app monitor stores a copy of the telemetry
         *        data that RUM collects using CloudWatch Logs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataStorage(DataStorage dataStorage);

        /**
         * <p>
         * A structure that contains information about whether this app monitor stores a copy of the telemetry data that
         * RUM collects using CloudWatch Logs.
         * </p>
         * This is a convenience method that creates an instance of the {@link DataStorage.Builder} avoiding the need to
         * create one manually via {@link DataStorage#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DataStorage.Builder#build()} is called immediately and its result
         * is passed to {@link #dataStorage(DataStorage)}.
         * 
         * @param dataStorage
         *        a consumer that will call methods on {@link DataStorage.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dataStorage(DataStorage)
         */
        default Builder dataStorage(Consumer<DataStorage.Builder> dataStorage) {
            return dataStorage(DataStorage.builder().applyMutation(dataStorage).build());
        }

        /**
         * <p>
         * Specifies whether this app monitor allows the web client to define and send custom events.
         * </p>
         * <p>
         * For more information about custom events, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-custom-events.html">Send
         * custom events</a>.
         * </p>
         * 
         * @param customEvents
         *        Specifies whether this app monitor allows the web client to define and send custom events.</p>
         *        <p>
         *        For more information about custom events, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-custom-events.html"
         *        >Send custom events</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customEvents(CustomEvents customEvents);

        /**
         * <p>
         * Specifies whether this app monitor allows the web client to define and send custom events.
         * </p>
         * <p>
         * For more information about custom events, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-custom-events.html">Send
         * custom events</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link CustomEvents.Builder} avoiding the need
         * to create one manually via {@link CustomEvents#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CustomEvents.Builder#build()} is called immediately and its
         * result is passed to {@link #customEvents(CustomEvents)}.
         * 
         * @param customEvents
         *        a consumer that will call methods on {@link CustomEvents.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #customEvents(CustomEvents)
         */
        default Builder customEvents(Consumer<CustomEvents.Builder> customEvents) {
            return customEvents(CustomEvents.builder().applyMutation(customEvents).build());
        }

        /**
         * <p>
         * A structure that contains the configuration for how an app monitor can deobfuscate stack traces.
         * </p>
         * 
         * @param deobfuscationConfiguration
         *        A structure that contains the configuration for how an app monitor can deobfuscate stack traces.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deobfuscationConfiguration(DeobfuscationConfiguration deobfuscationConfiguration);

        /**
         * <p>
         * A structure that contains the configuration for how an app monitor can deobfuscate stack traces.
         * </p>
         * This is a convenience method that creates an instance of the {@link DeobfuscationConfiguration.Builder}
         * avoiding the need to create one manually via {@link DeobfuscationConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DeobfuscationConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #deobfuscationConfiguration(DeobfuscationConfiguration)}.
         * 
         * @param deobfuscationConfiguration
         *        a consumer that will call methods on {@link DeobfuscationConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #deobfuscationConfiguration(DeobfuscationConfiguration)
         */
        default Builder deobfuscationConfiguration(Consumer<DeobfuscationConfiguration.Builder> deobfuscationConfiguration) {
            return deobfuscationConfiguration(DeobfuscationConfiguration.builder().applyMutation(deobfuscationConfiguration)
                    .build());
        }

        /**
         * <p>
         * The platform type for this app monitor. Valid values are <code>Web</code> for web applications ,
         * <code>Android</code> for Android applications, and <code>iOS</code> for IOS applications.
         * </p>
         * 
         * @param platform
         *        The platform type for this app monitor. Valid values are <code>Web</code> for web applications ,
         *        <code>Android</code> for Android applications, and <code>iOS</code> for IOS applications.
         * @see AppMonitorPlatform
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AppMonitorPlatform
         */
        Builder platform(String platform);

        /**
         * <p>
         * The platform type for this app monitor. Valid values are <code>Web</code> for web applications ,
         * <code>Android</code> for Android applications, and <code>iOS</code> for IOS applications.
         * </p>
         * 
         * @param platform
         *        The platform type for this app monitor. Valid values are <code>Web</code> for web applications ,
         *        <code>Android</code> for Android applications, and <code>iOS</code> for IOS applications.
         * @see AppMonitorPlatform
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AppMonitorPlatform
         */
        Builder platform(AppMonitorPlatform platform);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String domain;

        private List<String> domainList = DefaultSdkAutoConstructList.getInstance();

        private String id;

        private String created;

        private String lastModified;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private String state;

        private AppMonitorConfiguration appMonitorConfiguration;

        private DataStorage dataStorage;

        private CustomEvents customEvents;

        private DeobfuscationConfiguration deobfuscationConfiguration;

        private String platform;

        private BuilderImpl() {
        }

        private BuilderImpl(AppMonitor model) {
            name(model.name);
            domain(model.domain);
            domainList(model.domainList);
            id(model.id);
            created(model.created);
            lastModified(model.lastModified);
            tags(model.tags);
            state(model.state);
            appMonitorConfiguration(model.appMonitorConfiguration);
            dataStorage(model.dataStorage);
            customEvents(model.customEvents);
            deobfuscationConfiguration(model.deobfuscationConfiguration);
            platform(model.platform);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDomain() {
            return domain;
        }

        public final void setDomain(String domain) {
            this.domain = domain;
        }

        @Override
        public final Builder domain(String domain) {
            this.domain = domain;
            return this;
        }

        public final Collection<String> getDomainList() {
            if (domainList instanceof SdkAutoConstructList) {
                return null;
            }
            return domainList;
        }

        public final void setDomainList(Collection<String> domainList) {
            this.domainList = AppMonitorDomainListCopier.copy(domainList);
        }

        @Override
        public final Builder domainList(Collection<String> domainList) {
            this.domainList = AppMonitorDomainListCopier.copy(domainList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder domainList(String... domainList) {
            domainList(Arrays.asList(domainList));
            return this;
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getCreated() {
            return created;
        }

        public final void setCreated(String created) {
            this.created = created;
        }

        @Override
        public final Builder created(String created) {
            this.created = created;
            return this;
        }

        public final String getLastModified() {
            return lastModified;
        }

        public final void setLastModified(String lastModified) {
            this.lastModified = lastModified;
        }

        @Override
        public final Builder lastModified(String lastModified) {
            this.lastModified = lastModified;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(StateEnum state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final AppMonitorConfiguration.Builder getAppMonitorConfiguration() {
            return appMonitorConfiguration != null ? appMonitorConfiguration.toBuilder() : null;
        }

        public final void setAppMonitorConfiguration(AppMonitorConfiguration.BuilderImpl appMonitorConfiguration) {
            this.appMonitorConfiguration = appMonitorConfiguration != null ? appMonitorConfiguration.build() : null;
        }

        @Override
        public final Builder appMonitorConfiguration(AppMonitorConfiguration appMonitorConfiguration) {
            this.appMonitorConfiguration = appMonitorConfiguration;
            return this;
        }

        public final DataStorage.Builder getDataStorage() {
            return dataStorage != null ? dataStorage.toBuilder() : null;
        }

        public final void setDataStorage(DataStorage.BuilderImpl dataStorage) {
            this.dataStorage = dataStorage != null ? dataStorage.build() : null;
        }

        @Override
        public final Builder dataStorage(DataStorage dataStorage) {
            this.dataStorage = dataStorage;
            return this;
        }

        public final CustomEvents.Builder getCustomEvents() {
            return customEvents != null ? customEvents.toBuilder() : null;
        }

        public final void setCustomEvents(CustomEvents.BuilderImpl customEvents) {
            this.customEvents = customEvents != null ? customEvents.build() : null;
        }

        @Override
        public final Builder customEvents(CustomEvents customEvents) {
            this.customEvents = customEvents;
            return this;
        }

        public final DeobfuscationConfiguration.Builder getDeobfuscationConfiguration() {
            return deobfuscationConfiguration != null ? deobfuscationConfiguration.toBuilder() : null;
        }

        public final void setDeobfuscationConfiguration(DeobfuscationConfiguration.BuilderImpl deobfuscationConfiguration) {
            this.deobfuscationConfiguration = deobfuscationConfiguration != null ? deobfuscationConfiguration.build() : null;
        }

        @Override
        public final Builder deobfuscationConfiguration(DeobfuscationConfiguration deobfuscationConfiguration) {
            this.deobfuscationConfiguration = deobfuscationConfiguration;
            return this;
        }

        public final String getPlatform() {
            return platform;
        }

        public final void setPlatform(String platform) {
            this.platform = platform;
        }

        @Override
        public final Builder platform(String platform) {
            this.platform = platform;
            return this;
        }

        @Override
        public final Builder platform(AppMonitorPlatform platform) {
            this.platform(platform == null ? null : platform.toString());
            return this;
        }

        @Override
        public AppMonitor build() {
            return new AppMonitor(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
