/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rum;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.rum.internal.RumServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.rum.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.rum.model.AccessDeniedException;
import software.amazon.awssdk.services.rum.model.BatchCreateRumMetricDefinitionsRequest;
import software.amazon.awssdk.services.rum.model.BatchCreateRumMetricDefinitionsResponse;
import software.amazon.awssdk.services.rum.model.BatchDeleteRumMetricDefinitionsRequest;
import software.amazon.awssdk.services.rum.model.BatchDeleteRumMetricDefinitionsResponse;
import software.amazon.awssdk.services.rum.model.BatchGetRumMetricDefinitionsRequest;
import software.amazon.awssdk.services.rum.model.BatchGetRumMetricDefinitionsResponse;
import software.amazon.awssdk.services.rum.model.ConflictException;
import software.amazon.awssdk.services.rum.model.CreateAppMonitorRequest;
import software.amazon.awssdk.services.rum.model.CreateAppMonitorResponse;
import software.amazon.awssdk.services.rum.model.DeleteAppMonitorRequest;
import software.amazon.awssdk.services.rum.model.DeleteAppMonitorResponse;
import software.amazon.awssdk.services.rum.model.DeleteResourcePolicyRequest;
import software.amazon.awssdk.services.rum.model.DeleteResourcePolicyResponse;
import software.amazon.awssdk.services.rum.model.DeleteRumMetricsDestinationRequest;
import software.amazon.awssdk.services.rum.model.DeleteRumMetricsDestinationResponse;
import software.amazon.awssdk.services.rum.model.GetAppMonitorDataRequest;
import software.amazon.awssdk.services.rum.model.GetAppMonitorDataResponse;
import software.amazon.awssdk.services.rum.model.GetAppMonitorRequest;
import software.amazon.awssdk.services.rum.model.GetAppMonitorResponse;
import software.amazon.awssdk.services.rum.model.GetResourcePolicyRequest;
import software.amazon.awssdk.services.rum.model.GetResourcePolicyResponse;
import software.amazon.awssdk.services.rum.model.InternalServerException;
import software.amazon.awssdk.services.rum.model.InvalidPolicyRevisionIdException;
import software.amazon.awssdk.services.rum.model.ListAppMonitorsRequest;
import software.amazon.awssdk.services.rum.model.ListAppMonitorsResponse;
import software.amazon.awssdk.services.rum.model.ListRumMetricsDestinationsRequest;
import software.amazon.awssdk.services.rum.model.ListRumMetricsDestinationsResponse;
import software.amazon.awssdk.services.rum.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.rum.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.rum.model.MalformedPolicyDocumentException;
import software.amazon.awssdk.services.rum.model.PolicyNotFoundException;
import software.amazon.awssdk.services.rum.model.PolicySizeLimitExceededException;
import software.amazon.awssdk.services.rum.model.PutResourcePolicyRequest;
import software.amazon.awssdk.services.rum.model.PutResourcePolicyResponse;
import software.amazon.awssdk.services.rum.model.PutRumEventsRequest;
import software.amazon.awssdk.services.rum.model.PutRumEventsResponse;
import software.amazon.awssdk.services.rum.model.PutRumMetricsDestinationRequest;
import software.amazon.awssdk.services.rum.model.PutRumMetricsDestinationResponse;
import software.amazon.awssdk.services.rum.model.ResourceNotFoundException;
import software.amazon.awssdk.services.rum.model.RumException;
import software.amazon.awssdk.services.rum.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.rum.model.TagResourceRequest;
import software.amazon.awssdk.services.rum.model.TagResourceResponse;
import software.amazon.awssdk.services.rum.model.ThrottlingException;
import software.amazon.awssdk.services.rum.model.UntagResourceRequest;
import software.amazon.awssdk.services.rum.model.UntagResourceResponse;
import software.amazon.awssdk.services.rum.model.UpdateAppMonitorRequest;
import software.amazon.awssdk.services.rum.model.UpdateAppMonitorResponse;
import software.amazon.awssdk.services.rum.model.UpdateRumMetricDefinitionRequest;
import software.amazon.awssdk.services.rum.model.UpdateRumMetricDefinitionResponse;
import software.amazon.awssdk.services.rum.model.ValidationException;
import software.amazon.awssdk.services.rum.transform.BatchCreateRumMetricDefinitionsRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.BatchDeleteRumMetricDefinitionsRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.BatchGetRumMetricDefinitionsRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.CreateAppMonitorRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.DeleteAppMonitorRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.DeleteResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.DeleteRumMetricsDestinationRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.GetAppMonitorDataRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.GetAppMonitorRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.GetResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.ListAppMonitorsRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.ListRumMetricsDestinationsRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.PutResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.PutRumEventsRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.PutRumMetricsDestinationRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.UpdateAppMonitorRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.UpdateRumMetricDefinitionRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link RumClient}.
 *
 * @see RumClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRumClient implements RumClient {
    private static final Logger log = Logger.loggerFor(DefaultRumClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRumClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "RUM" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Specifies the extended metrics and custom metrics that you want a CloudWatch RUM app monitor to send to a
     * destination. Valid destinations include CloudWatch and Evidently.
     * </p>
     * <p>
     * By default, RUM app monitors send some metrics to CloudWatch. These default metrics are listed in <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-metrics.html">CloudWatch
     * metrics that you can collect with CloudWatch RUM</a>.
     * </p>
     * <p>
     * In addition to these default metrics, you can choose to send extended metrics, custom metrics, or both.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Extended metrics let you send metrics with additional dimensions that aren't included in the default metrics. You
     * can also send extended metrics to both Evidently and CloudWatch. The valid dimension names for the additional
     * dimensions for extended metrics are <code>BrowserName</code>, <code>CountryCode</code>, <code>DeviceType</code>,
     * <code>FileType</code>, <code>OSName</code>, and <code>PageId</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-vended-metrics.html">
     * Extended metrics that you can send to CloudWatch and CloudWatch Evidently</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Custom metrics are metrics that you define. You can send custom metrics to CloudWatch. CloudWatch Evidently, or
     * both. With custom metrics, you can use any metric name and namespace. To derive the metrics, you can use any
     * custom events, built-in events, custom attributes, or default attributes.
     * </p>
     * <p>
     * You can't send custom metrics to the <code>AWS/RUM</code> namespace. You must send custom metrics to a custom
     * namespace that you define. The namespace that you use can't start with <code>AWS/</code>. CloudWatch RUM prepends
     * <code>RUM/CustomMetrics/</code> to the custom namespace that you define, so the final namespace for your metrics
     * in CloudWatch is <code>RUM/CustomMetrics/<i>your-custom-namespace</i> </code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The maximum number of metric definitions that you can specify in one <code>BatchCreateRumMetricDefinitions</code>
     * operation is 200.
     * </p>
     * <p>
     * The maximum number of metric definitions that one destination can contain is 2000.
     * </p>
     * <p>
     * Extended metrics sent to CloudWatch and RUM custom metrics are charged as CloudWatch custom metrics. Each
     * combination of additional dimension name and dimension value counts as a custom metric. For more information, see
     * <a href="https://aws.amazon.com/cloudwatch/pricing/">Amazon CloudWatch Pricing</a>.
     * </p>
     * <p>
     * You must have already created a destination for the metrics before you send them. For more information, see <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_PutRumMetricsDestination.html">
     * PutRumMetricsDestination</a>.
     * </p>
     * <p>
     * If some metric definitions specified in a <code>BatchCreateRumMetricDefinitions</code> operations are not valid,
     * those metric definitions fail and return errors, but all valid metric definitions in the same operation still
     * succeed.
     * </p>
     *
     * @param batchCreateRumMetricDefinitionsRequest
     * @return Result of the BatchCreateRumMetricDefinitions operation returned by the service.
     * @throws ConflictException
     *         This operation attempted to create a resource that already exists.
     * @throws ServiceQuotaExceededException
     *         This request exceeds a service quota.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.BatchCreateRumMetricDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/BatchCreateRumMetricDefinitions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchCreateRumMetricDefinitionsResponse batchCreateRumMetricDefinitions(
            BatchCreateRumMetricDefinitionsRequest batchCreateRumMetricDefinitionsRequest) throws ConflictException,
            ServiceQuotaExceededException, ResourceNotFoundException, InternalServerException, ValidationException,
            ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException, RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchCreateRumMetricDefinitionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchCreateRumMetricDefinitionsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "MalformedPolicyDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MalformedPolicyDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MalformedPolicyDocumentException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InvalidPolicyRevisionIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPolicyRevisionIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPolicyRevisionIdException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "PolicySizeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicySizeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PolicySizeLimitExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "PolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicyNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(PolicyNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchCreateRumMetricDefinitionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                batchCreateRumMetricDefinitionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchCreateRumMetricDefinitions");

            return clientHandler
                    .execute(new ClientExecutionParams<BatchCreateRumMetricDefinitionsRequest, BatchCreateRumMetricDefinitionsResponse>()
                            .withOperationName("BatchCreateRumMetricDefinitions").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(batchCreateRumMetricDefinitionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new BatchCreateRumMetricDefinitionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the specified metrics from being sent to an extended metrics destination.
     * </p>
     * <p>
     * If some metric definition IDs specified in a <code>BatchDeleteRumMetricDefinitions</code> operations are not
     * valid, those metric definitions fail and return errors, but all valid metric definition IDs in the same operation
     * are still deleted.
     * </p>
     * <p>
     * The maximum number of metric definitions that you can specify in one <code>BatchDeleteRumMetricDefinitions</code>
     * operation is 200.
     * </p>
     *
     * @param batchDeleteRumMetricDefinitionsRequest
     * @return Result of the BatchDeleteRumMetricDefinitions operation returned by the service.
     * @throws ConflictException
     *         This operation attempted to create a resource that already exists.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.BatchDeleteRumMetricDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/BatchDeleteRumMetricDefinitions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchDeleteRumMetricDefinitionsResponse batchDeleteRumMetricDefinitions(
            BatchDeleteRumMetricDefinitionsRequest batchDeleteRumMetricDefinitionsRequest) throws ConflictException,
            ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException, AccessDeniedException,
            AwsServiceException, SdkClientException, RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchDeleteRumMetricDefinitionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchDeleteRumMetricDefinitionsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "MalformedPolicyDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MalformedPolicyDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MalformedPolicyDocumentException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InvalidPolicyRevisionIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPolicyRevisionIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPolicyRevisionIdException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "PolicySizeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicySizeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PolicySizeLimitExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "PolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicyNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(PolicyNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchDeleteRumMetricDefinitionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                batchDeleteRumMetricDefinitionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDeleteRumMetricDefinitions");

            return clientHandler
                    .execute(new ClientExecutionParams<BatchDeleteRumMetricDefinitionsRequest, BatchDeleteRumMetricDefinitionsResponse>()
                            .withOperationName("BatchDeleteRumMetricDefinitions").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(batchDeleteRumMetricDefinitionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new BatchDeleteRumMetricDefinitionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the list of metrics and dimensions that a RUM app monitor is sending to a single destination.
     * </p>
     *
     * @param batchGetRumMetricDefinitionsRequest
     * @return Result of the BatchGetRumMetricDefinitions operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.BatchGetRumMetricDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/BatchGetRumMetricDefinitions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchGetRumMetricDefinitionsResponse batchGetRumMetricDefinitions(
            BatchGetRumMetricDefinitionsRequest batchGetRumMetricDefinitionsRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, AccessDeniedException, AwsServiceException, SdkClientException,
            RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchGetRumMetricDefinitionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchGetRumMetricDefinitionsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "MalformedPolicyDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MalformedPolicyDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MalformedPolicyDocumentException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InvalidPolicyRevisionIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPolicyRevisionIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPolicyRevisionIdException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "PolicySizeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicySizeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PolicySizeLimitExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "PolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicyNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(PolicyNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchGetRumMetricDefinitionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetRumMetricDefinitionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetRumMetricDefinitions");

            return clientHandler
                    .execute(new ClientExecutionParams<BatchGetRumMetricDefinitionsRequest, BatchGetRumMetricDefinitionsResponse>()
                            .withOperationName("BatchGetRumMetricDefinitions").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(batchGetRumMetricDefinitionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new BatchGetRumMetricDefinitionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a Amazon CloudWatch RUM app monitor, which collects telemetry data from your application and sends that
     * data to RUM. The data includes performance and reliability information such as page load time, client-side
     * errors, and user behavior.
     * </p>
     * <p>
     * You use this operation only to create a new app monitor. To update an existing app monitor, use <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_UpdateAppMonitor.html"
     * >UpdateAppMonitor</a> instead.
     * </p>
     * <p>
     * After you create an app monitor, sign in to the CloudWatch RUM console to get the JavaScript code snippet to add
     * to your web application. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-find-code-snippet.html">How
     * do I find a code snippet that I've already generated?</a>
     * </p>
     *
     * @param createAppMonitorRequest
     * @return Result of the CreateAppMonitor operation returned by the service.
     * @throws ConflictException
     *         This operation attempted to create a resource that already exists.
     * @throws ServiceQuotaExceededException
     *         This request exceeds a service quota.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.CreateAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/CreateAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateAppMonitorResponse createAppMonitor(CreateAppMonitorRequest createAppMonitorRequest) throws ConflictException,
            ServiceQuotaExceededException, ResourceNotFoundException, InternalServerException, ValidationException,
            ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException, RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAppMonitorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateAppMonitorResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "MalformedPolicyDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MalformedPolicyDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MalformedPolicyDocumentException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InvalidPolicyRevisionIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPolicyRevisionIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPolicyRevisionIdException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "PolicySizeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicySizeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PolicySizeLimitExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "PolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicyNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(PolicyNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAppMonitorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAppMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAppMonitor");

            return clientHandler.execute(new ClientExecutionParams<CreateAppMonitorRequest, CreateAppMonitorResponse>()
                    .withOperationName("CreateAppMonitor").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createAppMonitorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateAppMonitorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an existing app monitor. This immediately stops the collection of data.
     * </p>
     *
     * @param deleteAppMonitorRequest
     * @return Result of the DeleteAppMonitor operation returned by the service.
     * @throws ConflictException
     *         This operation attempted to create a resource that already exists.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.DeleteAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/DeleteAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteAppMonitorResponse deleteAppMonitor(DeleteAppMonitorRequest deleteAppMonitorRequest) throws ConflictException,
            ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException, AccessDeniedException,
            AwsServiceException, SdkClientException, RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAppMonitorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteAppMonitorResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "MalformedPolicyDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MalformedPolicyDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MalformedPolicyDocumentException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InvalidPolicyRevisionIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPolicyRevisionIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPolicyRevisionIdException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "PolicySizeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicySizeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PolicySizeLimitExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "PolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicyNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(PolicyNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAppMonitorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAppMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAppMonitor");

            return clientHandler.execute(new ClientExecutionParams<DeleteAppMonitorRequest, DeleteAppMonitorResponse>()
                    .withOperationName("DeleteAppMonitor").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteAppMonitorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAppMonitorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the association of a resource-based policy from an app monitor.
     * </p>
     *
     * @param deleteResourcePolicyRequest
     * @return Result of the DeleteResourcePolicy operation returned by the service.
     * @throws InvalidPolicyRevisionIdException
     *         The policy revision ID that you provided doeesn't match the latest policy revision ID.
     * @throws ConflictException
     *         This operation attempted to create a resource that already exists.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws PolicyNotFoundException
     *         The resource-based policy doesn't exist on this app monitor.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.DeleteResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/DeleteResourcePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteResourcePolicyResponse deleteResourcePolicy(DeleteResourcePolicyRequest deleteResourcePolicyRequest)
            throws InvalidPolicyRevisionIdException, ConflictException, ResourceNotFoundException, InternalServerException,
            PolicyNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteResourcePolicyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "MalformedPolicyDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MalformedPolicyDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MalformedPolicyDocumentException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InvalidPolicyRevisionIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPolicyRevisionIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPolicyRevisionIdException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "PolicySizeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicySizeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PolicySizeLimitExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "PolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicyNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(PolicyNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourcePolicy");

            return clientHandler.execute(new ClientExecutionParams<DeleteResourcePolicyRequest, DeleteResourcePolicyResponse>()
                    .withOperationName("DeleteResourcePolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteResourcePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteResourcePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a destination for CloudWatch RUM extended metrics, so that the specified app monitor stops sending
     * extended metrics to that destination.
     * </p>
     *
     * @param deleteRumMetricsDestinationRequest
     * @return Result of the DeleteRumMetricsDestination operation returned by the service.
     * @throws ConflictException
     *         This operation attempted to create a resource that already exists.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.DeleteRumMetricsDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/DeleteRumMetricsDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteRumMetricsDestinationResponse deleteRumMetricsDestination(
            DeleteRumMetricsDestinationRequest deleteRumMetricsDestinationRequest) throws ConflictException,
            ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException, AccessDeniedException,
            AwsServiceException, SdkClientException, RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteRumMetricsDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteRumMetricsDestinationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "MalformedPolicyDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MalformedPolicyDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MalformedPolicyDocumentException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InvalidPolicyRevisionIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPolicyRevisionIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPolicyRevisionIdException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "PolicySizeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicySizeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PolicySizeLimitExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "PolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicyNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(PolicyNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRumMetricsDestinationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRumMetricsDestinationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRumMetricsDestination");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteRumMetricsDestinationRequest, DeleteRumMetricsDestinationResponse>()
                            .withOperationName("DeleteRumMetricsDestination").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteRumMetricsDestinationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteRumMetricsDestinationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the complete configuration information for one app monitor.
     * </p>
     *
     * @param getAppMonitorRequest
     * @return Result of the GetAppMonitor operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.GetAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/GetAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetAppMonitorResponse getAppMonitor(GetAppMonitorRequest getAppMonitorRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAppMonitorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAppMonitorResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "MalformedPolicyDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MalformedPolicyDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MalformedPolicyDocumentException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InvalidPolicyRevisionIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPolicyRevisionIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPolicyRevisionIdException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "PolicySizeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicySizeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PolicySizeLimitExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "PolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicyNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(PolicyNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAppMonitorRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAppMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAppMonitor");

            return clientHandler.execute(new ClientExecutionParams<GetAppMonitorRequest, GetAppMonitorResponse>()
                    .withOperationName("GetAppMonitor").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getAppMonitorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAppMonitorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the raw performance events that RUM has collected from your web application, so that you can do your
     * own processing or analysis of this data.
     * </p>
     *
     * @param getAppMonitorDataRequest
     * @return Result of the GetAppMonitorData operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.GetAppMonitorData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/GetAppMonitorData" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetAppMonitorDataResponse getAppMonitorData(GetAppMonitorDataRequest getAppMonitorDataRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAppMonitorDataResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAppMonitorDataResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "MalformedPolicyDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MalformedPolicyDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MalformedPolicyDocumentException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InvalidPolicyRevisionIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPolicyRevisionIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPolicyRevisionIdException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "PolicySizeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicySizeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PolicySizeLimitExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "PolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicyNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(PolicyNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAppMonitorDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAppMonitorDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAppMonitorData");

            return clientHandler.execute(new ClientExecutionParams<GetAppMonitorDataRequest, GetAppMonitorDataResponse>()
                    .withOperationName("GetAppMonitorData").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getAppMonitorDataRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAppMonitorDataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Use this operation to retrieve information about a resource-based policy that is attached to an app monitor.
     * </p>
     *
     * @param getResourcePolicyRequest
     * @return Result of the GetResourcePolicy operation returned by the service.
     * @throws ConflictException
     *         This operation attempted to create a resource that already exists.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws PolicyNotFoundException
     *         The resource-based policy doesn't exist on this app monitor.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.GetResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/GetResourcePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetResourcePolicyResponse getResourcePolicy(GetResourcePolicyRequest getResourcePolicyRequest)
            throws ConflictException, ResourceNotFoundException, InternalServerException, PolicyNotFoundException,
            ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetResourcePolicyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "MalformedPolicyDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MalformedPolicyDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MalformedPolicyDocumentException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InvalidPolicyRevisionIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPolicyRevisionIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPolicyRevisionIdException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "PolicySizeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicySizeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PolicySizeLimitExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "PolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicyNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(PolicyNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourcePolicy");

            return clientHandler.execute(new ClientExecutionParams<GetResourcePolicyRequest, GetResourcePolicyResponse>()
                    .withOperationName("GetResourcePolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getResourcePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourcePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the Amazon CloudWatch RUM app monitors in the account.
     * </p>
     *
     * @param listAppMonitorsRequest
     * @return Result of the ListAppMonitors operation returned by the service.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.ListAppMonitors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/ListAppMonitors" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAppMonitorsResponse listAppMonitors(ListAppMonitorsRequest listAppMonitorsRequest) throws InternalServerException,
            ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAppMonitorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAppMonitorsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "MalformedPolicyDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MalformedPolicyDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MalformedPolicyDocumentException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InvalidPolicyRevisionIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPolicyRevisionIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPolicyRevisionIdException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "PolicySizeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicySizeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PolicySizeLimitExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "PolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicyNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(PolicyNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAppMonitorsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAppMonitorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAppMonitors");

            return clientHandler.execute(new ClientExecutionParams<ListAppMonitorsRequest, ListAppMonitorsResponse>()
                    .withOperationName("ListAppMonitors").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listAppMonitorsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAppMonitorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of destinations that you have created to receive RUM extended metrics, for the specified app
     * monitor.
     * </p>
     * <p>
     * For more information about extended metrics, see <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_AddRumMetrcs.html">AddRumMetrics</a>.
     * </p>
     *
     * @param listRumMetricsDestinationsRequest
     * @return Result of the ListRumMetricsDestinations operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.ListRumMetricsDestinations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/ListRumMetricsDestinations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRumMetricsDestinationsResponse listRumMetricsDestinations(
            ListRumMetricsDestinationsRequest listRumMetricsDestinationsRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, AccessDeniedException, AwsServiceException, SdkClientException,
            RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRumMetricsDestinationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListRumMetricsDestinationsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "MalformedPolicyDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MalformedPolicyDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MalformedPolicyDocumentException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InvalidPolicyRevisionIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPolicyRevisionIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPolicyRevisionIdException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "PolicySizeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicySizeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PolicySizeLimitExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "PolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicyNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(PolicyNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRumMetricsDestinationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRumMetricsDestinationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRumMetricsDestinations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListRumMetricsDestinationsRequest, ListRumMetricsDestinationsResponse>()
                            .withOperationName("ListRumMetricsDestinations").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listRumMetricsDestinationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListRumMetricsDestinationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Displays the tags associated with a CloudWatch RUM resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "MalformedPolicyDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MalformedPolicyDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MalformedPolicyDocumentException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InvalidPolicyRevisionIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPolicyRevisionIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPolicyRevisionIdException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "PolicySizeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicySizeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PolicySizeLimitExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "PolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicyNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(PolicyNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Use this operation to assign a resource-based policy to a CloudWatch RUM app monitor to control access to it.
     * Each app monitor can have one resource-based policy. The maximum size of the policy is 4 KB. To learn more about
     * using resource policies with RUM, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-resource-policies.html">Using
     * resource-based policies with CloudWatch RUM</a>.
     * </p>
     *
     * @param putResourcePolicyRequest
     * @return Result of the PutResourcePolicy operation returned by the service.
     * @throws PolicySizeLimitExceededException
     *         The policy document is too large. The limit is 4 KB.
     * @throws InvalidPolicyRevisionIdException
     *         The policy revision ID that you provided doeesn't match the latest policy revision ID.
     * @throws ConflictException
     *         This operation attempted to create a resource that already exists.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws MalformedPolicyDocumentException
     *         The policy document that you specified is not formatted correctly.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.PutResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/PutResourcePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutResourcePolicyResponse putResourcePolicy(PutResourcePolicyRequest putResourcePolicyRequest)
            throws PolicySizeLimitExceededException, InvalidPolicyRevisionIdException, ConflictException,
            ResourceNotFoundException, InternalServerException, MalformedPolicyDocumentException, ValidationException,
            ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException, RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutResourcePolicyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "MalformedPolicyDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MalformedPolicyDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MalformedPolicyDocumentException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InvalidPolicyRevisionIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPolicyRevisionIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPolicyRevisionIdException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "PolicySizeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicySizeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PolicySizeLimitExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "PolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicyNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(PolicyNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutResourcePolicy");

            return clientHandler.execute(new ClientExecutionParams<PutResourcePolicyRequest, PutResourcePolicyResponse>()
                    .withOperationName("PutResourcePolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putResourcePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutResourcePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Sends telemetry events about your application performance and user behavior to CloudWatch RUM. The code snippet
     * that RUM generates for you to add to your application includes <code>PutRumEvents</code> operations to send this
     * data to RUM.
     * </p>
     * <p>
     * Each <code>PutRumEvents</code> operation can send a batch of events from one user session.
     * </p>
     *
     * @param putRumEventsRequest
     * @return Result of the PutRumEvents operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.PutRumEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/PutRumEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutRumEventsResponse putRumEvents(PutRumEventsRequest putRumEventsRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutRumEventsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutRumEventsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "MalformedPolicyDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MalformedPolicyDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MalformedPolicyDocumentException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InvalidPolicyRevisionIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPolicyRevisionIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPolicyRevisionIdException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "PolicySizeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicySizeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PolicySizeLimitExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "PolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicyNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(PolicyNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putRumEventsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRumEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRumEvents");
            String hostPrefix = "dataplane.";
            String resolvedHostExpression = "dataplane.";

            return clientHandler.execute(new ClientExecutionParams<PutRumEventsRequest, PutRumEventsResponse>()
                    .withOperationName("PutRumEvents").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .hostPrefixExpression(resolvedHostExpression).withRequestConfiguration(clientConfiguration)
                    .withInput(putRumEventsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutRumEventsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates a destination to receive extended metrics from CloudWatch RUM. You can send extended metrics
     * to CloudWatch or to a CloudWatch Evidently experiment.
     * </p>
     * <p>
     * For more information about extended metrics, see <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_BatchCreateRumMetricDefinitions.html"
     * >BatchCreateRumMetricDefinitions</a>.
     * </p>
     *
     * @param putRumMetricsDestinationRequest
     * @return Result of the PutRumMetricsDestination operation returned by the service.
     * @throws ConflictException
     *         This operation attempted to create a resource that already exists.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.PutRumMetricsDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/PutRumMetricsDestination" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutRumMetricsDestinationResponse putRumMetricsDestination(
            PutRumMetricsDestinationRequest putRumMetricsDestinationRequest) throws ConflictException, ResourceNotFoundException,
            InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutRumMetricsDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutRumMetricsDestinationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "MalformedPolicyDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MalformedPolicyDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MalformedPolicyDocumentException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InvalidPolicyRevisionIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPolicyRevisionIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPolicyRevisionIdException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "PolicySizeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicySizeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PolicySizeLimitExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "PolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicyNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(PolicyNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putRumMetricsDestinationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRumMetricsDestinationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRumMetricsDestination");

            return clientHandler
                    .execute(new ClientExecutionParams<PutRumMetricsDestinationRequest, PutRumMetricsDestinationResponse>()
                            .withOperationName("PutRumMetricsDestination").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(putRumMetricsDestinationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutRumMetricsDestinationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified CloudWatch RUM resource. Currently, the only
     * resources that can be tagged app monitors.
     * </p>
     * <p>
     * Tags can help you organize and categorize your resources. You can also use them to scope user permissions by
     * granting a user permission to access or change only resources with certain tag values.
     * </p>
     * <p>
     * Tags don't have any semantic meaning to Amazon Web Services and are interpreted strictly as strings of
     * characters.
     * </p>
     * <p>
     * You can use the <code>TagResource</code> action with a resource that already has tags. If you specify a new tag
     * key for the resource, this tag is appended to the list of tags associated with the alarm. If you specify a tag
     * key that is already associated with the resource, the new tag value that you specify replaces the previous value
     * for that tag.
     * </p>
     * <p>
     * You can associate as many as 50 tags with a resource.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon
     * Web Services resources</a>.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, AwsServiceException, SdkClientException, RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "MalformedPolicyDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MalformedPolicyDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MalformedPolicyDocumentException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InvalidPolicyRevisionIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPolicyRevisionIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPolicyRevisionIdException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "PolicySizeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicySizeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PolicySizeLimitExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "PolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicyNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(PolicyNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes one or more tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, AwsServiceException, SdkClientException, RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "MalformedPolicyDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MalformedPolicyDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MalformedPolicyDocumentException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InvalidPolicyRevisionIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPolicyRevisionIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPolicyRevisionIdException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "PolicySizeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicySizeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PolicySizeLimitExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "PolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicyNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(PolicyNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the configuration of an existing app monitor. When you use this operation, only the parts of the app
     * monitor configuration that you specify in this operation are changed. For any parameters that you omit, the
     * existing values are kept.
     * </p>
     * <p>
     * You can't use this operation to change the tags of an existing app monitor. To change the tags of an existing app
     * monitor, use <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_TagResource.html">TagResource</a>.
     * </p>
     * <p>
     * To create a new app monitor, use <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_CreateAppMonitor.html"
     * >CreateAppMonitor</a>.
     * </p>
     * <p>
     * After you update an app monitor, sign in to the CloudWatch RUM console to get the updated JavaScript code snippet
     * to add to your web application. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-find-code-snippet.html">How
     * do I find a code snippet that I've already generated?</a>
     * </p>
     *
     * @param updateAppMonitorRequest
     * @return Result of the UpdateAppMonitor operation returned by the service.
     * @throws ConflictException
     *         This operation attempted to create a resource that already exists.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.UpdateAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/UpdateAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateAppMonitorResponse updateAppMonitor(UpdateAppMonitorRequest updateAppMonitorRequest) throws ConflictException,
            ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException, AccessDeniedException,
            AwsServiceException, SdkClientException, RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateAppMonitorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateAppMonitorResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "MalformedPolicyDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MalformedPolicyDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MalformedPolicyDocumentException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InvalidPolicyRevisionIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPolicyRevisionIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPolicyRevisionIdException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "PolicySizeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicySizeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PolicySizeLimitExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "PolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicyNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(PolicyNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAppMonitorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAppMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAppMonitor");

            return clientHandler.execute(new ClientExecutionParams<UpdateAppMonitorRequest, UpdateAppMonitorResponse>()
                    .withOperationName("UpdateAppMonitor").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateAppMonitorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateAppMonitorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Modifies one existing metric definition for CloudWatch RUM extended metrics. For more information about extended
     * metrics, see <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_BatchCreateRumMetricsDefinitions.html"
     * >BatchCreateRumMetricsDefinitions</a>.
     * </p>
     *
     * @param updateRumMetricDefinitionRequest
     * @return Result of the UpdateRumMetricDefinition operation returned by the service.
     * @throws ConflictException
     *         This operation attempted to create a resource that already exists.
     * @throws ServiceQuotaExceededException
     *         This request exceeds a service quota.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws InternalServerException
     *         Internal service exception.
     * @throws ValidationException
     *         One of the arguments for the request is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RumException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RumClient.UpdateRumMetricDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/UpdateRumMetricDefinition" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateRumMetricDefinitionResponse updateRumMetricDefinition(
            UpdateRumMetricDefinitionRequest updateRumMetricDefinitionRequest) throws ConflictException,
            ServiceQuotaExceededException, ResourceNotFoundException, InternalServerException, ValidationException,
            ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException, RumException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateRumMetricDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateRumMetricDefinitionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "MalformedPolicyDocumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MalformedPolicyDocumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MalformedPolicyDocumentException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InvalidPolicyRevisionIdException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPolicyRevisionIdException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPolicyRevisionIdException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "PolicySizeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicySizeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PolicySizeLimitExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "PolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PolicyNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(PolicyNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateRumMetricDefinitionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRumMetricDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRumMetricDefinition");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateRumMetricDefinitionRequest, UpdateRumMetricDefinitionResponse>()
                            .withOperationName("UpdateRumMetricDefinition").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateRumMetricDefinitionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateRumMetricDefinitionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        RumServiceClientConfigurationBuilder serviceConfigBuilder = new RumServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(RumException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    @Override
    public final RumServiceClientConfiguration serviceClientConfiguration() {
        return new RumServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
