/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rum.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Use this structure to define one extended metric or custom metric that RUM will send to CloudWatch or CloudWatch
 * Evidently. For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-vended-metrics.html"> Additional
 * metrics that you can send to CloudWatch and CloudWatch Evidently</a>.
 * </p>
 * <p>
 * This structure is validated differently for extended metrics and custom metrics. For extended metrics that are sent
 * to the <code>AWS/RUM</code> namespace, the following validations apply:
 * </p>
 * <ul>
 * <li>
 * <p>
 * The <code>Namespace</code> parameter must be omitted or set to <code>AWS/RUM</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * Only certain combinations of values for <code>Name</code>, <code>ValueKey</code>, and <code>EventPattern</code> are
 * valid. In addition to what is displayed in the list below, the <code>EventPattern</code> can also include information
 * used by the <code>DimensionKeys</code> field.
 * </p>
 * <ul>
 * <li>
 * <p>
 * If <code>Name</code> is <code>PerformanceNavigationDuration</code>, then <code>ValueKey</code>must be
 * <code>event_details.duration</code> and the <code>EventPattern</code> must include
 * <code>{"event_type":["com.amazon.rum.performance_navigation_event"]}</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * If <code>Name</code> is <code>PerformanceResourceDuration</code>, then <code>ValueKey</code>must be
 * <code>event_details.duration</code> and the <code>EventPattern</code> must include
 * <code>{"event_type":["com.amazon.rum.performance_resource_event"]}</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * If <code>Name</code> is <code>NavigationSatisfiedTransaction</code>, then <code>ValueKey</code>must be null and the
 * <code>EventPattern</code> must include
 * <code>{ "event_type": ["com.amazon.rum.performance_navigation_event"], "event_details": { "duration": [{ "numeric": ["&gt;",2000] }] } }</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * If <code>Name</code> is <code>NavigationToleratedTransaction</code>, then <code>ValueKey</code>must be null and the
 * <code>EventPattern</code> must include
 * <code>{ "event_type": ["com.amazon.rum.performance_navigation_event"], "event_details": { "duration": [{ "numeric": ["&gt;=",2000,"&lt;"8000] }] } }</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * If <code>Name</code> is <code>NavigationFrustratedTransaction</code>, then <code>ValueKey</code>must be null and the
 * <code>EventPattern</code> must include
 * <code>{ "event_type": ["com.amazon.rum.performance_navigation_event"], "event_details": { "duration": [{ "numeric": ["&gt;=",8000] }] } }</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * If <code>Name</code> is <code>WebVitalsCumulativeLayoutShift</code>, then <code>ValueKey</code>must be
 * <code>event_details.value</code> and the <code>EventPattern</code> must include
 * <code>{"event_type":["com.amazon.rum.cumulative_layout_shift_event"]}</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * If <code>Name</code> is <code>WebVitalsFirstInputDelay</code>, then <code>ValueKey</code>must be
 * <code>event_details.value</code> and the <code>EventPattern</code> must include
 * <code>{"event_type":["com.amazon.rum.first_input_delay_event"]}</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * If <code>Name</code> is <code>WebVitalsLargestContentfulPaint</code>, then <code>ValueKey</code>must be
 * <code>event_details.value</code> and the <code>EventPattern</code> must include
 * <code>{"event_type":["com.amazon.rum.largest_contentful_paint_event"]}</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * If <code>Name</code> is <code>JsErrorCount</code>, then <code>ValueKey</code>must be null and the
 * <code>EventPattern</code> must include <code>{"event_type":["com.amazon.rum.js_error_event"]}</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * If <code>Name</code> is <code>HttpErrorCount</code>, then <code>ValueKey</code>must be null and the
 * <code>EventPattern</code> must include <code>{"event_type":["com.amazon.rum.http_event"]}</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * If <code>Name</code> is <code>SessionCount</code>, then <code>ValueKey</code>must be null and the
 * <code>EventPattern</code> must include <code>{"event_type":["com.amazon.rum.session_start_event"]}</code>
 * </p>
 * </li>
 * </ul>
 * </li>
 * </ul>
 * <p>
 * For custom metrics, the following validation rules apply:
 * </p>
 * <ul>
 * <li>
 * <p>
 * The namespace can't be omitted and can't be <code>AWS/RUM</code>. You can use the <code>AWS/RUM</code> namespace only
 * for extended metrics.
 * </p>
 * </li>
 * <li>
 * <p>
 * All dimensions listed in the <code>DimensionKeys</code> field must be present in the value of
 * <code>EventPattern</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * The values that you specify for <code>ValueKey</code>, <code>EventPattern</code>, and <code>DimensionKeys</code> must
 * be fields in RUM events, so all first-level keys in these fields must be one of the keys in the list later in this
 * section.
 * </p>
 * </li>
 * <li>
 * <p>
 * If you set a value for <code>EventPattern</code>, it must be a JSON object.
 * </p>
 * </li>
 * <li>
 * <p>
 * For every non-empty <code>event_details</code>, there must be a non-empty <code>event_type</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * If <code>EventPattern</code> contains an <code>event_details</code> field, it must also contain an
 * <code>event_type</code>. For every built-in <code>event_type</code> that you use, you must use a value for
 * <code>event_details</code> that corresponds to that <code>event_type</code>. For information about event details that
 * correspond to event types, see <a href=
 * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-datacollected.html#CloudWatch-RUM-datacollected-eventDetails"
 * > RUM event details</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * In <code>EventPattern</code>, any JSON array must contain only one value.
 * </p>
 * </li>
 * </ul>
 * <p>
 * Valid key values for first-level keys in the <code>ValueKey</code>, <code>EventPattern</code>, and
 * <code>DimensionKeys</code> fields:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>account_id</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>application_Id</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>application_version</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>application_name</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>batch_id</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>event_details</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>event_id</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>event_interaction</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>event_timestamp</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>event_type</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>event_version</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>log_stream</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>metadata</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>sessionId</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>user_details</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>userId</code>
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MetricDefinitionRequest implements SdkPojo, Serializable,
        ToCopyableBuilder<MetricDefinitionRequest.Builder, MetricDefinitionRequest> {
    private static final SdkField<Map<String, String>> DIMENSION_KEYS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("DimensionKeys")
            .getter(getter(MetricDefinitionRequest::dimensionKeys))
            .setter(setter(Builder::dimensionKeys))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DimensionKeys").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> EVENT_PATTERN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EventPattern").getter(getter(MetricDefinitionRequest::eventPattern))
            .setter(setter(Builder::eventPattern))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventPattern").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(MetricDefinitionRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Namespace").getter(getter(MetricDefinitionRequest::namespace)).setter(setter(Builder::namespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Namespace").build()).build();

    private static final SdkField<String> UNIT_LABEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UnitLabel").getter(getter(MetricDefinitionRequest::unitLabel)).setter(setter(Builder::unitLabel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UnitLabel").build()).build();

    private static final SdkField<String> VALUE_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ValueKey").getter(getter(MetricDefinitionRequest::valueKey)).setter(setter(Builder::valueKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ValueKey").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DIMENSION_KEYS_FIELD,
            EVENT_PATTERN_FIELD, NAME_FIELD, NAMESPACE_FIELD, UNIT_LABEL_FIELD, VALUE_KEY_FIELD));

    private static final long serialVersionUID = 1L;

    private final Map<String, String> dimensionKeys;

    private final String eventPattern;

    private final String name;

    private final String namespace;

    private final String unitLabel;

    private final String valueKey;

    private MetricDefinitionRequest(BuilderImpl builder) {
        this.dimensionKeys = builder.dimensionKeys;
        this.eventPattern = builder.eventPattern;
        this.name = builder.name;
        this.namespace = builder.namespace;
        this.unitLabel = builder.unitLabel;
        this.valueKey = builder.valueKey;
    }

    /**
     * For responses, this returns true if the service returned a value for the DimensionKeys property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDimensionKeys() {
        return dimensionKeys != null && !(dimensionKeys instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Use this field only if you are sending the metric to CloudWatch.
     * </p>
     * <p>
     * This field is a map of field paths to dimension names. It defines the dimensions to associate with this metric in
     * CloudWatch. For extended metrics, valid values for the entries in this field are the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>"metadata.pageId": "PageId"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"metadata.browserName": "BrowserName"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"metadata.deviceType": "DeviceType"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"metadata.osName": "OSName"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"metadata.countryCode": "CountryCode"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"event_details.fileType": "FileType"</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For both extended metrics and custom metrics, all dimensions listed in this field must also be included in
     * <code>EventPattern</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDimensionKeys} method.
     * </p>
     * 
     * @return Use this field only if you are sending the metric to CloudWatch.</p>
     *         <p>
     *         This field is a map of field paths to dimension names. It defines the dimensions to associate with this
     *         metric in CloudWatch. For extended metrics, valid values for the entries in this field are the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>"metadata.pageId": "PageId"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"metadata.browserName": "BrowserName"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"metadata.deviceType": "DeviceType"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"metadata.osName": "OSName"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"metadata.countryCode": "CountryCode"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"event_details.fileType": "FileType"</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For both extended metrics and custom metrics, all dimensions listed in this field must also be included
     *         in <code>EventPattern</code>.
     */
    public final Map<String, String> dimensionKeys() {
        return dimensionKeys;
    }

    /**
     * <p>
     * The pattern that defines the metric, specified as a JSON object. RUM checks events that happen in a user's
     * session against the pattern, and events that match the pattern are sent to the metric destination.
     * </p>
     * <p>
     * When you define extended metrics, the metric definition is not valid if <code>EventPattern</code> is omitted.
     * </p>
     * <p>
     * Example event patterns:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>'{ "event_type": ["com.amazon.rum.js_error_event"], "metadata": { "browserName": [ "Chrome", "Safari" ], } }'</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>'{ "event_type": ["com.amazon.rum.performance_navigation_event"], "metadata": { "browserName": [ "Chrome", "Firefox" ] }, "event_details": { "duration": [{ "numeric": [ "&lt;", 2000 ] }] } }'</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>'{ "event_type": ["com.amazon.rum.performance_navigation_event"], "metadata": { "browserName": [ "Chrome", "Safari" ], "countryCode": [ "US" ] }, "event_details": { "duration": [{ "numeric": [ "&gt;=", 2000, "&lt;", 8000 ] }] } }'</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the metrics destination' is <code>CloudWatch</code> and the event also matches a value in
     * <code>DimensionKeys</code>, then the metric is published with the specified dimensions.
     * </p>
     * 
     * @return The pattern that defines the metric, specified as a JSON object. RUM checks events that happen in a
     *         user's session against the pattern, and events that match the pattern are sent to the metric
     *         destination.</p>
     *         <p>
     *         When you define extended metrics, the metric definition is not valid if <code>EventPattern</code> is
     *         omitted.
     *         </p>
     *         <p>
     *         Example event patterns:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>'{ "event_type": ["com.amazon.rum.js_error_event"], "metadata": { "browserName": [ "Chrome", "Safari" ], } }'</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>'{ "event_type": ["com.amazon.rum.performance_navigation_event"], "metadata": { "browserName": [ "Chrome", "Firefox" ] }, "event_details": { "duration": [{ "numeric": [ "&lt;", 2000 ] }] } }'</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>'{ "event_type": ["com.amazon.rum.performance_navigation_event"], "metadata": { "browserName": [ "Chrome", "Safari" ], "countryCode": [ "US" ] }, "event_details": { "duration": [{ "numeric": [ "&gt;=", 2000, "&lt;", 8000 ] }] } }'</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If the metrics destination' is <code>CloudWatch</code> and the event also matches a value in
     *         <code>DimensionKeys</code>, then the metric is published with the specified dimensions.
     */
    public final String eventPattern() {
        return eventPattern;
    }

    /**
     * <p>
     * The name for the metric that is defined in this structure. For custom metrics, you can specify any name that you
     * like. For extended metrics, valid values are the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PerformanceNavigationDuration</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PerformanceResourceDuration </code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NavigationSatisfiedTransaction</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NavigationToleratedTransaction</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NavigationFrustratedTransaction</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WebVitalsCumulativeLayoutShift</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WebVitalsFirstInputDelay</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WebVitalsLargestContentfulPaint</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>JsErrorCount</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>HttpErrorCount</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SessionCount</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The name for the metric that is defined in this structure. For custom metrics, you can specify any name
     *         that you like. For extended metrics, valid values are the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PerformanceNavigationDuration</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PerformanceResourceDuration </code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NavigationSatisfiedTransaction</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NavigationToleratedTransaction</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NavigationFrustratedTransaction</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>WebVitalsCumulativeLayoutShift</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>WebVitalsFirstInputDelay</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>WebVitalsLargestContentfulPaint</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>JsErrorCount</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>HttpErrorCount</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SessionCount</code>
     *         </p>
     *         </li>
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * If this structure is for a custom metric instead of an extended metrics, use this parameter to define the metric
     * namespace for that custom metric. Do not specify this parameter if this structure is for an extended metric.
     * </p>
     * <p>
     * You cannot use any string that starts with <code>AWS/</code> for your namespace.
     * </p>
     * 
     * @return If this structure is for a custom metric instead of an extended metrics, use this parameter to define the
     *         metric namespace for that custom metric. Do not specify this parameter if this structure is for an
     *         extended metric.</p>
     *         <p>
     *         You cannot use any string that starts with <code>AWS/</code> for your namespace.
     */
    public final String namespace() {
        return namespace;
    }

    /**
     * <p>
     * The CloudWatch metric unit to use for this metric. If you omit this field, the metric is recorded with no unit.
     * </p>
     * 
     * @return The CloudWatch metric unit to use for this metric. If you omit this field, the metric is recorded with no
     *         unit.
     */
    public final String unitLabel() {
        return unitLabel;
    }

    /**
     * <p>
     * The field within the event object that the metric value is sourced from.
     * </p>
     * <p>
     * If you omit this field, a hardcoded value of 1 is pushed as the metric value. This is useful if you just want to
     * count the number of events that the filter catches.
     * </p>
     * <p>
     * If this metric is sent to CloudWatch Evidently, this field will be passed to Evidently raw and Evidently will
     * handle data extraction from the event.
     * </p>
     * 
     * @return The field within the event object that the metric value is sourced from.</p>
     *         <p>
     *         If you omit this field, a hardcoded value of 1 is pushed as the metric value. This is useful if you just
     *         want to count the number of events that the filter catches.
     *         </p>
     *         <p>
     *         If this metric is sent to CloudWatch Evidently, this field will be passed to Evidently raw and Evidently
     *         will handle data extraction from the event.
     */
    public final String valueKey() {
        return valueKey;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasDimensionKeys() ? dimensionKeys() : null);
        hashCode = 31 * hashCode + Objects.hashCode(eventPattern());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(namespace());
        hashCode = 31 * hashCode + Objects.hashCode(unitLabel());
        hashCode = 31 * hashCode + Objects.hashCode(valueKey());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MetricDefinitionRequest)) {
            return false;
        }
        MetricDefinitionRequest other = (MetricDefinitionRequest) obj;
        return hasDimensionKeys() == other.hasDimensionKeys() && Objects.equals(dimensionKeys(), other.dimensionKeys())
                && Objects.equals(eventPattern(), other.eventPattern()) && Objects.equals(name(), other.name())
                && Objects.equals(namespace(), other.namespace()) && Objects.equals(unitLabel(), other.unitLabel())
                && Objects.equals(valueKey(), other.valueKey());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MetricDefinitionRequest").add("DimensionKeys", hasDimensionKeys() ? dimensionKeys() : null)
                .add("EventPattern", eventPattern()).add("Name", name()).add("Namespace", namespace())
                .add("UnitLabel", unitLabel()).add("ValueKey", valueKey()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DimensionKeys":
            return Optional.ofNullable(clazz.cast(dimensionKeys()));
        case "EventPattern":
            return Optional.ofNullable(clazz.cast(eventPattern()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Namespace":
            return Optional.ofNullable(clazz.cast(namespace()));
        case "UnitLabel":
            return Optional.ofNullable(clazz.cast(unitLabel()));
        case "ValueKey":
            return Optional.ofNullable(clazz.cast(valueKey()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MetricDefinitionRequest, T> g) {
        return obj -> g.apply((MetricDefinitionRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MetricDefinitionRequest> {
        /**
         * <p>
         * Use this field only if you are sending the metric to CloudWatch.
         * </p>
         * <p>
         * This field is a map of field paths to dimension names. It defines the dimensions to associate with this
         * metric in CloudWatch. For extended metrics, valid values for the entries in this field are the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>"metadata.pageId": "PageId"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"metadata.browserName": "BrowserName"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"metadata.deviceType": "DeviceType"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"metadata.osName": "OSName"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"metadata.countryCode": "CountryCode"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"event_details.fileType": "FileType"</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * For both extended metrics and custom metrics, all dimensions listed in this field must also be included in
         * <code>EventPattern</code>.
         * </p>
         * 
         * @param dimensionKeys
         *        Use this field only if you are sending the metric to CloudWatch.</p>
         *        <p>
         *        This field is a map of field paths to dimension names. It defines the dimensions to associate with
         *        this metric in CloudWatch. For extended metrics, valid values for the entries in this field are the
         *        following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>"metadata.pageId": "PageId"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"metadata.browserName": "BrowserName"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"metadata.deviceType": "DeviceType"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"metadata.osName": "OSName"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"metadata.countryCode": "CountryCode"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"event_details.fileType": "FileType"</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For both extended metrics and custom metrics, all dimensions listed in this field must also be
         *        included in <code>EventPattern</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensionKeys(Map<String, String> dimensionKeys);

        /**
         * <p>
         * The pattern that defines the metric, specified as a JSON object. RUM checks events that happen in a user's
         * session against the pattern, and events that match the pattern are sent to the metric destination.
         * </p>
         * <p>
         * When you define extended metrics, the metric definition is not valid if <code>EventPattern</code> is omitted.
         * </p>
         * <p>
         * Example event patterns:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>'{ "event_type": ["com.amazon.rum.js_error_event"], "metadata": { "browserName": [ "Chrome", "Safari" ], } }'</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>'{ "event_type": ["com.amazon.rum.performance_navigation_event"], "metadata": { "browserName": [ "Chrome", "Firefox" ] }, "event_details": { "duration": [{ "numeric": [ "&lt;", 2000 ] }] } }'</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>'{ "event_type": ["com.amazon.rum.performance_navigation_event"], "metadata": { "browserName": [ "Chrome", "Safari" ], "countryCode": [ "US" ] }, "event_details": { "duration": [{ "numeric": [ "&gt;=", 2000, "&lt;", 8000 ] }] } }'</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * If the metrics destination' is <code>CloudWatch</code> and the event also matches a value in
         * <code>DimensionKeys</code>, then the metric is published with the specified dimensions.
         * </p>
         * 
         * @param eventPattern
         *        The pattern that defines the metric, specified as a JSON object. RUM checks events that happen in a
         *        user's session against the pattern, and events that match the pattern are sent to the metric
         *        destination.</p>
         *        <p>
         *        When you define extended metrics, the metric definition is not valid if <code>EventPattern</code> is
         *        omitted.
         *        </p>
         *        <p>
         *        Example event patterns:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>'{ "event_type": ["com.amazon.rum.js_error_event"], "metadata": { "browserName": [ "Chrome", "Safari" ], } }'</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>'{ "event_type": ["com.amazon.rum.performance_navigation_event"], "metadata": { "browserName": [ "Chrome", "Firefox" ] }, "event_details": { "duration": [{ "numeric": [ "&lt;", 2000 ] }] } }'</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>'{ "event_type": ["com.amazon.rum.performance_navigation_event"], "metadata": { "browserName": [ "Chrome", "Safari" ], "countryCode": [ "US" ] }, "event_details": { "duration": [{ "numeric": [ "&gt;=", 2000, "&lt;", 8000 ] }] } }'</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If the metrics destination' is <code>CloudWatch</code> and the event also matches a value in
         *        <code>DimensionKeys</code>, then the metric is published with the specified dimensions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventPattern(String eventPattern);

        /**
         * <p>
         * The name for the metric that is defined in this structure. For custom metrics, you can specify any name that
         * you like. For extended metrics, valid values are the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PerformanceNavigationDuration</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PerformanceResourceDuration </code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NavigationSatisfiedTransaction</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NavigationToleratedTransaction</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NavigationFrustratedTransaction</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>WebVitalsCumulativeLayoutShift</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>WebVitalsFirstInputDelay</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>WebVitalsLargestContentfulPaint</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>JsErrorCount</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>HttpErrorCount</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SessionCount</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param name
         *        The name for the metric that is defined in this structure. For custom metrics, you can specify any
         *        name that you like. For extended metrics, valid values are the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PerformanceNavigationDuration</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PerformanceResourceDuration </code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NavigationSatisfiedTransaction</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NavigationToleratedTransaction</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NavigationFrustratedTransaction</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>WebVitalsCumulativeLayoutShift</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>WebVitalsFirstInputDelay</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>WebVitalsLargestContentfulPaint</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>JsErrorCount</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>HttpErrorCount</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SessionCount</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * If this structure is for a custom metric instead of an extended metrics, use this parameter to define the
         * metric namespace for that custom metric. Do not specify this parameter if this structure is for an extended
         * metric.
         * </p>
         * <p>
         * You cannot use any string that starts with <code>AWS/</code> for your namespace.
         * </p>
         * 
         * @param namespace
         *        If this structure is for a custom metric instead of an extended metrics, use this parameter to define
         *        the metric namespace for that custom metric. Do not specify this parameter if this structure is for an
         *        extended metric.</p>
         *        <p>
         *        You cannot use any string that starts with <code>AWS/</code> for your namespace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(String namespace);

        /**
         * <p>
         * The CloudWatch metric unit to use for this metric. If you omit this field, the metric is recorded with no
         * unit.
         * </p>
         * 
         * @param unitLabel
         *        The CloudWatch metric unit to use for this metric. If you omit this field, the metric is recorded with
         *        no unit.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unitLabel(String unitLabel);

        /**
         * <p>
         * The field within the event object that the metric value is sourced from.
         * </p>
         * <p>
         * If you omit this field, a hardcoded value of 1 is pushed as the metric value. This is useful if you just want
         * to count the number of events that the filter catches.
         * </p>
         * <p>
         * If this metric is sent to CloudWatch Evidently, this field will be passed to Evidently raw and Evidently will
         * handle data extraction from the event.
         * </p>
         * 
         * @param valueKey
         *        The field within the event object that the metric value is sourced from.</p>
         *        <p>
         *        If you omit this field, a hardcoded value of 1 is pushed as the metric value. This is useful if you
         *        just want to count the number of events that the filter catches.
         *        </p>
         *        <p>
         *        If this metric is sent to CloudWatch Evidently, this field will be passed to Evidently raw and
         *        Evidently will handle data extraction from the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder valueKey(String valueKey);
    }

    static final class BuilderImpl implements Builder {
        private Map<String, String> dimensionKeys = DefaultSdkAutoConstructMap.getInstance();

        private String eventPattern;

        private String name;

        private String namespace;

        private String unitLabel;

        private String valueKey;

        private BuilderImpl() {
        }

        private BuilderImpl(MetricDefinitionRequest model) {
            dimensionKeys(model.dimensionKeys);
            eventPattern(model.eventPattern);
            name(model.name);
            namespace(model.namespace);
            unitLabel(model.unitLabel);
            valueKey(model.valueKey);
        }

        public final Map<String, String> getDimensionKeys() {
            if (dimensionKeys instanceof SdkAutoConstructMap) {
                return null;
            }
            return dimensionKeys;
        }

        public final void setDimensionKeys(Map<String, String> dimensionKeys) {
            this.dimensionKeys = DimensionKeysMapCopier.copy(dimensionKeys);
        }

        @Override
        public final Builder dimensionKeys(Map<String, String> dimensionKeys) {
            this.dimensionKeys = DimensionKeysMapCopier.copy(dimensionKeys);
            return this;
        }

        public final String getEventPattern() {
            return eventPattern;
        }

        public final void setEventPattern(String eventPattern) {
            this.eventPattern = eventPattern;
        }

        @Override
        public final Builder eventPattern(String eventPattern) {
            this.eventPattern = eventPattern;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getNamespace() {
            return namespace;
        }

        public final void setNamespace(String namespace) {
            this.namespace = namespace;
        }

        @Override
        public final Builder namespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        public final String getUnitLabel() {
            return unitLabel;
        }

        public final void setUnitLabel(String unitLabel) {
            this.unitLabel = unitLabel;
        }

        @Override
        public final Builder unitLabel(String unitLabel) {
            this.unitLabel = unitLabel;
            return this;
        }

        public final String getValueKey() {
            return valueKey;
        }

        public final void setValueKey(String valueKey) {
            this.valueKey = valueKey;
        }

        @Override
        public final Builder valueKey(String valueKey) {
            this.valueKey = valueKey;
            return this;
        }

        @Override
        public MetricDefinitionRequest build() {
            return new MetricDefinitionRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
