/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rtbfabric;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.rtbfabric.internal.RtbFabricServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.rtbfabric.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.rtbfabric.model.AcceptLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.AcceptLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.AccessDeniedException;
import software.amazon.awssdk.services.rtbfabric.model.ConflictException;
import software.amazon.awssdk.services.rtbfabric.model.CreateInboundExternalLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.CreateInboundExternalLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.CreateLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.CreateLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.CreateOutboundExternalLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.CreateOutboundExternalLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.CreateRequesterGatewayRequest;
import software.amazon.awssdk.services.rtbfabric.model.CreateRequesterGatewayResponse;
import software.amazon.awssdk.services.rtbfabric.model.CreateResponderGatewayRequest;
import software.amazon.awssdk.services.rtbfabric.model.CreateResponderGatewayResponse;
import software.amazon.awssdk.services.rtbfabric.model.DeleteInboundExternalLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.DeleteInboundExternalLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.DeleteLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.DeleteLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.DeleteOutboundExternalLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.DeleteOutboundExternalLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.DeleteRequesterGatewayRequest;
import software.amazon.awssdk.services.rtbfabric.model.DeleteRequesterGatewayResponse;
import software.amazon.awssdk.services.rtbfabric.model.DeleteResponderGatewayRequest;
import software.amazon.awssdk.services.rtbfabric.model.DeleteResponderGatewayResponse;
import software.amazon.awssdk.services.rtbfabric.model.GetInboundExternalLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.GetInboundExternalLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.GetLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.GetLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.GetOutboundExternalLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.GetOutboundExternalLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.GetRequesterGatewayRequest;
import software.amazon.awssdk.services.rtbfabric.model.GetRequesterGatewayResponse;
import software.amazon.awssdk.services.rtbfabric.model.GetResponderGatewayRequest;
import software.amazon.awssdk.services.rtbfabric.model.GetResponderGatewayResponse;
import software.amazon.awssdk.services.rtbfabric.model.InternalServerException;
import software.amazon.awssdk.services.rtbfabric.model.ListLinksRequest;
import software.amazon.awssdk.services.rtbfabric.model.ListLinksResponse;
import software.amazon.awssdk.services.rtbfabric.model.ListRequesterGatewaysRequest;
import software.amazon.awssdk.services.rtbfabric.model.ListRequesterGatewaysResponse;
import software.amazon.awssdk.services.rtbfabric.model.ListResponderGatewaysRequest;
import software.amazon.awssdk.services.rtbfabric.model.ListResponderGatewaysResponse;
import software.amazon.awssdk.services.rtbfabric.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.rtbfabric.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.rtbfabric.model.RejectLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.RejectLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.ResourceNotFoundException;
import software.amazon.awssdk.services.rtbfabric.model.RtbFabricException;
import software.amazon.awssdk.services.rtbfabric.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.rtbfabric.model.TagResourceRequest;
import software.amazon.awssdk.services.rtbfabric.model.TagResourceResponse;
import software.amazon.awssdk.services.rtbfabric.model.ThrottlingException;
import software.amazon.awssdk.services.rtbfabric.model.UntagResourceRequest;
import software.amazon.awssdk.services.rtbfabric.model.UntagResourceResponse;
import software.amazon.awssdk.services.rtbfabric.model.UpdateLinkModuleFlowRequest;
import software.amazon.awssdk.services.rtbfabric.model.UpdateLinkModuleFlowResponse;
import software.amazon.awssdk.services.rtbfabric.model.UpdateLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.UpdateLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.UpdateRequesterGatewayRequest;
import software.amazon.awssdk.services.rtbfabric.model.UpdateRequesterGatewayResponse;
import software.amazon.awssdk.services.rtbfabric.model.UpdateResponderGatewayRequest;
import software.amazon.awssdk.services.rtbfabric.model.UpdateResponderGatewayResponse;
import software.amazon.awssdk.services.rtbfabric.model.ValidationException;
import software.amazon.awssdk.services.rtbfabric.transform.AcceptLinkRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.CreateInboundExternalLinkRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.CreateLinkRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.CreateOutboundExternalLinkRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.CreateRequesterGatewayRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.CreateResponderGatewayRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.DeleteInboundExternalLinkRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.DeleteLinkRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.DeleteOutboundExternalLinkRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.DeleteRequesterGatewayRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.DeleteResponderGatewayRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.GetInboundExternalLinkRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.GetLinkRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.GetOutboundExternalLinkRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.GetRequesterGatewayRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.GetResponderGatewayRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.ListLinksRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.ListRequesterGatewaysRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.ListResponderGatewaysRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.RejectLinkRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.UpdateLinkModuleFlowRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.UpdateLinkRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.UpdateRequesterGatewayRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.UpdateResponderGatewayRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.waiters.RtbFabricWaiter;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link RtbFabricClient}.
 *
 * @see RtbFabricClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRtbFabricClient implements RtbFabricClient {
    private static final Logger log = Logger.loggerFor(DefaultRtbFabricClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRtbFabricClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "RTBFabric" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Accepts a link request between gateways.
     * </p>
     * <p>
     * When a requester gateway requests to link with a responder gateway, the responder can use this operation to
     * accept the link request and establish the connection.
     * </p>
     *
     * @param acceptLinkRequest
     * @return Result of the AcceptLink operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request could not be completed because the resource does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have sufficient access to perform this action.
     * @throws ConflictException
     *         The request could not be completed because of a conflict in the current state of the resource.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.AcceptLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/AcceptLink" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AcceptLinkResponse acceptLink(AcceptLinkRequest acceptLinkRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ConflictException, InternalServerException, ValidationException,
            AwsServiceException, SdkClientException, RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AcceptLinkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AcceptLinkResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(acceptLinkRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acceptLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptLink");

            return clientHandler.execute(new ClientExecutionParams<AcceptLinkRequest, AcceptLinkResponse>()
                    .withOperationName("AcceptLink").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(acceptLinkRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AcceptLinkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an inbound external link.
     * </p>
     *
     * @param createInboundExternalLinkRequest
     * @return Result of the CreateInboundExternalLink operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request could not be completed because you exceeded a service quota.
     * @throws ResourceNotFoundException
     *         The request could not be completed because the resource does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have sufficient access to perform this action.
     * @throws ConflictException
     *         The request could not be completed because of a conflict in the current state of the resource.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.CreateInboundExternalLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/CreateInboundExternalLink"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateInboundExternalLinkResponse createInboundExternalLink(
            CreateInboundExternalLinkRequest createInboundExternalLinkRequest) throws ServiceQuotaExceededException,
            ResourceNotFoundException, ThrottlingException, AccessDeniedException, ConflictException, InternalServerException,
            ValidationException, AwsServiceException, SdkClientException, RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateInboundExternalLinkResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateInboundExternalLinkResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createInboundExternalLinkRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createInboundExternalLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateInboundExternalLink");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateInboundExternalLinkRequest, CreateInboundExternalLinkResponse>()
                            .withOperationName("CreateInboundExternalLink").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createInboundExternalLinkRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateInboundExternalLinkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new link between gateways.
     * </p>
     * <p>
     * Establishes a connection that allows gateways to communicate and exchange bid requests and responses.
     * </p>
     *
     * @param createLinkRequest
     * @return Result of the CreateLink operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request could not be completed because you exceeded a service quota.
     * @throws ResourceNotFoundException
     *         The request could not be completed because the resource does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have sufficient access to perform this action.
     * @throws ConflictException
     *         The request could not be completed because of a conflict in the current state of the resource.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.CreateLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/CreateLink" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateLinkResponse createLink(CreateLinkRequest createLinkRequest) throws ServiceQuotaExceededException,
            ResourceNotFoundException, ThrottlingException, AccessDeniedException, ConflictException, InternalServerException,
            ValidationException, AwsServiceException, SdkClientException, RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateLinkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateLinkResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createLinkRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLink");

            return clientHandler.execute(new ClientExecutionParams<CreateLinkRequest, CreateLinkResponse>()
                    .withOperationName("CreateLink").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createLinkRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateLinkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an outbound external link.
     * </p>
     *
     * @param createOutboundExternalLinkRequest
     * @return Result of the CreateOutboundExternalLink operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request could not be completed because you exceeded a service quota.
     * @throws ResourceNotFoundException
     *         The request could not be completed because the resource does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.CreateOutboundExternalLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/CreateOutboundExternalLink"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateOutboundExternalLinkResponse createOutboundExternalLink(
            CreateOutboundExternalLinkRequest createOutboundExternalLinkRequest) throws ServiceQuotaExceededException,
            ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException, ValidationException,
            AwsServiceException, SdkClientException, RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateOutboundExternalLinkResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateOutboundExternalLinkResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createOutboundExternalLinkRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createOutboundExternalLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateOutboundExternalLink");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateOutboundExternalLinkRequest, CreateOutboundExternalLinkResponse>()
                            .withOperationName("CreateOutboundExternalLink").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createOutboundExternalLinkRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateOutboundExternalLinkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a requester gateway.
     * </p>
     *
     * @param createRequesterGatewayRequest
     * @return Result of the CreateRequesterGateway operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request could not be completed because you exceeded a service quota.
     * @throws ResourceNotFoundException
     *         The request could not be completed because the resource does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.CreateRequesterGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/CreateRequesterGateway"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateRequesterGatewayResponse createRequesterGateway(CreateRequesterGatewayRequest createRequesterGatewayRequest)
            throws ServiceQuotaExceededException, ResourceNotFoundException, ThrottlingException, AccessDeniedException,
            InternalServerException, ValidationException, AwsServiceException, SdkClientException, RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateRequesterGatewayResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateRequesterGatewayResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRequesterGatewayRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRequesterGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRequesterGateway");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateRequesterGatewayRequest, CreateRequesterGatewayResponse>()
                            .withOperationName("CreateRequesterGateway").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createRequesterGatewayRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateRequesterGatewayRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a responder gateway.
     * </p>
     * <important>
     * <p>
     * A domain name or managed endpoint is required.
     * </p>
     * </important>
     *
     * @param createResponderGatewayRequest
     * @return Result of the CreateResponderGateway operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request could not be completed because you exceeded a service quota.
     * @throws ResourceNotFoundException
     *         The request could not be completed because the resource does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.CreateResponderGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/CreateResponderGateway"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateResponderGatewayResponse createResponderGateway(CreateResponderGatewayRequest createResponderGatewayRequest)
            throws ServiceQuotaExceededException, ResourceNotFoundException, ThrottlingException, AccessDeniedException,
            InternalServerException, ValidationException, AwsServiceException, SdkClientException, RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateResponderGatewayResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateResponderGatewayResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createResponderGatewayRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createResponderGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateResponderGateway");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateResponderGatewayRequest, CreateResponderGatewayResponse>()
                            .withOperationName("CreateResponderGateway").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createResponderGatewayRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateResponderGatewayRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an inbound external link.
     * </p>
     *
     * @param deleteInboundExternalLinkRequest
     * @return Result of the DeleteInboundExternalLink operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request could not be completed because the resource does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have sufficient access to perform this action.
     * @throws ConflictException
     *         The request could not be completed because of a conflict in the current state of the resource.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.DeleteInboundExternalLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/DeleteInboundExternalLink"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteInboundExternalLinkResponse deleteInboundExternalLink(
            DeleteInboundExternalLinkRequest deleteInboundExternalLinkRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ConflictException, InternalServerException, ValidationException,
            AwsServiceException, SdkClientException, RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteInboundExternalLinkResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteInboundExternalLinkResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteInboundExternalLinkRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteInboundExternalLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteInboundExternalLink");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteInboundExternalLinkRequest, DeleteInboundExternalLinkResponse>()
                            .withOperationName("DeleteInboundExternalLink").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteInboundExternalLinkRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteInboundExternalLinkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a link between gateways.
     * </p>
     * <p>
     * Permanently removes the connection between gateways. This action cannot be undone.
     * </p>
     *
     * @param deleteLinkRequest
     * @return Result of the DeleteLink operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request could not be completed because the resource does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have sufficient access to perform this action.
     * @throws ConflictException
     *         The request could not be completed because of a conflict in the current state of the resource.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.DeleteLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/DeleteLink" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteLinkResponse deleteLink(DeleteLinkRequest deleteLinkRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ConflictException, InternalServerException, ValidationException,
            AwsServiceException, SdkClientException, RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteLinkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteLinkResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteLinkRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLink");

            return clientHandler.execute(new ClientExecutionParams<DeleteLinkRequest, DeleteLinkResponse>()
                    .withOperationName("DeleteLink").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteLinkRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteLinkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an outbound external link.
     * </p>
     *
     * @param deleteOutboundExternalLinkRequest
     * @return Result of the DeleteOutboundExternalLink operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request could not be completed because the resource does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have sufficient access to perform this action.
     * @throws ConflictException
     *         The request could not be completed because of a conflict in the current state of the resource.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.DeleteOutboundExternalLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/DeleteOutboundExternalLink"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteOutboundExternalLinkResponse deleteOutboundExternalLink(
            DeleteOutboundExternalLinkRequest deleteOutboundExternalLinkRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ConflictException, InternalServerException, ValidationException,
            AwsServiceException, SdkClientException, RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteOutboundExternalLinkResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteOutboundExternalLinkResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteOutboundExternalLinkRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteOutboundExternalLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteOutboundExternalLink");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteOutboundExternalLinkRequest, DeleteOutboundExternalLinkResponse>()
                            .withOperationName("DeleteOutboundExternalLink").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteOutboundExternalLinkRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteOutboundExternalLinkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a requester gateway.
     * </p>
     *
     * @param deleteRequesterGatewayRequest
     * @return Result of the DeleteRequesterGateway operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request could not be completed because the resource does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.DeleteRequesterGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/DeleteRequesterGateway"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteRequesterGatewayResponse deleteRequesterGateway(DeleteRequesterGatewayRequest deleteRequesterGatewayRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException,
            ValidationException, AwsServiceException, SdkClientException, RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteRequesterGatewayResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteRequesterGatewayResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRequesterGatewayRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRequesterGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRequesterGateway");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteRequesterGatewayRequest, DeleteRequesterGatewayResponse>()
                            .withOperationName("DeleteRequesterGateway").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteRequesterGatewayRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteRequesterGatewayRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a responder gateway.
     * </p>
     *
     * @param deleteResponderGatewayRequest
     * @return Result of the DeleteResponderGateway operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request could not be completed because the resource does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.DeleteResponderGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/DeleteResponderGateway"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteResponderGatewayResponse deleteResponderGateway(DeleteResponderGatewayRequest deleteResponderGatewayRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException,
            ValidationException, AwsServiceException, SdkClientException, RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteResponderGatewayResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteResponderGatewayResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteResponderGatewayRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResponderGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResponderGateway");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteResponderGatewayRequest, DeleteResponderGatewayResponse>()
                            .withOperationName("DeleteResponderGateway").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteResponderGatewayRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteResponderGatewayRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about an inbound external link.
     * </p>
     *
     * @param getInboundExternalLinkRequest
     * @return Result of the GetInboundExternalLink operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request could not be completed because the resource does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.GetInboundExternalLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/GetInboundExternalLink"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetInboundExternalLinkResponse getInboundExternalLink(GetInboundExternalLinkRequest getInboundExternalLinkRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException,
            ValidationException, AwsServiceException, SdkClientException, RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetInboundExternalLinkResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetInboundExternalLinkResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getInboundExternalLinkRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInboundExternalLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInboundExternalLink");

            return clientHandler
                    .execute(new ClientExecutionParams<GetInboundExternalLinkRequest, GetInboundExternalLinkResponse>()
                            .withOperationName("GetInboundExternalLink").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getInboundExternalLinkRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetInboundExternalLinkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about a link between gateways.
     * </p>
     * <p>
     * Returns detailed information about the link configuration, status, and associated gateways.
     * </p>
     *
     * @param getLinkRequest
     * @return Result of the GetLink operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request could not be completed because the resource does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have sufficient access to perform this action.
     * @throws ConflictException
     *         The request could not be completed because of a conflict in the current state of the resource.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.GetLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/GetLink" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetLinkResponse getLink(GetLinkRequest getLinkRequest) throws ResourceNotFoundException, ThrottlingException,
            AccessDeniedException, ConflictException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetLinkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetLinkResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getLinkRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLink");

            return clientHandler.execute(new ClientExecutionParams<GetLinkRequest, GetLinkResponse>()
                    .withOperationName("GetLink").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getLinkRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetLinkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about an outbound external link.
     * </p>
     *
     * @param getOutboundExternalLinkRequest
     * @return Result of the GetOutboundExternalLink operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request could not be completed because the resource does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.GetOutboundExternalLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/GetOutboundExternalLink"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetOutboundExternalLinkResponse getOutboundExternalLink(GetOutboundExternalLinkRequest getOutboundExternalLinkRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException,
            ValidationException, AwsServiceException, SdkClientException, RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetOutboundExternalLinkResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetOutboundExternalLinkResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getOutboundExternalLinkRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getOutboundExternalLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOutboundExternalLink");

            return clientHandler
                    .execute(new ClientExecutionParams<GetOutboundExternalLinkRequest, GetOutboundExternalLinkResponse>()
                            .withOperationName("GetOutboundExternalLink").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getOutboundExternalLinkRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetOutboundExternalLinkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about a requester gateway.
     * </p>
     *
     * @param getRequesterGatewayRequest
     * @return Result of the GetRequesterGateway operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request could not be completed because the resource does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.GetRequesterGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/GetRequesterGateway" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetRequesterGatewayResponse getRequesterGateway(GetRequesterGatewayRequest getRequesterGatewayRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException,
            ValidationException, AwsServiceException, SdkClientException, RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRequesterGatewayResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetRequesterGatewayResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRequesterGatewayRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRequesterGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRequesterGateway");

            return clientHandler.execute(new ClientExecutionParams<GetRequesterGatewayRequest, GetRequesterGatewayResponse>()
                    .withOperationName("GetRequesterGateway").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getRequesterGatewayRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetRequesterGatewayRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about a responder gateway.
     * </p>
     *
     * @param getResponderGatewayRequest
     * @return Result of the GetResponderGateway operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request could not be completed because the resource does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.GetResponderGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/GetResponderGateway" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetResponderGatewayResponse getResponderGateway(GetResponderGatewayRequest getResponderGatewayRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException,
            ValidationException, AwsServiceException, SdkClientException, RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResponderGatewayResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetResponderGatewayResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResponderGatewayRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResponderGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResponderGateway");

            return clientHandler.execute(new ClientExecutionParams<GetResponderGatewayRequest, GetResponderGatewayResponse>()
                    .withOperationName("GetResponderGateway").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getResponderGatewayRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResponderGatewayRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists links associated with gateways.
     * </p>
     * <p>
     * Returns a list of all links for the specified gateways, including their status and configuration details.
     * </p>
     *
     * @param listLinksRequest
     * @return Result of the ListLinks operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request could not be completed because the resource does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.ListLinks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/ListLinks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListLinksResponse listLinks(ListLinksRequest listLinksRequest) throws ResourceNotFoundException, ThrottlingException,
            AccessDeniedException, InternalServerException, ValidationException, AwsServiceException, SdkClientException,
            RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListLinksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListLinksResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listLinksRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLinksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLinks");

            return clientHandler.execute(new ClientExecutionParams<ListLinksRequest, ListLinksResponse>()
                    .withOperationName("ListLinks").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listLinksRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListLinksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists requester gateways.
     * </p>
     *
     * @param listRequesterGatewaysRequest
     * @return Result of the ListRequesterGateways operation returned by the service.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.ListRequesterGateways
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/ListRequesterGateways"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRequesterGatewaysResponse listRequesterGateways(ListRequesterGatewaysRequest listRequesterGatewaysRequest)
            throws InternalServerException, ValidationException, AwsServiceException, SdkClientException, RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRequesterGatewaysResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListRequesterGatewaysResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRequesterGatewaysRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRequesterGatewaysRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRequesterGateways");

            return clientHandler.execute(new ClientExecutionParams<ListRequesterGatewaysRequest, ListRequesterGatewaysResponse>()
                    .withOperationName("ListRequesterGateways").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listRequesterGatewaysRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListRequesterGatewaysRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists reponder gateways.
     * </p>
     *
     * @param listResponderGatewaysRequest
     * @return Result of the ListResponderGateways operation returned by the service.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.ListResponderGateways
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/ListResponderGateways"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResponderGatewaysResponse listResponderGateways(ListResponderGatewaysRequest listResponderGatewaysRequest)
            throws InternalServerException, ValidationException, AwsServiceException, SdkClientException, RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResponderGatewaysResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListResponderGatewaysResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listResponderGatewaysRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResponderGatewaysRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResponderGateways");

            return clientHandler.execute(new ClientExecutionParams<ListResponderGatewaysRequest, ListResponderGatewaysResponse>()
                    .withOperationName("ListResponderGateways").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listResponderGatewaysRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListResponderGatewaysRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists tags for a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request could not be completed because the resource does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException,
            ValidationException, AwsServiceException, SdkClientException, RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Rejects a link request between gateways.
     * </p>
     * <p>
     * When a requester gateway requests to link with a responder gateway, the responder can use this operation to
     * decline the link request.
     * </p>
     *
     * @param rejectLinkRequest
     * @return Result of the RejectLink operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request could not be completed because the resource does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have sufficient access to perform this action.
     * @throws ConflictException
     *         The request could not be completed because of a conflict in the current state of the resource.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.RejectLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/RejectLink" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RejectLinkResponse rejectLink(RejectLinkRequest rejectLinkRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ConflictException, InternalServerException, ValidationException,
            AwsServiceException, SdkClientException, RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RejectLinkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RejectLinkResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(rejectLinkRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rejectLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RejectLink");

            return clientHandler.execute(new ClientExecutionParams<RejectLinkRequest, RejectLinkResponse>()
                    .withOperationName("RejectLink").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(rejectLinkRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RejectLinkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request could not be completed because the resource does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a tag or tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request could not be completed because the resource does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the configuration of a link between gateways.
     * </p>
     * <p>
     * Allows you to modify settings and parameters for an existing link.
     * </p>
     *
     * @param updateLinkRequest
     * @return Result of the UpdateLink operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request could not be completed because the resource does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have sufficient access to perform this action.
     * @throws ConflictException
     *         The request could not be completed because of a conflict in the current state of the resource.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.UpdateLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/UpdateLink" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateLinkResponse updateLink(UpdateLinkRequest updateLinkRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ConflictException, InternalServerException, ValidationException,
            AwsServiceException, SdkClientException, RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateLinkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateLinkResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateLinkRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLink");

            return clientHandler.execute(new ClientExecutionParams<UpdateLinkRequest, UpdateLinkResponse>()
                    .withOperationName("UpdateLink").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(updateLinkRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateLinkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a link module flow.
     * </p>
     *
     * @param updateLinkModuleFlowRequest
     * @return Result of the UpdateLinkModuleFlow operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request could not be completed because you exceeded a service quota.
     * @throws ResourceNotFoundException
     *         The request could not be completed because the resource does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have sufficient access to perform this action.
     * @throws ConflictException
     *         The request could not be completed because of a conflict in the current state of the resource.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.UpdateLinkModuleFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/UpdateLinkModuleFlow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateLinkModuleFlowResponse updateLinkModuleFlow(UpdateLinkModuleFlowRequest updateLinkModuleFlowRequest)
            throws ServiceQuotaExceededException, ResourceNotFoundException, ThrottlingException, AccessDeniedException,
            ConflictException, InternalServerException, ValidationException, AwsServiceException, SdkClientException,
            RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateLinkModuleFlowResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateLinkModuleFlowResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateLinkModuleFlowRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLinkModuleFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLinkModuleFlow");

            return clientHandler.execute(new ClientExecutionParams<UpdateLinkModuleFlowRequest, UpdateLinkModuleFlowResponse>()
                    .withOperationName("UpdateLinkModuleFlow").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateLinkModuleFlowRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateLinkModuleFlowRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a requester gateway.
     * </p>
     *
     * @param updateRequesterGatewayRequest
     * @return Result of the UpdateRequesterGateway operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request could not be completed because the resource does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have sufficient access to perform this action.
     * @throws ConflictException
     *         The request could not be completed because of a conflict in the current state of the resource.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.UpdateRequesterGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/UpdateRequesterGateway"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateRequesterGatewayResponse updateRequesterGateway(UpdateRequesterGatewayRequest updateRequesterGatewayRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, ConflictException,
            InternalServerException, ValidationException, AwsServiceException, SdkClientException, RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateRequesterGatewayResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateRequesterGatewayResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateRequesterGatewayRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRequesterGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRequesterGateway");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateRequesterGatewayRequest, UpdateRequesterGatewayResponse>()
                            .withOperationName("UpdateRequesterGateway").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateRequesterGatewayRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateRequesterGatewayRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a responder gateway.
     * </p>
     *
     * @param updateResponderGatewayRequest
     * @return Result of the UpdateResponderGateway operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request could not be completed because the resource does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have sufficient access to perform this action.
     * @throws ConflictException
     *         The request could not be completed because of a conflict in the current state of the resource.
     * @throws InternalServerException
     *         The request could not be completed because of an internal server error. Try your call again.
     * @throws ValidationException
     *         The request could not be completed because it fails satisfy the constraints specified by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws RtbFabricException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample RtbFabricClient.UpdateResponderGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/UpdateResponderGateway"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateResponderGatewayResponse updateResponderGateway(UpdateResponderGatewayRequest updateResponderGatewayRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, ConflictException,
            InternalServerException, ValidationException, AwsServiceException, SdkClientException, RtbFabricException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateResponderGatewayResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateResponderGatewayResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateResponderGatewayRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateResponderGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateResponderGateway");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateResponderGatewayRequest, UpdateResponderGatewayResponse>()
                            .withOperationName("UpdateResponderGateway").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateResponderGatewayRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateResponderGatewayRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Create an instance of {@link RtbFabricWaiter} using this client.
     * <p>
     * Waiters created via this method are managed by the SDK and resources will be released when the service client is
     * closed.
     *
     * @return an instance of {@link RtbFabricWaiter}
     */
    @Override
    public RtbFabricWaiter waiter() {
        return RtbFabricWaiter.builder().client(this).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        RtbFabricServiceClientConfigurationBuilder serviceConfigBuilder = new RtbFabricServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(RtbFabricException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    @Override
    public final RtbFabricServiceClientConfiguration serviceClientConfiguration() {
        return new RtbFabricServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
