/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.resourcegroups;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.resourcegroups.model.BadRequestException;
import software.amazon.awssdk.services.resourcegroups.model.CreateGroupRequest;
import software.amazon.awssdk.services.resourcegroups.model.CreateGroupResponse;
import software.amazon.awssdk.services.resourcegroups.model.DeleteGroupRequest;
import software.amazon.awssdk.services.resourcegroups.model.DeleteGroupResponse;
import software.amazon.awssdk.services.resourcegroups.model.ForbiddenException;
import software.amazon.awssdk.services.resourcegroups.model.GetAccountSettingsRequest;
import software.amazon.awssdk.services.resourcegroups.model.GetAccountSettingsResponse;
import software.amazon.awssdk.services.resourcegroups.model.GetGroupConfigurationRequest;
import software.amazon.awssdk.services.resourcegroups.model.GetGroupConfigurationResponse;
import software.amazon.awssdk.services.resourcegroups.model.GetGroupQueryRequest;
import software.amazon.awssdk.services.resourcegroups.model.GetGroupQueryResponse;
import software.amazon.awssdk.services.resourcegroups.model.GetGroupRequest;
import software.amazon.awssdk.services.resourcegroups.model.GetGroupResponse;
import software.amazon.awssdk.services.resourcegroups.model.GetTagsRequest;
import software.amazon.awssdk.services.resourcegroups.model.GetTagsResponse;
import software.amazon.awssdk.services.resourcegroups.model.GroupResourcesRequest;
import software.amazon.awssdk.services.resourcegroups.model.GroupResourcesResponse;
import software.amazon.awssdk.services.resourcegroups.model.InternalServerErrorException;
import software.amazon.awssdk.services.resourcegroups.model.ListGroupResourcesRequest;
import software.amazon.awssdk.services.resourcegroups.model.ListGroupResourcesResponse;
import software.amazon.awssdk.services.resourcegroups.model.ListGroupsRequest;
import software.amazon.awssdk.services.resourcegroups.model.ListGroupsResponse;
import software.amazon.awssdk.services.resourcegroups.model.MethodNotAllowedException;
import software.amazon.awssdk.services.resourcegroups.model.NotFoundException;
import software.amazon.awssdk.services.resourcegroups.model.PutGroupConfigurationRequest;
import software.amazon.awssdk.services.resourcegroups.model.PutGroupConfigurationResponse;
import software.amazon.awssdk.services.resourcegroups.model.ResourceGroupsException;
import software.amazon.awssdk.services.resourcegroups.model.SearchResourcesRequest;
import software.amazon.awssdk.services.resourcegroups.model.SearchResourcesResponse;
import software.amazon.awssdk.services.resourcegroups.model.TagRequest;
import software.amazon.awssdk.services.resourcegroups.model.TagResponse;
import software.amazon.awssdk.services.resourcegroups.model.TooManyRequestsException;
import software.amazon.awssdk.services.resourcegroups.model.UnauthorizedException;
import software.amazon.awssdk.services.resourcegroups.model.UngroupResourcesRequest;
import software.amazon.awssdk.services.resourcegroups.model.UngroupResourcesResponse;
import software.amazon.awssdk.services.resourcegroups.model.UntagRequest;
import software.amazon.awssdk.services.resourcegroups.model.UntagResponse;
import software.amazon.awssdk.services.resourcegroups.model.UpdateAccountSettingsRequest;
import software.amazon.awssdk.services.resourcegroups.model.UpdateAccountSettingsResponse;
import software.amazon.awssdk.services.resourcegroups.model.UpdateGroupQueryRequest;
import software.amazon.awssdk.services.resourcegroups.model.UpdateGroupQueryResponse;
import software.amazon.awssdk.services.resourcegroups.model.UpdateGroupRequest;
import software.amazon.awssdk.services.resourcegroups.model.UpdateGroupResponse;
import software.amazon.awssdk.services.resourcegroups.transform.CreateGroupRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.DeleteGroupRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.GetAccountSettingsRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.GetGroupConfigurationRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.GetGroupQueryRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.GetGroupRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.GetTagsRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.GroupResourcesRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.ListGroupResourcesRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.ListGroupsRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.PutGroupConfigurationRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.SearchResourcesRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.TagRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.UngroupResourcesRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.UntagRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.UpdateAccountSettingsRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.UpdateGroupQueryRequestMarshaller;
import software.amazon.awssdk.services.resourcegroups.transform.UpdateGroupRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ResourceGroupsClient}.
 *
 * @see ResourceGroupsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultResourceGroupsClient implements ResourceGroupsClient {
    private static final Logger log = Logger.loggerFor(DefaultResourceGroupsClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final ResourceGroupsServiceClientConfiguration serviceClientConfiguration;

    protected DefaultResourceGroupsClient(ResourceGroupsServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a resource group with the specified name and description. You can optionally include either a resource
     * query or a service configuration. For more information about constructing a resource query, see <a
     * href="https://docs.aws.amazon.com/ARG/latest/userguide/getting_started-query.html">Build queries and groups in
     * Resource Groups</a> in the <i>Resource Groups User Guide</i>. For more information about service-linked groups
     * and service configurations, see <a
     * href="https://docs.aws.amazon.com/ARG/latest/APIReference/about-slg.html">Service configurations for Resource
     * Groups</a>.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:CreateGroup</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param createGroupRequest
     * @return Result of the CreateGroup operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.CreateGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/CreateGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateGroupResponse createGroup(CreateGroupRequest createGroupRequest) throws BadRequestException, ForbiddenException,
            MethodNotAllowedException, TooManyRequestsException, InternalServerErrorException, AwsServiceException,
            SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGroup");

            return clientHandler.execute(new ClientExecutionParams<CreateGroupRequest, CreateGroupResponse>()
                    .withOperationName("CreateGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified resource group. Deleting a resource group does not delete any resources that are members of
     * the group; it only deletes the group structure.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:DeleteGroup</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteGroupRequest
     * @return Result of the DeleteGroup operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.DeleteGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/DeleteGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteGroupResponse deleteGroup(DeleteGroupRequest deleteGroupRequest) throws BadRequestException, ForbiddenException,
            NotFoundException, MethodNotAllowedException, TooManyRequestsException, InternalServerErrorException,
            AwsServiceException, SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGroup");

            return clientHandler.execute(new ClientExecutionParams<DeleteGroupRequest, DeleteGroupResponse>()
                    .withOperationName("DeleteGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the current status of optional features in Resource Groups.
     * </p>
     *
     * @param getAccountSettingsRequest
     * @return Result of the GetAccountSettings operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.GetAccountSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GetAccountSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAccountSettingsResponse getAccountSettings(GetAccountSettingsRequest getAccountSettingsRequest)
            throws BadRequestException, ForbiddenException, MethodNotAllowedException, TooManyRequestsException,
            InternalServerErrorException, AwsServiceException, SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAccountSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetAccountSettingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccountSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccountSettings");

            return clientHandler.execute(new ClientExecutionParams<GetAccountSettingsRequest, GetAccountSettingsResponse>()
                    .withOperationName("GetAccountSettings").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getAccountSettingsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAccountSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a specified resource group.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:GetGroup</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param getGroupRequest
     * @return Result of the GetGroup operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.GetGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GetGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetGroupResponse getGroup(GetGroupRequest getGroupRequest) throws BadRequestException, ForbiddenException,
            NotFoundException, MethodNotAllowedException, TooManyRequestsException, InternalServerErrorException,
            AwsServiceException, SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGroup");

            return clientHandler.execute(new ClientExecutionParams<GetGroupRequest, GetGroupResponse>()
                    .withOperationName("GetGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getGroupRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the service configuration associated with the specified resource group. For details about the service
     * configuration syntax, see <a href="https://docs.aws.amazon.com/ARG/latest/APIReference/about-slg.html">Service
     * configurations for Resource Groups</a>.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:GetGroupConfiguration</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param getGroupConfigurationRequest
     * @return Result of the GetGroupConfiguration operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.GetGroupConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GetGroupConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetGroupConfigurationResponse getGroupConfiguration(GetGroupConfigurationRequest getGroupConfigurationRequest)
            throws BadRequestException, ForbiddenException, NotFoundException, MethodNotAllowedException,
            TooManyRequestsException, InternalServerErrorException, AwsServiceException, SdkClientException,
            ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetGroupConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetGroupConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGroupConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGroupConfiguration");

            return clientHandler.execute(new ClientExecutionParams<GetGroupConfigurationRequest, GetGroupConfigurationResponse>()
                    .withOperationName("GetGroupConfiguration").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getGroupConfigurationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetGroupConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the resource query associated with the specified resource group. For more information about resource
     * queries, see <a
     * href="https://docs.aws.amazon.com/ARG/latest/userguide/gettingstarted-query.html#gettingstarted-query-cli-tag"
     * >Create a tag-based group in Resource Groups</a>.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:GetGroupQuery</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param getGroupQueryRequest
     * @return Result of the GetGroupQuery operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.GetGroupQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GetGroupQuery" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetGroupQueryResponse getGroupQuery(GetGroupQueryRequest getGroupQueryRequest) throws BadRequestException,
            ForbiddenException, NotFoundException, MethodNotAllowedException, TooManyRequestsException,
            InternalServerErrorException, AwsServiceException, SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetGroupQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetGroupQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGroupQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGroupQuery");

            return clientHandler.execute(new ClientExecutionParams<GetGroupQueryRequest, GetGroupQueryResponse>()
                    .withOperationName("GetGroupQuery").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getGroupQueryRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetGroupQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of tags that are associated with a resource group, specified by an ARN.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:GetTags</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param getTagsRequest
     * @return Result of the GetTags operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.GetTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GetTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTagsResponse getTags(GetTagsRequest getTagsRequest) throws BadRequestException, ForbiddenException,
            NotFoundException, MethodNotAllowedException, TooManyRequestsException, InternalServerErrorException,
            AwsServiceException, SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTags");

            return clientHandler.execute(new ClientExecutionParams<GetTagsRequest, GetTagsResponse>()
                    .withOperationName("GetTags").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getTagsRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds the specified resources to the specified group.
     * </p>
     * <important>
     * <p>
     * You can use this operation with only resource groups that are configured with the following types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AWS::EC2::HostManagement</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AWS::EC2::CapacityReservationPool</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * Other resource group type and resource types aren't currently supported by this operation.
     * </p>
     * </important>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:GroupResources</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param groupResourcesRequest
     * @return Result of the GroupResources operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.GroupResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/GroupResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GroupResourcesResponse groupResources(GroupResourcesRequest groupResourcesRequest) throws BadRequestException,
            ForbiddenException, NotFoundException, MethodNotAllowedException, TooManyRequestsException,
            InternalServerErrorException, AwsServiceException, SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GroupResourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GroupResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, groupResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GroupResources");

            return clientHandler.execute(new ClientExecutionParams<GroupResourcesRequest, GroupResourcesResponse>()
                    .withOperationName("GroupResources").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(groupResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GroupResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of ARNs of the resources that are members of a specified resource group.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:ListGroupResources</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cloudformation:DescribeStacks</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cloudformation:ListStackResources</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>tag:GetResources</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param listGroupResourcesRequest
     * @return Result of the ListGroupResources operation returned by the service.
     * @throws UnauthorizedException
     *         The request was rejected because it doesn't have valid credentials for the target resource.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.ListGroupResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/ListGroupResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListGroupResourcesResponse listGroupResources(ListGroupResourcesRequest listGroupResourcesRequest)
            throws UnauthorizedException, BadRequestException, ForbiddenException, NotFoundException, MethodNotAllowedException,
            TooManyRequestsException, InternalServerErrorException, AwsServiceException, SdkClientException,
            ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListGroupResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListGroupResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGroupResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGroupResources");

            return clientHandler.execute(new ClientExecutionParams<ListGroupResourcesRequest, ListGroupResourcesResponse>()
                    .withOperationName("ListGroupResources").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listGroupResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListGroupResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of existing Resource Groups in your account.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:ListGroups</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param listGroupsRequest
     * @return Result of the ListGroups operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.ListGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/ListGroups" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListGroupsResponse listGroups(ListGroupsRequest listGroupsRequest) throws BadRequestException, ForbiddenException,
            MethodNotAllowedException, TooManyRequestsException, InternalServerErrorException, AwsServiceException,
            SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListGroupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGroups");

            return clientHandler
                    .execute(new ClientExecutionParams<ListGroupsRequest, ListGroupsResponse>().withOperationName("ListGroups")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listGroupsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Attaches a service configuration to the specified group. This occurs asynchronously, and can take time to
     * complete. You can use <a>GetGroupConfiguration</a> to check the status of the update.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:PutGroupConfiguration</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param putGroupConfigurationRequest
     * @return Result of the PutGroupConfiguration operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.PutGroupConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/PutGroupConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutGroupConfigurationResponse putGroupConfiguration(PutGroupConfigurationRequest putGroupConfigurationRequest)
            throws BadRequestException, ForbiddenException, NotFoundException, MethodNotAllowedException,
            TooManyRequestsException, InternalServerErrorException, AwsServiceException, SdkClientException,
            ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutGroupConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutGroupConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putGroupConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutGroupConfiguration");

            return clientHandler.execute(new ClientExecutionParams<PutGroupConfigurationRequest, PutGroupConfigurationResponse>()
                    .withOperationName("PutGroupConfiguration").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putGroupConfigurationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutGroupConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of Amazon Web Services resource identifiers that matches the specified query. The query uses the
     * same format as a resource query in a <a>CreateGroup</a> or <a>UpdateGroupQuery</a> operation.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:SearchResources</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cloudformation:DescribeStacks</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cloudformation:ListStackResources</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>tag:GetResources</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param searchResourcesRequest
     * @return Result of the SearchResources operation returned by the service.
     * @throws UnauthorizedException
     *         The request was rejected because it doesn't have valid credentials for the target resource.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.SearchResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/SearchResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SearchResourcesResponse searchResources(SearchResourcesRequest searchResourcesRequest) throws UnauthorizedException,
            BadRequestException, ForbiddenException, MethodNotAllowedException, TooManyRequestsException,
            InternalServerErrorException, AwsServiceException, SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SearchResourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SearchResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchResources");

            return clientHandler.execute(new ClientExecutionParams<SearchResourcesRequest, SearchResourcesResponse>()
                    .withOperationName("SearchResources").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(searchResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SearchResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds tags to a resource group with the specified ARN. Existing tags on a resource group are not changed if they
     * are not specified in the request parameters.
     * </p>
     * <important>
     * <p>
     * Do not store personally identifiable information (PII) or other confidential or sensitive information in tags. We
     * use tags to provide you with billing and administration services. Tags are not intended to be used for private or
     * sensitive data.
     * </p>
     * </important>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:Tag</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param tagRequest
     * @return Result of the Tag operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.Tag
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/Tag" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResponse tag(TagRequest tagRequest) throws BadRequestException, ForbiddenException, NotFoundException,
            MethodNotAllowedException, TooManyRequestsException, InternalServerErrorException, AwsServiceException,
            SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagRequest.overrideConfiguration()
                .orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Tag");

            return clientHandler.execute(new ClientExecutionParams<TagRequest, TagResponse>().withOperationName("Tag")
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(tagRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new TagRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the specified resources from the specified group. This operation works only with static groups that you
     * populated using the <a>GroupResources</a> operation. It doesn't work with any resource groups that are
     * automatically populated by tag-based or CloudFormation stack-based queries.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:UngroupResources</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param ungroupResourcesRequest
     * @return Result of the UngroupResources operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.UngroupResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/UngroupResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UngroupResourcesResponse ungroupResources(UngroupResourcesRequest ungroupResourcesRequest) throws BadRequestException,
            ForbiddenException, NotFoundException, MethodNotAllowedException, TooManyRequestsException,
            InternalServerErrorException, AwsServiceException, SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UngroupResourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UngroupResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, ungroupResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UngroupResources");

            return clientHandler.execute(new ClientExecutionParams<UngroupResourcesRequest, UngroupResourcesResponse>()
                    .withOperationName("UngroupResources").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(ungroupResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UngroupResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes tags from a specified resource group.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:Untag</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param untagRequest
     * @return Result of the Untag operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.Untag
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/Untag" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResponse untag(UntagRequest untagRequest) throws BadRequestException, ForbiddenException, NotFoundException,
            MethodNotAllowedException, TooManyRequestsException, InternalServerErrorException, AwsServiceException,
            SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Untag");

            return clientHandler.execute(new ClientExecutionParams<UntagRequest, UntagResponse>().withOperationName("Untag")
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(untagRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new UntagRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Turns on or turns off optional features in Resource Groups.
     * </p>
     * <p>
     * The preceding example shows that the request to turn on group lifecycle events is <code>IN_PROGRESS</code>. You
     * can call the <a>GetAccountSettings</a> operation to check for completion by looking for
     * <code>GroupLifecycleEventsStatus</code> to change to <code>ACTIVE</code>.
     * </p>
     *
     * @param updateAccountSettingsRequest
     * @return Result of the UpdateAccountSettings operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.UpdateAccountSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/UpdateAccountSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateAccountSettingsResponse updateAccountSettings(UpdateAccountSettingsRequest updateAccountSettingsRequest)
            throws BadRequestException, ForbiddenException, MethodNotAllowedException, TooManyRequestsException,
            InternalServerErrorException, AwsServiceException, SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateAccountSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateAccountSettingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAccountSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAccountSettings");

            return clientHandler.execute(new ClientExecutionParams<UpdateAccountSettingsRequest, UpdateAccountSettingsResponse>()
                    .withOperationName("UpdateAccountSettings").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateAccountSettingsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateAccountSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the description for an existing group. You cannot update the name of a resource group.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:UpdateGroup</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateGroupRequest
     * @return Result of the UpdateGroup operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.UpdateGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/UpdateGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateGroupResponse updateGroup(UpdateGroupRequest updateGroupRequest) throws BadRequestException, ForbiddenException,
            NotFoundException, MethodNotAllowedException, TooManyRequestsException, InternalServerErrorException,
            AwsServiceException, SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGroup");

            return clientHandler.execute(new ClientExecutionParams<UpdateGroupRequest, UpdateGroupResponse>()
                    .withOperationName("UpdateGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the resource query of a group. For more information about resource queries, see <a
     * href="https://docs.aws.amazon.com/ARG/latest/userguide/gettingstarted-query.html#gettingstarted-query-cli-tag"
     * >Create a tag-based group in Resource Groups</a>.
     * </p>
     * <p>
     * <b>Minimum permissions</b>
     * </p>
     * <p>
     * To run this command, you must have the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>resource-groups:UpdateGroupQuery</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateGroupQueryRequest
     * @return Result of the UpdateGroupQuery operation returned by the service.
     * @throws BadRequestException
     *         The request includes one or more parameters that violate validation rules.
     * @throws ForbiddenException
     *         The caller isn't authorized to make the request. Check permissions.
     * @throws NotFoundException
     *         One or more of the specified resources don't exist.
     * @throws MethodNotAllowedException
     *         The request uses an HTTP method that isn't allowed for the specified resource.
     * @throws TooManyRequestsException
     *         You've exceeded throttling limits by making too many requests in a period of time.
     * @throws InternalServerErrorException
     *         An internal error occurred while processing the request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ResourceGroupsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ResourceGroupsClient.UpdateGroupQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/resource-groups-2017-11-27/UpdateGroupQuery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateGroupQueryResponse updateGroupQuery(UpdateGroupQueryRequest updateGroupQueryRequest) throws BadRequestException,
            ForbiddenException, NotFoundException, MethodNotAllowedException, TooManyRequestsException,
            InternalServerErrorException, AwsServiceException, SdkClientException, ResourceGroupsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateGroupQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateGroupQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGroupQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Resource Groups");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGroupQuery");

            return clientHandler.execute(new ClientExecutionParams<UpdateGroupQueryRequest, UpdateGroupQueryResponse>()
                    .withOperationName("UpdateGroupQuery").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateGroupQueryRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateGroupQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ResourceGroupsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MethodNotAllowedException")
                                .exceptionBuilderSupplier(MethodNotAllowedException::builder).httpStatusCode(405).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    @Override
    public final ResourceGroupsServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
