/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.protocolrestjsoncustomized;

import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.protocolrestjsoncustomized.model.ProtocolRestJsonCustomizedException;
import software.amazon.awssdk.services.protocolrestjsoncustomized.model.SimpleRequest;
import software.amazon.awssdk.services.protocolrestjsoncustomized.model.SimpleResponse;
import software.amazon.awssdk.services.protocolrestjsoncustomized.transform.SimpleRequestMarshaller;

/**
 * Internal implementation of {@link ProtocolRestJsonCustomizedClient}.
 *
 * @see ProtocolRestJsonCustomizedClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultProtocolRestJsonCustomizedClient implements ProtocolRestJsonCustomizedClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultProtocolRestJsonCustomizedClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * Invokes the Simple operation.
     *
     * @param simpleRequest
     * @return Result of the Simple operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ProtocolRestJsonCustomizedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ProtocolRestJsonCustomizedClient.Simple
     */
    @Override
    public SimpleResponse simple(SimpleRequest simpleRequest) throws AwsServiceException, SdkClientException,
            ProtocolRestJsonCustomizedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SimpleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SimpleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<SimpleRequest, SimpleResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(simpleRequest)
                .withMarshaller(new SimpleRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ProtocolRestJsonCustomizedException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
