/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.protocolquery;

import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.query.AwsQueryProtocolFactory;
import software.amazon.awssdk.services.protocolquery.model.AllTypesRequest;
import software.amazon.awssdk.services.protocolquery.model.AllTypesResponse;
import software.amazon.awssdk.services.protocolquery.model.EmptyModeledException;
import software.amazon.awssdk.services.protocolquery.model.IdempotentOperationRequest;
import software.amazon.awssdk.services.protocolquery.model.IdempotentOperationResponse;
import software.amazon.awssdk.services.protocolquery.model.ImplicitPayloadException;
import software.amazon.awssdk.services.protocolquery.model.ProtocolQueryException;
import software.amazon.awssdk.services.protocolquery.model.QueryTypesRequest;
import software.amazon.awssdk.services.protocolquery.model.QueryTypesResponse;
import software.amazon.awssdk.services.protocolquery.transform.AllTypesRequestMarshaller;
import software.amazon.awssdk.services.protocolquery.transform.IdempotentOperationRequestMarshaller;
import software.amazon.awssdk.services.protocolquery.transform.QueryTypesRequestMarshaller;

/**
 * Internal implementation of {@link ProtocolQueryClient}.
 *
 * @see ProtocolQueryClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultProtocolQueryClient implements ProtocolQueryClient {
    private final SyncClientHandler clientHandler;

    private final AwsQueryProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultProtocolQueryClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * Invokes the AllTypes operation.
     *
     * @param allTypesRequest
     * @return Result of the AllTypes operation returned by the service.
     * @throws EmptyModeledException
     * @throws ImplicitPayloadException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ProtocolQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ProtocolQueryClient.AllTypes
     */
    @Override
    public AllTypesResponse allTypes(AllTypesRequest allTypesRequest) throws EmptyModeledException, ImplicitPayloadException,
            AwsServiceException, SdkClientException, ProtocolQueryException {

        HttpResponseHandler<AllTypesResponse> responseHandler = protocolFactory.createResponseHandler(AllTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<AllTypesRequest, AllTypesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(allTypesRequest)
                .withMarshaller(new AllTypesRequestMarshaller(protocolFactory)));
    }

    /**
     * Invokes the IdempotentOperation operation.
     *
     * @param idempotentOperationRequest
     * @return Result of the IdempotentOperation operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ProtocolQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ProtocolQueryClient.IdempotentOperation
     */
    @Override
    public IdempotentOperationResponse idempotentOperation(IdempotentOperationRequest idempotentOperationRequest)
            throws AwsServiceException, SdkClientException, ProtocolQueryException {

        HttpResponseHandler<IdempotentOperationResponse> responseHandler = protocolFactory
                .createResponseHandler(IdempotentOperationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<IdempotentOperationRequest, IdempotentOperationResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(idempotentOperationRequest).withMarshaller(new IdempotentOperationRequestMarshaller(protocolFactory)));
    }

    /**
     * Invokes the QueryTypes operation.
     *
     * @param queryTypesRequest
     * @return Result of the QueryTypes operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ProtocolQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ProtocolQueryClient.QueryTypes
     */
    @Override
    public QueryTypesResponse queryTypes(QueryTypesRequest queryTypesRequest) throws AwsServiceException, SdkClientException,
            ProtocolQueryException {

        HttpResponseHandler<QueryTypesResponse> responseHandler = protocolFactory
                .createResponseHandler(QueryTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<QueryTypesRequest, QueryTypesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(queryTypesRequest)
                .withMarshaller(new QueryTypesRequestMarshaller(protocolFactory)));
    }

    private AwsQueryProtocolFactory init() {
        return AwsQueryProtocolFactory
                .builder()
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EmptyModeledException")
                                .exceptionBuilderSupplier(EmptyModeledException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ImplicitPayloadException")
                                .exceptionBuilderSupplier(ImplicitPayloadException::builder).build())
                .clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(ProtocolQueryException::builder)
                .build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
