/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.protocoljsonrpc;

import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.protocoljsonrpc.model.AllTypesRequest;
import software.amazon.awssdk.services.protocoljsonrpc.model.AllTypesResponse;
import software.amazon.awssdk.services.protocoljsonrpc.model.EmptyModeledException;
import software.amazon.awssdk.services.protocoljsonrpc.model.FurtherNestedContainersRequest;
import software.amazon.awssdk.services.protocoljsonrpc.model.FurtherNestedContainersResponse;
import software.amazon.awssdk.services.protocoljsonrpc.model.IdempotentOperationRequest;
import software.amazon.awssdk.services.protocoljsonrpc.model.IdempotentOperationResponse;
import software.amazon.awssdk.services.protocoljsonrpc.model.ImplicitPayloadException;
import software.amazon.awssdk.services.protocoljsonrpc.model.NestedContainersRequest;
import software.amazon.awssdk.services.protocoljsonrpc.model.NestedContainersResponse;
import software.amazon.awssdk.services.protocoljsonrpc.model.OperationWithNoInputOrOutputRequest;
import software.amazon.awssdk.services.protocoljsonrpc.model.OperationWithNoInputOrOutputResponse;
import software.amazon.awssdk.services.protocoljsonrpc.model.ProtocolJsonRpcException;
import software.amazon.awssdk.services.protocoljsonrpc.transform.AllTypesRequestMarshaller;
import software.amazon.awssdk.services.protocoljsonrpc.transform.FurtherNestedContainersRequestMarshaller;
import software.amazon.awssdk.services.protocoljsonrpc.transform.IdempotentOperationRequestMarshaller;
import software.amazon.awssdk.services.protocoljsonrpc.transform.NestedContainersRequestMarshaller;
import software.amazon.awssdk.services.protocoljsonrpc.transform.OperationWithNoInputOrOutputRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ProtocolJsonRpcAsyncClient}.
 *
 * @see ProtocolJsonRpcAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultProtocolJsonRpcAsyncClient implements ProtocolJsonRpcAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultProtocolJsonRpcAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultProtocolJsonRpcAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * Invokes the AllTypes operation asynchronously.
     *
     * @param allTypesRequest
     * @return A Java Future containing the result of the AllTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EmptyModeledException</li>
     *         <li>ImplicitPayloadException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolJsonRpcException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ProtocolJsonRpcAsyncClient.AllTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/jsonrpc-2016-03-11/AllTypes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AllTypesResponse> allTypes(AllTypesRequest allTypesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AllTypesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    AllTypesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<AllTypesRequest, AllTypesResponse>()
                    .withMarshaller(new AllTypesRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(allTypesRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the FurtherNestedContainers operation asynchronously.
     *
     * @param furtherNestedContainersRequest
     * @return A Java Future containing the result of the FurtherNestedContainers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolJsonRpcException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ProtocolJsonRpcAsyncClient.FurtherNestedContainers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/jsonrpc-2016-03-11/FurtherNestedContainers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<FurtherNestedContainersResponse> furtherNestedContainers(
            FurtherNestedContainersRequest furtherNestedContainersRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<FurtherNestedContainersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, FurtherNestedContainersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler
                    .execute(new ClientExecutionParams<FurtherNestedContainersRequest, FurtherNestedContainersResponse>()
                            .withMarshaller(new FurtherNestedContainersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(furtherNestedContainersRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the IdempotentOperation operation asynchronously.
     *
     * @param idempotentOperationRequest
     * @return A Java Future containing the result of the IdempotentOperation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolJsonRpcException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ProtocolJsonRpcAsyncClient.IdempotentOperation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/jsonrpc-2016-03-11/IdempotentOperation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<IdempotentOperationResponse> idempotentOperation(
            IdempotentOperationRequest idempotentOperationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<IdempotentOperationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, IdempotentOperationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<IdempotentOperationRequest, IdempotentOperationResponse>()
                    .withMarshaller(new IdempotentOperationRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(idempotentOperationRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the NestedContainers operation asynchronously.
     *
     * @param nestedContainersRequest
     * @return A Java Future containing the result of the NestedContainers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolJsonRpcException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ProtocolJsonRpcAsyncClient.NestedContainers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/jsonrpc-2016-03-11/NestedContainers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<NestedContainersResponse> nestedContainers(NestedContainersRequest nestedContainersRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<NestedContainersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, NestedContainersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<NestedContainersRequest, NestedContainersResponse>()
                    .withMarshaller(new NestedContainersRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(nestedContainersRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the OperationWithNoInputOrOutput operation asynchronously.
     *
     * @param operationWithNoInputOrOutputRequest
     * @return A Java Future containing the result of the OperationWithNoInputOrOutput operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ProtocolJsonRpcException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ProtocolJsonRpcAsyncClient.OperationWithNoInputOrOutput
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/jsonrpc-2016-03-11/OperationWithNoInputOrOutput"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<OperationWithNoInputOrOutputResponse> operationWithNoInputOrOutput(
            OperationWithNoInputOrOutputRequest operationWithNoInputOrOutputRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<OperationWithNoInputOrOutputResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, OperationWithNoInputOrOutputResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler
                    .execute(new ClientExecutionParams<OperationWithNoInputOrOutputRequest, OperationWithNoInputOrOutputResponse>()
                            .withMarshaller(new OperationWithNoInputOrOutputRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(operationWithNoInputOrOutputRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ProtocolJsonRpcException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ImplicitPayloadException")
                                .exceptionBuilderSupplier(ImplicitPayloadException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EmptyModeledException")
                                .exceptionBuilderSupplier(EmptyModeledException::builder).build());
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
