/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.protocolec2;

import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.query.AwsEc2ProtocolFactory;
import software.amazon.awssdk.services.protocolec2.model.AllTypesRequest;
import software.amazon.awssdk.services.protocolec2.model.AllTypesResponse;
import software.amazon.awssdk.services.protocolec2.model.Ec2TypesRequest;
import software.amazon.awssdk.services.protocolec2.model.Ec2TypesResponse;
import software.amazon.awssdk.services.protocolec2.model.EmptyModeledException;
import software.amazon.awssdk.services.protocolec2.model.IdempotentOperationRequest;
import software.amazon.awssdk.services.protocolec2.model.IdempotentOperationResponse;
import software.amazon.awssdk.services.protocolec2.model.ImplicitPayloadException;
import software.amazon.awssdk.services.protocolec2.model.ProtocolEc2Exception;
import software.amazon.awssdk.services.protocolec2.transform.AllTypesRequestMarshaller;
import software.amazon.awssdk.services.protocolec2.transform.Ec2TypesRequestMarshaller;
import software.amazon.awssdk.services.protocolec2.transform.IdempotentOperationRequestMarshaller;

/**
 * Internal implementation of {@link ProtocolEc2Client}.
 *
 * @see ProtocolEc2Client#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultProtocolEc2Client implements ProtocolEc2Client {
    private final SyncClientHandler clientHandler;

    private final AwsEc2ProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultProtocolEc2Client(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * Invokes the AllTypes operation.
     *
     * @param allTypesRequest
     * @return Result of the AllTypes operation returned by the service.
     * @throws EmptyModeledException
     * @throws ImplicitPayloadException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ProtocolEc2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ProtocolEc2Client.AllTypes
     */
    @Override
    public AllTypesResponse allTypes(AllTypesRequest allTypesRequest) throws EmptyModeledException, ImplicitPayloadException,
            AwsServiceException, SdkClientException, ProtocolEc2Exception {

        HttpResponseHandler<AllTypesResponse> responseHandler = protocolFactory.createResponseHandler(AllTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<AllTypesRequest, AllTypesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(allTypesRequest)
                .withMarshaller(new AllTypesRequestMarshaller(protocolFactory)));
    }

    /**
     * Invokes the Ec2Types operation.
     *
     * @param ec2TypesRequest
     * @return Result of the Ec2Types operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ProtocolEc2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ProtocolEc2Client.Ec2Types
     */
    @Override
    public Ec2TypesResponse ec2Types(Ec2TypesRequest ec2TypesRequest) throws AwsServiceException, SdkClientException,
            ProtocolEc2Exception {

        HttpResponseHandler<Ec2TypesResponse> responseHandler = protocolFactory.createResponseHandler(Ec2TypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<Ec2TypesRequest, Ec2TypesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(ec2TypesRequest)
                .withMarshaller(new Ec2TypesRequestMarshaller(protocolFactory)));
    }

    /**
     * Invokes the IdempotentOperation operation.
     *
     * @param idempotentOperationRequest
     * @return Result of the IdempotentOperation operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ProtocolEc2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ProtocolEc2Client.IdempotentOperation
     */
    @Override
    public IdempotentOperationResponse idempotentOperation(IdempotentOperationRequest idempotentOperationRequest)
            throws AwsServiceException, SdkClientException, ProtocolEc2Exception {

        HttpResponseHandler<IdempotentOperationResponse> responseHandler = protocolFactory
                .createResponseHandler(IdempotentOperationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<IdempotentOperationRequest, IdempotentOperationResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(idempotentOperationRequest).withMarshaller(new IdempotentOperationRequestMarshaller(protocolFactory)));
    }

    private AwsEc2ProtocolFactory init() {
        return AwsEc2ProtocolFactory
                .builder()
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EmptyModeledException")
                                .exceptionBuilderSupplier(EmptyModeledException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ImplicitPayloadException")
                                .exceptionBuilderSupplier(ImplicitPayloadException::builder).build())
                .clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(ProtocolEc2Exception::builder).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
