/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.protocolec2;

import java.util.ArrayList;
import java.util.List;
import javax.annotation.Generated;
import org.w3c.dom.Node;
import software.amazon.awssdk.AmazonClientException;
import software.amazon.awssdk.AmazonServiceException;
import software.amazon.awssdk.AmazonWebServiceRequest;
import software.amazon.awssdk.Request;
import software.amazon.awssdk.SdkBaseException;
import software.amazon.awssdk.SdkClientException;
import software.amazon.awssdk.annotation.SdkInternalApi;
import software.amazon.awssdk.client.AwsSyncClientParams;
import software.amazon.awssdk.client.ClientExecutionParams;
import software.amazon.awssdk.client.ClientHandler;
import software.amazon.awssdk.client.ClientHandlerParams;
import software.amazon.awssdk.client.SdkClientHandler;
import software.amazon.awssdk.http.DefaultErrorResponseHandler;
import software.amazon.awssdk.http.StaxResponseHandler;
import software.amazon.awssdk.runtime.transform.LegacyErrorUnmarshaller;
import software.amazon.awssdk.runtime.transform.Unmarshaller;
import software.amazon.awssdk.runtime.transform.VoidStaxUnmarshaller;
import software.amazon.awssdk.services.protocolec2.model.AllTypesRequest;
import software.amazon.awssdk.services.protocolec2.model.AllTypesResponse;
import software.amazon.awssdk.services.protocolec2.model.DryRunResult;
import software.amazon.awssdk.services.protocolec2.model.DryRunSupportedRequest;
import software.amazon.awssdk.services.protocolec2.model.Ec2TypesRequest;
import software.amazon.awssdk.services.protocolec2.model.Ec2TypesResponse;
import software.amazon.awssdk.services.protocolec2.model.IdempotentOperationRequest;
import software.amazon.awssdk.services.protocolec2.model.IdempotentOperationResponse;
import software.amazon.awssdk.services.protocolec2.model.ProtocolEc2Exception;
import software.amazon.awssdk.services.protocolec2.transform.AllTypesRequestMarshaller;
import software.amazon.awssdk.services.protocolec2.transform.AllTypesResponseUnmarshaller;
import software.amazon.awssdk.services.protocolec2.transform.Ec2TypesRequestMarshaller;
import software.amazon.awssdk.services.protocolec2.transform.Ec2TypesResponseUnmarshaller;
import software.amazon.awssdk.services.protocolec2.transform.IdempotentOperationRequestMarshaller;
import software.amazon.awssdk.services.protocolec2.transform.IdempotentOperationResponseUnmarshaller;

/**
 * Internal implementation of {@link ProtocolEc2Client}.
 *
 * @see ProtocolEc2Client#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultProtocolEc2Client implements ProtocolEc2Client {
    private final ClientHandler clientHandler;

    private final List<Unmarshaller<AmazonServiceException, Node>> exceptionUnmarshallers;

    private final AwsSyncClientParams clientParams;

    protected DefaultProtocolEc2Client(AwsSyncClientParams clientParams) {
        this.clientHandler = new SdkClientHandler(new ClientHandlerParams().withClientParams(clientParams)
                .withCalculateCrc32FromCompressedDataEnabled(false));
        this.clientParams = clientParams;
        this.exceptionUnmarshallers = init();
    }

    /**
     *
     * @param allTypesRequest
     * @return Result of the AllTypes operation returned by the service.
     * @sample ProtocolEc2Client.AllTypes
     */
    @Override
    public AllTypesResponse allTypes(AllTypesRequest allTypesRequest) throws SdkBaseException, SdkClientException,
            ProtocolEc2Exception {

        StaxResponseHandler<AllTypesResponse> responseHandler = new StaxResponseHandler<AllTypesResponse>(
                new AllTypesResponseUnmarshaller());

        DefaultErrorResponseHandler errorResponseHandler = new DefaultErrorResponseHandler(exceptionUnmarshallers);

        return clientHandler.execute(new ClientExecutionParams<AllTypesRequest, AllTypesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(allTypesRequest)
                .withMarshaller(new AllTypesRequestMarshaller()));
    }

    /**
     *
     * @param ec2TypesRequest
     * @return Result of the Ec2Types operation returned by the service.
     * @sample ProtocolEc2Client.Ec2Types
     */
    @Override
    public Ec2TypesResponse ec2Types(Ec2TypesRequest ec2TypesRequest) throws SdkBaseException, SdkClientException,
            ProtocolEc2Exception {

        StaxResponseHandler<Ec2TypesResponse> responseHandler = new StaxResponseHandler<Ec2TypesResponse>(
                new Ec2TypesResponseUnmarshaller());

        DefaultErrorResponseHandler errorResponseHandler = new DefaultErrorResponseHandler(exceptionUnmarshallers);

        return clientHandler.execute(new ClientExecutionParams<Ec2TypesRequest, Ec2TypesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(ec2TypesRequest)
                .withMarshaller(new Ec2TypesRequestMarshaller()));
    }

    /**
     *
     * @param idempotentOperationRequest
     * @return Result of the IdempotentOperation operation returned by the service.
     * @sample ProtocolEc2Client.IdempotentOperation
     */
    @Override
    public IdempotentOperationResponse idempotentOperation(IdempotentOperationRequest idempotentOperationRequest)
            throws SdkBaseException, SdkClientException, ProtocolEc2Exception {

        StaxResponseHandler<IdempotentOperationResponse> responseHandler = new StaxResponseHandler<IdempotentOperationResponse>(
                new IdempotentOperationResponseUnmarshaller());

        DefaultErrorResponseHandler errorResponseHandler = new DefaultErrorResponseHandler(exceptionUnmarshallers);

        return clientHandler.execute(new ClientExecutionParams<IdempotentOperationRequest, IdempotentOperationResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(idempotentOperationRequest).withMarshaller(new IdempotentOperationRequestMarshaller()));
    }

    private List<Unmarshaller<AmazonServiceException, Node>> init() {
        List<Unmarshaller<AmazonServiceException, Node>> unmarshallers = new ArrayList<>();
        unmarshallers.add(new LegacyErrorUnmarshaller(ProtocolEc2Exception.class));
        return unmarshallers;
    }

    @Override
    public void close() throws Exception {
        clientHandler.close();
    }

    public <X extends AmazonWebServiceRequest> DryRunResult<X> dryRun(DryRunSupportedRequest<X> request) {
        Request<X> dryRunRequest = request.getDryRunRequest();
        try {
            StaxResponseHandler<DryRunResult> responseHandler = new StaxResponseHandler<DryRunResult>(new VoidStaxUnmarshaller());

            clientHandler.execute(new ClientExecutionParams<Request, DryRunResult>().withMarshaller(null)
                    .withResponseHandler(responseHandler).withInput(dryRunRequest));
            throw new AmazonClientException("Unrecognized service response for the dry-run request.");
        } catch (AmazonServiceException ase) {
            if (ase.getErrorCode().equals("DryRunOperation") && ase.getStatusCode() == 412) {
                return new DryRunResult<X>(true, request, ase.getMessage(), ase);
            } else if (ase.getErrorCode().equals("UnauthorizedOperation") && ase.getStatusCode() == 403) {
                return new DryRunResult<X>(false, request, ase.getMessage(), ase);
            }
            throw new AmazonClientException("Unrecognized service response for the dry-run request.", ase);
        }
    }
}
