/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pinpoint.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the settings for a multivariate split activity in a journey. This type of activity sends participants down
 * one of as many as five paths (including a default <i>Else</i> path) in a journey, based on conditions that you
 * specify.
 * </p>
 * <note>
 * <p>
 * To create multivariate split activities that send participants down different paths based on push notification events
 * (such as Open or Received events), your mobile app has to specify the User ID and Endpoint ID values. For more
 * information, see <a href="https://docs.aws.amazon.com/pinpoint/latest/developerguide/integrate.html">Integrating
 * Amazon Pinpoint with your application</a> in the <i>Amazon Pinpoint Developer Guide</i>.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MultiConditionalSplitActivity implements SdkPojo, Serializable,
        ToCopyableBuilder<MultiConditionalSplitActivity.Builder, MultiConditionalSplitActivity> {
    private static final SdkField<List<MultiConditionalBranch>> BRANCHES_FIELD = SdkField
            .<List<MultiConditionalBranch>> builder(MarshallingType.LIST)
            .memberName("Branches")
            .getter(getter(MultiConditionalSplitActivity::branches))
            .setter(setter(Builder::branches))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Branches").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MultiConditionalBranch> builder(MarshallingType.SDK_POJO)
                                            .constructor(MultiConditionalBranch::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DEFAULT_ACTIVITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DefaultActivity").getter(getter(MultiConditionalSplitActivity::defaultActivity))
            .setter(setter(Builder::defaultActivity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultActivity").build()).build();

    private static final SdkField<WaitTime> EVALUATION_WAIT_TIME_FIELD = SdkField.<WaitTime> builder(MarshallingType.SDK_POJO)
            .memberName("EvaluationWaitTime").getter(getter(MultiConditionalSplitActivity::evaluationWaitTime))
            .setter(setter(Builder::evaluationWaitTime)).constructor(WaitTime::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EvaluationWaitTime").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BRANCHES_FIELD,
            DEFAULT_ACTIVITY_FIELD, EVALUATION_WAIT_TIME_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<MultiConditionalBranch> branches;

    private final String defaultActivity;

    private final WaitTime evaluationWaitTime;

    private MultiConditionalSplitActivity(BuilderImpl builder) {
        this.branches = builder.branches;
        this.defaultActivity = builder.defaultActivity;
        this.evaluationWaitTime = builder.evaluationWaitTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the Branches property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasBranches() {
        return branches != null && !(branches instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The paths for the activity, including the conditions for entering each path and the activity to perform for each
     * path.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasBranches} method.
     * </p>
     * 
     * @return The paths for the activity, including the conditions for entering each path and the activity to perform
     *         for each path.
     */
    public final List<MultiConditionalBranch> branches() {
        return branches;
    }

    /**
     * <p>
     * The unique identifier for the activity to perform for participants who don't meet any of the conditions specified
     * for other paths in the activity.
     * </p>
     * 
     * @return The unique identifier for the activity to perform for participants who don't meet any of the conditions
     *         specified for other paths in the activity.
     */
    public final String defaultActivity() {
        return defaultActivity;
    }

    /**
     * <p>
     * The amount of time to wait or the date and time when Amazon Pinpoint determines whether the conditions are met.
     * </p>
     * 
     * @return The amount of time to wait or the date and time when Amazon Pinpoint determines whether the conditions
     *         are met.
     */
    public final WaitTime evaluationWaitTime() {
        return evaluationWaitTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasBranches() ? branches() : null);
        hashCode = 31 * hashCode + Objects.hashCode(defaultActivity());
        hashCode = 31 * hashCode + Objects.hashCode(evaluationWaitTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MultiConditionalSplitActivity)) {
            return false;
        }
        MultiConditionalSplitActivity other = (MultiConditionalSplitActivity) obj;
        return hasBranches() == other.hasBranches() && Objects.equals(branches(), other.branches())
                && Objects.equals(defaultActivity(), other.defaultActivity())
                && Objects.equals(evaluationWaitTime(), other.evaluationWaitTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MultiConditionalSplitActivity").add("Branches", hasBranches() ? branches() : null)
                .add("DefaultActivity", defaultActivity()).add("EvaluationWaitTime", evaluationWaitTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Branches":
            return Optional.ofNullable(clazz.cast(branches()));
        case "DefaultActivity":
            return Optional.ofNullable(clazz.cast(defaultActivity()));
        case "EvaluationWaitTime":
            return Optional.ofNullable(clazz.cast(evaluationWaitTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Branches", BRANCHES_FIELD);
        map.put("DefaultActivity", DEFAULT_ACTIVITY_FIELD);
        map.put("EvaluationWaitTime", EVALUATION_WAIT_TIME_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<MultiConditionalSplitActivity, T> g) {
        return obj -> g.apply((MultiConditionalSplitActivity) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MultiConditionalSplitActivity> {
        /**
         * <p>
         * The paths for the activity, including the conditions for entering each path and the activity to perform for
         * each path.
         * </p>
         * 
         * @param branches
         *        The paths for the activity, including the conditions for entering each path and the activity to
         *        perform for each path.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder branches(Collection<MultiConditionalBranch> branches);

        /**
         * <p>
         * The paths for the activity, including the conditions for entering each path and the activity to perform for
         * each path.
         * </p>
         * 
         * @param branches
         *        The paths for the activity, including the conditions for entering each path and the activity to
         *        perform for each path.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder branches(MultiConditionalBranch... branches);

        /**
         * <p>
         * The paths for the activity, including the conditions for entering each path and the activity to perform for
         * each path.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.pinpoint.model.MultiConditionalBranch.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.pinpoint.model.MultiConditionalBranch#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.pinpoint.model.MultiConditionalBranch.Builder#build()} is called
         * immediately and its result is passed to {@link #branches(List<MultiConditionalBranch>)}.
         * 
         * @param branches
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.pinpoint.model.MultiConditionalBranch.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #branches(java.util.Collection<MultiConditionalBranch>)
         */
        Builder branches(Consumer<MultiConditionalBranch.Builder>... branches);

        /**
         * <p>
         * The unique identifier for the activity to perform for participants who don't meet any of the conditions
         * specified for other paths in the activity.
         * </p>
         * 
         * @param defaultActivity
         *        The unique identifier for the activity to perform for participants who don't meet any of the
         *        conditions specified for other paths in the activity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultActivity(String defaultActivity);

        /**
         * <p>
         * The amount of time to wait or the date and time when Amazon Pinpoint determines whether the conditions are
         * met.
         * </p>
         * 
         * @param evaluationWaitTime
         *        The amount of time to wait or the date and time when Amazon Pinpoint determines whether the conditions
         *        are met.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evaluationWaitTime(WaitTime evaluationWaitTime);

        /**
         * <p>
         * The amount of time to wait or the date and time when Amazon Pinpoint determines whether the conditions are
         * met.
         * </p>
         * This is a convenience method that creates an instance of the {@link WaitTime.Builder} avoiding the need to
         * create one manually via {@link WaitTime#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link WaitTime.Builder#build()} is called immediately and its result is
         * passed to {@link #evaluationWaitTime(WaitTime)}.
         * 
         * @param evaluationWaitTime
         *        a consumer that will call methods on {@link WaitTime.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #evaluationWaitTime(WaitTime)
         */
        default Builder evaluationWaitTime(Consumer<WaitTime.Builder> evaluationWaitTime) {
            return evaluationWaitTime(WaitTime.builder().applyMutation(evaluationWaitTime).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private List<MultiConditionalBranch> branches = DefaultSdkAutoConstructList.getInstance();

        private String defaultActivity;

        private WaitTime evaluationWaitTime;

        private BuilderImpl() {
        }

        private BuilderImpl(MultiConditionalSplitActivity model) {
            branches(model.branches);
            defaultActivity(model.defaultActivity);
            evaluationWaitTime(model.evaluationWaitTime);
        }

        public final List<MultiConditionalBranch.Builder> getBranches() {
            List<MultiConditionalBranch.Builder> result = ListOfMultiConditionalBranchCopier.copyToBuilder(this.branches);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setBranches(Collection<MultiConditionalBranch.BuilderImpl> branches) {
            this.branches = ListOfMultiConditionalBranchCopier.copyFromBuilder(branches);
        }

        @Override
        public final Builder branches(Collection<MultiConditionalBranch> branches) {
            this.branches = ListOfMultiConditionalBranchCopier.copy(branches);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder branches(MultiConditionalBranch... branches) {
            branches(Arrays.asList(branches));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder branches(Consumer<MultiConditionalBranch.Builder>... branches) {
            branches(Stream.of(branches).map(c -> MultiConditionalBranch.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getDefaultActivity() {
            return defaultActivity;
        }

        public final void setDefaultActivity(String defaultActivity) {
            this.defaultActivity = defaultActivity;
        }

        @Override
        public final Builder defaultActivity(String defaultActivity) {
            this.defaultActivity = defaultActivity;
            return this;
        }

        public final WaitTime.Builder getEvaluationWaitTime() {
            return evaluationWaitTime != null ? evaluationWaitTime.toBuilder() : null;
        }

        public final void setEvaluationWaitTime(WaitTime.BuilderImpl evaluationWaitTime) {
            this.evaluationWaitTime = evaluationWaitTime != null ? evaluationWaitTime.build() : null;
        }

        @Override
        public final Builder evaluationWaitTime(WaitTime evaluationWaitTime) {
            this.evaluationWaitTime = evaluationWaitTime;
            return this;
        }

        @Override
        public MultiConditionalSplitActivity build() {
            return new MultiConditionalSplitActivity(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
