/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.personalizeruntime.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information on a promotion. A promotion defines additional business rules that apply to a configurable
 * subset of recommended items.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Promotion implements SdkPojo, Serializable, ToCopyableBuilder<Promotion.Builder, Promotion> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Promotion::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<Integer> PERCENT_PROMOTED_ITEMS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("percentPromotedItems").getter(getter(Promotion::percentPromotedItems))
            .setter(setter(Builder::percentPromotedItems))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("percentPromotedItems").build())
            .build();

    private static final SdkField<String> FILTER_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("filterArn").getter(getter(Promotion::filterArn)).setter(setter(Builder::filterArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("filterArn").build()).build();

    private static final SdkField<Map<String, String>> FILTER_VALUES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("filterValues")
            .getter(getter(Promotion::filterValues))
            .setter(setter(Builder::filterValues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("filterValues").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            PERCENT_PROMOTED_ITEMS_FIELD, FILTER_ARN_FIELD, FILTER_VALUES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final Integer percentPromotedItems;

    private final String filterArn;

    private final Map<String, String> filterValues;

    private Promotion(BuilderImpl builder) {
        this.name = builder.name;
        this.percentPromotedItems = builder.percentPromotedItems;
        this.filterArn = builder.filterArn;
        this.filterValues = builder.filterValues;
    }

    /**
     * <p>
     * The name of the promotion.
     * </p>
     * 
     * @return The name of the promotion.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The percentage of recommended items to apply the promotion to.
     * </p>
     * 
     * @return The percentage of recommended items to apply the promotion to.
     */
    public final Integer percentPromotedItems() {
        return percentPromotedItems;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the filter used by the promotion. This filter defines the criteria for promoted
     * items. For more information, see <a
     * href="https://docs.aws.amazon.com/personalize/latest/dg/promoting-items.html#promotion-filters">Promotion
     * filters</a>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the filter used by the promotion. This filter defines the criteria for
     *         promoted items. For more information, see <a
     *         href="https://docs.aws.amazon.com/personalize/latest/dg/promoting-items.html#promotion-filters">Promotion
     *         filters</a>.
     */
    public final String filterArn() {
        return filterArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the FilterValues property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFilterValues() {
        return filterValues != null && !(filterValues instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The values to use when promoting items. For each placeholder parameter in your promotion's filter expression,
     * provide the parameter name (in matching case) as a key and the filter value(s) as the corresponding value.
     * Separate multiple values for one parameter with a comma.
     * </p>
     * <p>
     * For filter expressions that use an <code>INCLUDE</code> element to include items, you must provide values for all
     * parameters that are defined in the expression. For filters with expressions that use an <code>EXCLUDE</code>
     * element to exclude items, you can omit the <code>filter-values</code>. In this case, Amazon Personalize doesn't
     * use that portion of the expression to filter recommendations.
     * </p>
     * <p>
     * For more information on creating filters, see <a
     * href="https://docs.aws.amazon.com/personalize/latest/dg/filter.html">Filtering recommendations and user
     * segments</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFilterValues} method.
     * </p>
     * 
     * @return The values to use when promoting items. For each placeholder parameter in your promotion's filter
     *         expression, provide the parameter name (in matching case) as a key and the filter value(s) as the
     *         corresponding value. Separate multiple values for one parameter with a comma. </p>
     *         <p>
     *         For filter expressions that use an <code>INCLUDE</code> element to include items, you must provide values
     *         for all parameters that are defined in the expression. For filters with expressions that use an
     *         <code>EXCLUDE</code> element to exclude items, you can omit the <code>filter-values</code>. In this case,
     *         Amazon Personalize doesn't use that portion of the expression to filter recommendations.
     *         </p>
     *         <p>
     *         For more information on creating filters, see <a
     *         href="https://docs.aws.amazon.com/personalize/latest/dg/filter.html">Filtering recommendations and user
     *         segments</a>.
     */
    public final Map<String, String> filterValues() {
        return filterValues;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(percentPromotedItems());
        hashCode = 31 * hashCode + Objects.hashCode(filterArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasFilterValues() ? filterValues() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Promotion)) {
            return false;
        }
        Promotion other = (Promotion) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(percentPromotedItems(), other.percentPromotedItems())
                && Objects.equals(filterArn(), other.filterArn()) && hasFilterValues() == other.hasFilterValues()
                && Objects.equals(filterValues(), other.filterValues());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Promotion").add("Name", name()).add("PercentPromotedItems", percentPromotedItems())
                .add("FilterArn", filterArn())
                .add("FilterValues", filterValues() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "percentPromotedItems":
            return Optional.ofNullable(clazz.cast(percentPromotedItems()));
        case "filterArn":
            return Optional.ofNullable(clazz.cast(filterArn()));
        case "filterValues":
            return Optional.ofNullable(clazz.cast(filterValues()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("name", NAME_FIELD);
        map.put("percentPromotedItems", PERCENT_PROMOTED_ITEMS_FIELD);
        map.put("filterArn", FILTER_ARN_FIELD);
        map.put("filterValues", FILTER_VALUES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Promotion, T> g) {
        return obj -> g.apply((Promotion) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Promotion> {
        /**
         * <p>
         * The name of the promotion.
         * </p>
         * 
         * @param name
         *        The name of the promotion.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The percentage of recommended items to apply the promotion to.
         * </p>
         * 
         * @param percentPromotedItems
         *        The percentage of recommended items to apply the promotion to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder percentPromotedItems(Integer percentPromotedItems);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the filter used by the promotion. This filter defines the criteria for
         * promoted items. For more information, see <a
         * href="https://docs.aws.amazon.com/personalize/latest/dg/promoting-items.html#promotion-filters">Promotion
         * filters</a>.
         * </p>
         * 
         * @param filterArn
         *        The Amazon Resource Name (ARN) of the filter used by the promotion. This filter defines the criteria
         *        for promoted items. For more information, see <a
         *        href="https://docs.aws.amazon.com/personalize/latest/dg/promoting-items.html#promotion-filters"
         *        >Promotion filters</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filterArn(String filterArn);

        /**
         * <p>
         * The values to use when promoting items. For each placeholder parameter in your promotion's filter expression,
         * provide the parameter name (in matching case) as a key and the filter value(s) as the corresponding value.
         * Separate multiple values for one parameter with a comma.
         * </p>
         * <p>
         * For filter expressions that use an <code>INCLUDE</code> element to include items, you must provide values for
         * all parameters that are defined in the expression. For filters with expressions that use an
         * <code>EXCLUDE</code> element to exclude items, you can omit the <code>filter-values</code>. In this case,
         * Amazon Personalize doesn't use that portion of the expression to filter recommendations.
         * </p>
         * <p>
         * For more information on creating filters, see <a
         * href="https://docs.aws.amazon.com/personalize/latest/dg/filter.html">Filtering recommendations and user
         * segments</a>.
         * </p>
         * 
         * @param filterValues
         *        The values to use when promoting items. For each placeholder parameter in your promotion's filter
         *        expression, provide the parameter name (in matching case) as a key and the filter value(s) as the
         *        corresponding value. Separate multiple values for one parameter with a comma. </p>
         *        <p>
         *        For filter expressions that use an <code>INCLUDE</code> element to include items, you must provide
         *        values for all parameters that are defined in the expression. For filters with expressions that use an
         *        <code>EXCLUDE</code> element to exclude items, you can omit the <code>filter-values</code>. In this
         *        case, Amazon Personalize doesn't use that portion of the expression to filter recommendations.
         *        </p>
         *        <p>
         *        For more information on creating filters, see <a
         *        href="https://docs.aws.amazon.com/personalize/latest/dg/filter.html">Filtering recommendations and
         *        user segments</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filterValues(Map<String, String> filterValues);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private Integer percentPromotedItems;

        private String filterArn;

        private Map<String, String> filterValues = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Promotion model) {
            name(model.name);
            percentPromotedItems(model.percentPromotedItems);
            filterArn(model.filterArn);
            filterValues(model.filterValues);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Integer getPercentPromotedItems() {
            return percentPromotedItems;
        }

        public final void setPercentPromotedItems(Integer percentPromotedItems) {
            this.percentPromotedItems = percentPromotedItems;
        }

        @Override
        public final Builder percentPromotedItems(Integer percentPromotedItems) {
            this.percentPromotedItems = percentPromotedItems;
            return this;
        }

        public final String getFilterArn() {
            return filterArn;
        }

        public final void setFilterArn(String filterArn) {
            this.filterArn = filterArn;
        }

        @Override
        public final Builder filterArn(String filterArn) {
            this.filterArn = filterArn;
            return this;
        }

        public final Map<String, String> getFilterValues() {
            if (filterValues instanceof SdkAutoConstructMap) {
                return null;
            }
            return filterValues;
        }

        public final void setFilterValues(Map<String, String> filterValues) {
            this.filterValues = FilterValuesCopier.copy(filterValues);
        }

        @Override
        public final Builder filterValues(Map<String, String> filterValues) {
            this.filterValues = FilterValuesCopier.copy(filterValues);
            return this;
        }

        @Override
        public Promotion build() {
            return new Promotion(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
