/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.partnercentralchannel;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.partnercentralchannel.internal.PartnerCentralChannelServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.partnercentralchannel.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.partnercentralchannel.model.AcceptChannelHandshakeRequest;
import software.amazon.awssdk.services.partnercentralchannel.model.AcceptChannelHandshakeResponse;
import software.amazon.awssdk.services.partnercentralchannel.model.AccessDeniedException;
import software.amazon.awssdk.services.partnercentralchannel.model.CancelChannelHandshakeRequest;
import software.amazon.awssdk.services.partnercentralchannel.model.CancelChannelHandshakeResponse;
import software.amazon.awssdk.services.partnercentralchannel.model.ConflictException;
import software.amazon.awssdk.services.partnercentralchannel.model.CreateChannelHandshakeRequest;
import software.amazon.awssdk.services.partnercentralchannel.model.CreateChannelHandshakeResponse;
import software.amazon.awssdk.services.partnercentralchannel.model.CreateProgramManagementAccountRequest;
import software.amazon.awssdk.services.partnercentralchannel.model.CreateProgramManagementAccountResponse;
import software.amazon.awssdk.services.partnercentralchannel.model.CreateRelationshipRequest;
import software.amazon.awssdk.services.partnercentralchannel.model.CreateRelationshipResponse;
import software.amazon.awssdk.services.partnercentralchannel.model.DeleteProgramManagementAccountRequest;
import software.amazon.awssdk.services.partnercentralchannel.model.DeleteProgramManagementAccountResponse;
import software.amazon.awssdk.services.partnercentralchannel.model.DeleteRelationshipRequest;
import software.amazon.awssdk.services.partnercentralchannel.model.DeleteRelationshipResponse;
import software.amazon.awssdk.services.partnercentralchannel.model.GetRelationshipRequest;
import software.amazon.awssdk.services.partnercentralchannel.model.GetRelationshipResponse;
import software.amazon.awssdk.services.partnercentralchannel.model.InternalServerException;
import software.amazon.awssdk.services.partnercentralchannel.model.ListChannelHandshakesRequest;
import software.amazon.awssdk.services.partnercentralchannel.model.ListChannelHandshakesResponse;
import software.amazon.awssdk.services.partnercentralchannel.model.ListProgramManagementAccountsRequest;
import software.amazon.awssdk.services.partnercentralchannel.model.ListProgramManagementAccountsResponse;
import software.amazon.awssdk.services.partnercentralchannel.model.ListRelationshipsRequest;
import software.amazon.awssdk.services.partnercentralchannel.model.ListRelationshipsResponse;
import software.amazon.awssdk.services.partnercentralchannel.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.partnercentralchannel.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.partnercentralchannel.model.PartnerCentralChannelException;
import software.amazon.awssdk.services.partnercentralchannel.model.RejectChannelHandshakeRequest;
import software.amazon.awssdk.services.partnercentralchannel.model.RejectChannelHandshakeResponse;
import software.amazon.awssdk.services.partnercentralchannel.model.ResourceNotFoundException;
import software.amazon.awssdk.services.partnercentralchannel.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.partnercentralchannel.model.TagResourceRequest;
import software.amazon.awssdk.services.partnercentralchannel.model.TagResourceResponse;
import software.amazon.awssdk.services.partnercentralchannel.model.ThrottlingException;
import software.amazon.awssdk.services.partnercentralchannel.model.UntagResourceRequest;
import software.amazon.awssdk.services.partnercentralchannel.model.UntagResourceResponse;
import software.amazon.awssdk.services.partnercentralchannel.model.UpdateProgramManagementAccountRequest;
import software.amazon.awssdk.services.partnercentralchannel.model.UpdateProgramManagementAccountResponse;
import software.amazon.awssdk.services.partnercentralchannel.model.UpdateRelationshipRequest;
import software.amazon.awssdk.services.partnercentralchannel.model.UpdateRelationshipResponse;
import software.amazon.awssdk.services.partnercentralchannel.model.ValidationException;
import software.amazon.awssdk.services.partnercentralchannel.transform.AcceptChannelHandshakeRequestMarshaller;
import software.amazon.awssdk.services.partnercentralchannel.transform.CancelChannelHandshakeRequestMarshaller;
import software.amazon.awssdk.services.partnercentralchannel.transform.CreateChannelHandshakeRequestMarshaller;
import software.amazon.awssdk.services.partnercentralchannel.transform.CreateProgramManagementAccountRequestMarshaller;
import software.amazon.awssdk.services.partnercentralchannel.transform.CreateRelationshipRequestMarshaller;
import software.amazon.awssdk.services.partnercentralchannel.transform.DeleteProgramManagementAccountRequestMarshaller;
import software.amazon.awssdk.services.partnercentralchannel.transform.DeleteRelationshipRequestMarshaller;
import software.amazon.awssdk.services.partnercentralchannel.transform.GetRelationshipRequestMarshaller;
import software.amazon.awssdk.services.partnercentralchannel.transform.ListChannelHandshakesRequestMarshaller;
import software.amazon.awssdk.services.partnercentralchannel.transform.ListProgramManagementAccountsRequestMarshaller;
import software.amazon.awssdk.services.partnercentralchannel.transform.ListRelationshipsRequestMarshaller;
import software.amazon.awssdk.services.partnercentralchannel.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.partnercentralchannel.transform.RejectChannelHandshakeRequestMarshaller;
import software.amazon.awssdk.services.partnercentralchannel.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.partnercentralchannel.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.partnercentralchannel.transform.UpdateProgramManagementAccountRequestMarshaller;
import software.amazon.awssdk.services.partnercentralchannel.transform.UpdateRelationshipRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link PartnerCentralChannelClient}.
 *
 * @see PartnerCentralChannelClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPartnerCentralChannelClient implements PartnerCentralChannelClient {
    private static final Logger log = Logger.loggerFor(DefaultPartnerCentralChannelClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultPartnerCentralChannelClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "PartnerCentral_Channel" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Accepts a pending channel handshake request from another AWS account.
     * </p>
     *
     * @param acceptChannelHandshakeRequest
     * @return Result of the AcceptChannelHandshake operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ValidationException
     *         The request failed validation due to invalid input parameters.
     * @throws ThrottlingException
     *         The request was throttled due to too many requests being sent in a short period.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient permissions.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PartnerCentralChannelException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PartnerCentralChannelClient.AcceptChannelHandshake
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/partnercentral-channel-2024-03-18/AcceptChannelHandshake"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AcceptChannelHandshakeResponse acceptChannelHandshake(AcceptChannelHandshakeRequest acceptChannelHandshakeRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, PartnerCentralChannelException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AcceptChannelHandshakeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AcceptChannelHandshakeResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(acceptChannelHandshakeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acceptChannelHandshakeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PartnerCentral Channel");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptChannelHandshake");

            return clientHandler
                    .execute(new ClientExecutionParams<AcceptChannelHandshakeRequest, AcceptChannelHandshakeResponse>()
                            .withOperationName("AcceptChannelHandshake").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(acceptChannelHandshakeRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AcceptChannelHandshakeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Cancels a pending channel handshake request.
     * </p>
     *
     * @param cancelChannelHandshakeRequest
     * @return Result of the CancelChannelHandshake operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ValidationException
     *         The request failed validation due to invalid input parameters.
     * @throws ThrottlingException
     *         The request was throttled due to too many requests being sent in a short period.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient permissions.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PartnerCentralChannelException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PartnerCentralChannelClient.CancelChannelHandshake
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/partnercentral-channel-2024-03-18/CancelChannelHandshake"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CancelChannelHandshakeResponse cancelChannelHandshake(CancelChannelHandshakeRequest cancelChannelHandshakeRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, PartnerCentralChannelException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelChannelHandshakeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CancelChannelHandshakeResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelChannelHandshakeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelChannelHandshakeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PartnerCentral Channel");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelChannelHandshake");

            return clientHandler
                    .execute(new ClientExecutionParams<CancelChannelHandshakeRequest, CancelChannelHandshakeResponse>()
                            .withOperationName("CancelChannelHandshake").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(cancelChannelHandshakeRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CancelChannelHandshakeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new channel handshake request to establish a partnership with another AWS account.
     * </p>
     *
     * @param createChannelHandshakeRequest
     * @return Result of the CreateChannelHandshake operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ValidationException
     *         The request failed validation due to invalid input parameters.
     * @throws ThrottlingException
     *         The request was throttled due to too many requests being sent in a short period.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient permissions.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the resource.
     * @throws ServiceQuotaExceededException
     *         The request would exceed a service quota limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PartnerCentralChannelException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PartnerCentralChannelClient.CreateChannelHandshake
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/partnercentral-channel-2024-03-18/CreateChannelHandshake"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateChannelHandshakeResponse createChannelHandshake(CreateChannelHandshakeRequest createChannelHandshakeRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, ConflictException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            PartnerCentralChannelException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateChannelHandshakeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateChannelHandshakeResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createChannelHandshakeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createChannelHandshakeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PartnerCentral Channel");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateChannelHandshake");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateChannelHandshakeRequest, CreateChannelHandshakeResponse>()
                            .withOperationName("CreateChannelHandshake").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createChannelHandshakeRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateChannelHandshakeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new program management account for managing partner relationships.
     * </p>
     *
     * @param createProgramManagementAccountRequest
     * @return Result of the CreateProgramManagementAccount operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ValidationException
     *         The request failed validation due to invalid input parameters.
     * @throws ThrottlingException
     *         The request was throttled due to too many requests being sent in a short period.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient permissions.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the resource.
     * @throws ServiceQuotaExceededException
     *         The request would exceed a service quota limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PartnerCentralChannelException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PartnerCentralChannelClient.CreateProgramManagementAccount
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/partnercentral-channel-2024-03-18/CreateProgramManagementAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateProgramManagementAccountResponse createProgramManagementAccount(
            CreateProgramManagementAccountRequest createProgramManagementAccountRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, ConflictException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, PartnerCentralChannelException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateProgramManagementAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateProgramManagementAccountResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createProgramManagementAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createProgramManagementAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PartnerCentral Channel");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProgramManagementAccount");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateProgramManagementAccountRequest, CreateProgramManagementAccountResponse>()
                            .withOperationName("CreateProgramManagementAccount").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createProgramManagementAccountRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateProgramManagementAccountRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new partner relationship between accounts.
     * </p>
     *
     * @param createRelationshipRequest
     * @return Result of the CreateRelationship operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ValidationException
     *         The request failed validation due to invalid input parameters.
     * @throws ThrottlingException
     *         The request was throttled due to too many requests being sent in a short period.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient permissions.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the resource.
     * @throws ServiceQuotaExceededException
     *         The request would exceed a service quota limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PartnerCentralChannelException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PartnerCentralChannelClient.CreateRelationship
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/partnercentral-channel-2024-03-18/CreateRelationship"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateRelationshipResponse createRelationship(CreateRelationshipRequest createRelationshipRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, ConflictException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            PartnerCentralChannelException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateRelationshipResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateRelationshipResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRelationshipRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRelationshipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PartnerCentral Channel");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRelationship");

            return clientHandler.execute(new ClientExecutionParams<CreateRelationshipRequest, CreateRelationshipResponse>()
                    .withOperationName("CreateRelationship").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createRelationshipRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateRelationshipRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a program management account.
     * </p>
     *
     * @param deleteProgramManagementAccountRequest
     * @return Result of the DeleteProgramManagementAccount operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ValidationException
     *         The request failed validation due to invalid input parameters.
     * @throws ThrottlingException
     *         The request was throttled due to too many requests being sent in a short period.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient permissions.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PartnerCentralChannelException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PartnerCentralChannelClient.DeleteProgramManagementAccount
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/partnercentral-channel-2024-03-18/DeleteProgramManagementAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteProgramManagementAccountResponse deleteProgramManagementAccount(
            DeleteProgramManagementAccountRequest deleteProgramManagementAccountRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, ConflictException,
            AwsServiceException, SdkClientException, PartnerCentralChannelException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteProgramManagementAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteProgramManagementAccountResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteProgramManagementAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteProgramManagementAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PartnerCentral Channel");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProgramManagementAccount");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteProgramManagementAccountRequest, DeleteProgramManagementAccountResponse>()
                            .withOperationName("DeleteProgramManagementAccount").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteProgramManagementAccountRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteProgramManagementAccountRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a partner relationship.
     * </p>
     *
     * @param deleteRelationshipRequest
     * @return Result of the DeleteRelationship operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ValidationException
     *         The request failed validation due to invalid input parameters.
     * @throws ThrottlingException
     *         The request was throttled due to too many requests being sent in a short period.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient permissions.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PartnerCentralChannelException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PartnerCentralChannelClient.DeleteRelationship
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/partnercentral-channel-2024-03-18/DeleteRelationship"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteRelationshipResponse deleteRelationship(DeleteRelationshipRequest deleteRelationshipRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, ConflictException, AwsServiceException, SdkClientException, PartnerCentralChannelException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteRelationshipResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteRelationshipResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRelationshipRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRelationshipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PartnerCentral Channel");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRelationship");

            return clientHandler.execute(new ClientExecutionParams<DeleteRelationshipRequest, DeleteRelationshipResponse>()
                    .withOperationName("DeleteRelationship").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteRelationshipRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteRelationshipRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves details of a specific partner relationship.
     * </p>
     *
     * @param getRelationshipRequest
     * @return Result of the GetRelationship operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ValidationException
     *         The request failed validation due to invalid input parameters.
     * @throws ThrottlingException
     *         The request was throttled due to too many requests being sent in a short period.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient permissions.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PartnerCentralChannelException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PartnerCentralChannelClient.GetRelationship
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/partnercentral-channel-2024-03-18/GetRelationship"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetRelationshipResponse getRelationship(GetRelationshipRequest getRelationshipRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, PartnerCentralChannelException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRelationshipResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetRelationshipResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRelationshipRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRelationshipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PartnerCentral Channel");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRelationship");

            return clientHandler.execute(new ClientExecutionParams<GetRelationshipRequest, GetRelationshipResponse>()
                    .withOperationName("GetRelationship").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getRelationshipRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetRelationshipRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists channel handshakes based on specified criteria.
     * </p>
     *
     * @param listChannelHandshakesRequest
     * @return Result of the ListChannelHandshakes operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ValidationException
     *         The request failed validation due to invalid input parameters.
     * @throws ThrottlingException
     *         The request was throttled due to too many requests being sent in a short period.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient permissions.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PartnerCentralChannelException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PartnerCentralChannelClient.ListChannelHandshakes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/partnercentral-channel-2024-03-18/ListChannelHandshakes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListChannelHandshakesResponse listChannelHandshakes(ListChannelHandshakesRequest listChannelHandshakesRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, PartnerCentralChannelException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListChannelHandshakesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListChannelHandshakesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listChannelHandshakesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChannelHandshakesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PartnerCentral Channel");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChannelHandshakes");

            return clientHandler.execute(new ClientExecutionParams<ListChannelHandshakesRequest, ListChannelHandshakesResponse>()
                    .withOperationName("ListChannelHandshakes").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listChannelHandshakesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListChannelHandshakesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists program management accounts based on specified criteria.
     * </p>
     *
     * @param listProgramManagementAccountsRequest
     * @return Result of the ListProgramManagementAccounts operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ValidationException
     *         The request failed validation due to invalid input parameters.
     * @throws ThrottlingException
     *         The request was throttled due to too many requests being sent in a short period.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient permissions.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PartnerCentralChannelException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PartnerCentralChannelClient.ListProgramManagementAccounts
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/partnercentral-channel-2024-03-18/ListProgramManagementAccounts"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListProgramManagementAccountsResponse listProgramManagementAccounts(
            ListProgramManagementAccountsRequest listProgramManagementAccountsRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, PartnerCentralChannelException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListProgramManagementAccountsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListProgramManagementAccountsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listProgramManagementAccountsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listProgramManagementAccountsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PartnerCentral Channel");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProgramManagementAccounts");

            return clientHandler
                    .execute(new ClientExecutionParams<ListProgramManagementAccountsRequest, ListProgramManagementAccountsResponse>()
                            .withOperationName("ListProgramManagementAccounts").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listProgramManagementAccountsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListProgramManagementAccountsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists partner relationships based on specified criteria.
     * </p>
     *
     * @param listRelationshipsRequest
     * @return Result of the ListRelationships operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ValidationException
     *         The request failed validation due to invalid input parameters.
     * @throws ThrottlingException
     *         The request was throttled due to too many requests being sent in a short period.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient permissions.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PartnerCentralChannelException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PartnerCentralChannelClient.ListRelationships
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/partnercentral-channel-2024-03-18/ListRelationships"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRelationshipsResponse listRelationships(ListRelationshipsRequest listRelationshipsRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, PartnerCentralChannelException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRelationshipsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListRelationshipsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRelationshipsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRelationshipsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PartnerCentral Channel");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRelationships");

            return clientHandler.execute(new ClientExecutionParams<ListRelationshipsRequest, ListRelationshipsResponse>()
                    .withOperationName("ListRelationships").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listRelationshipsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListRelationshipsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists tags associated with a specific resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ValidationException
     *         The request failed validation due to invalid input parameters.
     * @throws ThrottlingException
     *         The request was throttled due to too many requests being sent in a short period.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient permissions.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PartnerCentralChannelException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PartnerCentralChannelClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/partnercentral-channel-2024-03-18/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, PartnerCentralChannelException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PartnerCentral Channel");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Rejects a pending channel handshake request.
     * </p>
     *
     * @param rejectChannelHandshakeRequest
     * @return Result of the RejectChannelHandshake operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ValidationException
     *         The request failed validation due to invalid input parameters.
     * @throws ThrottlingException
     *         The request was throttled due to too many requests being sent in a short period.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient permissions.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PartnerCentralChannelException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PartnerCentralChannelClient.RejectChannelHandshake
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/partnercentral-channel-2024-03-18/RejectChannelHandshake"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RejectChannelHandshakeResponse rejectChannelHandshake(RejectChannelHandshakeRequest rejectChannelHandshakeRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, PartnerCentralChannelException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RejectChannelHandshakeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RejectChannelHandshakeResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(rejectChannelHandshakeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rejectChannelHandshakeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PartnerCentral Channel");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RejectChannelHandshake");

            return clientHandler
                    .execute(new ClientExecutionParams<RejectChannelHandshakeRequest, RejectChannelHandshakeResponse>()
                            .withOperationName("RejectChannelHandshake").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(rejectChannelHandshakeRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RejectChannelHandshakeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds or updates tags for a specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ValidationException
     *         The request failed validation due to invalid input parameters.
     * @throws ThrottlingException
     *         The request was throttled due to too many requests being sent in a short period.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient permissions.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PartnerCentralChannelException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PartnerCentralChannelClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/partnercentral-channel-2024-03-18/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, ConflictException,
            AwsServiceException, SdkClientException, PartnerCentralChannelException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PartnerCentral Channel");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from a specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ValidationException
     *         The request failed validation due to invalid input parameters.
     * @throws ThrottlingException
     *         The request was throttled due to too many requests being sent in a short period.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient permissions.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PartnerCentralChannelException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PartnerCentralChannelClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/partnercentral-channel-2024-03-18/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, ConflictException,
            AwsServiceException, SdkClientException, PartnerCentralChannelException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PartnerCentral Channel");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the properties of a program management account.
     * </p>
     *
     * @param updateProgramManagementAccountRequest
     * @return Result of the UpdateProgramManagementAccount operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ValidationException
     *         The request failed validation due to invalid input parameters.
     * @throws ThrottlingException
     *         The request was throttled due to too many requests being sent in a short period.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient permissions.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PartnerCentralChannelException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PartnerCentralChannelClient.UpdateProgramManagementAccount
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/partnercentral-channel-2024-03-18/UpdateProgramManagementAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateProgramManagementAccountResponse updateProgramManagementAccount(
            UpdateProgramManagementAccountRequest updateProgramManagementAccountRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, ConflictException,
            AwsServiceException, SdkClientException, PartnerCentralChannelException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateProgramManagementAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateProgramManagementAccountResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateProgramManagementAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateProgramManagementAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PartnerCentral Channel");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProgramManagementAccount");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateProgramManagementAccountRequest, UpdateProgramManagementAccountResponse>()
                            .withOperationName("UpdateProgramManagementAccount").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateProgramManagementAccountRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateProgramManagementAccountRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the properties of a partner relationship.
     * </p>
     *
     * @param updateRelationshipRequest
     * @return Result of the UpdateRelationship operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ValidationException
     *         The request failed validation due to invalid input parameters.
     * @throws ThrottlingException
     *         The request was throttled due to too many requests being sent in a short period.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient permissions.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PartnerCentralChannelException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PartnerCentralChannelClient.UpdateRelationship
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/partnercentral-channel-2024-03-18/UpdateRelationship"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateRelationshipResponse updateRelationship(UpdateRelationshipRequest updateRelationshipRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, ConflictException, AwsServiceException, SdkClientException, PartnerCentralChannelException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateRelationshipResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateRelationshipResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateRelationshipRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRelationshipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PartnerCentral Channel");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRelationship");

            return clientHandler.execute(new ClientExecutionParams<UpdateRelationshipRequest, UpdateRelationshipResponse>()
                    .withOperationName("UpdateRelationship").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateRelationshipRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateRelationshipRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        PartnerCentralChannelServiceClientConfigurationBuilder serviceConfigBuilder = new PartnerCentralChannelServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PartnerCentralChannelException::builder).protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0");
    }

    @Override
    public final PartnerCentralChannelServiceClientConfiguration serviceClientConfiguration() {
        return new PartnerCentralChannelServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
