/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.opsworkscm;

import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.opsworkscm.model.AssociateNodeRequest;
import software.amazon.awssdk.services.opsworkscm.model.AssociateNodeResponse;
import software.amazon.awssdk.services.opsworkscm.model.CreateBackupRequest;
import software.amazon.awssdk.services.opsworkscm.model.CreateBackupResponse;
import software.amazon.awssdk.services.opsworkscm.model.CreateServerRequest;
import software.amazon.awssdk.services.opsworkscm.model.CreateServerResponse;
import software.amazon.awssdk.services.opsworkscm.model.DeleteBackupRequest;
import software.amazon.awssdk.services.opsworkscm.model.DeleteBackupResponse;
import software.amazon.awssdk.services.opsworkscm.model.DeleteServerRequest;
import software.amazon.awssdk.services.opsworkscm.model.DeleteServerResponse;
import software.amazon.awssdk.services.opsworkscm.model.DescribeAccountAttributesRequest;
import software.amazon.awssdk.services.opsworkscm.model.DescribeAccountAttributesResponse;
import software.amazon.awssdk.services.opsworkscm.model.DescribeBackupsRequest;
import software.amazon.awssdk.services.opsworkscm.model.DescribeBackupsResponse;
import software.amazon.awssdk.services.opsworkscm.model.DescribeEventsRequest;
import software.amazon.awssdk.services.opsworkscm.model.DescribeEventsResponse;
import software.amazon.awssdk.services.opsworkscm.model.DescribeNodeAssociationStatusRequest;
import software.amazon.awssdk.services.opsworkscm.model.DescribeNodeAssociationStatusResponse;
import software.amazon.awssdk.services.opsworkscm.model.DescribeServersRequest;
import software.amazon.awssdk.services.opsworkscm.model.DescribeServersResponse;
import software.amazon.awssdk.services.opsworkscm.model.DisassociateNodeRequest;
import software.amazon.awssdk.services.opsworkscm.model.DisassociateNodeResponse;
import software.amazon.awssdk.services.opsworkscm.model.ExportServerEngineAttributeRequest;
import software.amazon.awssdk.services.opsworkscm.model.ExportServerEngineAttributeResponse;
import software.amazon.awssdk.services.opsworkscm.model.InvalidNextTokenException;
import software.amazon.awssdk.services.opsworkscm.model.InvalidStateException;
import software.amazon.awssdk.services.opsworkscm.model.LimitExceededException;
import software.amazon.awssdk.services.opsworkscm.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.opsworkscm.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.opsworkscm.model.OpsWorksCmException;
import software.amazon.awssdk.services.opsworkscm.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.opsworkscm.model.ResourceNotFoundException;
import software.amazon.awssdk.services.opsworkscm.model.RestoreServerRequest;
import software.amazon.awssdk.services.opsworkscm.model.RestoreServerResponse;
import software.amazon.awssdk.services.opsworkscm.model.StartMaintenanceRequest;
import software.amazon.awssdk.services.opsworkscm.model.StartMaintenanceResponse;
import software.amazon.awssdk.services.opsworkscm.model.TagResourceRequest;
import software.amazon.awssdk.services.opsworkscm.model.TagResourceResponse;
import software.amazon.awssdk.services.opsworkscm.model.UntagResourceRequest;
import software.amazon.awssdk.services.opsworkscm.model.UntagResourceResponse;
import software.amazon.awssdk.services.opsworkscm.model.UpdateServerEngineAttributesRequest;
import software.amazon.awssdk.services.opsworkscm.model.UpdateServerEngineAttributesResponse;
import software.amazon.awssdk.services.opsworkscm.model.UpdateServerRequest;
import software.amazon.awssdk.services.opsworkscm.model.UpdateServerResponse;
import software.amazon.awssdk.services.opsworkscm.model.ValidationException;
import software.amazon.awssdk.services.opsworkscm.transform.AssociateNodeRequestMarshaller;
import software.amazon.awssdk.services.opsworkscm.transform.CreateBackupRequestMarshaller;
import software.amazon.awssdk.services.opsworkscm.transform.CreateServerRequestMarshaller;
import software.amazon.awssdk.services.opsworkscm.transform.DeleteBackupRequestMarshaller;
import software.amazon.awssdk.services.opsworkscm.transform.DeleteServerRequestMarshaller;
import software.amazon.awssdk.services.opsworkscm.transform.DescribeAccountAttributesRequestMarshaller;
import software.amazon.awssdk.services.opsworkscm.transform.DescribeBackupsRequestMarshaller;
import software.amazon.awssdk.services.opsworkscm.transform.DescribeEventsRequestMarshaller;
import software.amazon.awssdk.services.opsworkscm.transform.DescribeNodeAssociationStatusRequestMarshaller;
import software.amazon.awssdk.services.opsworkscm.transform.DescribeServersRequestMarshaller;
import software.amazon.awssdk.services.opsworkscm.transform.DisassociateNodeRequestMarshaller;
import software.amazon.awssdk.services.opsworkscm.transform.ExportServerEngineAttributeRequestMarshaller;
import software.amazon.awssdk.services.opsworkscm.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.opsworkscm.transform.RestoreServerRequestMarshaller;
import software.amazon.awssdk.services.opsworkscm.transform.StartMaintenanceRequestMarshaller;
import software.amazon.awssdk.services.opsworkscm.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.opsworkscm.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.opsworkscm.transform.UpdateServerEngineAttributesRequestMarshaller;
import software.amazon.awssdk.services.opsworkscm.transform.UpdateServerRequestMarshaller;

/**
 * Internal implementation of {@link OpsWorksCmClient}.
 *
 * @see OpsWorksCmClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultOpsWorksCmClient implements OpsWorksCmClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultOpsWorksCmClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Associates a new node with the server. For more information about how to disassociate a node, see
     * <a>DisassociateNode</a>.
     * </p>
     * <p>
     * On a Chef server: This command is an alternative to <code>knife bootstrap</code>.
     * </p>
     * <p>
     * Example (Chef):
     * <code>aws opsworks-cm associate-node --server-name <i>MyServer</i> --node-name <i>MyManagedNode</i> --engine-attributes "Name=<i>CHEF_ORGANIZATION</i>,Value=default" "Name=<i>CHEF_NODE_PUBLIC_KEY</i>,Value=<i>public-key-pem</i>"</code>
     * </p>
     * <p>
     * On a Puppet server, this command is an alternative to the <code>puppet cert sign</code> command that signs a
     * Puppet node CSR.
     * </p>
     * <p>
     * Example (Chef):
     * <code>aws opsworks-cm associate-node --server-name <i>MyServer</i> --node-name <i>MyManagedNode</i> --engine-attributes "Name=<i>PUPPET_NODE_CSR</i>,Value=<i>csr-pem</i>"</code>
     * </p>
     * <p>
     * A node can can only be associated with servers that are in a <code>HEALTHY</code> state. Otherwise, an
     * <code>InvalidStateException</code> is thrown. A <code>ResourceNotFoundException</code> is thrown when the server
     * does not exist. A <code>ValidationException</code> is raised when parameters of the request are not valid. The
     * AssociateNode API call can be integrated into Auto Scaling configurations, AWS Cloudformation templates, or the
     * user data of a server's instance.
     * </p>
     *
     * @param associateNodeRequest
     * @return Result of the AssociateNode operation returned by the service.
     * @throws InvalidStateException
     *         The resource is in a state that does not allow you to perform a specified action.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist, or access was denied.
     * @throws ValidationException
     *         One or more of the provided request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OpsWorksCmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OpsWorksCmClient.AssociateNode
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/opsworkscm-2016-11-01/AssociateNode" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AssociateNodeResponse associateNode(AssociateNodeRequest associateNodeRequest) throws InvalidStateException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, OpsWorksCmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateNodeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AssociateNodeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<AssociateNodeRequest, AssociateNodeResponse>()
                .withOperationName("AssociateNode").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(associateNodeRequest)
                .withMarshaller(new AssociateNodeRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates an application-level backup of a server. While the server is in the <code>BACKING_UP</code> state, the
     * server cannot be changed, and no additional backup can be created.
     * </p>
     * <p>
     * Backups can be created for servers in <code>RUNNING</code>, <code>HEALTHY</code>, and <code>UNHEALTHY</code>
     * states. By default, you can create a maximum of 50 manual backups.
     * </p>
     * <p>
     * This operation is asynchronous.
     * </p>
     * <p>
     * A <code>LimitExceededException</code> is thrown when the maximum number of manual backups is reached. An
     * <code>InvalidStateException</code> is thrown when the server is not in any of the following states: RUNNING,
     * HEALTHY, or UNHEALTHY. A <code>ResourceNotFoundException</code> is thrown when the server is not found. A
     * <code>ValidationException</code> is thrown when parameters of the request are not valid.
     * </p>
     *
     * @param createBackupRequest
     * @return Result of the CreateBackup operation returned by the service.
     * @throws InvalidStateException
     *         The resource is in a state that does not allow you to perform a specified action.
     * @throws LimitExceededException
     *         The limit of servers or backups has been reached.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist, or access was denied.
     * @throws ValidationException
     *         One or more of the provided request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OpsWorksCmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OpsWorksCmClient.CreateBackup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/opsworkscm-2016-11-01/CreateBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateBackupResponse createBackup(CreateBackupRequest createBackupRequest) throws InvalidStateException,
            LimitExceededException, ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException,
            OpsWorksCmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateBackupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateBackupRequest, CreateBackupResponse>()
                .withOperationName("CreateBackup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createBackupRequest)
                .withMarshaller(new CreateBackupRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates and immedately starts a new server. The server is ready to use when it is in the <code>HEALTHY</code>
     * state. By default, you can create a maximum of 10 servers.
     * </p>
     * <p>
     * This operation is asynchronous.
     * </p>
     * <p>
     * A <code>LimitExceededException</code> is thrown when you have created the maximum number of servers (10). A
     * <code>ResourceAlreadyExistsException</code> is thrown when a server with the same name already exists in the
     * account. A <code>ResourceNotFoundException</code> is thrown when you specify a backup ID that is not valid or is
     * for a backup that does not exist. A <code>ValidationException</code> is thrown when parameters of the request are
     * not valid.
     * </p>
     * <p>
     * If you do not specify a security group by adding the <code>SecurityGroupIds</code> parameter, AWS OpsWorks
     * creates a new security group.
     * </p>
     * <p>
     * <i>Chef Automate:</i> The default security group opens the Chef server to the world on TCP port 443. If a KeyName
     * is present, AWS OpsWorks enables SSH access. SSH is also open to the world on TCP port 22.
     * </p>
     * <p>
     * <i>Puppet Enterprise:</i> The default security group opens TCP ports 22, 443, 4433, 8140, 8142, 8143, and 8170.
     * If a KeyName is present, AWS OpsWorks enables SSH access. SSH is also open to the world on TCP port 22.
     * </p>
     * <p>
     * By default, your server is accessible from any IP address. We recommend that you update your security group rules
     * to allow access from known IP addresses and address ranges only. To edit security group rules, open Security
     * Groups in the navigation pane of the EC2 management console.
     * </p>
     * <p>
     * To specify your own domain for a server, and provide your own self-signed or CA-signed certificate and private
     * key, specify values for <code>CustomDomain</code>, <code>CustomCertificate</code>, and
     * <code>CustomPrivateKey</code>.
     * </p>
     *
     * @param createServerRequest
     * @return Result of the CreateServer operation returned by the service.
     * @throws LimitExceededException
     *         The limit of servers or backups has been reached.
     * @throws ResourceAlreadyExistsException
     *         The requested resource cannot be created because it already exists.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist, or access was denied.
     * @throws ValidationException
     *         One or more of the provided request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OpsWorksCmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OpsWorksCmClient.CreateServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/opsworkscm-2016-11-01/CreateServer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateServerResponse createServer(CreateServerRequest createServerRequest) throws LimitExceededException,
            ResourceAlreadyExistsException, ResourceNotFoundException, ValidationException, AwsServiceException,
            SdkClientException, OpsWorksCmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateServerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateServerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateServerRequest, CreateServerResponse>()
                .withOperationName("CreateServer").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createServerRequest)
                .withMarshaller(new CreateServerRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a backup. You can delete both manual and automated backups. This operation is asynchronous.
     * </p>
     * <p>
     * An <code>InvalidStateException</code> is thrown when a backup deletion is already in progress. A
     * <code>ResourceNotFoundException</code> is thrown when the backup does not exist. A
     * <code>ValidationException</code> is thrown when parameters of the request are not valid.
     * </p>
     *
     * @param deleteBackupRequest
     * @return Result of the DeleteBackup operation returned by the service.
     * @throws InvalidStateException
     *         The resource is in a state that does not allow you to perform a specified action.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist, or access was denied.
     * @throws ValidationException
     *         One or more of the provided request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OpsWorksCmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OpsWorksCmClient.DeleteBackup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/opsworkscm-2016-11-01/DeleteBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteBackupResponse deleteBackup(DeleteBackupRequest deleteBackupRequest) throws InvalidStateException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, OpsWorksCmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteBackupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteBackupRequest, DeleteBackupResponse>()
                .withOperationName("DeleteBackup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteBackupRequest)
                .withMarshaller(new DeleteBackupRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes the server and the underlying AWS CloudFormation stacks (including the server's EC2 instance). When you
     * run this command, the server state is updated to <code>DELETING</code>. After the server is deleted, it is no
     * longer returned by <code>DescribeServer</code> requests. If the AWS CloudFormation stack cannot be deleted, the
     * server cannot be deleted.
     * </p>
     * <p>
     * This operation is asynchronous.
     * </p>
     * <p>
     * An <code>InvalidStateException</code> is thrown when a server deletion is already in progress. A
     * <code>ResourceNotFoundException</code> is thrown when the server does not exist. A
     * <code>ValidationException</code> is raised when parameters of the request are not valid.
     * </p>
     * <p>
     * </p>
     *
     * @param deleteServerRequest
     * @return Result of the DeleteServer operation returned by the service.
     * @throws InvalidStateException
     *         The resource is in a state that does not allow you to perform a specified action.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist, or access was denied.
     * @throws ValidationException
     *         One or more of the provided request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OpsWorksCmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OpsWorksCmClient.DeleteServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/opsworkscm-2016-11-01/DeleteServer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteServerResponse deleteServer(DeleteServerRequest deleteServerRequest) throws InvalidStateException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, OpsWorksCmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteServerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteServerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteServerRequest, DeleteServerResponse>()
                .withOperationName("DeleteServer").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteServerRequest)
                .withMarshaller(new DeleteServerRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Describes your OpsWorks-CM account attributes.
     * </p>
     * <p>
     * This operation is synchronous.
     * </p>
     *
     * @param describeAccountAttributesRequest
     * @return Result of the DescribeAccountAttributes operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OpsWorksCmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OpsWorksCmClient.DescribeAccountAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/opsworkscm-2016-11-01/DescribeAccountAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAccountAttributesResponse describeAccountAttributes(
            DescribeAccountAttributesRequest describeAccountAttributesRequest) throws AwsServiceException, SdkClientException,
            OpsWorksCmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAccountAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeAccountAttributesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeAccountAttributesRequest, DescribeAccountAttributesResponse>()
                        .withOperationName("DescribeAccountAttributes").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeAccountAttributesRequest)
                        .withMarshaller(new DescribeAccountAttributesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Describes backups. The results are ordered by time, with newest backups first. If you do not specify a BackupId
     * or ServerName, the command returns all backups.
     * </p>
     * <p>
     * This operation is synchronous.
     * </p>
     * <p>
     * A <code>ResourceNotFoundException</code> is thrown when the backup does not exist. A
     * <code>ValidationException</code> is raised when parameters of the request are not valid.
     * </p>
     *
     * @param describeBackupsRequest
     * @return Result of the DescribeBackups operation returned by the service.
     * @throws ValidationException
     *         One or more of the provided request parameters are not valid.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist, or access was denied.
     * @throws InvalidNextTokenException
     *         This occurs when the provided nextToken is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OpsWorksCmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OpsWorksCmClient.DescribeBackups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/opsworkscm-2016-11-01/DescribeBackups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeBackupsResponse describeBackups(DescribeBackupsRequest describeBackupsRequest) throws ValidationException,
            ResourceNotFoundException, InvalidNextTokenException, AwsServiceException, SdkClientException, OpsWorksCmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeBackupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeBackupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeBackupsRequest, DescribeBackupsResponse>()
                .withOperationName("DescribeBackups").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeBackupsRequest)
                .withMarshaller(new DescribeBackupsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Describes events for a specified server. Results are ordered by time, with newest events first.
     * </p>
     * <p>
     * This operation is synchronous.
     * </p>
     * <p>
     * A <code>ResourceNotFoundException</code> is thrown when the server does not exist. A
     * <code>ValidationException</code> is raised when parameters of the request are not valid.
     * </p>
     *
     * @param describeEventsRequest
     * @return Result of the DescribeEvents operation returned by the service.
     * @throws ValidationException
     *         One or more of the provided request parameters are not valid.
     * @throws InvalidNextTokenException
     *         This occurs when the provided nextToken is not valid.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist, or access was denied.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OpsWorksCmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OpsWorksCmClient.DescribeEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/opsworkscm-2016-11-01/DescribeEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeEventsResponse describeEvents(DescribeEventsRequest describeEventsRequest) throws ValidationException,
            InvalidNextTokenException, ResourceNotFoundException, AwsServiceException, SdkClientException, OpsWorksCmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEventsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeEventsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeEventsRequest, DescribeEventsResponse>()
                .withOperationName("DescribeEvents").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeEventsRequest)
                .withMarshaller(new DescribeEventsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the current status of an existing association or disassociation request.
     * </p>
     * <p>
     * A <code>ResourceNotFoundException</code> is thrown when no recent association or disassociation request with the
     * specified token is found, or when the server does not exist. A <code>ValidationException</code> is raised when
     * parameters of the request are not valid.
     * </p>
     *
     * @param describeNodeAssociationStatusRequest
     * @return Result of the DescribeNodeAssociationStatus operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist, or access was denied.
     * @throws ValidationException
     *         One or more of the provided request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OpsWorksCmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OpsWorksCmClient.DescribeNodeAssociationStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/opsworkscm-2016-11-01/DescribeNodeAssociationStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeNodeAssociationStatusResponse describeNodeAssociationStatus(
            DescribeNodeAssociationStatusRequest describeNodeAssociationStatusRequest) throws ResourceNotFoundException,
            ValidationException, AwsServiceException, SdkClientException, OpsWorksCmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeNodeAssociationStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeNodeAssociationStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeNodeAssociationStatusRequest, DescribeNodeAssociationStatusResponse>()
                        .withOperationName("DescribeNodeAssociationStatus").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeNodeAssociationStatusRequest)
                        .withMarshaller(new DescribeNodeAssociationStatusRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists all configuration management servers that are identified with your account. Only the stored results from
     * Amazon DynamoDB are returned. AWS OpsWorks CM does not query other services.
     * </p>
     * <p>
     * This operation is synchronous.
     * </p>
     * <p>
     * A <code>ResourceNotFoundException</code> is thrown when the server does not exist. A
     * <code>ValidationException</code> is raised when parameters of the request are not valid.
     * </p>
     *
     * @param describeServersRequest
     * @return Result of the DescribeServers operation returned by the service.
     * @throws ValidationException
     *         One or more of the provided request parameters are not valid.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist, or access was denied.
     * @throws InvalidNextTokenException
     *         This occurs when the provided nextToken is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OpsWorksCmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OpsWorksCmClient.DescribeServers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/opsworkscm-2016-11-01/DescribeServers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeServersResponse describeServers(DescribeServersRequest describeServersRequest) throws ValidationException,
            ResourceNotFoundException, InvalidNextTokenException, AwsServiceException, SdkClientException, OpsWorksCmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeServersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeServersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeServersRequest, DescribeServersResponse>()
                .withOperationName("DescribeServers").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeServersRequest)
                .withMarshaller(new DescribeServersRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Disassociates a node from an AWS OpsWorks CM server, and removes the node from the server's managed nodes. After
     * a node is disassociated, the node key pair is no longer valid for accessing the configuration manager's API. For
     * more information about how to associate a node, see <a>AssociateNode</a>.
     * </p>
     * <p>
     * A node can can only be disassociated from a server that is in a <code>HEALTHY</code> state. Otherwise, an
     * <code>InvalidStateException</code> is thrown. A <code>ResourceNotFoundException</code> is thrown when the server
     * does not exist. A <code>ValidationException</code> is raised when parameters of the request are not valid.
     * </p>
     *
     * @param disassociateNodeRequest
     * @return Result of the DisassociateNode operation returned by the service.
     * @throws InvalidStateException
     *         The resource is in a state that does not allow you to perform a specified action.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist, or access was denied.
     * @throws ValidationException
     *         One or more of the provided request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OpsWorksCmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OpsWorksCmClient.DisassociateNode
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/opsworkscm-2016-11-01/DisassociateNode" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DisassociateNodeResponse disassociateNode(DisassociateNodeRequest disassociateNodeRequest)
            throws InvalidStateException, ResourceNotFoundException, ValidationException, AwsServiceException,
            SdkClientException, OpsWorksCmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateNodeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DisassociateNodeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DisassociateNodeRequest, DisassociateNodeResponse>()
                .withOperationName("DisassociateNode").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(disassociateNodeRequest)
                .withMarshaller(new DisassociateNodeRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Exports a specified server engine attribute as a base64-encoded string. For example, you can export user data
     * that you can use in EC2 to associate nodes with a server.
     * </p>
     * <p>
     * This operation is synchronous.
     * </p>
     * <p>
     * A <code>ValidationException</code> is raised when parameters of the request are not valid. A
     * <code>ResourceNotFoundException</code> is thrown when the server does not exist. An
     * <code>InvalidStateException</code> is thrown when the server is in any of the following states: CREATING,
     * TERMINATED, FAILED or DELETING.
     * </p>
     *
     * @param exportServerEngineAttributeRequest
     * @return Result of the ExportServerEngineAttribute operation returned by the service.
     * @throws ValidationException
     *         One or more of the provided request parameters are not valid.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist, or access was denied.
     * @throws InvalidStateException
     *         The resource is in a state that does not allow you to perform a specified action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OpsWorksCmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OpsWorksCmClient.ExportServerEngineAttribute
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/opsworkscm-2016-11-01/ExportServerEngineAttribute"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ExportServerEngineAttributeResponse exportServerEngineAttribute(
            ExportServerEngineAttributeRequest exportServerEngineAttributeRequest) throws ValidationException,
            ResourceNotFoundException, InvalidStateException, AwsServiceException, SdkClientException, OpsWorksCmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ExportServerEngineAttributeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ExportServerEngineAttributeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ExportServerEngineAttributeRequest, ExportServerEngineAttributeResponse>()
                        .withOperationName("ExportServerEngineAttribute").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(exportServerEngineAttributeRequest)
                        .withMarshaller(new ExportServerEngineAttributeRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of tags that are applied to the specified AWS OpsWorks for Chef Automate or AWS OpsWorks for
     * Puppet Enterprise servers or backups.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist, or access was denied.
     * @throws ValidationException
     *         One or more of the provided request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OpsWorksCmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OpsWorksCmClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/opsworkscm-2016-11-01/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, OpsWorksCmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Restores a backup to a server that is in a <code>CONNECTION_LOST</code>, <code>HEALTHY</code>,
     * <code>RUNNING</code>, <code>UNHEALTHY</code>, or <code>TERMINATED</code> state. When you run RestoreServer, the
     * server's EC2 instance is deleted, and a new EC2 instance is configured. RestoreServer maintains the existing
     * server endpoint, so configuration management of the server's client devices (nodes) should continue to work.
     * </p>
     * <p>
     * Restoring from a backup is performed by creating a new EC2 instance. If restoration is successful, and the server
     * is in a <code>HEALTHY</code> state, AWS OpsWorks CM switches traffic over to the new instance. After restoration
     * is finished, the old EC2 instance is maintained in a <code>Running</code> or <code>Stopped</code> state, but is
     * eventually terminated.
     * </p>
     * <p>
     * This operation is asynchronous.
     * </p>
     * <p>
     * An <code>InvalidStateException</code> is thrown when the server is not in a valid state. A
     * <code>ResourceNotFoundException</code> is thrown when the server does not exist. A
     * <code>ValidationException</code> is raised when parameters of the request are not valid.
     * </p>
     *
     * @param restoreServerRequest
     * @return Result of the RestoreServer operation returned by the service.
     * @throws InvalidStateException
     *         The resource is in a state that does not allow you to perform a specified action.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist, or access was denied.
     * @throws ValidationException
     *         One or more of the provided request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OpsWorksCmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OpsWorksCmClient.RestoreServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/opsworkscm-2016-11-01/RestoreServer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RestoreServerResponse restoreServer(RestoreServerRequest restoreServerRequest) throws InvalidStateException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, OpsWorksCmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RestoreServerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RestoreServerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<RestoreServerRequest, RestoreServerResponse>()
                .withOperationName("RestoreServer").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(restoreServerRequest)
                .withMarshaller(new RestoreServerRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Manually starts server maintenance. This command can be useful if an earlier maintenance attempt failed, and the
     * underlying cause of maintenance failure has been resolved. The server is in an <code>UNDER_MAINTENANCE</code>
     * state while maintenance is in progress.
     * </p>
     * <p>
     * Maintenance can only be started on servers in <code>HEALTHY</code> and <code>UNHEALTHY</code> states. Otherwise,
     * an <code>InvalidStateException</code> is thrown. A <code>ResourceNotFoundException</code> is thrown when the
     * server does not exist. A <code>ValidationException</code> is raised when parameters of the request are not valid.
     * </p>
     *
     * @param startMaintenanceRequest
     * @return Result of the StartMaintenance operation returned by the service.
     * @throws InvalidStateException
     *         The resource is in a state that does not allow you to perform a specified action.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist, or access was denied.
     * @throws ValidationException
     *         One or more of the provided request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OpsWorksCmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OpsWorksCmClient.StartMaintenance
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/opsworkscm-2016-11-01/StartMaintenance" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StartMaintenanceResponse startMaintenance(StartMaintenanceRequest startMaintenanceRequest)
            throws InvalidStateException, ResourceNotFoundException, ValidationException, AwsServiceException,
            SdkClientException, OpsWorksCmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartMaintenanceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartMaintenanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<StartMaintenanceRequest, StartMaintenanceResponse>()
                .withOperationName("StartMaintenance").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(startMaintenanceRequest)
                .withMarshaller(new StartMaintenanceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Applies tags to an AWS OpsWorks for Chef Automate or AWS OpsWorks for Puppet Enterprise server, or to server
     * backups.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist, or access was denied.
     * @throws ValidationException
     *         One or more of the provided request parameters are not valid.
     * @throws InvalidStateException
     *         The resource is in a state that does not allow you to perform a specified action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OpsWorksCmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OpsWorksCmClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/opsworkscm-2016-11-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            ValidationException, InvalidStateException, AwsServiceException, SdkClientException, OpsWorksCmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                .withOperationName("TagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes specified tags from an AWS OpsWorks-CM server or backup.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist, or access was denied.
     * @throws ValidationException
     *         One or more of the provided request parameters are not valid.
     * @throws InvalidStateException
     *         The resource is in a state that does not allow you to perform a specified action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OpsWorksCmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OpsWorksCmClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/opsworkscm-2016-11-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            ValidationException, InvalidStateException, AwsServiceException, SdkClientException, OpsWorksCmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                .withOperationName("UntagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates settings for a server.
     * </p>
     * <p>
     * This operation is synchronous.
     * </p>
     *
     * @param updateServerRequest
     * @return Result of the UpdateServer operation returned by the service.
     * @throws InvalidStateException
     *         The resource is in a state that does not allow you to perform a specified action.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist, or access was denied.
     * @throws ValidationException
     *         One or more of the provided request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OpsWorksCmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OpsWorksCmClient.UpdateServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/opsworkscm-2016-11-01/UpdateServer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateServerResponse updateServer(UpdateServerRequest updateServerRequest) throws InvalidStateException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, OpsWorksCmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateServerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateServerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateServerRequest, UpdateServerResponse>()
                .withOperationName("UpdateServer").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateServerRequest)
                .withMarshaller(new UpdateServerRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates engine-specific attributes on a specified server. The server enters the <code>MODIFYING</code> state when
     * this operation is in progress. Only one update can occur at a time. You can use this command to reset a Chef
     * server's public key (<code>CHEF_PIVOTAL_KEY</code>) or a Puppet server's admin password (
     * <code>PUPPET_ADMIN_PASSWORD</code>).
     * </p>
     * <p>
     * This operation is asynchronous.
     * </p>
     * <p>
     * This operation can only be called for servers in <code>HEALTHY</code> or <code>UNHEALTHY</code> states.
     * Otherwise, an <code>InvalidStateException</code> is raised. A <code>ResourceNotFoundException</code> is thrown
     * when the server does not exist. A <code>ValidationException</code> is raised when parameters of the request are
     * not valid.
     * </p>
     *
     * @param updateServerEngineAttributesRequest
     * @return Result of the UpdateServerEngineAttributes operation returned by the service.
     * @throws InvalidStateException
     *         The resource is in a state that does not allow you to perform a specified action.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist, or access was denied.
     * @throws ValidationException
     *         One or more of the provided request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OpsWorksCmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OpsWorksCmClient.UpdateServerEngineAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/opsworkscm-2016-11-01/UpdateServerEngineAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateServerEngineAttributesResponse updateServerEngineAttributes(
            UpdateServerEngineAttributesRequest updateServerEngineAttributesRequest) throws InvalidStateException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, OpsWorksCmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateServerEngineAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateServerEngineAttributesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<UpdateServerEngineAttributesRequest, UpdateServerEngineAttributesResponse>()
                        .withOperationName("UpdateServerEngineAttributes").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(updateServerEngineAttributesRequest)
                        .withMarshaller(new UpdateServerEngineAttributesRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(OpsWorksCmException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidStateException")
                                .exceptionBuilderSupplier(InvalidStateException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
