/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.observabilityadmin.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A summary of a telemetry rule's key properties.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TelemetryRuleSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<TelemetryRuleSummary.Builder, TelemetryRuleSummary> {
    private static final SdkField<String> RULE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RuleName").getter(getter(TelemetryRuleSummary::ruleName)).setter(setter(Builder::ruleName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleName").build()).build();

    private static final SdkField<String> RULE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RuleArn").getter(getter(TelemetryRuleSummary::ruleArn)).setter(setter(Builder::ruleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleArn").build()).build();

    private static final SdkField<Long> CREATED_TIME_STAMP_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("CreatedTimeStamp").getter(getter(TelemetryRuleSummary::createdTimeStamp))
            .setter(setter(Builder::createdTimeStamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTimeStamp").build()).build();

    private static final SdkField<Long> LAST_UPDATE_TIME_STAMP_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("LastUpdateTimeStamp").getter(getter(TelemetryRuleSummary::lastUpdateTimeStamp))
            .setter(setter(Builder::lastUpdateTimeStamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdateTimeStamp").build())
            .build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(TelemetryRuleSummary::resourceTypeAsString))
            .setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<String> TELEMETRY_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TelemetryType").getter(getter(TelemetryRuleSummary::telemetryTypeAsString))
            .setter(setter(Builder::telemetryType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TelemetryType").build()).build();

    private static final SdkField<List<String>> TELEMETRY_SOURCE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("TelemetrySourceTypes")
            .getter(getter(TelemetryRuleSummary::telemetrySourceTypesAsStrings))
            .setter(setter(Builder::telemetrySourceTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TelemetrySourceTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RULE_NAME_FIELD,
            RULE_ARN_FIELD, CREATED_TIME_STAMP_FIELD, LAST_UPDATE_TIME_STAMP_FIELD, RESOURCE_TYPE_FIELD, TELEMETRY_TYPE_FIELD,
            TELEMETRY_SOURCE_TYPES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String ruleName;

    private final String ruleArn;

    private final Long createdTimeStamp;

    private final Long lastUpdateTimeStamp;

    private final String resourceType;

    private final String telemetryType;

    private final List<String> telemetrySourceTypes;

    private TelemetryRuleSummary(BuilderImpl builder) {
        this.ruleName = builder.ruleName;
        this.ruleArn = builder.ruleArn;
        this.createdTimeStamp = builder.createdTimeStamp;
        this.lastUpdateTimeStamp = builder.lastUpdateTimeStamp;
        this.resourceType = builder.resourceType;
        this.telemetryType = builder.telemetryType;
        this.telemetrySourceTypes = builder.telemetrySourceTypes;
    }

    /**
     * <p>
     * The name of the telemetry rule.
     * </p>
     * 
     * @return The name of the telemetry rule.
     */
    public final String ruleName() {
        return ruleName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the telemetry rule.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the telemetry rule.
     */
    public final String ruleArn() {
        return ruleArn;
    }

    /**
     * <p>
     * The timestamp when the telemetry rule was created.
     * </p>
     * 
     * @return The timestamp when the telemetry rule was created.
     */
    public final Long createdTimeStamp() {
        return createdTimeStamp;
    }

    /**
     * <p>
     * The timestamp when the telemetry rule was last modified.
     * </p>
     * 
     * @return The timestamp when the telemetry rule was last modified.
     */
    public final Long lastUpdateTimeStamp() {
        return lastUpdateTimeStamp;
    }

    /**
     * <p>
     * The type of Amazon Web Services resource the rule applies to.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of Amazon Web Services resource the rule applies to.
     * @see ResourceType
     */
    public final ResourceType resourceType() {
        return ResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The type of Amazon Web Services resource the rule applies to.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of Amazon Web Services resource the rule applies to.
     * @see ResourceType
     */
    public final String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * <p>
     * The type of telemetry (Logs, Metrics, or Traces) the rule configures.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #telemetryType}
     * will return {@link TelemetryType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #telemetryTypeAsString}.
     * </p>
     * 
     * @return The type of telemetry (Logs, Metrics, or Traces) the rule configures.
     * @see TelemetryType
     */
    public final TelemetryType telemetryType() {
        return TelemetryType.fromValue(telemetryType);
    }

    /**
     * <p>
     * The type of telemetry (Logs, Metrics, or Traces) the rule configures.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #telemetryType}
     * will return {@link TelemetryType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #telemetryTypeAsString}.
     * </p>
     * 
     * @return The type of telemetry (Logs, Metrics, or Traces) the rule configures.
     * @see TelemetryType
     */
    public final String telemetryTypeAsString() {
        return telemetryType;
    }

    /**
     * <p>
     * The types of telemetry sources configured for this rule, such as VPC Flow Logs or EKS audit logs.
     * TelemetrySourceTypes must be correlated with the specific resource type.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTelemetrySourceTypes} method.
     * </p>
     * 
     * @return The types of telemetry sources configured for this rule, such as VPC Flow Logs or EKS audit logs.
     *         TelemetrySourceTypes must be correlated with the specific resource type.
     */
    public final List<TelemetrySourceType> telemetrySourceTypes() {
        return TelemetrySourceTypesCopier.copyStringToEnum(telemetrySourceTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the TelemetrySourceTypes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTelemetrySourceTypes() {
        return telemetrySourceTypes != null && !(telemetrySourceTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The types of telemetry sources configured for this rule, such as VPC Flow Logs or EKS audit logs.
     * TelemetrySourceTypes must be correlated with the specific resource type.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTelemetrySourceTypes} method.
     * </p>
     * 
     * @return The types of telemetry sources configured for this rule, such as VPC Flow Logs or EKS audit logs.
     *         TelemetrySourceTypes must be correlated with the specific resource type.
     */
    public final List<String> telemetrySourceTypesAsStrings() {
        return telemetrySourceTypes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ruleName());
        hashCode = 31 * hashCode + Objects.hashCode(ruleArn());
        hashCode = 31 * hashCode + Objects.hashCode(createdTimeStamp());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdateTimeStamp());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(telemetryTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTelemetrySourceTypes() ? telemetrySourceTypesAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TelemetryRuleSummary)) {
            return false;
        }
        TelemetryRuleSummary other = (TelemetryRuleSummary) obj;
        return Objects.equals(ruleName(), other.ruleName()) && Objects.equals(ruleArn(), other.ruleArn())
                && Objects.equals(createdTimeStamp(), other.createdTimeStamp())
                && Objects.equals(lastUpdateTimeStamp(), other.lastUpdateTimeStamp())
                && Objects.equals(resourceTypeAsString(), other.resourceTypeAsString())
                && Objects.equals(telemetryTypeAsString(), other.telemetryTypeAsString())
                && hasTelemetrySourceTypes() == other.hasTelemetrySourceTypes()
                && Objects.equals(telemetrySourceTypesAsStrings(), other.telemetrySourceTypesAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TelemetryRuleSummary").add("RuleName", ruleName()).add("RuleArn", ruleArn())
                .add("CreatedTimeStamp", createdTimeStamp()).add("LastUpdateTimeStamp", lastUpdateTimeStamp())
                .add("ResourceType", resourceTypeAsString()).add("TelemetryType", telemetryTypeAsString())
                .add("TelemetrySourceTypes", hasTelemetrySourceTypes() ? telemetrySourceTypesAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RuleName":
            return Optional.ofNullable(clazz.cast(ruleName()));
        case "RuleArn":
            return Optional.ofNullable(clazz.cast(ruleArn()));
        case "CreatedTimeStamp":
            return Optional.ofNullable(clazz.cast(createdTimeStamp()));
        case "LastUpdateTimeStamp":
            return Optional.ofNullable(clazz.cast(lastUpdateTimeStamp()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "TelemetryType":
            return Optional.ofNullable(clazz.cast(telemetryTypeAsString()));
        case "TelemetrySourceTypes":
            return Optional.ofNullable(clazz.cast(telemetrySourceTypesAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("RuleName", RULE_NAME_FIELD);
        map.put("RuleArn", RULE_ARN_FIELD);
        map.put("CreatedTimeStamp", CREATED_TIME_STAMP_FIELD);
        map.put("LastUpdateTimeStamp", LAST_UPDATE_TIME_STAMP_FIELD);
        map.put("ResourceType", RESOURCE_TYPE_FIELD);
        map.put("TelemetryType", TELEMETRY_TYPE_FIELD);
        map.put("TelemetrySourceTypes", TELEMETRY_SOURCE_TYPES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<TelemetryRuleSummary, T> g) {
        return obj -> g.apply((TelemetryRuleSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TelemetryRuleSummary> {
        /**
         * <p>
         * The name of the telemetry rule.
         * </p>
         * 
         * @param ruleName
         *        The name of the telemetry rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleName(String ruleName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the telemetry rule.
         * </p>
         * 
         * @param ruleArn
         *        The Amazon Resource Name (ARN) of the telemetry rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleArn(String ruleArn);

        /**
         * <p>
         * The timestamp when the telemetry rule was created.
         * </p>
         * 
         * @param createdTimeStamp
         *        The timestamp when the telemetry rule was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTimeStamp(Long createdTimeStamp);

        /**
         * <p>
         * The timestamp when the telemetry rule was last modified.
         * </p>
         * 
         * @param lastUpdateTimeStamp
         *        The timestamp when the telemetry rule was last modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdateTimeStamp(Long lastUpdateTimeStamp);

        /**
         * <p>
         * The type of Amazon Web Services resource the rule applies to.
         * </p>
         * 
         * @param resourceType
         *        The type of Amazon Web Services resource the rule applies to.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The type of Amazon Web Services resource the rule applies to.
         * </p>
         * 
         * @param resourceType
         *        The type of Amazon Web Services resource the rule applies to.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(ResourceType resourceType);

        /**
         * <p>
         * The type of telemetry (Logs, Metrics, or Traces) the rule configures.
         * </p>
         * 
         * @param telemetryType
         *        The type of telemetry (Logs, Metrics, or Traces) the rule configures.
         * @see TelemetryType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TelemetryType
         */
        Builder telemetryType(String telemetryType);

        /**
         * <p>
         * The type of telemetry (Logs, Metrics, or Traces) the rule configures.
         * </p>
         * 
         * @param telemetryType
         *        The type of telemetry (Logs, Metrics, or Traces) the rule configures.
         * @see TelemetryType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TelemetryType
         */
        Builder telemetryType(TelemetryType telemetryType);

        /**
         * <p>
         * The types of telemetry sources configured for this rule, such as VPC Flow Logs or EKS audit logs.
         * TelemetrySourceTypes must be correlated with the specific resource type.
         * </p>
         * 
         * @param telemetrySourceTypes
         *        The types of telemetry sources configured for this rule, such as VPC Flow Logs or EKS audit logs.
         *        TelemetrySourceTypes must be correlated with the specific resource type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder telemetrySourceTypesWithStrings(Collection<String> telemetrySourceTypes);

        /**
         * <p>
         * The types of telemetry sources configured for this rule, such as VPC Flow Logs or EKS audit logs.
         * TelemetrySourceTypes must be correlated with the specific resource type.
         * </p>
         * 
         * @param telemetrySourceTypes
         *        The types of telemetry sources configured for this rule, such as VPC Flow Logs or EKS audit logs.
         *        TelemetrySourceTypes must be correlated with the specific resource type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder telemetrySourceTypesWithStrings(String... telemetrySourceTypes);

        /**
         * <p>
         * The types of telemetry sources configured for this rule, such as VPC Flow Logs or EKS audit logs.
         * TelemetrySourceTypes must be correlated with the specific resource type.
         * </p>
         * 
         * @param telemetrySourceTypes
         *        The types of telemetry sources configured for this rule, such as VPC Flow Logs or EKS audit logs.
         *        TelemetrySourceTypes must be correlated with the specific resource type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder telemetrySourceTypes(Collection<TelemetrySourceType> telemetrySourceTypes);

        /**
         * <p>
         * The types of telemetry sources configured for this rule, such as VPC Flow Logs or EKS audit logs.
         * TelemetrySourceTypes must be correlated with the specific resource type.
         * </p>
         * 
         * @param telemetrySourceTypes
         *        The types of telemetry sources configured for this rule, such as VPC Flow Logs or EKS audit logs.
         *        TelemetrySourceTypes must be correlated with the specific resource type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder telemetrySourceTypes(TelemetrySourceType... telemetrySourceTypes);
    }

    static final class BuilderImpl implements Builder {
        private String ruleName;

        private String ruleArn;

        private Long createdTimeStamp;

        private Long lastUpdateTimeStamp;

        private String resourceType;

        private String telemetryType;

        private List<String> telemetrySourceTypes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(TelemetryRuleSummary model) {
            ruleName(model.ruleName);
            ruleArn(model.ruleArn);
            createdTimeStamp(model.createdTimeStamp);
            lastUpdateTimeStamp(model.lastUpdateTimeStamp);
            resourceType(model.resourceType);
            telemetryType(model.telemetryType);
            telemetrySourceTypesWithStrings(model.telemetrySourceTypes);
        }

        public final String getRuleName() {
            return ruleName;
        }

        public final void setRuleName(String ruleName) {
            this.ruleName = ruleName;
        }

        @Override
        public final Builder ruleName(String ruleName) {
            this.ruleName = ruleName;
            return this;
        }

        public final String getRuleArn() {
            return ruleArn;
        }

        public final void setRuleArn(String ruleArn) {
            this.ruleArn = ruleArn;
        }

        @Override
        public final Builder ruleArn(String ruleArn) {
            this.ruleArn = ruleArn;
            return this;
        }

        public final Long getCreatedTimeStamp() {
            return createdTimeStamp;
        }

        public final void setCreatedTimeStamp(Long createdTimeStamp) {
            this.createdTimeStamp = createdTimeStamp;
        }

        @Override
        public final Builder createdTimeStamp(Long createdTimeStamp) {
            this.createdTimeStamp = createdTimeStamp;
            return this;
        }

        public final Long getLastUpdateTimeStamp() {
            return lastUpdateTimeStamp;
        }

        public final void setLastUpdateTimeStamp(Long lastUpdateTimeStamp) {
            this.lastUpdateTimeStamp = lastUpdateTimeStamp;
        }

        @Override
        public final Builder lastUpdateTimeStamp(Long lastUpdateTimeStamp) {
            this.lastUpdateTimeStamp = lastUpdateTimeStamp;
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(ResourceType resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        public final String getTelemetryType() {
            return telemetryType;
        }

        public final void setTelemetryType(String telemetryType) {
            this.telemetryType = telemetryType;
        }

        @Override
        public final Builder telemetryType(String telemetryType) {
            this.telemetryType = telemetryType;
            return this;
        }

        @Override
        public final Builder telemetryType(TelemetryType telemetryType) {
            this.telemetryType(telemetryType == null ? null : telemetryType.toString());
            return this;
        }

        public final Collection<String> getTelemetrySourceTypes() {
            if (telemetrySourceTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return telemetrySourceTypes;
        }

        public final void setTelemetrySourceTypes(Collection<String> telemetrySourceTypes) {
            this.telemetrySourceTypes = TelemetrySourceTypesCopier.copy(telemetrySourceTypes);
        }

        @Override
        public final Builder telemetrySourceTypesWithStrings(Collection<String> telemetrySourceTypes) {
            this.telemetrySourceTypes = TelemetrySourceTypesCopier.copy(telemetrySourceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder telemetrySourceTypesWithStrings(String... telemetrySourceTypes) {
            telemetrySourceTypesWithStrings(Arrays.asList(telemetrySourceTypes));
            return this;
        }

        @Override
        public final Builder telemetrySourceTypes(Collection<TelemetrySourceType> telemetrySourceTypes) {
            this.telemetrySourceTypes = TelemetrySourceTypesCopier.copyEnumToString(telemetrySourceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder telemetrySourceTypes(TelemetrySourceType... telemetrySourceTypes) {
            telemetrySourceTypes(Arrays.asList(telemetrySourceTypes));
            return this;
        }

        @Override
        public TelemetryRuleSummary build() {
            return new TelemetryRuleSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
