/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.networkfirewall.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A single Suricata rules specification, for use in a stateful rule group. Use this option to specify a simple Suricata
 * rule with protocol, source and destination, ports, direction, and rule options. For information about the Suricata
 * <code>Rules</code> format, see <a href="https://suricata.readthedocs.io/en/suricata-7.0.3/rules/intro.html">Rules
 * Format</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StatefulRule implements SdkPojo, Serializable, ToCopyableBuilder<StatefulRule.Builder, StatefulRule> {
    private static final SdkField<String> ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Action")
            .getter(getter(StatefulRule::actionAsString)).setter(setter(Builder::action))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Action").build()).build();

    private static final SdkField<Header> HEADER_FIELD = SdkField.<Header> builder(MarshallingType.SDK_POJO).memberName("Header")
            .getter(getter(StatefulRule::header)).setter(setter(Builder::header)).constructor(Header::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Header").build()).build();

    private static final SdkField<List<RuleOption>> RULE_OPTIONS_FIELD = SdkField
            .<List<RuleOption>> builder(MarshallingType.LIST)
            .memberName("RuleOptions")
            .getter(getter(StatefulRule::ruleOptions))
            .setter(setter(Builder::ruleOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleOptions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RuleOption> builder(MarshallingType.SDK_POJO)
                                            .constructor(RuleOption::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACTION_FIELD, HEADER_FIELD,
            RULE_OPTIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String action;

    private final Header header;

    private final List<RuleOption> ruleOptions;

    private StatefulRule(BuilderImpl builder) {
        this.action = builder.action;
        this.header = builder.header;
        this.ruleOptions = builder.ruleOptions;
    }

    /**
     * <p>
     * Defines what Network Firewall should do with the packets in a traffic flow when the flow matches the stateful
     * rule criteria. For all actions, Network Firewall performs the specified action and discontinues stateful
     * inspection of the traffic flow.
     * </p>
     * <p>
     * The actions for a stateful rule are defined as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>PASS</b> - Permits the packets to go to the intended destination.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DROP</b> - Blocks the packets from going to the intended destination and sends an alert log message, if alert
     * logging is configured in the <a>Firewall</a> <a>LoggingConfiguration</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ALERT</b> - Sends an alert log message, if alert logging is configured in the <a>Firewall</a>
     * <a>LoggingConfiguration</a>.
     * </p>
     * <p>
     * You can use this action to test a rule that you intend to use to drop traffic. You can enable the rule with
     * <code>ALERT</code> action, verify in the logs that the rule is filtering as you want, then change the action to
     * <code>DROP</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>REJECT</b> - Drops traffic that matches the conditions of the stateful rule, and sends a TCP reset packet back
     * to sender of the packet. A TCP reset packet is a packet with no payload and an RST bit contained in the TCP
     * header flags. REJECT is available only for TCP traffic. This option doesn't support FTP or IMAP protocols.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link StatefulAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #actionAsString}.
     * </p>
     * 
     * @return Defines what Network Firewall should do with the packets in a traffic flow when the flow matches the
     *         stateful rule criteria. For all actions, Network Firewall performs the specified action and discontinues
     *         stateful inspection of the traffic flow. </p>
     *         <p>
     *         The actions for a stateful rule are defined as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>PASS</b> - Permits the packets to go to the intended destination.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DROP</b> - Blocks the packets from going to the intended destination and sends an alert log message,
     *         if alert logging is configured in the <a>Firewall</a> <a>LoggingConfiguration</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ALERT</b> - Sends an alert log message, if alert logging is configured in the <a>Firewall</a>
     *         <a>LoggingConfiguration</a>.
     *         </p>
     *         <p>
     *         You can use this action to test a rule that you intend to use to drop traffic. You can enable the rule
     *         with <code>ALERT</code> action, verify in the logs that the rule is filtering as you want, then change
     *         the action to <code>DROP</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>REJECT</b> - Drops traffic that matches the conditions of the stateful rule, and sends a TCP reset
     *         packet back to sender of the packet. A TCP reset packet is a packet with no payload and an RST bit
     *         contained in the TCP header flags. REJECT is available only for TCP traffic. This option doesn't support
     *         FTP or IMAP protocols.
     *         </p>
     *         </li>
     * @see StatefulAction
     */
    public final StatefulAction action() {
        return StatefulAction.fromValue(action);
    }

    /**
     * <p>
     * Defines what Network Firewall should do with the packets in a traffic flow when the flow matches the stateful
     * rule criteria. For all actions, Network Firewall performs the specified action and discontinues stateful
     * inspection of the traffic flow.
     * </p>
     * <p>
     * The actions for a stateful rule are defined as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>PASS</b> - Permits the packets to go to the intended destination.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DROP</b> - Blocks the packets from going to the intended destination and sends an alert log message, if alert
     * logging is configured in the <a>Firewall</a> <a>LoggingConfiguration</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ALERT</b> - Sends an alert log message, if alert logging is configured in the <a>Firewall</a>
     * <a>LoggingConfiguration</a>.
     * </p>
     * <p>
     * You can use this action to test a rule that you intend to use to drop traffic. You can enable the rule with
     * <code>ALERT</code> action, verify in the logs that the rule is filtering as you want, then change the action to
     * <code>DROP</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>REJECT</b> - Drops traffic that matches the conditions of the stateful rule, and sends a TCP reset packet back
     * to sender of the packet. A TCP reset packet is a packet with no payload and an RST bit contained in the TCP
     * header flags. REJECT is available only for TCP traffic. This option doesn't support FTP or IMAP protocols.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link StatefulAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #actionAsString}.
     * </p>
     * 
     * @return Defines what Network Firewall should do with the packets in a traffic flow when the flow matches the
     *         stateful rule criteria. For all actions, Network Firewall performs the specified action and discontinues
     *         stateful inspection of the traffic flow. </p>
     *         <p>
     *         The actions for a stateful rule are defined as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>PASS</b> - Permits the packets to go to the intended destination.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DROP</b> - Blocks the packets from going to the intended destination and sends an alert log message,
     *         if alert logging is configured in the <a>Firewall</a> <a>LoggingConfiguration</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ALERT</b> - Sends an alert log message, if alert logging is configured in the <a>Firewall</a>
     *         <a>LoggingConfiguration</a>.
     *         </p>
     *         <p>
     *         You can use this action to test a rule that you intend to use to drop traffic. You can enable the rule
     *         with <code>ALERT</code> action, verify in the logs that the rule is filtering as you want, then change
     *         the action to <code>DROP</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>REJECT</b> - Drops traffic that matches the conditions of the stateful rule, and sends a TCP reset
     *         packet back to sender of the packet. A TCP reset packet is a packet with no payload and an RST bit
     *         contained in the TCP header flags. REJECT is available only for TCP traffic. This option doesn't support
     *         FTP or IMAP protocols.
     *         </p>
     *         </li>
     * @see StatefulAction
     */
    public final String actionAsString() {
        return action;
    }

    /**
     * <p>
     * The stateful inspection criteria for this rule, used to inspect traffic flows.
     * </p>
     * 
     * @return The stateful inspection criteria for this rule, used to inspect traffic flows.
     */
    public final Header header() {
        return header;
    }

    /**
     * For responses, this returns true if the service returned a value for the RuleOptions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRuleOptions() {
        return ruleOptions != null && !(ruleOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Additional options for the rule. These are the Suricata <code>RuleOptions</code> settings.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRuleOptions} method.
     * </p>
     * 
     * @return Additional options for the rule. These are the Suricata <code>RuleOptions</code> settings.
     */
    public final List<RuleOption> ruleOptions() {
        return ruleOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(actionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(header());
        hashCode = 31 * hashCode + Objects.hashCode(hasRuleOptions() ? ruleOptions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StatefulRule)) {
            return false;
        }
        StatefulRule other = (StatefulRule) obj;
        return Objects.equals(actionAsString(), other.actionAsString()) && Objects.equals(header(), other.header())
                && hasRuleOptions() == other.hasRuleOptions() && Objects.equals(ruleOptions(), other.ruleOptions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StatefulRule").add("Action", actionAsString()).add("Header", header())
                .add("RuleOptions", hasRuleOptions() ? ruleOptions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Action":
            return Optional.ofNullable(clazz.cast(actionAsString()));
        case "Header":
            return Optional.ofNullable(clazz.cast(header()));
        case "RuleOptions":
            return Optional.ofNullable(clazz.cast(ruleOptions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Action", ACTION_FIELD);
        map.put("Header", HEADER_FIELD);
        map.put("RuleOptions", RULE_OPTIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<StatefulRule, T> g) {
        return obj -> g.apply((StatefulRule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StatefulRule> {
        /**
         * <p>
         * Defines what Network Firewall should do with the packets in a traffic flow when the flow matches the stateful
         * rule criteria. For all actions, Network Firewall performs the specified action and discontinues stateful
         * inspection of the traffic flow.
         * </p>
         * <p>
         * The actions for a stateful rule are defined as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>PASS</b> - Permits the packets to go to the intended destination.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DROP</b> - Blocks the packets from going to the intended destination and sends an alert log message, if
         * alert logging is configured in the <a>Firewall</a> <a>LoggingConfiguration</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ALERT</b> - Sends an alert log message, if alert logging is configured in the <a>Firewall</a>
         * <a>LoggingConfiguration</a>.
         * </p>
         * <p>
         * You can use this action to test a rule that you intend to use to drop traffic. You can enable the rule with
         * <code>ALERT</code> action, verify in the logs that the rule is filtering as you want, then change the action
         * to <code>DROP</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>REJECT</b> - Drops traffic that matches the conditions of the stateful rule, and sends a TCP reset packet
         * back to sender of the packet. A TCP reset packet is a packet with no payload and an RST bit contained in the
         * TCP header flags. REJECT is available only for TCP traffic. This option doesn't support FTP or IMAP
         * protocols.
         * </p>
         * </li>
         * </ul>
         * 
         * @param action
         *        Defines what Network Firewall should do with the packets in a traffic flow when the flow matches the
         *        stateful rule criteria. For all actions, Network Firewall performs the specified action and
         *        discontinues stateful inspection of the traffic flow. </p>
         *        <p>
         *        The actions for a stateful rule are defined as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>PASS</b> - Permits the packets to go to the intended destination.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DROP</b> - Blocks the packets from going to the intended destination and sends an alert log
         *        message, if alert logging is configured in the <a>Firewall</a> <a>LoggingConfiguration</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ALERT</b> - Sends an alert log message, if alert logging is configured in the <a>Firewall</a>
         *        <a>LoggingConfiguration</a>.
         *        </p>
         *        <p>
         *        You can use this action to test a rule that you intend to use to drop traffic. You can enable the rule
         *        with <code>ALERT</code> action, verify in the logs that the rule is filtering as you want, then change
         *        the action to <code>DROP</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>REJECT</b> - Drops traffic that matches the conditions of the stateful rule, and sends a TCP reset
         *        packet back to sender of the packet. A TCP reset packet is a packet with no payload and an RST bit
         *        contained in the TCP header flags. REJECT is available only for TCP traffic. This option doesn't
         *        support FTP or IMAP protocols.
         *        </p>
         *        </li>
         * @see StatefulAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StatefulAction
         */
        Builder action(String action);

        /**
         * <p>
         * Defines what Network Firewall should do with the packets in a traffic flow when the flow matches the stateful
         * rule criteria. For all actions, Network Firewall performs the specified action and discontinues stateful
         * inspection of the traffic flow.
         * </p>
         * <p>
         * The actions for a stateful rule are defined as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>PASS</b> - Permits the packets to go to the intended destination.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DROP</b> - Blocks the packets from going to the intended destination and sends an alert log message, if
         * alert logging is configured in the <a>Firewall</a> <a>LoggingConfiguration</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ALERT</b> - Sends an alert log message, if alert logging is configured in the <a>Firewall</a>
         * <a>LoggingConfiguration</a>.
         * </p>
         * <p>
         * You can use this action to test a rule that you intend to use to drop traffic. You can enable the rule with
         * <code>ALERT</code> action, verify in the logs that the rule is filtering as you want, then change the action
         * to <code>DROP</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>REJECT</b> - Drops traffic that matches the conditions of the stateful rule, and sends a TCP reset packet
         * back to sender of the packet. A TCP reset packet is a packet with no payload and an RST bit contained in the
         * TCP header flags. REJECT is available only for TCP traffic. This option doesn't support FTP or IMAP
         * protocols.
         * </p>
         * </li>
         * </ul>
         * 
         * @param action
         *        Defines what Network Firewall should do with the packets in a traffic flow when the flow matches the
         *        stateful rule criteria. For all actions, Network Firewall performs the specified action and
         *        discontinues stateful inspection of the traffic flow. </p>
         *        <p>
         *        The actions for a stateful rule are defined as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>PASS</b> - Permits the packets to go to the intended destination.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DROP</b> - Blocks the packets from going to the intended destination and sends an alert log
         *        message, if alert logging is configured in the <a>Firewall</a> <a>LoggingConfiguration</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ALERT</b> - Sends an alert log message, if alert logging is configured in the <a>Firewall</a>
         *        <a>LoggingConfiguration</a>.
         *        </p>
         *        <p>
         *        You can use this action to test a rule that you intend to use to drop traffic. You can enable the rule
         *        with <code>ALERT</code> action, verify in the logs that the rule is filtering as you want, then change
         *        the action to <code>DROP</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>REJECT</b> - Drops traffic that matches the conditions of the stateful rule, and sends a TCP reset
         *        packet back to sender of the packet. A TCP reset packet is a packet with no payload and an RST bit
         *        contained in the TCP header flags. REJECT is available only for TCP traffic. This option doesn't
         *        support FTP or IMAP protocols.
         *        </p>
         *        </li>
         * @see StatefulAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StatefulAction
         */
        Builder action(StatefulAction action);

        /**
         * <p>
         * The stateful inspection criteria for this rule, used to inspect traffic flows.
         * </p>
         * 
         * @param header
         *        The stateful inspection criteria for this rule, used to inspect traffic flows.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder header(Header header);

        /**
         * <p>
         * The stateful inspection criteria for this rule, used to inspect traffic flows.
         * </p>
         * This is a convenience method that creates an instance of the {@link Header.Builder} avoiding the need to
         * create one manually via {@link Header#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Header.Builder#build()} is called immediately and its result is
         * passed to {@link #header(Header)}.
         * 
         * @param header
         *        a consumer that will call methods on {@link Header.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #header(Header)
         */
        default Builder header(Consumer<Header.Builder> header) {
            return header(Header.builder().applyMutation(header).build());
        }

        /**
         * <p>
         * Additional options for the rule. These are the Suricata <code>RuleOptions</code> settings.
         * </p>
         * 
         * @param ruleOptions
         *        Additional options for the rule. These are the Suricata <code>RuleOptions</code> settings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleOptions(Collection<RuleOption> ruleOptions);

        /**
         * <p>
         * Additional options for the rule. These are the Suricata <code>RuleOptions</code> settings.
         * </p>
         * 
         * @param ruleOptions
         *        Additional options for the rule. These are the Suricata <code>RuleOptions</code> settings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleOptions(RuleOption... ruleOptions);

        /**
         * <p>
         * Additional options for the rule. These are the Suricata <code>RuleOptions</code> settings.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.networkfirewall.model.RuleOption.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.networkfirewall.model.RuleOption#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.networkfirewall.model.RuleOption.Builder#build()} is called
         * immediately and its result is passed to {@link #ruleOptions(List<RuleOption>)}.
         * 
         * @param ruleOptions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.networkfirewall.model.RuleOption.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ruleOptions(java.util.Collection<RuleOption>)
         */
        Builder ruleOptions(Consumer<RuleOption.Builder>... ruleOptions);
    }

    static final class BuilderImpl implements Builder {
        private String action;

        private Header header;

        private List<RuleOption> ruleOptions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(StatefulRule model) {
            action(model.action);
            header(model.header);
            ruleOptions(model.ruleOptions);
        }

        public final String getAction() {
            return action;
        }

        public final void setAction(String action) {
            this.action = action;
        }

        @Override
        public final Builder action(String action) {
            this.action = action;
            return this;
        }

        @Override
        public final Builder action(StatefulAction action) {
            this.action(action == null ? null : action.toString());
            return this;
        }

        public final Header.Builder getHeader() {
            return header != null ? header.toBuilder() : null;
        }

        public final void setHeader(Header.BuilderImpl header) {
            this.header = header != null ? header.build() : null;
        }

        @Override
        public final Builder header(Header header) {
            this.header = header;
            return this;
        }

        public final List<RuleOption.Builder> getRuleOptions() {
            List<RuleOption.Builder> result = RuleOptionsCopier.copyToBuilder(this.ruleOptions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRuleOptions(Collection<RuleOption.BuilderImpl> ruleOptions) {
            this.ruleOptions = RuleOptionsCopier.copyFromBuilder(ruleOptions);
        }

        @Override
        public final Builder ruleOptions(Collection<RuleOption> ruleOptions) {
            this.ruleOptions = RuleOptionsCopier.copy(ruleOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ruleOptions(RuleOption... ruleOptions) {
            ruleOptions(Arrays.asList(ruleOptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ruleOptions(Consumer<RuleOption.Builder>... ruleOptions) {
            ruleOptions(Stream.of(ruleOptions).map(c -> RuleOption.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public StatefulRule build() {
            return new StatefulRule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
