/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.networkfirewall.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines where Network Firewall sends logs for the firewall for one log type. This is used in
 * <a>LoggingConfiguration</a>. You can send each type of log to an Amazon S3 bucket, a CloudWatch log group, or a
 * Firehose delivery stream.
 * </p>
 * <p>
 * Network Firewall generates logs for stateful rule groups. You can save alert, flow, and TLS log types.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LogDestinationConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<LogDestinationConfig.Builder, LogDestinationConfig> {
    private static final SdkField<String> LOG_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LogType").getter(getter(LogDestinationConfig::logTypeAsString)).setter(setter(Builder::logType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogType").build()).build();

    private static final SdkField<String> LOG_DESTINATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LogDestinationType").getter(getter(LogDestinationConfig::logDestinationTypeAsString))
            .setter(setter(Builder::logDestinationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogDestinationType").build())
            .build();

    private static final SdkField<Map<String, String>> LOG_DESTINATION_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("LogDestination")
            .getter(getter(LogDestinationConfig::logDestination))
            .setter(setter(Builder::logDestination))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogDestination").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LOG_TYPE_FIELD,
            LOG_DESTINATION_TYPE_FIELD, LOG_DESTINATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String logType;

    private final String logDestinationType;

    private final Map<String, String> logDestination;

    private LogDestinationConfig(BuilderImpl builder) {
        this.logType = builder.logType;
        this.logDestinationType = builder.logDestinationType;
        this.logDestination = builder.logDestination;
    }

    /**
     * <p>
     * The type of log to record. You can record the following types of logs from your Network Firewall stateful engine.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ALERT</code> - Logs for traffic that matches your stateful rules and that have an action that sends an
     * alert. A stateful rule sends alerts for the rule actions DROP, ALERT, and REJECT. For more information, see
     * <a>StatefulRule</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FLOW</code> - Standard network traffic flow logs. The stateful rules engine records flow logs for all
     * network traffic that it receives. Each flow log record captures the network flow for a specific standard
     * stateless rule group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TLS</code> - Logs for events that are related to TLS inspection. For more information, see <a
     * href="https://docs.aws.amazon.com/network-firewall/latest/developerguide/tls-inspection-configurations.html"
     * >Inspecting SSL/TLS traffic with TLS inspection configurations</a> in the <i>Network Firewall Developer
     * Guide</i>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #logType} will
     * return {@link LogType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #logTypeAsString}.
     * </p>
     * 
     * @return The type of log to record. You can record the following types of logs from your Network Firewall stateful
     *         engine.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ALERT</code> - Logs for traffic that matches your stateful rules and that have an action that sends
     *         an alert. A stateful rule sends alerts for the rule actions DROP, ALERT, and REJECT. For more
     *         information, see <a>StatefulRule</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FLOW</code> - Standard network traffic flow logs. The stateful rules engine records flow logs for
     *         all network traffic that it receives. Each flow log record captures the network flow for a specific
     *         standard stateless rule group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TLS</code> - Logs for events that are related to TLS inspection. For more information, see <a
     *         href="https://docs.aws.amazon.com/network-firewall/latest/developerguide/tls-inspection-configurations.html"
     *         >Inspecting SSL/TLS traffic with TLS inspection configurations</a> in the <i>Network Firewall Developer
     *         Guide</i>.
     *         </p>
     *         </li>
     * @see LogType
     */
    public final LogType logType() {
        return LogType.fromValue(logType);
    }

    /**
     * <p>
     * The type of log to record. You can record the following types of logs from your Network Firewall stateful engine.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ALERT</code> - Logs for traffic that matches your stateful rules and that have an action that sends an
     * alert. A stateful rule sends alerts for the rule actions DROP, ALERT, and REJECT. For more information, see
     * <a>StatefulRule</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FLOW</code> - Standard network traffic flow logs. The stateful rules engine records flow logs for all
     * network traffic that it receives. Each flow log record captures the network flow for a specific standard
     * stateless rule group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TLS</code> - Logs for events that are related to TLS inspection. For more information, see <a
     * href="https://docs.aws.amazon.com/network-firewall/latest/developerguide/tls-inspection-configurations.html"
     * >Inspecting SSL/TLS traffic with TLS inspection configurations</a> in the <i>Network Firewall Developer
     * Guide</i>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #logType} will
     * return {@link LogType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #logTypeAsString}.
     * </p>
     * 
     * @return The type of log to record. You can record the following types of logs from your Network Firewall stateful
     *         engine.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ALERT</code> - Logs for traffic that matches your stateful rules and that have an action that sends
     *         an alert. A stateful rule sends alerts for the rule actions DROP, ALERT, and REJECT. For more
     *         information, see <a>StatefulRule</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FLOW</code> - Standard network traffic flow logs. The stateful rules engine records flow logs for
     *         all network traffic that it receives. Each flow log record captures the network flow for a specific
     *         standard stateless rule group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TLS</code> - Logs for events that are related to TLS inspection. For more information, see <a
     *         href="https://docs.aws.amazon.com/network-firewall/latest/developerguide/tls-inspection-configurations.html"
     *         >Inspecting SSL/TLS traffic with TLS inspection configurations</a> in the <i>Network Firewall Developer
     *         Guide</i>.
     *         </p>
     *         </li>
     * @see LogType
     */
    public final String logTypeAsString() {
        return logType;
    }

    /**
     * <p>
     * The type of storage destination to send these logs to. You can send logs to an Amazon S3 bucket, a CloudWatch log
     * group, or a Firehose delivery stream.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #logDestinationType} will return {@link LogDestinationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #logDestinationTypeAsString}.
     * </p>
     * 
     * @return The type of storage destination to send these logs to. You can send logs to an Amazon S3 bucket, a
     *         CloudWatch log group, or a Firehose delivery stream.
     * @see LogDestinationType
     */
    public final LogDestinationType logDestinationType() {
        return LogDestinationType.fromValue(logDestinationType);
    }

    /**
     * <p>
     * The type of storage destination to send these logs to. You can send logs to an Amazon S3 bucket, a CloudWatch log
     * group, or a Firehose delivery stream.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #logDestinationType} will return {@link LogDestinationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #logDestinationTypeAsString}.
     * </p>
     * 
     * @return The type of storage destination to send these logs to. You can send logs to an Amazon S3 bucket, a
     *         CloudWatch log group, or a Firehose delivery stream.
     * @see LogDestinationType
     */
    public final String logDestinationTypeAsString() {
        return logDestinationType;
    }

    /**
     * For responses, this returns true if the service returned a value for the LogDestination property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLogDestination() {
        return logDestination != null && !(logDestination instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The named location for the logs, provided in a key:value mapping that is specific to the chosen destination type.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For an Amazon S3 bucket, provide the name of the bucket, with key <code>bucketName</code>, and optionally provide
     * a prefix, with key <code>prefix</code>.
     * </p>
     * <p>
     * The following example specifies an Amazon S3 bucket named <code>DOC-EXAMPLE-BUCKET</code> and the prefix
     * <code>alerts</code>:
     * </p>
     * <p>
     * <code>"LogDestination": { "bucketName": "DOC-EXAMPLE-BUCKET", "prefix": "alerts" }</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * For a CloudWatch log group, provide the name of the CloudWatch log group, with key <code>logGroup</code>. The
     * following example specifies a log group named <code>alert-log-group</code>:
     * </p>
     * <p>
     * <code>"LogDestination": { "logGroup": "alert-log-group" }</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * For a Firehose delivery stream, provide the name of the delivery stream, with key <code>deliveryStream</code>.
     * The following example specifies a delivery stream named <code>alert-delivery-stream</code>:
     * </p>
     * <p>
     * <code>"LogDestination": { "deliveryStream": "alert-delivery-stream" }</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLogDestination} method.
     * </p>
     * 
     * @return The named location for the logs, provided in a key:value mapping that is specific to the chosen
     *         destination type. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For an Amazon S3 bucket, provide the name of the bucket, with key <code>bucketName</code>, and optionally
     *         provide a prefix, with key <code>prefix</code>.
     *         </p>
     *         <p>
     *         The following example specifies an Amazon S3 bucket named <code>DOC-EXAMPLE-BUCKET</code> and the prefix
     *         <code>alerts</code>:
     *         </p>
     *         <p>
     *         <code>"LogDestination": { "bucketName": "DOC-EXAMPLE-BUCKET", "prefix": "alerts" }</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For a CloudWatch log group, provide the name of the CloudWatch log group, with key <code>logGroup</code>.
     *         The following example specifies a log group named <code>alert-log-group</code>:
     *         </p>
     *         <p>
     *         <code>"LogDestination": { "logGroup": "alert-log-group" }</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For a Firehose delivery stream, provide the name of the delivery stream, with key
     *         <code>deliveryStream</code>. The following example specifies a delivery stream named
     *         <code>alert-delivery-stream</code>:
     *         </p>
     *         <p>
     *         <code>"LogDestination": { "deliveryStream": "alert-delivery-stream" }</code>
     *         </p>
     *         </li>
     */
    public final Map<String, String> logDestination() {
        return logDestination;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(logTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(logDestinationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasLogDestination() ? logDestination() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LogDestinationConfig)) {
            return false;
        }
        LogDestinationConfig other = (LogDestinationConfig) obj;
        return Objects.equals(logTypeAsString(), other.logTypeAsString())
                && Objects.equals(logDestinationTypeAsString(), other.logDestinationTypeAsString())
                && hasLogDestination() == other.hasLogDestination() && Objects.equals(logDestination(), other.logDestination());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LogDestinationConfig").add("LogType", logTypeAsString())
                .add("LogDestinationType", logDestinationTypeAsString())
                .add("LogDestination", hasLogDestination() ? logDestination() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "LogType":
            return Optional.ofNullable(clazz.cast(logTypeAsString()));
        case "LogDestinationType":
            return Optional.ofNullable(clazz.cast(logDestinationTypeAsString()));
        case "LogDestination":
            return Optional.ofNullable(clazz.cast(logDestination()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("LogType", LOG_TYPE_FIELD);
        map.put("LogDestinationType", LOG_DESTINATION_TYPE_FIELD);
        map.put("LogDestination", LOG_DESTINATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<LogDestinationConfig, T> g) {
        return obj -> g.apply((LogDestinationConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LogDestinationConfig> {
        /**
         * <p>
         * The type of log to record. You can record the following types of logs from your Network Firewall stateful
         * engine.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ALERT</code> - Logs for traffic that matches your stateful rules and that have an action that sends an
         * alert. A stateful rule sends alerts for the rule actions DROP, ALERT, and REJECT. For more information, see
         * <a>StatefulRule</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FLOW</code> - Standard network traffic flow logs. The stateful rules engine records flow logs for all
         * network traffic that it receives. Each flow log record captures the network flow for a specific standard
         * stateless rule group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TLS</code> - Logs for events that are related to TLS inspection. For more information, see <a href=
         * "https://docs.aws.amazon.com/network-firewall/latest/developerguide/tls-inspection-configurations.html"
         * >Inspecting SSL/TLS traffic with TLS inspection configurations</a> in the <i>Network Firewall Developer
         * Guide</i>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param logType
         *        The type of log to record. You can record the following types of logs from your Network Firewall
         *        stateful engine.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ALERT</code> - Logs for traffic that matches your stateful rules and that have an action that
         *        sends an alert. A stateful rule sends alerts for the rule actions DROP, ALERT, and REJECT. For more
         *        information, see <a>StatefulRule</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FLOW</code> - Standard network traffic flow logs. The stateful rules engine records flow logs
         *        for all network traffic that it receives. Each flow log record captures the network flow for a
         *        specific standard stateless rule group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TLS</code> - Logs for events that are related to TLS inspection. For more information, see <a
         *        href=
         *        "https://docs.aws.amazon.com/network-firewall/latest/developerguide/tls-inspection-configurations.html"
         *        >Inspecting SSL/TLS traffic with TLS inspection configurations</a> in the <i>Network Firewall
         *        Developer Guide</i>.
         *        </p>
         *        </li>
         * @see LogType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogType
         */
        Builder logType(String logType);

        /**
         * <p>
         * The type of log to record. You can record the following types of logs from your Network Firewall stateful
         * engine.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ALERT</code> - Logs for traffic that matches your stateful rules and that have an action that sends an
         * alert. A stateful rule sends alerts for the rule actions DROP, ALERT, and REJECT. For more information, see
         * <a>StatefulRule</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FLOW</code> - Standard network traffic flow logs. The stateful rules engine records flow logs for all
         * network traffic that it receives. Each flow log record captures the network flow for a specific standard
         * stateless rule group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TLS</code> - Logs for events that are related to TLS inspection. For more information, see <a href=
         * "https://docs.aws.amazon.com/network-firewall/latest/developerguide/tls-inspection-configurations.html"
         * >Inspecting SSL/TLS traffic with TLS inspection configurations</a> in the <i>Network Firewall Developer
         * Guide</i>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param logType
         *        The type of log to record. You can record the following types of logs from your Network Firewall
         *        stateful engine.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ALERT</code> - Logs for traffic that matches your stateful rules and that have an action that
         *        sends an alert. A stateful rule sends alerts for the rule actions DROP, ALERT, and REJECT. For more
         *        information, see <a>StatefulRule</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FLOW</code> - Standard network traffic flow logs. The stateful rules engine records flow logs
         *        for all network traffic that it receives. Each flow log record captures the network flow for a
         *        specific standard stateless rule group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TLS</code> - Logs for events that are related to TLS inspection. For more information, see <a
         *        href=
         *        "https://docs.aws.amazon.com/network-firewall/latest/developerguide/tls-inspection-configurations.html"
         *        >Inspecting SSL/TLS traffic with TLS inspection configurations</a> in the <i>Network Firewall
         *        Developer Guide</i>.
         *        </p>
         *        </li>
         * @see LogType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogType
         */
        Builder logType(LogType logType);

        /**
         * <p>
         * The type of storage destination to send these logs to. You can send logs to an Amazon S3 bucket, a CloudWatch
         * log group, or a Firehose delivery stream.
         * </p>
         * 
         * @param logDestinationType
         *        The type of storage destination to send these logs to. You can send logs to an Amazon S3 bucket, a
         *        CloudWatch log group, or a Firehose delivery stream.
         * @see LogDestinationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogDestinationType
         */
        Builder logDestinationType(String logDestinationType);

        /**
         * <p>
         * The type of storage destination to send these logs to. You can send logs to an Amazon S3 bucket, a CloudWatch
         * log group, or a Firehose delivery stream.
         * </p>
         * 
         * @param logDestinationType
         *        The type of storage destination to send these logs to. You can send logs to an Amazon S3 bucket, a
         *        CloudWatch log group, or a Firehose delivery stream.
         * @see LogDestinationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogDestinationType
         */
        Builder logDestinationType(LogDestinationType logDestinationType);

        /**
         * <p>
         * The named location for the logs, provided in a key:value mapping that is specific to the chosen destination
         * type.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For an Amazon S3 bucket, provide the name of the bucket, with key <code>bucketName</code>, and optionally
         * provide a prefix, with key <code>prefix</code>.
         * </p>
         * <p>
         * The following example specifies an Amazon S3 bucket named <code>DOC-EXAMPLE-BUCKET</code> and the prefix
         * <code>alerts</code>:
         * </p>
         * <p>
         * <code>"LogDestination": { "bucketName": "DOC-EXAMPLE-BUCKET", "prefix": "alerts" }</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * For a CloudWatch log group, provide the name of the CloudWatch log group, with key <code>logGroup</code>. The
         * following example specifies a log group named <code>alert-log-group</code>:
         * </p>
         * <p>
         * <code>"LogDestination": { "logGroup": "alert-log-group" }</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * For a Firehose delivery stream, provide the name of the delivery stream, with key <code>deliveryStream</code>
         * . The following example specifies a delivery stream named <code>alert-delivery-stream</code>:
         * </p>
         * <p>
         * <code>"LogDestination": { "deliveryStream": "alert-delivery-stream" }</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param logDestination
         *        The named location for the logs, provided in a key:value mapping that is specific to the chosen
         *        destination type. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For an Amazon S3 bucket, provide the name of the bucket, with key <code>bucketName</code>, and
         *        optionally provide a prefix, with key <code>prefix</code>.
         *        </p>
         *        <p>
         *        The following example specifies an Amazon S3 bucket named <code>DOC-EXAMPLE-BUCKET</code> and the
         *        prefix <code>alerts</code>:
         *        </p>
         *        <p>
         *        <code>"LogDestination": { "bucketName": "DOC-EXAMPLE-BUCKET", "prefix": "alerts" }</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For a CloudWatch log group, provide the name of the CloudWatch log group, with key
         *        <code>logGroup</code>. The following example specifies a log group named <code>alert-log-group</code>:
         *        </p>
         *        <p>
         *        <code>"LogDestination": { "logGroup": "alert-log-group" }</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For a Firehose delivery stream, provide the name of the delivery stream, with key
         *        <code>deliveryStream</code>. The following example specifies a delivery stream named
         *        <code>alert-delivery-stream</code>:
         *        </p>
         *        <p>
         *        <code>"LogDestination": { "deliveryStream": "alert-delivery-stream" }</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logDestination(Map<String, String> logDestination);
    }

    static final class BuilderImpl implements Builder {
        private String logType;

        private String logDestinationType;

        private Map<String, String> logDestination = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(LogDestinationConfig model) {
            logType(model.logType);
            logDestinationType(model.logDestinationType);
            logDestination(model.logDestination);
        }

        public final String getLogType() {
            return logType;
        }

        public final void setLogType(String logType) {
            this.logType = logType;
        }

        @Override
        public final Builder logType(String logType) {
            this.logType = logType;
            return this;
        }

        @Override
        public final Builder logType(LogType logType) {
            this.logType(logType == null ? null : logType.toString());
            return this;
        }

        public final String getLogDestinationType() {
            return logDestinationType;
        }

        public final void setLogDestinationType(String logDestinationType) {
            this.logDestinationType = logDestinationType;
        }

        @Override
        public final Builder logDestinationType(String logDestinationType) {
            this.logDestinationType = logDestinationType;
            return this;
        }

        @Override
        public final Builder logDestinationType(LogDestinationType logDestinationType) {
            this.logDestinationType(logDestinationType == null ? null : logDestinationType.toString());
            return this;
        }

        public final Map<String, String> getLogDestination() {
            if (logDestination instanceof SdkAutoConstructMap) {
                return null;
            }
            return logDestination;
        }

        public final void setLogDestination(Map<String, String> logDestination) {
            this.logDestination = LogDestinationMapCopier.copy(logDestination);
        }

        @Override
        public final Builder logDestination(Map<String, String> logDestination) {
            this.logDestination = LogDestinationMapCopier.copy(logDestination);
            return this;
        }

        @Override
        public LogDestinationConfig build() {
            return new LogDestinationConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
