/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.networkfirewall.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about a flow operation, such as related statuses, unique identifiers, and all filters defined in
 * the operation.
 * </p>
 * <p>
 * Flow operations let you manage the flows tracked in the flow table, also known as the firewall table.
 * </p>
 * <p>
 * A flow is network traffic that is monitored by a firewall, either by stateful or stateless rules. For traffic to be
 * considered part of a flow, it must share Destination, DestinationPort, Direction, Protocol, Source, and SourcePort.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FlowOperation implements SdkPojo, Serializable, ToCopyableBuilder<FlowOperation.Builder, FlowOperation> {
    private static final SdkField<Integer> MINIMUM_FLOW_AGE_IN_SECONDS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("MinimumFlowAgeInSeconds")
            .getter(getter(FlowOperation::minimumFlowAgeInSeconds)).setter(setter(Builder::minimumFlowAgeInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinimumFlowAgeInSeconds").build())
            .build();

    private static final SdkField<List<FlowFilter>> FLOW_FILTERS_FIELD = SdkField
            .<List<FlowFilter>> builder(MarshallingType.LIST)
            .memberName("FlowFilters")
            .getter(getter(FlowOperation::flowFilters))
            .setter(setter(Builder::flowFilters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FlowFilters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<FlowFilter> builder(MarshallingType.SDK_POJO)
                                            .constructor(FlowFilter::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            MINIMUM_FLOW_AGE_IN_SECONDS_FIELD, FLOW_FILTERS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Integer minimumFlowAgeInSeconds;

    private final List<FlowFilter> flowFilters;

    private FlowOperation(BuilderImpl builder) {
        this.minimumFlowAgeInSeconds = builder.minimumFlowAgeInSeconds;
        this.flowFilters = builder.flowFilters;
    }

    /**
     * <p>
     * The reqested <code>FlowOperation</code> ignores flows with an age (in seconds) lower than
     * <code>MinimumFlowAgeInSeconds</code>. You provide this for start commands.
     * </p>
     * 
     * @return The reqested <code>FlowOperation</code> ignores flows with an age (in seconds) lower than
     *         <code>MinimumFlowAgeInSeconds</code>. You provide this for start commands.
     */
    public final Integer minimumFlowAgeInSeconds() {
        return minimumFlowAgeInSeconds;
    }

    /**
     * For responses, this returns true if the service returned a value for the FlowFilters property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFlowFilters() {
        return flowFilters != null && !(flowFilters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Defines the scope a flow operation. You can use up to 20 filters to configure a single flow operation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFlowFilters} method.
     * </p>
     * 
     * @return Defines the scope a flow operation. You can use up to 20 filters to configure a single flow operation.
     */
    public final List<FlowFilter> flowFilters() {
        return flowFilters;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(minimumFlowAgeInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(hasFlowFilters() ? flowFilters() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FlowOperation)) {
            return false;
        }
        FlowOperation other = (FlowOperation) obj;
        return Objects.equals(minimumFlowAgeInSeconds(), other.minimumFlowAgeInSeconds())
                && hasFlowFilters() == other.hasFlowFilters() && Objects.equals(flowFilters(), other.flowFilters());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FlowOperation").add("MinimumFlowAgeInSeconds", minimumFlowAgeInSeconds())
                .add("FlowFilters", hasFlowFilters() ? flowFilters() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MinimumFlowAgeInSeconds":
            return Optional.ofNullable(clazz.cast(minimumFlowAgeInSeconds()));
        case "FlowFilters":
            return Optional.ofNullable(clazz.cast(flowFilters()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("MinimumFlowAgeInSeconds", MINIMUM_FLOW_AGE_IN_SECONDS_FIELD);
        map.put("FlowFilters", FLOW_FILTERS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<FlowOperation, T> g) {
        return obj -> g.apply((FlowOperation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FlowOperation> {
        /**
         * <p>
         * The reqested <code>FlowOperation</code> ignores flows with an age (in seconds) lower than
         * <code>MinimumFlowAgeInSeconds</code>. You provide this for start commands.
         * </p>
         * 
         * @param minimumFlowAgeInSeconds
         *        The reqested <code>FlowOperation</code> ignores flows with an age (in seconds) lower than
         *        <code>MinimumFlowAgeInSeconds</code>. You provide this for start commands.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minimumFlowAgeInSeconds(Integer minimumFlowAgeInSeconds);

        /**
         * <p>
         * Defines the scope a flow operation. You can use up to 20 filters to configure a single flow operation.
         * </p>
         * 
         * @param flowFilters
         *        Defines the scope a flow operation. You can use up to 20 filters to configure a single flow operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder flowFilters(Collection<FlowFilter> flowFilters);

        /**
         * <p>
         * Defines the scope a flow operation. You can use up to 20 filters to configure a single flow operation.
         * </p>
         * 
         * @param flowFilters
         *        Defines the scope a flow operation. You can use up to 20 filters to configure a single flow operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder flowFilters(FlowFilter... flowFilters);

        /**
         * <p>
         * Defines the scope a flow operation. You can use up to 20 filters to configure a single flow operation.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.networkfirewall.model.FlowFilter.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.networkfirewall.model.FlowFilter#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.networkfirewall.model.FlowFilter.Builder#build()} is called
         * immediately and its result is passed to {@link #flowFilters(List<FlowFilter>)}.
         * 
         * @param flowFilters
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.networkfirewall.model.FlowFilter.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #flowFilters(java.util.Collection<FlowFilter>)
         */
        Builder flowFilters(Consumer<FlowFilter.Builder>... flowFilters);
    }

    static final class BuilderImpl implements Builder {
        private Integer minimumFlowAgeInSeconds;

        private List<FlowFilter> flowFilters = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(FlowOperation model) {
            minimumFlowAgeInSeconds(model.minimumFlowAgeInSeconds);
            flowFilters(model.flowFilters);
        }

        public final Integer getMinimumFlowAgeInSeconds() {
            return minimumFlowAgeInSeconds;
        }

        public final void setMinimumFlowAgeInSeconds(Integer minimumFlowAgeInSeconds) {
            this.minimumFlowAgeInSeconds = minimumFlowAgeInSeconds;
        }

        @Override
        public final Builder minimumFlowAgeInSeconds(Integer minimumFlowAgeInSeconds) {
            this.minimumFlowAgeInSeconds = minimumFlowAgeInSeconds;
            return this;
        }

        public final List<FlowFilter.Builder> getFlowFilters() {
            List<FlowFilter.Builder> result = FlowFiltersCopier.copyToBuilder(this.flowFilters);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setFlowFilters(Collection<FlowFilter.BuilderImpl> flowFilters) {
            this.flowFilters = FlowFiltersCopier.copyFromBuilder(flowFilters);
        }

        @Override
        public final Builder flowFilters(Collection<FlowFilter> flowFilters) {
            this.flowFilters = FlowFiltersCopier.copy(flowFilters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder flowFilters(FlowFilter... flowFilters) {
            flowFilters(Arrays.asList(flowFilters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder flowFilters(Consumer<FlowFilter.Builder>... flowFilters) {
            flowFilters(Stream.of(flowFilters).map(c -> FlowFilter.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public FlowOperation build() {
            return new FlowOperation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
