/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.networkfirewall.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A report that captures key activity from the last 30 days of network traffic monitored by your firewall.
 * </p>
 * <p>
 * You can generate up to one report per traffic type, per 30 day period. For example, when you successfully create an
 * HTTP traffic report, you cannot create another HTTP traffic report until 30 days pass. Alternatively, if you generate
 * a report that combines metrics on both HTTP and HTTPS traffic, you cannot create another report for either traffic
 * type until 30 days pass.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AnalysisReport implements SdkPojo, Serializable, ToCopyableBuilder<AnalysisReport.Builder, AnalysisReport> {
    private static final SdkField<String> ANALYSIS_REPORT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AnalysisReportId").getter(getter(AnalysisReport::analysisReportId))
            .setter(setter(Builder::analysisReportId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AnalysisReportId").build()).build();

    private static final SdkField<String> ANALYSIS_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AnalysisType").getter(getter(AnalysisReport::analysisTypeAsString))
            .setter(setter(Builder::analysisType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AnalysisType").build()).build();

    private static final SdkField<Instant> REPORT_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ReportTime").getter(getter(AnalysisReport::reportTime)).setter(setter(Builder::reportTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReportTime").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(AnalysisReport::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ANALYSIS_REPORT_ID_FIELD,
            ANALYSIS_TYPE_FIELD, REPORT_TIME_FIELD, STATUS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String analysisReportId;

    private final String analysisType;

    private final Instant reportTime;

    private final String status;

    private AnalysisReport(BuilderImpl builder) {
        this.analysisReportId = builder.analysisReportId;
        this.analysisType = builder.analysisType;
        this.reportTime = builder.reportTime;
        this.status = builder.status;
    }

    /**
     * <p>
     * The unique ID of the query that ran when you requested an analysis report.
     * </p>
     * 
     * @return The unique ID of the query that ran when you requested an analysis report.
     */
    public final String analysisReportId() {
        return analysisReportId;
    }

    /**
     * <p>
     * The type of traffic that will be used to generate a report.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #analysisType} will
     * return {@link EnabledAnalysisType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #analysisTypeAsString}.
     * </p>
     * 
     * @return The type of traffic that will be used to generate a report.
     * @see EnabledAnalysisType
     */
    public final EnabledAnalysisType analysisType() {
        return EnabledAnalysisType.fromValue(analysisType);
    }

    /**
     * <p>
     * The type of traffic that will be used to generate a report.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #analysisType} will
     * return {@link EnabledAnalysisType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #analysisTypeAsString}.
     * </p>
     * 
     * @return The type of traffic that will be used to generate a report.
     * @see EnabledAnalysisType
     */
    public final String analysisTypeAsString() {
        return analysisType;
    }

    /**
     * <p>
     * The date and time the analysis report was ran.
     * </p>
     * 
     * @return The date and time the analysis report was ran.
     */
    public final Instant reportTime() {
        return reportTime;
    }

    /**
     * <p>
     * The status of the analysis report you specify. Statuses include <code>RUNNING</code>, <code>COMPLETED</code>, or
     * <code>FAILED</code>.
     * </p>
     * 
     * @return The status of the analysis report you specify. Statuses include <code>RUNNING</code>,
     *         <code>COMPLETED</code>, or <code>FAILED</code>.
     */
    public final String status() {
        return status;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(analysisReportId());
        hashCode = 31 * hashCode + Objects.hashCode(analysisTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(reportTime());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AnalysisReport)) {
            return false;
        }
        AnalysisReport other = (AnalysisReport) obj;
        return Objects.equals(analysisReportId(), other.analysisReportId())
                && Objects.equals(analysisTypeAsString(), other.analysisTypeAsString())
                && Objects.equals(reportTime(), other.reportTime()) && Objects.equals(status(), other.status());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AnalysisReport").add("AnalysisReportId", analysisReportId())
                .add("AnalysisType", analysisTypeAsString()).add("ReportTime", reportTime()).add("Status", status()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AnalysisReportId":
            return Optional.ofNullable(clazz.cast(analysisReportId()));
        case "AnalysisType":
            return Optional.ofNullable(clazz.cast(analysisTypeAsString()));
        case "ReportTime":
            return Optional.ofNullable(clazz.cast(reportTime()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("AnalysisReportId", ANALYSIS_REPORT_ID_FIELD);
        map.put("AnalysisType", ANALYSIS_TYPE_FIELD);
        map.put("ReportTime", REPORT_TIME_FIELD);
        map.put("Status", STATUS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AnalysisReport, T> g) {
        return obj -> g.apply((AnalysisReport) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AnalysisReport> {
        /**
         * <p>
         * The unique ID of the query that ran when you requested an analysis report.
         * </p>
         * 
         * @param analysisReportId
         *        The unique ID of the query that ran when you requested an analysis report.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder analysisReportId(String analysisReportId);

        /**
         * <p>
         * The type of traffic that will be used to generate a report.
         * </p>
         * 
         * @param analysisType
         *        The type of traffic that will be used to generate a report.
         * @see EnabledAnalysisType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EnabledAnalysisType
         */
        Builder analysisType(String analysisType);

        /**
         * <p>
         * The type of traffic that will be used to generate a report.
         * </p>
         * 
         * @param analysisType
         *        The type of traffic that will be used to generate a report.
         * @see EnabledAnalysisType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EnabledAnalysisType
         */
        Builder analysisType(EnabledAnalysisType analysisType);

        /**
         * <p>
         * The date and time the analysis report was ran.
         * </p>
         * 
         * @param reportTime
         *        The date and time the analysis report was ran.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reportTime(Instant reportTime);

        /**
         * <p>
         * The status of the analysis report you specify. Statuses include <code>RUNNING</code>, <code>COMPLETED</code>,
         * or <code>FAILED</code>.
         * </p>
         * 
         * @param status
         *        The status of the analysis report you specify. Statuses include <code>RUNNING</code>,
         *        <code>COMPLETED</code>, or <code>FAILED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);
    }

    static final class BuilderImpl implements Builder {
        private String analysisReportId;

        private String analysisType;

        private Instant reportTime;

        private String status;

        private BuilderImpl() {
        }

        private BuilderImpl(AnalysisReport model) {
            analysisReportId(model.analysisReportId);
            analysisType(model.analysisType);
            reportTime(model.reportTime);
            status(model.status);
        }

        public final String getAnalysisReportId() {
            return analysisReportId;
        }

        public final void setAnalysisReportId(String analysisReportId) {
            this.analysisReportId = analysisReportId;
        }

        @Override
        public final Builder analysisReportId(String analysisReportId) {
            this.analysisReportId = analysisReportId;
            return this;
        }

        public final String getAnalysisType() {
            return analysisType;
        }

        public final void setAnalysisType(String analysisType) {
            this.analysisType = analysisType;
        }

        @Override
        public final Builder analysisType(String analysisType) {
            this.analysisType = analysisType;
            return this;
        }

        @Override
        public final Builder analysisType(EnabledAnalysisType analysisType) {
            this.analysisType(analysisType == null ? null : analysisType.toString());
            return this;
        }

        public final Instant getReportTime() {
            return reportTime;
        }

        public final void setReportTime(Instant reportTime) {
            this.reportTime = reportTime;
        }

        @Override
        public final Builder reportTime(Instant reportTime) {
            this.reportTime = reportTime;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public AnalysisReport build() {
            return new AnalysisReport(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
