/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mwaaserverless.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Detailed information about a workflow run execution, including timing, status, error information, and associated task
 * instances. This structure provides comprehensive visibility into the workflow execution lifecycle within the Amazon
 * Managed Workflows for Apache Airflow Serverless serverless environment. The service tracks execution across
 * distributed ECS worker tasks and provides detailed timing information, error diagnostics, and task instance
 * relationships to support effective monitoring and troubleshooting of complex workflow executions.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class WorkflowRunDetail implements SdkPojo, Serializable,
        ToCopyableBuilder<WorkflowRunDetail.Builder, WorkflowRunDetail> {
    private static final SdkField<String> WORKFLOW_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WorkflowArn").getter(getter(WorkflowRunDetail::workflowArn)).setter(setter(Builder::workflowArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WorkflowArn").build()).build();

    private static final SdkField<String> WORKFLOW_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WorkflowVersion").getter(getter(WorkflowRunDetail::workflowVersion))
            .setter(setter(Builder::workflowVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WorkflowVersion").build()).build();

    private static final SdkField<String> RUN_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("RunId")
            .getter(getter(WorkflowRunDetail::runId)).setter(setter(Builder::runId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RunId").build()).build();

    private static final SdkField<String> RUN_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RunType").getter(getter(WorkflowRunDetail::runTypeAsString)).setter(setter(Builder::runType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RunType").build()).build();

    private static final SdkField<Instant> STARTED_ON_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartedOn")
            .getter(getter(WorkflowRunDetail::startedOn))
            .setter(setter(Builder::startedOn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartedOn").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedAt")
            .getter(getter(WorkflowRunDetail::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> COMPLETED_ON_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CompletedOn")
            .getter(getter(WorkflowRunDetail::completedOn))
            .setter(setter(Builder::completedOn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompletedOn").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> MODIFIED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("ModifiedAt")
            .getter(getter(WorkflowRunDetail::modifiedAt))
            .setter(setter(Builder::modifiedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModifiedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Integer> DURATION_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Duration").getter(getter(WorkflowRunDetail::duration)).setter(setter(Builder::duration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Duration").build()).build();

    private static final SdkField<String> ERROR_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ErrorMessage").getter(getter(WorkflowRunDetail::errorMessage)).setter(setter(Builder::errorMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ErrorMessage").build()).build();

    private static final SdkField<List<String>> TASK_INSTANCES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("TaskInstances")
            .getter(getter(WorkflowRunDetail::taskInstances))
            .setter(setter(Builder::taskInstances))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TaskInstances").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RUN_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RunState").getter(getter(WorkflowRunDetail::runStateAsString)).setter(setter(Builder::runState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RunState").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(WORKFLOW_ARN_FIELD,
            WORKFLOW_VERSION_FIELD, RUN_ID_FIELD, RUN_TYPE_FIELD, STARTED_ON_FIELD, CREATED_AT_FIELD, COMPLETED_ON_FIELD,
            MODIFIED_AT_FIELD, DURATION_FIELD, ERROR_MESSAGE_FIELD, TASK_INSTANCES_FIELD, RUN_STATE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String workflowArn;

    private final String workflowVersion;

    private final String runId;

    private final String runType;

    private final Instant startedOn;

    private final Instant createdAt;

    private final Instant completedOn;

    private final Instant modifiedAt;

    private final Integer duration;

    private final String errorMessage;

    private final List<String> taskInstances;

    private final String runState;

    private WorkflowRunDetail(BuilderImpl builder) {
        this.workflowArn = builder.workflowArn;
        this.workflowVersion = builder.workflowVersion;
        this.runId = builder.runId;
        this.runType = builder.runType;
        this.startedOn = builder.startedOn;
        this.createdAt = builder.createdAt;
        this.completedOn = builder.completedOn;
        this.modifiedAt = builder.modifiedAt;
        this.duration = builder.duration;
        this.errorMessage = builder.errorMessage;
        this.taskInstances = builder.taskInstances;
        this.runState = builder.runState;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the workflow that contains this run.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the workflow that contains this run.
     */
    public final String workflowArn() {
        return workflowArn;
    }

    /**
     * <p>
     * The version of the workflow used for this run.
     * </p>
     * 
     * @return The version of the workflow used for this run.
     */
    public final String workflowVersion() {
        return workflowVersion;
    }

    /**
     * <p>
     * The unique identifier of this workflow run.
     * </p>
     * 
     * @return The unique identifier of this workflow run.
     */
    public final String runId() {
        return runId;
    }

    /**
     * <p>
     * The type of workflow run.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #runType} will
     * return {@link RunType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #runTypeAsString}.
     * </p>
     * 
     * @return The type of workflow run.
     * @see RunType
     */
    public final RunType runType() {
        return RunType.fromValue(runType);
    }

    /**
     * <p>
     * The type of workflow run.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #runType} will
     * return {@link RunType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #runTypeAsString}.
     * </p>
     * 
     * @return The type of workflow run.
     * @see RunType
     */
    public final String runTypeAsString() {
        return runType;
    }

    /**
     * <p>
     * The timestamp when the workflow run started execution, in ISO 8601 date-time format.
     * </p>
     * 
     * @return The timestamp when the workflow run started execution, in ISO 8601 date-time format.
     */
    public final Instant startedOn() {
        return startedOn;
    }

    /**
     * <p>
     * The timestamp when the workflow run was created, in ISO 8601 date-time format.
     * </p>
     * 
     * @return The timestamp when the workflow run was created, in ISO 8601 date-time format.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The timestamp when the workflow run completed execution, in ISO 8601 date-time format. This value is null if the
     * run is not complete.
     * </p>
     * 
     * @return The timestamp when the workflow run completed execution, in ISO 8601 date-time format. This value is null
     *         if the run is not complete.
     */
    public final Instant completedOn() {
        return completedOn;
    }

    /**
     * <p>
     * The timestamp when the workflow run was last modified, in ISO 8601 date-time format.
     * </p>
     * 
     * @return The timestamp when the workflow run was last modified, in ISO 8601 date-time format.
     */
    public final Instant modifiedAt() {
        return modifiedAt;
    }

    /**
     * <p>
     * The total duration of the workflow run execution in seconds. This value is null if the run is not complete.
     * </p>
     * 
     * @return The total duration of the workflow run execution in seconds. This value is null if the run is not
     *         complete.
     */
    public final Integer duration() {
        return duration;
    }

    /**
     * <p>
     * The error message if the workflow run failed. This value is null if the run completed successfully.
     * </p>
     * 
     * @return The error message if the workflow run failed. This value is null if the run completed successfully.
     */
    public final String errorMessage() {
        return errorMessage;
    }

    /**
     * For responses, this returns true if the service returned a value for the TaskInstances property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTaskInstances() {
        return taskInstances != null && !(taskInstances instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of task instance IDs that are part of this workflow run.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTaskInstances} method.
     * </p>
     * 
     * @return A list of task instance IDs that are part of this workflow run.
     */
    public final List<String> taskInstances() {
        return taskInstances;
    }

    /**
     * <p>
     * The current execution state of the workflow run.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #runState} will
     * return {@link WorkflowRunStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #runStateAsString}.
     * </p>
     * 
     * @return The current execution state of the workflow run.
     * @see WorkflowRunStatus
     */
    public final WorkflowRunStatus runState() {
        return WorkflowRunStatus.fromValue(runState);
    }

    /**
     * <p>
     * The current execution state of the workflow run.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #runState} will
     * return {@link WorkflowRunStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #runStateAsString}.
     * </p>
     * 
     * @return The current execution state of the workflow run.
     * @see WorkflowRunStatus
     */
    public final String runStateAsString() {
        return runState;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(workflowArn());
        hashCode = 31 * hashCode + Objects.hashCode(workflowVersion());
        hashCode = 31 * hashCode + Objects.hashCode(runId());
        hashCode = 31 * hashCode + Objects.hashCode(runTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(startedOn());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(completedOn());
        hashCode = 31 * hashCode + Objects.hashCode(modifiedAt());
        hashCode = 31 * hashCode + Objects.hashCode(duration());
        hashCode = 31 * hashCode + Objects.hashCode(errorMessage());
        hashCode = 31 * hashCode + Objects.hashCode(hasTaskInstances() ? taskInstances() : null);
        hashCode = 31 * hashCode + Objects.hashCode(runStateAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof WorkflowRunDetail)) {
            return false;
        }
        WorkflowRunDetail other = (WorkflowRunDetail) obj;
        return Objects.equals(workflowArn(), other.workflowArn()) && Objects.equals(workflowVersion(), other.workflowVersion())
                && Objects.equals(runId(), other.runId()) && Objects.equals(runTypeAsString(), other.runTypeAsString())
                && Objects.equals(startedOn(), other.startedOn()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(completedOn(), other.completedOn()) && Objects.equals(modifiedAt(), other.modifiedAt())
                && Objects.equals(duration(), other.duration()) && Objects.equals(errorMessage(), other.errorMessage())
                && hasTaskInstances() == other.hasTaskInstances() && Objects.equals(taskInstances(), other.taskInstances())
                && Objects.equals(runStateAsString(), other.runStateAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("WorkflowRunDetail").add("WorkflowArn", workflowArn()).add("WorkflowVersion", workflowVersion())
                .add("RunId", runId()).add("RunType", runTypeAsString()).add("StartedOn", startedOn())
                .add("CreatedAt", createdAt()).add("CompletedOn", completedOn()).add("ModifiedAt", modifiedAt())
                .add("Duration", duration()).add("ErrorMessage", errorMessage())
                .add("TaskInstances", hasTaskInstances() ? taskInstances() : null).add("RunState", runStateAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "WorkflowArn":
            return Optional.ofNullable(clazz.cast(workflowArn()));
        case "WorkflowVersion":
            return Optional.ofNullable(clazz.cast(workflowVersion()));
        case "RunId":
            return Optional.ofNullable(clazz.cast(runId()));
        case "RunType":
            return Optional.ofNullable(clazz.cast(runTypeAsString()));
        case "StartedOn":
            return Optional.ofNullable(clazz.cast(startedOn()));
        case "CreatedAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "CompletedOn":
            return Optional.ofNullable(clazz.cast(completedOn()));
        case "ModifiedAt":
            return Optional.ofNullable(clazz.cast(modifiedAt()));
        case "Duration":
            return Optional.ofNullable(clazz.cast(duration()));
        case "ErrorMessage":
            return Optional.ofNullable(clazz.cast(errorMessage()));
        case "TaskInstances":
            return Optional.ofNullable(clazz.cast(taskInstances()));
        case "RunState":
            return Optional.ofNullable(clazz.cast(runStateAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("WorkflowArn", WORKFLOW_ARN_FIELD);
        map.put("WorkflowVersion", WORKFLOW_VERSION_FIELD);
        map.put("RunId", RUN_ID_FIELD);
        map.put("RunType", RUN_TYPE_FIELD);
        map.put("StartedOn", STARTED_ON_FIELD);
        map.put("CreatedAt", CREATED_AT_FIELD);
        map.put("CompletedOn", COMPLETED_ON_FIELD);
        map.put("ModifiedAt", MODIFIED_AT_FIELD);
        map.put("Duration", DURATION_FIELD);
        map.put("ErrorMessage", ERROR_MESSAGE_FIELD);
        map.put("TaskInstances", TASK_INSTANCES_FIELD);
        map.put("RunState", RUN_STATE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<WorkflowRunDetail, T> g) {
        return obj -> g.apply((WorkflowRunDetail) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, WorkflowRunDetail> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the workflow that contains this run.
         * </p>
         * 
         * @param workflowArn
         *        The Amazon Resource Name (ARN) of the workflow that contains this run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workflowArn(String workflowArn);

        /**
         * <p>
         * The version of the workflow used for this run.
         * </p>
         * 
         * @param workflowVersion
         *        The version of the workflow used for this run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workflowVersion(String workflowVersion);

        /**
         * <p>
         * The unique identifier of this workflow run.
         * </p>
         * 
         * @param runId
         *        The unique identifier of this workflow run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder runId(String runId);

        /**
         * <p>
         * The type of workflow run.
         * </p>
         * 
         * @param runType
         *        The type of workflow run.
         * @see RunType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RunType
         */
        Builder runType(String runType);

        /**
         * <p>
         * The type of workflow run.
         * </p>
         * 
         * @param runType
         *        The type of workflow run.
         * @see RunType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RunType
         */
        Builder runType(RunType runType);

        /**
         * <p>
         * The timestamp when the workflow run started execution, in ISO 8601 date-time format.
         * </p>
         * 
         * @param startedOn
         *        The timestamp when the workflow run started execution, in ISO 8601 date-time format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startedOn(Instant startedOn);

        /**
         * <p>
         * The timestamp when the workflow run was created, in ISO 8601 date-time format.
         * </p>
         * 
         * @param createdAt
         *        The timestamp when the workflow run was created, in ISO 8601 date-time format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The timestamp when the workflow run completed execution, in ISO 8601 date-time format. This value is null if
         * the run is not complete.
         * </p>
         * 
         * @param completedOn
         *        The timestamp when the workflow run completed execution, in ISO 8601 date-time format. This value is
         *        null if the run is not complete.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completedOn(Instant completedOn);

        /**
         * <p>
         * The timestamp when the workflow run was last modified, in ISO 8601 date-time format.
         * </p>
         * 
         * @param modifiedAt
         *        The timestamp when the workflow run was last modified, in ISO 8601 date-time format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modifiedAt(Instant modifiedAt);

        /**
         * <p>
         * The total duration of the workflow run execution in seconds. This value is null if the run is not complete.
         * </p>
         * 
         * @param duration
         *        The total duration of the workflow run execution in seconds. This value is null if the run is not
         *        complete.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder duration(Integer duration);

        /**
         * <p>
         * The error message if the workflow run failed. This value is null if the run completed successfully.
         * </p>
         * 
         * @param errorMessage
         *        The error message if the workflow run failed. This value is null if the run completed successfully.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorMessage(String errorMessage);

        /**
         * <p>
         * A list of task instance IDs that are part of this workflow run.
         * </p>
         * 
         * @param taskInstances
         *        A list of task instance IDs that are part of this workflow run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskInstances(Collection<String> taskInstances);

        /**
         * <p>
         * A list of task instance IDs that are part of this workflow run.
         * </p>
         * 
         * @param taskInstances
         *        A list of task instance IDs that are part of this workflow run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskInstances(String... taskInstances);

        /**
         * <p>
         * The current execution state of the workflow run.
         * </p>
         * 
         * @param runState
         *        The current execution state of the workflow run.
         * @see WorkflowRunStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkflowRunStatus
         */
        Builder runState(String runState);

        /**
         * <p>
         * The current execution state of the workflow run.
         * </p>
         * 
         * @param runState
         *        The current execution state of the workflow run.
         * @see WorkflowRunStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkflowRunStatus
         */
        Builder runState(WorkflowRunStatus runState);
    }

    static final class BuilderImpl implements Builder {
        private String workflowArn;

        private String workflowVersion;

        private String runId;

        private String runType;

        private Instant startedOn;

        private Instant createdAt;

        private Instant completedOn;

        private Instant modifiedAt;

        private Integer duration;

        private String errorMessage;

        private List<String> taskInstances = DefaultSdkAutoConstructList.getInstance();

        private String runState;

        private BuilderImpl() {
        }

        private BuilderImpl(WorkflowRunDetail model) {
            workflowArn(model.workflowArn);
            workflowVersion(model.workflowVersion);
            runId(model.runId);
            runType(model.runType);
            startedOn(model.startedOn);
            createdAt(model.createdAt);
            completedOn(model.completedOn);
            modifiedAt(model.modifiedAt);
            duration(model.duration);
            errorMessage(model.errorMessage);
            taskInstances(model.taskInstances);
            runState(model.runState);
        }

        public final String getWorkflowArn() {
            return workflowArn;
        }

        public final void setWorkflowArn(String workflowArn) {
            this.workflowArn = workflowArn;
        }

        @Override
        public final Builder workflowArn(String workflowArn) {
            this.workflowArn = workflowArn;
            return this;
        }

        public final String getWorkflowVersion() {
            return workflowVersion;
        }

        public final void setWorkflowVersion(String workflowVersion) {
            this.workflowVersion = workflowVersion;
        }

        @Override
        public final Builder workflowVersion(String workflowVersion) {
            this.workflowVersion = workflowVersion;
            return this;
        }

        public final String getRunId() {
            return runId;
        }

        public final void setRunId(String runId) {
            this.runId = runId;
        }

        @Override
        public final Builder runId(String runId) {
            this.runId = runId;
            return this;
        }

        public final String getRunType() {
            return runType;
        }

        public final void setRunType(String runType) {
            this.runType = runType;
        }

        @Override
        public final Builder runType(String runType) {
            this.runType = runType;
            return this;
        }

        @Override
        public final Builder runType(RunType runType) {
            this.runType(runType == null ? null : runType.toString());
            return this;
        }

        public final Instant getStartedOn() {
            return startedOn;
        }

        public final void setStartedOn(Instant startedOn) {
            this.startedOn = startedOn;
        }

        @Override
        public final Builder startedOn(Instant startedOn) {
            this.startedOn = startedOn;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getCompletedOn() {
            return completedOn;
        }

        public final void setCompletedOn(Instant completedOn) {
            this.completedOn = completedOn;
        }

        @Override
        public final Builder completedOn(Instant completedOn) {
            this.completedOn = completedOn;
            return this;
        }

        public final Instant getModifiedAt() {
            return modifiedAt;
        }

        public final void setModifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
        }

        @Override
        public final Builder modifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
            return this;
        }

        public final Integer getDuration() {
            return duration;
        }

        public final void setDuration(Integer duration) {
            this.duration = duration;
        }

        @Override
        public final Builder duration(Integer duration) {
            this.duration = duration;
            return this;
        }

        public final String getErrorMessage() {
            return errorMessage;
        }

        public final void setErrorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
        }

        @Override
        public final Builder errorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
            return this;
        }

        public final Collection<String> getTaskInstances() {
            if (taskInstances instanceof SdkAutoConstructList) {
                return null;
            }
            return taskInstances;
        }

        public final void setTaskInstances(Collection<String> taskInstances) {
            this.taskInstances = TaskInstanceIdsCopier.copy(taskInstances);
        }

        @Override
        public final Builder taskInstances(Collection<String> taskInstances) {
            this.taskInstances = TaskInstanceIdsCopier.copy(taskInstances);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder taskInstances(String... taskInstances) {
            taskInstances(Arrays.asList(taskInstances));
            return this;
        }

        public final String getRunState() {
            return runState;
        }

        public final void setRunState(String runState) {
            this.runState = runState;
        }

        @Override
        public final Builder runState(String runState) {
            this.runState = runState;
            return this;
        }

        @Override
        public final Builder runState(WorkflowRunStatus runState) {
            this.runState(runState == null ? null : runState.toString());
            return this;
        }

        @Override
        public WorkflowRunDetail build() {
            return new WorkflowRunDetail(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
