/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mpa.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains details for the pending updates for an approval team, if applicable.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PendingUpdate implements SdkPojo, Serializable, ToCopyableBuilder<PendingUpdate.Builder, PendingUpdate> {
    private static final SdkField<String> VERSION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VersionId").getter(getter(PendingUpdate::versionId)).setter(setter(Builder::versionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VersionId").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(PendingUpdate::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<ApprovalStrategyResponse> APPROVAL_STRATEGY_FIELD = SdkField
            .<ApprovalStrategyResponse> builder(MarshallingType.SDK_POJO).memberName("ApprovalStrategy")
            .getter(getter(PendingUpdate::approvalStrategy)).setter(setter(Builder::approvalStrategy))
            .constructor(ApprovalStrategyResponse::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApprovalStrategy").build()).build();

    private static final SdkField<Integer> NUMBER_OF_APPROVERS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("NumberOfApprovers").getter(getter(PendingUpdate::numberOfApprovers))
            .setter(setter(Builder::numberOfApprovers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NumberOfApprovers").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(PendingUpdate::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STATUS_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusCode").getter(getter(PendingUpdate::statusCodeAsString)).setter(setter(Builder::statusCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusCode").build()).build();

    private static final SdkField<String> STATUS_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusMessage").getter(getter(PendingUpdate::statusMessage)).setter(setter(Builder::statusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusMessage").build()).build();

    private static final SdkField<List<GetApprovalTeamResponseApprover>> APPROVERS_FIELD = SdkField
            .<List<GetApprovalTeamResponseApprover>> builder(MarshallingType.LIST)
            .memberName("Approvers")
            .getter(getter(PendingUpdate::approvers))
            .setter(setter(Builder::approvers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Approvers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<GetApprovalTeamResponseApprover> builder(MarshallingType.SDK_POJO)
                                            .constructor(GetApprovalTeamResponseApprover::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> UPDATE_INITIATION_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("UpdateInitiationTime")
            .getter(getter(PendingUpdate::updateInitiationTime))
            .setter(setter(Builder::updateInitiationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdateInitiationTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VERSION_ID_FIELD,
            DESCRIPTION_FIELD, APPROVAL_STRATEGY_FIELD, NUMBER_OF_APPROVERS_FIELD, STATUS_FIELD, STATUS_CODE_FIELD,
            STATUS_MESSAGE_FIELD, APPROVERS_FIELD, UPDATE_INITIATION_TIME_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String versionId;

    private final String description;

    private final ApprovalStrategyResponse approvalStrategy;

    private final Integer numberOfApprovers;

    private final String status;

    private final String statusCode;

    private final String statusMessage;

    private final List<GetApprovalTeamResponseApprover> approvers;

    private final Instant updateInitiationTime;

    private PendingUpdate(BuilderImpl builder) {
        this.versionId = builder.versionId;
        this.description = builder.description;
        this.approvalStrategy = builder.approvalStrategy;
        this.numberOfApprovers = builder.numberOfApprovers;
        this.status = builder.status;
        this.statusCode = builder.statusCode;
        this.statusMessage = builder.statusMessage;
        this.approvers = builder.approvers;
        this.updateInitiationTime = builder.updateInitiationTime;
    }

    /**
     * <p>
     * Version ID for the team.
     * </p>
     * 
     * @return Version ID for the team.
     */
    public final String versionId() {
        return versionId;
    }

    /**
     * <p>
     * Description for the team.
     * </p>
     * 
     * @return Description for the team.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * An <code>ApprovalStrategyResponse</code> object. Contains details for how the team grants approval.
     * </p>
     * 
     * @return An <code>ApprovalStrategyResponse</code> object. Contains details for how the team grants approval.
     */
    public final ApprovalStrategyResponse approvalStrategy() {
        return approvalStrategy;
    }

    /**
     * <p>
     * Total number of approvers in the team.
     * </p>
     * 
     * @return Total number of approvers in the team.
     */
    public final Integer numberOfApprovers() {
        return numberOfApprovers;
    }

    /**
     * <p>
     * Status for the team. For more information, see <a
     * href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-health.html">Team health</a> in the <i>Multi-party
     * approval User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ApprovalTeamStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Status for the team. For more information, see <a
     *         href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-health.html">Team health</a> in the
     *         <i>Multi-party approval User Guide</i>.
     * @see ApprovalTeamStatus
     */
    public final ApprovalTeamStatus status() {
        return ApprovalTeamStatus.fromValue(status);
    }

    /**
     * <p>
     * Status for the team. For more information, see <a
     * href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-health.html">Team health</a> in the <i>Multi-party
     * approval User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ApprovalTeamStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Status for the team. For more information, see <a
     *         href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-health.html">Team health</a> in the
     *         <i>Multi-party approval User Guide</i>.
     * @see ApprovalTeamStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Status code for the update. For more information, see <a
     * href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-health.html">Team health</a> in the <i>Multi-party
     * approval User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statusCode} will
     * return {@link ApprovalTeamStatusCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusCodeAsString}.
     * </p>
     * 
     * @return Status code for the update. For more information, see <a
     *         href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-health.html">Team health</a> in the
     *         <i>Multi-party approval User Guide</i>.
     * @see ApprovalTeamStatusCode
     */
    public final ApprovalTeamStatusCode statusCode() {
        return ApprovalTeamStatusCode.fromValue(statusCode);
    }

    /**
     * <p>
     * Status code for the update. For more information, see <a
     * href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-health.html">Team health</a> in the <i>Multi-party
     * approval User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statusCode} will
     * return {@link ApprovalTeamStatusCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusCodeAsString}.
     * </p>
     * 
     * @return Status code for the update. For more information, see <a
     *         href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-health.html">Team health</a> in the
     *         <i>Multi-party approval User Guide</i>.
     * @see ApprovalTeamStatusCode
     */
    public final String statusCodeAsString() {
        return statusCode;
    }

    /**
     * <p>
     * Message describing the status for the team.
     * </p>
     * 
     * @return Message describing the status for the team.
     */
    public final String statusMessage() {
        return statusMessage;
    }

    /**
     * For responses, this returns true if the service returned a value for the Approvers property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasApprovers() {
        return approvers != null && !(approvers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of <code>GetApprovalTeamResponseApprover </code> objects. Contains details for the approvers in the
     * team.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasApprovers} method.
     * </p>
     * 
     * @return An array of <code>GetApprovalTeamResponseApprover </code> objects. Contains details for the approvers in
     *         the team.
     */
    public final List<GetApprovalTeamResponseApprover> approvers() {
        return approvers;
    }

    /**
     * <p>
     * Timestamp when the update request was initiated.
     * </p>
     * 
     * @return Timestamp when the update request was initiated.
     */
    public final Instant updateInitiationTime() {
        return updateInitiationTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(versionId());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(approvalStrategy());
        hashCode = 31 * hashCode + Objects.hashCode(numberOfApprovers());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusMessage());
        hashCode = 31 * hashCode + Objects.hashCode(hasApprovers() ? approvers() : null);
        hashCode = 31 * hashCode + Objects.hashCode(updateInitiationTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PendingUpdate)) {
            return false;
        }
        PendingUpdate other = (PendingUpdate) obj;
        return Objects.equals(versionId(), other.versionId()) && Objects.equals(description(), other.description())
                && Objects.equals(approvalStrategy(), other.approvalStrategy())
                && Objects.equals(numberOfApprovers(), other.numberOfApprovers())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusCodeAsString(), other.statusCodeAsString())
                && Objects.equals(statusMessage(), other.statusMessage()) && hasApprovers() == other.hasApprovers()
                && Objects.equals(approvers(), other.approvers())
                && Objects.equals(updateInitiationTime(), other.updateInitiationTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PendingUpdate").add("VersionId", versionId()).add("Description", description())
                .add("ApprovalStrategy", approvalStrategy()).add("NumberOfApprovers", numberOfApprovers())
                .add("Status", statusAsString()).add("StatusCode", statusCodeAsString()).add("StatusMessage", statusMessage())
                .add("Approvers", hasApprovers() ? approvers() : null).add("UpdateInitiationTime", updateInitiationTime())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VersionId":
            return Optional.ofNullable(clazz.cast(versionId()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "ApprovalStrategy":
            return Optional.ofNullable(clazz.cast(approvalStrategy()));
        case "NumberOfApprovers":
            return Optional.ofNullable(clazz.cast(numberOfApprovers()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StatusCode":
            return Optional.ofNullable(clazz.cast(statusCodeAsString()));
        case "StatusMessage":
            return Optional.ofNullable(clazz.cast(statusMessage()));
        case "Approvers":
            return Optional.ofNullable(clazz.cast(approvers()));
        case "UpdateInitiationTime":
            return Optional.ofNullable(clazz.cast(updateInitiationTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("VersionId", VERSION_ID_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("ApprovalStrategy", APPROVAL_STRATEGY_FIELD);
        map.put("NumberOfApprovers", NUMBER_OF_APPROVERS_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("StatusCode", STATUS_CODE_FIELD);
        map.put("StatusMessage", STATUS_MESSAGE_FIELD);
        map.put("Approvers", APPROVERS_FIELD);
        map.put("UpdateInitiationTime", UPDATE_INITIATION_TIME_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<PendingUpdate, T> g) {
        return obj -> g.apply((PendingUpdate) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PendingUpdate> {
        /**
         * <p>
         * Version ID for the team.
         * </p>
         * 
         * @param versionId
         *        Version ID for the team.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionId(String versionId);

        /**
         * <p>
         * Description for the team.
         * </p>
         * 
         * @param description
         *        Description for the team.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * An <code>ApprovalStrategyResponse</code> object. Contains details for how the team grants approval.
         * </p>
         * 
         * @param approvalStrategy
         *        An <code>ApprovalStrategyResponse</code> object. Contains details for how the team grants approval.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder approvalStrategy(ApprovalStrategyResponse approvalStrategy);

        /**
         * <p>
         * An <code>ApprovalStrategyResponse</code> object. Contains details for how the team grants approval.
         * </p>
         * This is a convenience method that creates an instance of the {@link ApprovalStrategyResponse.Builder}
         * avoiding the need to create one manually via {@link ApprovalStrategyResponse#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ApprovalStrategyResponse.Builder#build()} is called immediately
         * and its result is passed to {@link #approvalStrategy(ApprovalStrategyResponse)}.
         * 
         * @param approvalStrategy
         *        a consumer that will call methods on {@link ApprovalStrategyResponse.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #approvalStrategy(ApprovalStrategyResponse)
         */
        default Builder approvalStrategy(Consumer<ApprovalStrategyResponse.Builder> approvalStrategy) {
            return approvalStrategy(ApprovalStrategyResponse.builder().applyMutation(approvalStrategy).build());
        }

        /**
         * <p>
         * Total number of approvers in the team.
         * </p>
         * 
         * @param numberOfApprovers
         *        Total number of approvers in the team.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numberOfApprovers(Integer numberOfApprovers);

        /**
         * <p>
         * Status for the team. For more information, see <a
         * href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-health.html">Team health</a> in the <i>Multi-party
         * approval User Guide</i>.
         * </p>
         * 
         * @param status
         *        Status for the team. For more information, see <a
         *        href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-health.html">Team health</a> in the
         *        <i>Multi-party approval User Guide</i>.
         * @see ApprovalTeamStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ApprovalTeamStatus
         */
        Builder status(String status);

        /**
         * <p>
         * Status for the team. For more information, see <a
         * href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-health.html">Team health</a> in the <i>Multi-party
         * approval User Guide</i>.
         * </p>
         * 
         * @param status
         *        Status for the team. For more information, see <a
         *        href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-health.html">Team health</a> in the
         *        <i>Multi-party approval User Guide</i>.
         * @see ApprovalTeamStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ApprovalTeamStatus
         */
        Builder status(ApprovalTeamStatus status);

        /**
         * <p>
         * Status code for the update. For more information, see <a
         * href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-health.html">Team health</a> in the <i>Multi-party
         * approval User Guide</i>.
         * </p>
         * 
         * @param statusCode
         *        Status code for the update. For more information, see <a
         *        href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-health.html">Team health</a> in the
         *        <i>Multi-party approval User Guide</i>.
         * @see ApprovalTeamStatusCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ApprovalTeamStatusCode
         */
        Builder statusCode(String statusCode);

        /**
         * <p>
         * Status code for the update. For more information, see <a
         * href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-health.html">Team health</a> in the <i>Multi-party
         * approval User Guide</i>.
         * </p>
         * 
         * @param statusCode
         *        Status code for the update. For more information, see <a
         *        href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-health.html">Team health</a> in the
         *        <i>Multi-party approval User Guide</i>.
         * @see ApprovalTeamStatusCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ApprovalTeamStatusCode
         */
        Builder statusCode(ApprovalTeamStatusCode statusCode);

        /**
         * <p>
         * Message describing the status for the team.
         * </p>
         * 
         * @param statusMessage
         *        Message describing the status for the team.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusMessage(String statusMessage);

        /**
         * <p>
         * An array of <code>GetApprovalTeamResponseApprover </code> objects. Contains details for the approvers in the
         * team.
         * </p>
         * 
         * @param approvers
         *        An array of <code>GetApprovalTeamResponseApprover </code> objects. Contains details for the approvers
         *        in the team.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder approvers(Collection<GetApprovalTeamResponseApprover> approvers);

        /**
         * <p>
         * An array of <code>GetApprovalTeamResponseApprover </code> objects. Contains details for the approvers in the
         * team.
         * </p>
         * 
         * @param approvers
         *        An array of <code>GetApprovalTeamResponseApprover </code> objects. Contains details for the approvers
         *        in the team.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder approvers(GetApprovalTeamResponseApprover... approvers);

        /**
         * <p>
         * An array of <code>GetApprovalTeamResponseApprover </code> objects. Contains details for the approvers in the
         * team.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.mpa.model.GetApprovalTeamResponseApprover.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.mpa.model.GetApprovalTeamResponseApprover#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.mpa.model.GetApprovalTeamResponseApprover.Builder#build()} is called
         * immediately and its result is passed to {@link #approvers(List<GetApprovalTeamResponseApprover>)}.
         * 
         * @param approvers
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.mpa.model.GetApprovalTeamResponseApprover.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #approvers(java.util.Collection<GetApprovalTeamResponseApprover>)
         */
        Builder approvers(Consumer<GetApprovalTeamResponseApprover.Builder>... approvers);

        /**
         * <p>
         * Timestamp when the update request was initiated.
         * </p>
         * 
         * @param updateInitiationTime
         *        Timestamp when the update request was initiated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updateInitiationTime(Instant updateInitiationTime);
    }

    static final class BuilderImpl implements Builder {
        private String versionId;

        private String description;

        private ApprovalStrategyResponse approvalStrategy;

        private Integer numberOfApprovers;

        private String status;

        private String statusCode;

        private String statusMessage;

        private List<GetApprovalTeamResponseApprover> approvers = DefaultSdkAutoConstructList.getInstance();

        private Instant updateInitiationTime;

        private BuilderImpl() {
        }

        private BuilderImpl(PendingUpdate model) {
            versionId(model.versionId);
            description(model.description);
            approvalStrategy(model.approvalStrategy);
            numberOfApprovers(model.numberOfApprovers);
            status(model.status);
            statusCode(model.statusCode);
            statusMessage(model.statusMessage);
            approvers(model.approvers);
            updateInitiationTime(model.updateInitiationTime);
        }

        public final String getVersionId() {
            return versionId;
        }

        public final void setVersionId(String versionId) {
            this.versionId = versionId;
        }

        @Override
        public final Builder versionId(String versionId) {
            this.versionId = versionId;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final ApprovalStrategyResponse.Builder getApprovalStrategy() {
            return approvalStrategy != null ? approvalStrategy.toBuilder() : null;
        }

        public final void setApprovalStrategy(ApprovalStrategyResponse.BuilderImpl approvalStrategy) {
            this.approvalStrategy = approvalStrategy != null ? approvalStrategy.build() : null;
        }

        @Override
        public final Builder approvalStrategy(ApprovalStrategyResponse approvalStrategy) {
            this.approvalStrategy = approvalStrategy;
            return this;
        }

        public final Integer getNumberOfApprovers() {
            return numberOfApprovers;
        }

        public final void setNumberOfApprovers(Integer numberOfApprovers) {
            this.numberOfApprovers = numberOfApprovers;
        }

        @Override
        public final Builder numberOfApprovers(Integer numberOfApprovers) {
            this.numberOfApprovers = numberOfApprovers;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ApprovalTeamStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusCode() {
            return statusCode;
        }

        public final void setStatusCode(String statusCode) {
            this.statusCode = statusCode;
        }

        @Override
        public final Builder statusCode(String statusCode) {
            this.statusCode = statusCode;
            return this;
        }

        @Override
        public final Builder statusCode(ApprovalTeamStatusCode statusCode) {
            this.statusCode(statusCode == null ? null : statusCode.toString());
            return this;
        }

        public final String getStatusMessage() {
            return statusMessage;
        }

        public final void setStatusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
        }

        @Override
        public final Builder statusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
            return this;
        }

        public final List<GetApprovalTeamResponseApprover.Builder> getApprovers() {
            List<GetApprovalTeamResponseApprover.Builder> result = GetApprovalTeamResponseApproversCopier
                    .copyToBuilder(this.approvers);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setApprovers(Collection<GetApprovalTeamResponseApprover.BuilderImpl> approvers) {
            this.approvers = GetApprovalTeamResponseApproversCopier.copyFromBuilder(approvers);
        }

        @Override
        public final Builder approvers(Collection<GetApprovalTeamResponseApprover> approvers) {
            this.approvers = GetApprovalTeamResponseApproversCopier.copy(approvers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder approvers(GetApprovalTeamResponseApprover... approvers) {
            approvers(Arrays.asList(approvers));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder approvers(Consumer<GetApprovalTeamResponseApprover.Builder>... approvers) {
            approvers(Stream.of(approvers).map(c -> GetApprovalTeamResponseApprover.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Instant getUpdateInitiationTime() {
            return updateInitiationTime;
        }

        public final void setUpdateInitiationTime(Instant updateInitiationTime) {
            this.updateInitiationTime = updateInitiationTime;
        }

        @Override
        public final Builder updateInitiationTime(Instant updateInitiationTime) {
            this.updateInitiationTime = updateInitiationTime;
            return this;
        }

        @Override
        public PendingUpdate build() {
            return new PendingUpdate(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
