/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mpa;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.mpa.internal.MpaServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.mpa.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.mpa.model.AccessDeniedException;
import software.amazon.awssdk.services.mpa.model.CancelSessionRequest;
import software.amazon.awssdk.services.mpa.model.CancelSessionResponse;
import software.amazon.awssdk.services.mpa.model.ConflictException;
import software.amazon.awssdk.services.mpa.model.CreateApprovalTeamRequest;
import software.amazon.awssdk.services.mpa.model.CreateApprovalTeamResponse;
import software.amazon.awssdk.services.mpa.model.CreateIdentitySourceRequest;
import software.amazon.awssdk.services.mpa.model.CreateIdentitySourceResponse;
import software.amazon.awssdk.services.mpa.model.DeleteIdentitySourceRequest;
import software.amazon.awssdk.services.mpa.model.DeleteIdentitySourceResponse;
import software.amazon.awssdk.services.mpa.model.DeleteInactiveApprovalTeamVersionRequest;
import software.amazon.awssdk.services.mpa.model.DeleteInactiveApprovalTeamVersionResponse;
import software.amazon.awssdk.services.mpa.model.GetApprovalTeamRequest;
import software.amazon.awssdk.services.mpa.model.GetApprovalTeamResponse;
import software.amazon.awssdk.services.mpa.model.GetIdentitySourceRequest;
import software.amazon.awssdk.services.mpa.model.GetIdentitySourceResponse;
import software.amazon.awssdk.services.mpa.model.GetPolicyVersionRequest;
import software.amazon.awssdk.services.mpa.model.GetPolicyVersionResponse;
import software.amazon.awssdk.services.mpa.model.GetResourcePolicyRequest;
import software.amazon.awssdk.services.mpa.model.GetResourcePolicyResponse;
import software.amazon.awssdk.services.mpa.model.GetSessionRequest;
import software.amazon.awssdk.services.mpa.model.GetSessionResponse;
import software.amazon.awssdk.services.mpa.model.InternalServerException;
import software.amazon.awssdk.services.mpa.model.InvalidParameterException;
import software.amazon.awssdk.services.mpa.model.ListApprovalTeamsRequest;
import software.amazon.awssdk.services.mpa.model.ListApprovalTeamsResponse;
import software.amazon.awssdk.services.mpa.model.ListIdentitySourcesRequest;
import software.amazon.awssdk.services.mpa.model.ListIdentitySourcesResponse;
import software.amazon.awssdk.services.mpa.model.ListPoliciesRequest;
import software.amazon.awssdk.services.mpa.model.ListPoliciesResponse;
import software.amazon.awssdk.services.mpa.model.ListPolicyVersionsRequest;
import software.amazon.awssdk.services.mpa.model.ListPolicyVersionsResponse;
import software.amazon.awssdk.services.mpa.model.ListResourcePoliciesRequest;
import software.amazon.awssdk.services.mpa.model.ListResourcePoliciesResponse;
import software.amazon.awssdk.services.mpa.model.ListSessionsRequest;
import software.amazon.awssdk.services.mpa.model.ListSessionsResponse;
import software.amazon.awssdk.services.mpa.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.mpa.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.mpa.model.MpaException;
import software.amazon.awssdk.services.mpa.model.ResourceNotFoundException;
import software.amazon.awssdk.services.mpa.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.mpa.model.StartActiveApprovalTeamDeletionRequest;
import software.amazon.awssdk.services.mpa.model.StartActiveApprovalTeamDeletionResponse;
import software.amazon.awssdk.services.mpa.model.TagResourceRequest;
import software.amazon.awssdk.services.mpa.model.TagResourceResponse;
import software.amazon.awssdk.services.mpa.model.ThrottlingException;
import software.amazon.awssdk.services.mpa.model.TooManyTagsException;
import software.amazon.awssdk.services.mpa.model.UntagResourceRequest;
import software.amazon.awssdk.services.mpa.model.UntagResourceResponse;
import software.amazon.awssdk.services.mpa.model.UpdateApprovalTeamRequest;
import software.amazon.awssdk.services.mpa.model.UpdateApprovalTeamResponse;
import software.amazon.awssdk.services.mpa.model.ValidationException;
import software.amazon.awssdk.services.mpa.transform.CancelSessionRequestMarshaller;
import software.amazon.awssdk.services.mpa.transform.CreateApprovalTeamRequestMarshaller;
import software.amazon.awssdk.services.mpa.transform.CreateIdentitySourceRequestMarshaller;
import software.amazon.awssdk.services.mpa.transform.DeleteIdentitySourceRequestMarshaller;
import software.amazon.awssdk.services.mpa.transform.DeleteInactiveApprovalTeamVersionRequestMarshaller;
import software.amazon.awssdk.services.mpa.transform.GetApprovalTeamRequestMarshaller;
import software.amazon.awssdk.services.mpa.transform.GetIdentitySourceRequestMarshaller;
import software.amazon.awssdk.services.mpa.transform.GetPolicyVersionRequestMarshaller;
import software.amazon.awssdk.services.mpa.transform.GetResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.mpa.transform.GetSessionRequestMarshaller;
import software.amazon.awssdk.services.mpa.transform.ListApprovalTeamsRequestMarshaller;
import software.amazon.awssdk.services.mpa.transform.ListIdentitySourcesRequestMarshaller;
import software.amazon.awssdk.services.mpa.transform.ListPoliciesRequestMarshaller;
import software.amazon.awssdk.services.mpa.transform.ListPolicyVersionsRequestMarshaller;
import software.amazon.awssdk.services.mpa.transform.ListResourcePoliciesRequestMarshaller;
import software.amazon.awssdk.services.mpa.transform.ListSessionsRequestMarshaller;
import software.amazon.awssdk.services.mpa.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.mpa.transform.StartActiveApprovalTeamDeletionRequestMarshaller;
import software.amazon.awssdk.services.mpa.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.mpa.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.mpa.transform.UpdateApprovalTeamRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link MpaClient}.
 *
 * @see MpaClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMpaClient implements MpaClient {
    private static final Logger log = Logger.loggerFor(DefaultMpaClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMpaClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "MPA" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Cancels an approval session. For more information, see <a
     * href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-concepts.html">Session</a> in the <i>Multi-party
     * approval User Guide</i>.
     * </p>
     *
     * @param cancelSessionRequest
     * @return Result of the CancelSession operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check your permissions, and try again.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist. Check the resource ID, and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The service encountered an internal error. Try your request again. If the problem persists, contact
     *         Amazon Web Services Support.
     * @throws ConflictException
     *         The request cannot be completed because it conflicts with the current state of a resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MpaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MpaClient.CancelSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mpa-2022-07-26/CancelSession" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CancelSessionResponse cancelSession(CancelSessionRequest cancelSessionRequest) throws AccessDeniedException,
            ValidationException, ResourceNotFoundException, ThrottlingException, InternalServerException, ConflictException,
            AwsServiceException, SdkClientException, MpaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelSessionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelSessionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MPA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelSession");

            return clientHandler.execute(new ClientExecutionParams<CancelSessionRequest, CancelSessionResponse>()
                    .withOperationName("CancelSession").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(cancelSessionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CancelSessionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new approval team. For more information, see <a
     * href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-concepts.html">Approval team</a> in the <i>Multi-party
     * approval User Guide</i>.
     * </p>
     *
     * @param createApprovalTeamRequest
     * @return Result of the CreateApprovalTeam operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check your permissions, and try again.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ServiceQuotaExceededException
     *         The request exceeds the service quota for your account. Request a quota increase or reduce your request
     *         size.
     * @throws InternalServerException
     *         The service encountered an internal error. Try your request again. If the problem persists, contact
     *         Amazon Web Services Support.
     * @throws ConflictException
     *         The request cannot be completed because it conflicts with the current state of a resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MpaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MpaClient.CreateApprovalTeam
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mpa-2022-07-26/CreateApprovalTeam" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateApprovalTeamResponse createApprovalTeam(CreateApprovalTeamRequest createApprovalTeamRequest)
            throws AccessDeniedException, ValidationException, ThrottlingException, ServiceQuotaExceededException,
            InternalServerException, ConflictException, AwsServiceException, SdkClientException, MpaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateApprovalTeamResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateApprovalTeamResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createApprovalTeamRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createApprovalTeamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MPA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateApprovalTeam");

            return clientHandler.execute(new ClientExecutionParams<CreateApprovalTeamRequest, CreateApprovalTeamResponse>()
                    .withOperationName("CreateApprovalTeam").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createApprovalTeamRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateApprovalTeamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new identity source. For more information, see <a
     * href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-concepts.html">Identity Source</a> in the
     * <i>Multi-party approval User Guide</i>.
     * </p>
     *
     * @param createIdentitySourceRequest
     * @return Result of the CreateIdentitySource operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check your permissions, and try again.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ServiceQuotaExceededException
     *         The request exceeds the service quota for your account. Request a quota increase or reduce your request
     *         size.
     * @throws InternalServerException
     *         The service encountered an internal error. Try your request again. If the problem persists, contact
     *         Amazon Web Services Support.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MpaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MpaClient.CreateIdentitySource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mpa-2022-07-26/CreateIdentitySource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateIdentitySourceResponse createIdentitySource(CreateIdentitySourceRequest createIdentitySourceRequest)
            throws AccessDeniedException, ValidationException, ThrottlingException, ServiceQuotaExceededException,
            InternalServerException, AwsServiceException, SdkClientException, MpaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateIdentitySourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateIdentitySourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createIdentitySourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createIdentitySourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MPA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateIdentitySource");

            return clientHandler.execute(new ClientExecutionParams<CreateIdentitySourceRequest, CreateIdentitySourceResponse>()
                    .withOperationName("CreateIdentitySource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createIdentitySourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateIdentitySourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an identity source. For more information, see <a
     * href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-concepts.html">Identity Source</a> in the
     * <i>Multi-party approval User Guide</i>.
     * </p>
     *
     * @param deleteIdentitySourceRequest
     * @return Result of the DeleteIdentitySource operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check your permissions, and try again.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The service encountered an internal error. Try your request again. If the problem persists, contact
     *         Amazon Web Services Support.
     * @throws ConflictException
     *         The request cannot be completed because it conflicts with the current state of a resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MpaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MpaClient.DeleteIdentitySource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mpa-2022-07-26/DeleteIdentitySource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteIdentitySourceResponse deleteIdentitySource(DeleteIdentitySourceRequest deleteIdentitySourceRequest)
            throws AccessDeniedException, ValidationException, ThrottlingException, InternalServerException, ConflictException,
            AwsServiceException, SdkClientException, MpaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteIdentitySourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteIdentitySourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteIdentitySourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteIdentitySourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MPA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteIdentitySource");

            return clientHandler.execute(new ClientExecutionParams<DeleteIdentitySourceRequest, DeleteIdentitySourceResponse>()
                    .withOperationName("DeleteIdentitySource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteIdentitySourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteIdentitySourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an inactive approval team. For more information, see <a
     * href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-health.html">Team health</a> in the <i>Multi-party
     * approval User Guide</i>.
     * </p>
     * <p>
     * You can also use this operation to delete a team draft. For more information, see <a
     * href="https://docs.aws.amazon.com/mpa/latest/userguide/update-team.html#update-team-draft-status">Interacting
     * with drafts</a> in the <i>Multi-party approval User Guide</i>.
     * </p>
     *
     * @param deleteInactiveApprovalTeamVersionRequest
     * @return Result of the DeleteInactiveApprovalTeamVersion operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check your permissions, and try again.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist. Check the resource ID, and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The service encountered an internal error. Try your request again. If the problem persists, contact
     *         Amazon Web Services Support.
     * @throws ConflictException
     *         The request cannot be completed because it conflicts with the current state of a resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MpaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MpaClient.DeleteInactiveApprovalTeamVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mpa-2022-07-26/DeleteInactiveApprovalTeamVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteInactiveApprovalTeamVersionResponse deleteInactiveApprovalTeamVersion(
            DeleteInactiveApprovalTeamVersionRequest deleteInactiveApprovalTeamVersionRequest) throws AccessDeniedException,
            ValidationException, ResourceNotFoundException, ThrottlingException, InternalServerException, ConflictException,
            AwsServiceException, SdkClientException, MpaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteInactiveApprovalTeamVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteInactiveApprovalTeamVersionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteInactiveApprovalTeamVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteInactiveApprovalTeamVersionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MPA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteInactiveApprovalTeamVersion");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteInactiveApprovalTeamVersionRequest, DeleteInactiveApprovalTeamVersionResponse>()
                            .withOperationName("DeleteInactiveApprovalTeamVersion").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteInactiveApprovalTeamVersionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteInactiveApprovalTeamVersionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns details for an approval team.
     * </p>
     *
     * @param getApprovalTeamRequest
     * @return Result of the GetApprovalTeam operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check your permissions, and try again.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist. Check the resource ID, and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The service encountered an internal error. Try your request again. If the problem persists, contact
     *         Amazon Web Services Support.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MpaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MpaClient.GetApprovalTeam
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mpa-2022-07-26/GetApprovalTeam" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetApprovalTeamResponse getApprovalTeam(GetApprovalTeamRequest getApprovalTeamRequest) throws AccessDeniedException,
            ValidationException, ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, MpaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetApprovalTeamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetApprovalTeamResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getApprovalTeamRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getApprovalTeamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MPA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetApprovalTeam");

            return clientHandler.execute(new ClientExecutionParams<GetApprovalTeamRequest, GetApprovalTeamResponse>()
                    .withOperationName("GetApprovalTeam").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getApprovalTeamRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetApprovalTeamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns details for an identity source. For more information, see <a
     * href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-concepts.html">Identity Source</a> in the
     * <i>Multi-party approval User Guide</i>.
     * </p>
     *
     * @param getIdentitySourceRequest
     * @return Result of the GetIdentitySource operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check your permissions, and try again.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist. Check the resource ID, and try again.
     * @throws InternalServerException
     *         The service encountered an internal error. Try your request again. If the problem persists, contact
     *         Amazon Web Services Support.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MpaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MpaClient.GetIdentitySource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mpa-2022-07-26/GetIdentitySource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetIdentitySourceResponse getIdentitySource(GetIdentitySourceRequest getIdentitySourceRequest)
            throws AccessDeniedException, ValidationException, ThrottlingException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, MpaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetIdentitySourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetIdentitySourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getIdentitySourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getIdentitySourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MPA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetIdentitySource");

            return clientHandler.execute(new ClientExecutionParams<GetIdentitySourceRequest, GetIdentitySourceResponse>()
                    .withOperationName("GetIdentitySource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getIdentitySourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetIdentitySourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns details for the version of a policy. Policies define the permissions for team resources.
     * </p>
     * <p>
     * The protected operation for a service integration might require specific permissions. For more information, see
     * <a href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-integrations.html">How other services work with
     * Multi-party approval</a> in the <i>Multi-party approval User Guide</i>.
     * </p>
     *
     * @param getPolicyVersionRequest
     * @return Result of the GetPolicyVersion operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check your permissions, and try again.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist. Check the resource ID, and try again.
     * @throws InternalServerException
     *         The service encountered an internal error. Try your request again. If the problem persists, contact
     *         Amazon Web Services Support.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MpaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MpaClient.GetPolicyVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mpa-2022-07-26/GetPolicyVersion" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetPolicyVersionResponse getPolicyVersion(GetPolicyVersionRequest getPolicyVersionRequest)
            throws AccessDeniedException, ValidationException, ThrottlingException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, MpaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPolicyVersionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetPolicyVersionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPolicyVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPolicyVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MPA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPolicyVersion");

            return clientHandler.execute(new ClientExecutionParams<GetPolicyVersionRequest, GetPolicyVersionResponse>()
                    .withOperationName("GetPolicyVersion").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getPolicyVersionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetPolicyVersionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns details about a policy for a resource.
     * </p>
     *
     * @param getResourcePolicyRequest
     * @return Result of the GetResourcePolicy operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check your permissions, and try again.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InvalidParameterException
     *         The request contains an invalid parameter value.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist. Check the resource ID, and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MpaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MpaClient.GetResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mpa-2022-07-26/GetResourcePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetResourcePolicyResponse getResourcePolicy(GetResourcePolicyRequest getResourcePolicyRequest)
            throws AccessDeniedException, ValidationException, InvalidParameterException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, MpaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetResourcePolicyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MPA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourcePolicy");

            return clientHandler.execute(new ClientExecutionParams<GetResourcePolicyRequest, GetResourcePolicyResponse>()
                    .withOperationName("GetResourcePolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getResourcePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourcePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns details for an approval session. For more information, see <a
     * href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-concepts.html">Session</a> in the <i>Multi-party
     * approval User Guide</i>.
     * </p>
     *
     * @param getSessionRequest
     * @return Result of the GetSession operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check your permissions, and try again.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist. Check the resource ID, and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The service encountered an internal error. Try your request again. If the problem persists, contact
     *         Amazon Web Services Support.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MpaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MpaClient.GetSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mpa-2022-07-26/GetSession" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSessionResponse getSession(GetSessionRequest getSessionRequest) throws AccessDeniedException, ValidationException,
            ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException, SdkClientException,
            MpaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSessionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSessionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MPA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSession");

            return clientHandler.execute(new ClientExecutionParams<GetSessionRequest, GetSessionResponse>()
                    .withOperationName("GetSession").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getSessionRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSessionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of approval teams.
     * </p>
     *
     * @param listApprovalTeamsRequest
     * @return Result of the ListApprovalTeams operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check your permissions, and try again.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The service encountered an internal error. Try your request again. If the problem persists, contact
     *         Amazon Web Services Support.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MpaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MpaClient.ListApprovalTeams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mpa-2022-07-26/ListApprovalTeams" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListApprovalTeamsResponse listApprovalTeams(ListApprovalTeamsRequest listApprovalTeamsRequest)
            throws AccessDeniedException, ValidationException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, MpaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListApprovalTeamsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListApprovalTeamsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listApprovalTeamsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApprovalTeamsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MPA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApprovalTeams");

            return clientHandler.execute(new ClientExecutionParams<ListApprovalTeamsRequest, ListApprovalTeamsResponse>()
                    .withOperationName("ListApprovalTeams").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listApprovalTeamsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListApprovalTeamsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of identity sources. For more information, see <a
     * href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-concepts.html">Identity Source</a> in the
     * <i>Multi-party approval User Guide</i>.
     * </p>
     *
     * @param listIdentitySourcesRequest
     * @return Result of the ListIdentitySources operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check your permissions, and try again.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The service encountered an internal error. Try your request again. If the problem persists, contact
     *         Amazon Web Services Support.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MpaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MpaClient.ListIdentitySources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mpa-2022-07-26/ListIdentitySources" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListIdentitySourcesResponse listIdentitySources(ListIdentitySourcesRequest listIdentitySourcesRequest)
            throws AccessDeniedException, ValidationException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, MpaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListIdentitySourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListIdentitySourcesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listIdentitySourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listIdentitySourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MPA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListIdentitySources");

            return clientHandler.execute(new ClientExecutionParams<ListIdentitySourcesRequest, ListIdentitySourcesResponse>()
                    .withOperationName("ListIdentitySources").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listIdentitySourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListIdentitySourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of policies. Policies define the permissions for team resources.
     * </p>
     * <p>
     * The protected operation for a service integration might require specific permissions. For more information, see
     * <a href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-integrations.html">How other services work with
     * Multi-party approval</a> in the <i>Multi-party approval User Guide</i>.
     * </p>
     *
     * @param listPoliciesRequest
     * @return Result of the ListPolicies operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check your permissions, and try again.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The service encountered an internal error. Try your request again. If the problem persists, contact
     *         Amazon Web Services Support.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MpaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MpaClient.ListPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mpa-2022-07-26/ListPolicies" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPoliciesResponse listPolicies(ListPoliciesRequest listPoliciesRequest) throws AccessDeniedException,
            ValidationException, ThrottlingException, InternalServerException, AwsServiceException, SdkClientException,
            MpaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPoliciesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListPoliciesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPoliciesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPoliciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MPA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPolicies");

            return clientHandler.execute(new ClientExecutionParams<ListPoliciesRequest, ListPoliciesResponse>()
                    .withOperationName("ListPolicies").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listPoliciesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPoliciesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the versions for policies. Policies define the permissions for team resources.
     * </p>
     * <p>
     * The protected operation for a service integration might require specific permissions. For more information, see
     * <a href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-integrations.html">How other services work with
     * Multi-party approval</a> in the <i>Multi-party approval User Guide</i>.
     * </p>
     *
     * @param listPolicyVersionsRequest
     * @return Result of the ListPolicyVersions operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check your permissions, and try again.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist. Check the resource ID, and try again.
     * @throws InternalServerException
     *         The service encountered an internal error. Try your request again. If the problem persists, contact
     *         Amazon Web Services Support.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MpaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MpaClient.ListPolicyVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mpa-2022-07-26/ListPolicyVersions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPolicyVersionsResponse listPolicyVersions(ListPolicyVersionsRequest listPolicyVersionsRequest)
            throws AccessDeniedException, ValidationException, ThrottlingException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, MpaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPolicyVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListPolicyVersionsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPolicyVersionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPolicyVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MPA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPolicyVersions");

            return clientHandler.execute(new ClientExecutionParams<ListPolicyVersionsRequest, ListPolicyVersionsResponse>()
                    .withOperationName("ListPolicyVersions").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listPolicyVersionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPolicyVersionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of policies for a resource.
     * </p>
     *
     * @param listResourcePoliciesRequest
     * @return Result of the ListResourcePolicies operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check your permissions, and try again.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist. Check the resource ID, and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The service encountered an internal error. Try your request again. If the problem persists, contact
     *         Amazon Web Services Support.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MpaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MpaClient.ListResourcePolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mpa-2022-07-26/ListResourcePolicies" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListResourcePoliciesResponse listResourcePolicies(ListResourcePoliciesRequest listResourcePoliciesRequest)
            throws AccessDeniedException, ValidationException, ResourceNotFoundException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, MpaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResourcePoliciesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListResourcePoliciesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listResourcePoliciesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourcePoliciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MPA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResourcePolicies");

            return clientHandler.execute(new ClientExecutionParams<ListResourcePoliciesRequest, ListResourcePoliciesResponse>()
                    .withOperationName("ListResourcePolicies").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listResourcePoliciesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListResourcePoliciesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of approval sessions. For more information, see <a
     * href="https://docs.aws.amazon.com/mpa/latest/userguide/mpa-concepts.html">Session</a> in the <i>Multi-party
     * approval User Guide</i>.
     * </p>
     *
     * @param listSessionsRequest
     * @return Result of the ListSessions operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check your permissions, and try again.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist. Check the resource ID, and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The service encountered an internal error. Try your request again. If the problem persists, contact
     *         Amazon Web Services Support.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MpaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MpaClient.ListSessions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mpa-2022-07-26/ListSessions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListSessionsResponse listSessions(ListSessionsRequest listSessionsRequest) throws AccessDeniedException,
            ValidationException, ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, MpaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSessionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListSessionsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSessionsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSessionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MPA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSessions");

            return clientHandler.execute(new ClientExecutionParams<ListSessionsRequest, ListSessionsResponse>()
                    .withOperationName("ListSessions").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listSessionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListSessionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the tags for a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check your permissions, and try again.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist. Check the resource ID, and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The service encountered an internal error. Try your request again. If the problem persists, contact
     *         Amazon Web Services Support.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MpaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MpaClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mpa-2022-07-26/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws AccessDeniedException, ValidationException, ResourceNotFoundException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, MpaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MPA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts the deletion process for an active approval team.
     * </p>
     * <note>
     * <p>
     * <b>Deletions require team approval</b>
     * </p>
     * <p>
     * Requests to delete an active team must be approved by the team.
     * </p>
     * </note>
     *
     * @param startActiveApprovalTeamDeletionRequest
     * @return Result of the StartActiveApprovalTeamDeletion operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check your permissions, and try again.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist. Check the resource ID, and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The service encountered an internal error. Try your request again. If the problem persists, contact
     *         Amazon Web Services Support.
     * @throws ConflictException
     *         The request cannot be completed because it conflicts with the current state of a resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MpaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MpaClient.StartActiveApprovalTeamDeletion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mpa-2022-07-26/StartActiveApprovalTeamDeletion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartActiveApprovalTeamDeletionResponse startActiveApprovalTeamDeletion(
            StartActiveApprovalTeamDeletionRequest startActiveApprovalTeamDeletionRequest) throws AccessDeniedException,
            ValidationException, ResourceNotFoundException, ThrottlingException, InternalServerException, ConflictException,
            AwsServiceException, SdkClientException, MpaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartActiveApprovalTeamDeletionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartActiveApprovalTeamDeletionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startActiveApprovalTeamDeletionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                startActiveApprovalTeamDeletionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MPA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartActiveApprovalTeamDeletion");

            return clientHandler
                    .execute(new ClientExecutionParams<StartActiveApprovalTeamDeletionRequest, StartActiveApprovalTeamDeletionResponse>()
                            .withOperationName("StartActiveApprovalTeamDeletion").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(startActiveApprovalTeamDeletionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartActiveApprovalTeamDeletionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates a resource tag. Each tag is a label consisting of a user-defined key and value. Tags can help
     * you manage, identify, organize, search for, and filter resources.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check your permissions, and try again.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist. Check the resource ID, and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws TooManyTagsException
     *         The request exceeds the maximum number of tags allowed for this resource. Remove some tags, and try
     *         again.
     * @throws InternalServerException
     *         The service encountered an internal error. Try your request again. If the problem persists, contact
     *         Amazon Web Services Support.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MpaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MpaClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mpa-2022-07-26/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws AccessDeniedException,
            ValidationException, ResourceNotFoundException, ThrottlingException, TooManyTagsException, InternalServerException,
            AwsServiceException, SdkClientException, MpaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MPA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a resource tag. Each tag is a label consisting of a user-defined key and value. Tags can help you manage,
     * identify, organize, search for, and filter resources.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check your permissions, and try again.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist. Check the resource ID, and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The service encountered an internal error. Try your request again. If the problem persists, contact
     *         Amazon Web Services Support.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MpaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MpaClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mpa-2022-07-26/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws AccessDeniedException,
            ValidationException, ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, MpaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MPA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an approval team. You can request to update the team description, approval threshold, and approvers in
     * the team.
     * </p>
     * <note>
     * <p>
     * <b>Updates require team approval</b>
     * </p>
     * <p>
     * Updates to an active team must be approved by the team.
     * </p>
     * </note>
     *
     * @param updateApprovalTeamRequest
     * @return Result of the UpdateApprovalTeam operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check your permissions, and try again.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist. Check the resource ID, and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ServiceQuotaExceededException
     *         The request exceeds the service quota for your account. Request a quota increase or reduce your request
     *         size.
     * @throws InternalServerException
     *         The service encountered an internal error. Try your request again. If the problem persists, contact
     *         Amazon Web Services Support.
     * @throws ConflictException
     *         The request cannot be completed because it conflicts with the current state of a resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MpaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MpaClient.UpdateApprovalTeam
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mpa-2022-07-26/UpdateApprovalTeam" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateApprovalTeamResponse updateApprovalTeam(UpdateApprovalTeamRequest updateApprovalTeamRequest)
            throws AccessDeniedException, ValidationException, ResourceNotFoundException, ThrottlingException,
            ServiceQuotaExceededException, InternalServerException, ConflictException, AwsServiceException, SdkClientException,
            MpaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateApprovalTeamResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateApprovalTeamResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateApprovalTeamRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateApprovalTeamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MPA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateApprovalTeam");

            return clientHandler.execute(new ClientExecutionParams<UpdateApprovalTeamRequest, UpdateApprovalTeamResponse>()
                    .withOperationName("UpdateApprovalTeam").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateApprovalTeamRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateApprovalTeamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        MpaServiceClientConfigurationBuilder serviceConfigBuilder = new MpaServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(MpaException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    @Override
    public final MpaServiceClientConfiguration serviceClientConfiguration() {
        return new MpaServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
