/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediaconvert.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * Specified video input in a template.
 */
@Generated("software.amazon.awssdk:codegen")
public final class InputTemplate implements SdkPojo, Serializable, ToCopyableBuilder<InputTemplate.Builder, InputTemplate> {
    private static final SdkField<String> ADVANCED_INPUT_FILTER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AdvancedInputFilter").getter(getter(InputTemplate::advancedInputFilterAsString))
            .setter(setter(Builder::advancedInputFilter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("advancedInputFilter").build())
            .build();

    private static final SdkField<AdvancedInputFilterSettings> ADVANCED_INPUT_FILTER_SETTINGS_FIELD = SdkField
            .<AdvancedInputFilterSettings> builder(MarshallingType.SDK_POJO)
            .memberName("AdvancedInputFilterSettings")
            .getter(getter(InputTemplate::advancedInputFilterSettings))
            .setter(setter(Builder::advancedInputFilterSettings))
            .constructor(AdvancedInputFilterSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("advancedInputFilterSettings")
                    .build()).build();

    private static final SdkField<Map<String, AudioSelectorGroup>> AUDIO_SELECTOR_GROUPS_FIELD = SdkField
            .<Map<String, AudioSelectorGroup>> builder(MarshallingType.MAP)
            .memberName("AudioSelectorGroups")
            .getter(getter(InputTemplate::audioSelectorGroups))
            .setter(setter(Builder::audioSelectorGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("audioSelectorGroups").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<AudioSelectorGroup> builder(MarshallingType.SDK_POJO)
                                            .constructor(AudioSelectorGroup::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, AudioSelector>> AUDIO_SELECTORS_FIELD = SdkField
            .<Map<String, AudioSelector>> builder(MarshallingType.MAP)
            .memberName("AudioSelectors")
            .getter(getter(InputTemplate::audioSelectors))
            .setter(setter(Builder::audioSelectors))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("audioSelectors").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<AudioSelector> builder(MarshallingType.SDK_POJO)
                                            .constructor(AudioSelector::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, CaptionSelector>> CAPTION_SELECTORS_FIELD = SdkField
            .<Map<String, CaptionSelector>> builder(MarshallingType.MAP)
            .memberName("CaptionSelectors")
            .getter(getter(InputTemplate::captionSelectors))
            .setter(setter(Builder::captionSelectors))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("captionSelectors").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<CaptionSelector> builder(MarshallingType.SDK_POJO)
                                            .constructor(CaptionSelector::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Rectangle> CROP_FIELD = SdkField.<Rectangle> builder(MarshallingType.SDK_POJO)
            .memberName("Crop").getter(getter(InputTemplate::crop)).setter(setter(Builder::crop)).constructor(Rectangle::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("crop").build()).build();

    private static final SdkField<String> DEBLOCK_FILTER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DeblockFilter").getter(getter(InputTemplate::deblockFilterAsString))
            .setter(setter(Builder::deblockFilter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deblockFilter").build()).build();

    private static final SdkField<String> DENOISE_FILTER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DenoiseFilter").getter(getter(InputTemplate::denoiseFilterAsString))
            .setter(setter(Builder::denoiseFilter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("denoiseFilter").build()).build();

    private static final SdkField<String> DOLBY_VISION_METADATA_XML_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DolbyVisionMetadataXml").getter(getter(InputTemplate::dolbyVisionMetadataXml))
            .setter(setter(Builder::dolbyVisionMetadataXml))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dolbyVisionMetadataXml").build())
            .build();

    private static final SdkField<String> FILTER_ENABLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FilterEnable").getter(getter(InputTemplate::filterEnableAsString)).setter(setter(Builder::filterEnable))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("filterEnable").build()).build();

    private static final SdkField<Integer> FILTER_STRENGTH_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("FilterStrength").getter(getter(InputTemplate::filterStrength)).setter(setter(Builder::filterStrength))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("filterStrength").build()).build();

    private static final SdkField<ImageInserter> IMAGE_INSERTER_FIELD = SdkField
            .<ImageInserter> builder(MarshallingType.SDK_POJO).memberName("ImageInserter")
            .getter(getter(InputTemplate::imageInserter)).setter(setter(Builder::imageInserter))
            .constructor(ImageInserter::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imageInserter").build()).build();

    private static final SdkField<List<InputClipping>> INPUT_CLIPPINGS_FIELD = SdkField
            .<List<InputClipping>> builder(MarshallingType.LIST)
            .memberName("InputClippings")
            .getter(getter(InputTemplate::inputClippings))
            .setter(setter(Builder::inputClippings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inputClippings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<InputClipping> builder(MarshallingType.SDK_POJO)
                                            .constructor(InputClipping::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> INPUT_SCAN_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InputScanType").getter(getter(InputTemplate::inputScanTypeAsString))
            .setter(setter(Builder::inputScanType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inputScanType").build()).build();

    private static final SdkField<Rectangle> POSITION_FIELD = SdkField.<Rectangle> builder(MarshallingType.SDK_POJO)
            .memberName("Position").getter(getter(InputTemplate::position)).setter(setter(Builder::position))
            .constructor(Rectangle::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("position").build()).build();

    private static final SdkField<Integer> PROGRAM_NUMBER_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ProgramNumber").getter(getter(InputTemplate::programNumber)).setter(setter(Builder::programNumber))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("programNumber").build()).build();

    private static final SdkField<String> PSI_CONTROL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PsiControl").getter(getter(InputTemplate::psiControlAsString)).setter(setter(Builder::psiControl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("psiControl").build()).build();

    private static final SdkField<String> TIMECODE_SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TimecodeSource").getter(getter(InputTemplate::timecodeSourceAsString))
            .setter(setter(Builder::timecodeSource))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("timecodeSource").build()).build();

    private static final SdkField<String> TIMECODE_START_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TimecodeStart").getter(getter(InputTemplate::timecodeStart)).setter(setter(Builder::timecodeStart))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("timecodeStart").build()).build();

    private static final SdkField<VideoSelector> VIDEO_SELECTOR_FIELD = SdkField
            .<VideoSelector> builder(MarshallingType.SDK_POJO).memberName("VideoSelector")
            .getter(getter(InputTemplate::videoSelector)).setter(setter(Builder::videoSelector))
            .constructor(VideoSelector::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("videoSelector").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ADVANCED_INPUT_FILTER_FIELD,
            ADVANCED_INPUT_FILTER_SETTINGS_FIELD, AUDIO_SELECTOR_GROUPS_FIELD, AUDIO_SELECTORS_FIELD, CAPTION_SELECTORS_FIELD,
            CROP_FIELD, DEBLOCK_FILTER_FIELD, DENOISE_FILTER_FIELD, DOLBY_VISION_METADATA_XML_FIELD, FILTER_ENABLE_FIELD,
            FILTER_STRENGTH_FIELD, IMAGE_INSERTER_FIELD, INPUT_CLIPPINGS_FIELD, INPUT_SCAN_TYPE_FIELD, POSITION_FIELD,
            PROGRAM_NUMBER_FIELD, PSI_CONTROL_FIELD, TIMECODE_SOURCE_FIELD, TIMECODE_START_FIELD, VIDEO_SELECTOR_FIELD));

    private static final long serialVersionUID = 1L;

    private final String advancedInputFilter;

    private final AdvancedInputFilterSettings advancedInputFilterSettings;

    private final Map<String, AudioSelectorGroup> audioSelectorGroups;

    private final Map<String, AudioSelector> audioSelectors;

    private final Map<String, CaptionSelector> captionSelectors;

    private final Rectangle crop;

    private final String deblockFilter;

    private final String denoiseFilter;

    private final String dolbyVisionMetadataXml;

    private final String filterEnable;

    private final Integer filterStrength;

    private final ImageInserter imageInserter;

    private final List<InputClipping> inputClippings;

    private final String inputScanType;

    private final Rectangle position;

    private final Integer programNumber;

    private final String psiControl;

    private final String timecodeSource;

    private final String timecodeStart;

    private final VideoSelector videoSelector;

    private InputTemplate(BuilderImpl builder) {
        this.advancedInputFilter = builder.advancedInputFilter;
        this.advancedInputFilterSettings = builder.advancedInputFilterSettings;
        this.audioSelectorGroups = builder.audioSelectorGroups;
        this.audioSelectors = builder.audioSelectors;
        this.captionSelectors = builder.captionSelectors;
        this.crop = builder.crop;
        this.deblockFilter = builder.deblockFilter;
        this.denoiseFilter = builder.denoiseFilter;
        this.dolbyVisionMetadataXml = builder.dolbyVisionMetadataXml;
        this.filterEnable = builder.filterEnable;
        this.filterStrength = builder.filterStrength;
        this.imageInserter = builder.imageInserter;
        this.inputClippings = builder.inputClippings;
        this.inputScanType = builder.inputScanType;
        this.position = builder.position;
        this.programNumber = builder.programNumber;
        this.psiControl = builder.psiControl;
        this.timecodeSource = builder.timecodeSource;
        this.timecodeStart = builder.timecodeStart;
        this.videoSelector = builder.videoSelector;
    }

    /**
     * Use to remove noise, blocking, blurriness, or ringing from your input as a pre-filter step before encoding. The
     * Advanced input filter removes more types of compression artifacts and is an improvement when compared to basic
     * Deblock and Denoise filters. To remove video compression artifacts from your input and improve the video quality:
     * Choose Enabled. Additionally, this filter can help increase the video quality of your output relative to its
     * bitrate, since noisy inputs are more complex and require more bits to encode. To help restore loss of detail
     * after applying the filter, you can optionally add texture or sharpening as an additional step. Jobs that use this
     * feature incur pro-tier pricing. To not apply advanced input filtering: Choose Disabled. Note that you can still
     * apply basic filtering with Deblock and Denoise.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #advancedInputFilter} will return {@link AdvancedInputFilter#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #advancedInputFilterAsString}.
     * </p>
     * 
     * @return Use to remove noise, blocking, blurriness, or ringing from your input as a pre-filter step before
     *         encoding. The Advanced input filter removes more types of compression artifacts and is an improvement
     *         when compared to basic Deblock and Denoise filters. To remove video compression artifacts from your input
     *         and improve the video quality: Choose Enabled. Additionally, this filter can help increase the video
     *         quality of your output relative to its bitrate, since noisy inputs are more complex and require more bits
     *         to encode. To help restore loss of detail after applying the filter, you can optionally add texture or
     *         sharpening as an additional step. Jobs that use this feature incur pro-tier pricing. To not apply
     *         advanced input filtering: Choose Disabled. Note that you can still apply basic filtering with Deblock and
     *         Denoise.
     * @see AdvancedInputFilter
     */
    public final AdvancedInputFilter advancedInputFilter() {
        return AdvancedInputFilter.fromValue(advancedInputFilter);
    }

    /**
     * Use to remove noise, blocking, blurriness, or ringing from your input as a pre-filter step before encoding. The
     * Advanced input filter removes more types of compression artifacts and is an improvement when compared to basic
     * Deblock and Denoise filters. To remove video compression artifacts from your input and improve the video quality:
     * Choose Enabled. Additionally, this filter can help increase the video quality of your output relative to its
     * bitrate, since noisy inputs are more complex and require more bits to encode. To help restore loss of detail
     * after applying the filter, you can optionally add texture or sharpening as an additional step. Jobs that use this
     * feature incur pro-tier pricing. To not apply advanced input filtering: Choose Disabled. Note that you can still
     * apply basic filtering with Deblock and Denoise.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #advancedInputFilter} will return {@link AdvancedInputFilter#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #advancedInputFilterAsString}.
     * </p>
     * 
     * @return Use to remove noise, blocking, blurriness, or ringing from your input as a pre-filter step before
     *         encoding. The Advanced input filter removes more types of compression artifacts and is an improvement
     *         when compared to basic Deblock and Denoise filters. To remove video compression artifacts from your input
     *         and improve the video quality: Choose Enabled. Additionally, this filter can help increase the video
     *         quality of your output relative to its bitrate, since noisy inputs are more complex and require more bits
     *         to encode. To help restore loss of detail after applying the filter, you can optionally add texture or
     *         sharpening as an additional step. Jobs that use this feature incur pro-tier pricing. To not apply
     *         advanced input filtering: Choose Disabled. Note that you can still apply basic filtering with Deblock and
     *         Denoise.
     * @see AdvancedInputFilter
     */
    public final String advancedInputFilterAsString() {
        return advancedInputFilter;
    }

    /**
     * Optional settings for Advanced input filter when you set Advanced input filter to Enabled.
     * 
     * @return Optional settings for Advanced input filter when you set Advanced input filter to Enabled.
     */
    public final AdvancedInputFilterSettings advancedInputFilterSettings() {
        return advancedInputFilterSettings;
    }

    /**
     * For responses, this returns true if the service returned a value for the AudioSelectorGroups property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAudioSelectorGroups() {
        return audioSelectorGroups != null && !(audioSelectorGroups instanceof SdkAutoConstructMap);
    }

    /**
     * Use audio selector groups to combine multiple sidecar audio inputs so that you can assign them to a single output
     * audio tab (AudioDescription). Note that, if you're working with embedded audio, it's simpler to assign multiple
     * input tracks into a single audio selector rather than use an audio selector group.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAudioSelectorGroups} method.
     * </p>
     * 
     * @return Use audio selector groups to combine multiple sidecar audio inputs so that you can assign them to a
     *         single output audio tab (AudioDescription). Note that, if you're working with embedded audio, it's
     *         simpler to assign multiple input tracks into a single audio selector rather than use an audio selector
     *         group.
     */
    public final Map<String, AudioSelectorGroup> audioSelectorGroups() {
        return audioSelectorGroups;
    }

    /**
     * For responses, this returns true if the service returned a value for the AudioSelectors property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAudioSelectors() {
        return audioSelectors != null && !(audioSelectors instanceof SdkAutoConstructMap);
    }

    /**
     * Use Audio selectors (AudioSelectors) to specify a track or set of tracks from the input that you will use in your
     * outputs. You can use multiple Audio selectors per input.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAudioSelectors} method.
     * </p>
     * 
     * @return Use Audio selectors (AudioSelectors) to specify a track or set of tracks from the input that you will use
     *         in your outputs. You can use multiple Audio selectors per input.
     */
    public final Map<String, AudioSelector> audioSelectors() {
        return audioSelectors;
    }

    /**
     * For responses, this returns true if the service returned a value for the CaptionSelectors property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCaptionSelectors() {
        return captionSelectors != null && !(captionSelectors instanceof SdkAutoConstructMap);
    }

    /**
     * Use captions selectors to specify the captions data from your input that you use in your outputs. You can use up
     * to 100 captions selectors per input.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCaptionSelectors} method.
     * </p>
     * 
     * @return Use captions selectors to specify the captions data from your input that you use in your outputs. You can
     *         use up to 100 captions selectors per input.
     */
    public final Map<String, CaptionSelector> captionSelectors() {
        return captionSelectors;
    }

    /**
     * Use Cropping selection (crop) to specify the video area that the service will include in the output video frame.
     * If you specify a value here, it will override any value that you specify in the output setting Cropping selection
     * (crop).
     * 
     * @return Use Cropping selection (crop) to specify the video area that the service will include in the output video
     *         frame. If you specify a value here, it will override any value that you specify in the output setting
     *         Cropping selection (crop).
     */
    public final Rectangle crop() {
        return crop;
    }

    /**
     * Enable Deblock (InputDeblockFilter) to produce smoother motion in the output. Default is disabled. Only manually
     * controllable for MPEG2 and uncompressed video inputs.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #deblockFilter}
     * will return {@link InputDeblockFilter#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #deblockFilterAsString}.
     * </p>
     * 
     * @return Enable Deblock (InputDeblockFilter) to produce smoother motion in the output. Default is disabled. Only
     *         manually controllable for MPEG2 and uncompressed video inputs.
     * @see InputDeblockFilter
     */
    public final InputDeblockFilter deblockFilter() {
        return InputDeblockFilter.fromValue(deblockFilter);
    }

    /**
     * Enable Deblock (InputDeblockFilter) to produce smoother motion in the output. Default is disabled. Only manually
     * controllable for MPEG2 and uncompressed video inputs.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #deblockFilter}
     * will return {@link InputDeblockFilter#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #deblockFilterAsString}.
     * </p>
     * 
     * @return Enable Deblock (InputDeblockFilter) to produce smoother motion in the output. Default is disabled. Only
     *         manually controllable for MPEG2 and uncompressed video inputs.
     * @see InputDeblockFilter
     */
    public final String deblockFilterAsString() {
        return deblockFilter;
    }

    /**
     * Enable Denoise (InputDenoiseFilter) to filter noise from the input. Default is disabled. Only applicable to
     * MPEG2, H.264, H.265, and uncompressed video inputs.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #denoiseFilter}
     * will return {@link InputDenoiseFilter#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #denoiseFilterAsString}.
     * </p>
     * 
     * @return Enable Denoise (InputDenoiseFilter) to filter noise from the input. Default is disabled. Only applicable
     *         to MPEG2, H.264, H.265, and uncompressed video inputs.
     * @see InputDenoiseFilter
     */
    public final InputDenoiseFilter denoiseFilter() {
        return InputDenoiseFilter.fromValue(denoiseFilter);
    }

    /**
     * Enable Denoise (InputDenoiseFilter) to filter noise from the input. Default is disabled. Only applicable to
     * MPEG2, H.264, H.265, and uncompressed video inputs.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #denoiseFilter}
     * will return {@link InputDenoiseFilter#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #denoiseFilterAsString}.
     * </p>
     * 
     * @return Enable Denoise (InputDenoiseFilter) to filter noise from the input. Default is disabled. Only applicable
     *         to MPEG2, H.264, H.265, and uncompressed video inputs.
     * @see InputDenoiseFilter
     */
    public final String denoiseFilterAsString() {
        return denoiseFilter;
    }

    /**
     * Use this setting only when your video source has Dolby Vision studio mastering metadata that is carried in a
     * separate XML file. Specify the Amazon S3 location for the metadata XML file. MediaConvert uses this file to
     * provide global and frame-level metadata for Dolby Vision preprocessing. When you specify a file here and your
     * input also has interleaved global and frame level metadata, MediaConvert ignores the interleaved metadata and
     * uses only the the metadata from this external XML file. Note that your IAM service role must grant MediaConvert
     * read permissions to this file. For more information, see
     * https://docs.aws.amazon.com/mediaconvert/latest/ug/iam-role.html.
     * 
     * @return Use this setting only when your video source has Dolby Vision studio mastering metadata that is carried
     *         in a separate XML file. Specify the Amazon S3 location for the metadata XML file. MediaConvert uses this
     *         file to provide global and frame-level metadata for Dolby Vision preprocessing. When you specify a file
     *         here and your input also has interleaved global and frame level metadata, MediaConvert ignores the
     *         interleaved metadata and uses only the the metadata from this external XML file. Note that your IAM
     *         service role must grant MediaConvert read permissions to this file. For more information, see
     *         https://docs.aws.amazon.com/mediaconvert/latest/ug/iam-role.html.
     */
    public final String dolbyVisionMetadataXml() {
        return dolbyVisionMetadataXml;
    }

    /**
     * Specify whether to apply input filtering to improve the video quality of your input. To apply filtering depending
     * on your input type and quality: Choose Auto. To apply no filtering: Choose Disable. To apply filtering regardless
     * of your input type and quality: Choose Force. When you do, you must also specify a value for Filter strength.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #filterEnable} will
     * return {@link InputFilterEnable#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #filterEnableAsString}.
     * </p>
     * 
     * @return Specify whether to apply input filtering to improve the video quality of your input. To apply filtering
     *         depending on your input type and quality: Choose Auto. To apply no filtering: Choose Disable. To apply
     *         filtering regardless of your input type and quality: Choose Force. When you do, you must also specify a
     *         value for Filter strength.
     * @see InputFilterEnable
     */
    public final InputFilterEnable filterEnable() {
        return InputFilterEnable.fromValue(filterEnable);
    }

    /**
     * Specify whether to apply input filtering to improve the video quality of your input. To apply filtering depending
     * on your input type and quality: Choose Auto. To apply no filtering: Choose Disable. To apply filtering regardless
     * of your input type and quality: Choose Force. When you do, you must also specify a value for Filter strength.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #filterEnable} will
     * return {@link InputFilterEnable#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #filterEnableAsString}.
     * </p>
     * 
     * @return Specify whether to apply input filtering to improve the video quality of your input. To apply filtering
     *         depending on your input type and quality: Choose Auto. To apply no filtering: Choose Disable. To apply
     *         filtering regardless of your input type and quality: Choose Force. When you do, you must also specify a
     *         value for Filter strength.
     * @see InputFilterEnable
     */
    public final String filterEnableAsString() {
        return filterEnable;
    }

    /**
     * Specify the strength of the input filter. To apply an automatic amount of filtering based the compression
     * artifacts measured in your input: We recommend that you leave Filter strength blank and set Filter enable to
     * Auto. To manually apply filtering: Enter a value from 1 to 5, where 1 is the least amount of filtering and 5 is
     * the most. The value that you enter applies to the strength of the Deblock or Denoise filters, or to the strength
     * of the Advanced input filter.
     * 
     * @return Specify the strength of the input filter. To apply an automatic amount of filtering based the compression
     *         artifacts measured in your input: We recommend that you leave Filter strength blank and set Filter enable
     *         to Auto. To manually apply filtering: Enter a value from 1 to 5, where 1 is the least amount of filtering
     *         and 5 is the most. The value that you enter applies to the strength of the Deblock or Denoise filters, or
     *         to the strength of the Advanced input filter.
     */
    public final Integer filterStrength() {
        return filterStrength;
    }

    /**
     * Enable the image inserter feature to include a graphic overlay on your video. Enable or disable this feature for
     * each input individually. This setting is disabled by default.
     * 
     * @return Enable the image inserter feature to include a graphic overlay on your video. Enable or disable this
     *         feature for each input individually. This setting is disabled by default.
     */
    public final ImageInserter imageInserter() {
        return imageInserter;
    }

    /**
     * For responses, this returns true if the service returned a value for the InputClippings property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInputClippings() {
        return inputClippings != null && !(inputClippings instanceof SdkAutoConstructList);
    }

    /**
     * (InputClippings) contains sets of start and end times that together specify a portion of the input to be used in
     * the outputs. If you provide only a start time, the clip will be the entire input from that point to the end. If
     * you provide only an end time, it will be the entire input up to that point. When you specify more than one input
     * clip, the transcoding service creates the job outputs by stringing the clips together in the order you specify
     * them.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInputClippings} method.
     * </p>
     * 
     * @return (InputClippings) contains sets of start and end times that together specify a portion of the input to be
     *         used in the outputs. If you provide only a start time, the clip will be the entire input from that point
     *         to the end. If you provide only an end time, it will be the entire input up to that point. When you
     *         specify more than one input clip, the transcoding service creates the job outputs by stringing the clips
     *         together in the order you specify them.
     */
    public final List<InputClipping> inputClippings() {
        return inputClippings;
    }

    /**
     * When you have a progressive segmented frame (PsF) input, use this setting to flag the input as PsF. MediaConvert
     * doesn't automatically detect PsF. Therefore, flagging your input as PsF results in better preservation of video
     * quality when you do deinterlacing and frame rate conversion. If you don't specify, the default value is Auto
     * (AUTO). Auto is the correct setting for all inputs that are not PsF. Don't set this value to PsF when your input
     * is interlaced. Doing so creates horizontal interlacing artifacts.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #inputScanType}
     * will return {@link InputScanType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #inputScanTypeAsString}.
     * </p>
     * 
     * @return When you have a progressive segmented frame (PsF) input, use this setting to flag the input as PsF.
     *         MediaConvert doesn't automatically detect PsF. Therefore, flagging your input as PsF results in better
     *         preservation of video quality when you do deinterlacing and frame rate conversion. If you don't specify,
     *         the default value is Auto (AUTO). Auto is the correct setting for all inputs that are not PsF. Don't set
     *         this value to PsF when your input is interlaced. Doing so creates horizontal interlacing artifacts.
     * @see InputScanType
     */
    public final InputScanType inputScanType() {
        return InputScanType.fromValue(inputScanType);
    }

    /**
     * When you have a progressive segmented frame (PsF) input, use this setting to flag the input as PsF. MediaConvert
     * doesn't automatically detect PsF. Therefore, flagging your input as PsF results in better preservation of video
     * quality when you do deinterlacing and frame rate conversion. If you don't specify, the default value is Auto
     * (AUTO). Auto is the correct setting for all inputs that are not PsF. Don't set this value to PsF when your input
     * is interlaced. Doing so creates horizontal interlacing artifacts.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #inputScanType}
     * will return {@link InputScanType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #inputScanTypeAsString}.
     * </p>
     * 
     * @return When you have a progressive segmented frame (PsF) input, use this setting to flag the input as PsF.
     *         MediaConvert doesn't automatically detect PsF. Therefore, flagging your input as PsF results in better
     *         preservation of video quality when you do deinterlacing and frame rate conversion. If you don't specify,
     *         the default value is Auto (AUTO). Auto is the correct setting for all inputs that are not PsF. Don't set
     *         this value to PsF when your input is interlaced. Doing so creates horizontal interlacing artifacts.
     * @see InputScanType
     */
    public final String inputScanTypeAsString() {
        return inputScanType;
    }

    /**
     * Use Selection placement (position) to define the video area in your output frame. The area outside of the
     * rectangle that you specify here is black. If you specify a value here, it will override any value that you
     * specify in the output setting Selection placement (position). If you specify a value here, this will override any
     * AFD values in your input, even if you set Respond to AFD (RespondToAfd) to Respond (RESPOND). If you specify a
     * value here, this will ignore anything that you specify for the setting Scaling Behavior (scalingBehavior).
     * 
     * @return Use Selection placement (position) to define the video area in your output frame. The area outside of the
     *         rectangle that you specify here is black. If you specify a value here, it will override any value that
     *         you specify in the output setting Selection placement (position). If you specify a value here, this will
     *         override any AFD values in your input, even if you set Respond to AFD (RespondToAfd) to Respond
     *         (RESPOND). If you specify a value here, this will ignore anything that you specify for the setting
     *         Scaling Behavior (scalingBehavior).
     */
    public final Rectangle position() {
        return position;
    }

    /**
     * Use Program (programNumber) to select a specific program from within a multi-program transport stream. Note that
     * Quad 4K is not currently supported. Default is the first program within the transport stream. If the program you
     * specify doesn't exist, the transcoding service will use this default.
     * 
     * @return Use Program (programNumber) to select a specific program from within a multi-program transport stream.
     *         Note that Quad 4K is not currently supported. Default is the first program within the transport stream.
     *         If the program you specify doesn't exist, the transcoding service will use this default.
     */
    public final Integer programNumber() {
        return programNumber;
    }

    /**
     * Set PSI control (InputPsiControl) for transport stream inputs to specify which data the demux process to scans. *
     * Ignore PSI - Scan all PIDs for audio and video. * Use PSI - Scan only PSI data.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #psiControl} will
     * return {@link InputPsiControl#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #psiControlAsString}.
     * </p>
     * 
     * @return Set PSI control (InputPsiControl) for transport stream inputs to specify which data the demux process to
     *         scans. * Ignore PSI - Scan all PIDs for audio and video. * Use PSI - Scan only PSI data.
     * @see InputPsiControl
     */
    public final InputPsiControl psiControl() {
        return InputPsiControl.fromValue(psiControl);
    }

    /**
     * Set PSI control (InputPsiControl) for transport stream inputs to specify which data the demux process to scans. *
     * Ignore PSI - Scan all PIDs for audio and video. * Use PSI - Scan only PSI data.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #psiControl} will
     * return {@link InputPsiControl#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #psiControlAsString}.
     * </p>
     * 
     * @return Set PSI control (InputPsiControl) for transport stream inputs to specify which data the demux process to
     *         scans. * Ignore PSI - Scan all PIDs for audio and video. * Use PSI - Scan only PSI data.
     * @see InputPsiControl
     */
    public final String psiControlAsString() {
        return psiControl;
    }

    /**
     * Use this Timecode source setting, located under the input settings (InputTimecodeSource), to specify how the
     * service counts input video frames. This input frame count affects only the behavior of features that apply to a
     * single input at a time, such as input clipping and synchronizing some captions formats. Choose Embedded
     * (EMBEDDED) to use the timecodes in your input video. Choose Start at zero (ZEROBASED) to start the first frame at
     * zero. Choose Specified start (SPECIFIEDSTART) to start the first frame at the timecode that you specify in the
     * setting Start timecode (timecodeStart). If you don't specify a value for Timecode source, the service will use
     * Embedded by default. For more information about timecodes, see
     * https://docs.aws.amazon.com/console/mediaconvert/timecode.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #timecodeSource}
     * will return {@link InputTimecodeSource#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #timecodeSourceAsString}.
     * </p>
     * 
     * @return Use this Timecode source setting, located under the input settings (InputTimecodeSource), to specify how
     *         the service counts input video frames. This input frame count affects only the behavior of features that
     *         apply to a single input at a time, such as input clipping and synchronizing some captions formats. Choose
     *         Embedded (EMBEDDED) to use the timecodes in your input video. Choose Start at zero (ZEROBASED) to start
     *         the first frame at zero. Choose Specified start (SPECIFIEDSTART) to start the first frame at the timecode
     *         that you specify in the setting Start timecode (timecodeStart). If you don't specify a value for Timecode
     *         source, the service will use Embedded by default. For more information about timecodes, see
     *         https://docs.aws.amazon.com/console/mediaconvert/timecode.
     * @see InputTimecodeSource
     */
    public final InputTimecodeSource timecodeSource() {
        return InputTimecodeSource.fromValue(timecodeSource);
    }

    /**
     * Use this Timecode source setting, located under the input settings (InputTimecodeSource), to specify how the
     * service counts input video frames. This input frame count affects only the behavior of features that apply to a
     * single input at a time, such as input clipping and synchronizing some captions formats. Choose Embedded
     * (EMBEDDED) to use the timecodes in your input video. Choose Start at zero (ZEROBASED) to start the first frame at
     * zero. Choose Specified start (SPECIFIEDSTART) to start the first frame at the timecode that you specify in the
     * setting Start timecode (timecodeStart). If you don't specify a value for Timecode source, the service will use
     * Embedded by default. For more information about timecodes, see
     * https://docs.aws.amazon.com/console/mediaconvert/timecode.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #timecodeSource}
     * will return {@link InputTimecodeSource#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #timecodeSourceAsString}.
     * </p>
     * 
     * @return Use this Timecode source setting, located under the input settings (InputTimecodeSource), to specify how
     *         the service counts input video frames. This input frame count affects only the behavior of features that
     *         apply to a single input at a time, such as input clipping and synchronizing some captions formats. Choose
     *         Embedded (EMBEDDED) to use the timecodes in your input video. Choose Start at zero (ZEROBASED) to start
     *         the first frame at zero. Choose Specified start (SPECIFIEDSTART) to start the first frame at the timecode
     *         that you specify in the setting Start timecode (timecodeStart). If you don't specify a value for Timecode
     *         source, the service will use Embedded by default. For more information about timecodes, see
     *         https://docs.aws.amazon.com/console/mediaconvert/timecode.
     * @see InputTimecodeSource
     */
    public final String timecodeSourceAsString() {
        return timecodeSource;
    }

    /**
     * Specify the timecode that you want the service to use for this input's initial frame. To use this setting, you
     * must set the Timecode source setting, located under the input settings (InputTimecodeSource), to Specified start
     * (SPECIFIEDSTART). For more information about timecodes, see
     * https://docs.aws.amazon.com/console/mediaconvert/timecode.
     * 
     * @return Specify the timecode that you want the service to use for this input's initial frame. To use this
     *         setting, you must set the Timecode source setting, located under the input settings
     *         (InputTimecodeSource), to Specified start (SPECIFIEDSTART). For more information about timecodes, see
     *         https://docs.aws.amazon.com/console/mediaconvert/timecode.
     */
    public final String timecodeStart() {
        return timecodeStart;
    }

    /**
     * Input video selectors contain the video settings for the input. Each of your inputs can have up to one video
     * selector.
     * 
     * @return Input video selectors contain the video settings for the input. Each of your inputs can have up to one
     *         video selector.
     */
    public final VideoSelector videoSelector() {
        return videoSelector;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(advancedInputFilterAsString());
        hashCode = 31 * hashCode + Objects.hashCode(advancedInputFilterSettings());
        hashCode = 31 * hashCode + Objects.hashCode(hasAudioSelectorGroups() ? audioSelectorGroups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAudioSelectors() ? audioSelectors() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasCaptionSelectors() ? captionSelectors() : null);
        hashCode = 31 * hashCode + Objects.hashCode(crop());
        hashCode = 31 * hashCode + Objects.hashCode(deblockFilterAsString());
        hashCode = 31 * hashCode + Objects.hashCode(denoiseFilterAsString());
        hashCode = 31 * hashCode + Objects.hashCode(dolbyVisionMetadataXml());
        hashCode = 31 * hashCode + Objects.hashCode(filterEnableAsString());
        hashCode = 31 * hashCode + Objects.hashCode(filterStrength());
        hashCode = 31 * hashCode + Objects.hashCode(imageInserter());
        hashCode = 31 * hashCode + Objects.hashCode(hasInputClippings() ? inputClippings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(inputScanTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(position());
        hashCode = 31 * hashCode + Objects.hashCode(programNumber());
        hashCode = 31 * hashCode + Objects.hashCode(psiControlAsString());
        hashCode = 31 * hashCode + Objects.hashCode(timecodeSourceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(timecodeStart());
        hashCode = 31 * hashCode + Objects.hashCode(videoSelector());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InputTemplate)) {
            return false;
        }
        InputTemplate other = (InputTemplate) obj;
        return Objects.equals(advancedInputFilterAsString(), other.advancedInputFilterAsString())
                && Objects.equals(advancedInputFilterSettings(), other.advancedInputFilterSettings())
                && hasAudioSelectorGroups() == other.hasAudioSelectorGroups()
                && Objects.equals(audioSelectorGroups(), other.audioSelectorGroups())
                && hasAudioSelectors() == other.hasAudioSelectors() && Objects.equals(audioSelectors(), other.audioSelectors())
                && hasCaptionSelectors() == other.hasCaptionSelectors()
                && Objects.equals(captionSelectors(), other.captionSelectors()) && Objects.equals(crop(), other.crop())
                && Objects.equals(deblockFilterAsString(), other.deblockFilterAsString())
                && Objects.equals(denoiseFilterAsString(), other.denoiseFilterAsString())
                && Objects.equals(dolbyVisionMetadataXml(), other.dolbyVisionMetadataXml())
                && Objects.equals(filterEnableAsString(), other.filterEnableAsString())
                && Objects.equals(filterStrength(), other.filterStrength())
                && Objects.equals(imageInserter(), other.imageInserter()) && hasInputClippings() == other.hasInputClippings()
                && Objects.equals(inputClippings(), other.inputClippings())
                && Objects.equals(inputScanTypeAsString(), other.inputScanTypeAsString())
                && Objects.equals(position(), other.position()) && Objects.equals(programNumber(), other.programNumber())
                && Objects.equals(psiControlAsString(), other.psiControlAsString())
                && Objects.equals(timecodeSourceAsString(), other.timecodeSourceAsString())
                && Objects.equals(timecodeStart(), other.timecodeStart())
                && Objects.equals(videoSelector(), other.videoSelector());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InputTemplate").add("AdvancedInputFilter", advancedInputFilterAsString())
                .add("AdvancedInputFilterSettings", advancedInputFilterSettings())
                .add("AudioSelectorGroups", hasAudioSelectorGroups() ? audioSelectorGroups() : null)
                .add("AudioSelectors", hasAudioSelectors() ? audioSelectors() : null)
                .add("CaptionSelectors", hasCaptionSelectors() ? captionSelectors() : null).add("Crop", crop())
                .add("DeblockFilter", deblockFilterAsString()).add("DenoiseFilter", denoiseFilterAsString())
                .add("DolbyVisionMetadataXml", dolbyVisionMetadataXml()).add("FilterEnable", filterEnableAsString())
                .add("FilterStrength", filterStrength()).add("ImageInserter", imageInserter())
                .add("InputClippings", hasInputClippings() ? inputClippings() : null)
                .add("InputScanType", inputScanTypeAsString()).add("Position", position()).add("ProgramNumber", programNumber())
                .add("PsiControl", psiControlAsString()).add("TimecodeSource", timecodeSourceAsString())
                .add("TimecodeStart", timecodeStart()).add("VideoSelector", videoSelector()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AdvancedInputFilter":
            return Optional.ofNullable(clazz.cast(advancedInputFilterAsString()));
        case "AdvancedInputFilterSettings":
            return Optional.ofNullable(clazz.cast(advancedInputFilterSettings()));
        case "AudioSelectorGroups":
            return Optional.ofNullable(clazz.cast(audioSelectorGroups()));
        case "AudioSelectors":
            return Optional.ofNullable(clazz.cast(audioSelectors()));
        case "CaptionSelectors":
            return Optional.ofNullable(clazz.cast(captionSelectors()));
        case "Crop":
            return Optional.ofNullable(clazz.cast(crop()));
        case "DeblockFilter":
            return Optional.ofNullable(clazz.cast(deblockFilterAsString()));
        case "DenoiseFilter":
            return Optional.ofNullable(clazz.cast(denoiseFilterAsString()));
        case "DolbyVisionMetadataXml":
            return Optional.ofNullable(clazz.cast(dolbyVisionMetadataXml()));
        case "FilterEnable":
            return Optional.ofNullable(clazz.cast(filterEnableAsString()));
        case "FilterStrength":
            return Optional.ofNullable(clazz.cast(filterStrength()));
        case "ImageInserter":
            return Optional.ofNullable(clazz.cast(imageInserter()));
        case "InputClippings":
            return Optional.ofNullable(clazz.cast(inputClippings()));
        case "InputScanType":
            return Optional.ofNullable(clazz.cast(inputScanTypeAsString()));
        case "Position":
            return Optional.ofNullable(clazz.cast(position()));
        case "ProgramNumber":
            return Optional.ofNullable(clazz.cast(programNumber()));
        case "PsiControl":
            return Optional.ofNullable(clazz.cast(psiControlAsString()));
        case "TimecodeSource":
            return Optional.ofNullable(clazz.cast(timecodeSourceAsString()));
        case "TimecodeStart":
            return Optional.ofNullable(clazz.cast(timecodeStart()));
        case "VideoSelector":
            return Optional.ofNullable(clazz.cast(videoSelector()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InputTemplate, T> g) {
        return obj -> g.apply((InputTemplate) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InputTemplate> {
        /**
         * Use to remove noise, blocking, blurriness, or ringing from your input as a pre-filter step before encoding.
         * The Advanced input filter removes more types of compression artifacts and is an improvement when compared to
         * basic Deblock and Denoise filters. To remove video compression artifacts from your input and improve the
         * video quality: Choose Enabled. Additionally, this filter can help increase the video quality of your output
         * relative to its bitrate, since noisy inputs are more complex and require more bits to encode. To help restore
         * loss of detail after applying the filter, you can optionally add texture or sharpening as an additional step.
         * Jobs that use this feature incur pro-tier pricing. To not apply advanced input filtering: Choose Disabled.
         * Note that you can still apply basic filtering with Deblock and Denoise.
         * 
         * @param advancedInputFilter
         *        Use to remove noise, blocking, blurriness, or ringing from your input as a pre-filter step before
         *        encoding. The Advanced input filter removes more types of compression artifacts and is an improvement
         *        when compared to basic Deblock and Denoise filters. To remove video compression artifacts from your
         *        input and improve the video quality: Choose Enabled. Additionally, this filter can help increase the
         *        video quality of your output relative to its bitrate, since noisy inputs are more complex and require
         *        more bits to encode. To help restore loss of detail after applying the filter, you can optionally add
         *        texture or sharpening as an additional step. Jobs that use this feature incur pro-tier pricing. To not
         *        apply advanced input filtering: Choose Disabled. Note that you can still apply basic filtering with
         *        Deblock and Denoise.
         * @see AdvancedInputFilter
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AdvancedInputFilter
         */
        Builder advancedInputFilter(String advancedInputFilter);

        /**
         * Use to remove noise, blocking, blurriness, or ringing from your input as a pre-filter step before encoding.
         * The Advanced input filter removes more types of compression artifacts and is an improvement when compared to
         * basic Deblock and Denoise filters. To remove video compression artifacts from your input and improve the
         * video quality: Choose Enabled. Additionally, this filter can help increase the video quality of your output
         * relative to its bitrate, since noisy inputs are more complex and require more bits to encode. To help restore
         * loss of detail after applying the filter, you can optionally add texture or sharpening as an additional step.
         * Jobs that use this feature incur pro-tier pricing. To not apply advanced input filtering: Choose Disabled.
         * Note that you can still apply basic filtering with Deblock and Denoise.
         * 
         * @param advancedInputFilter
         *        Use to remove noise, blocking, blurriness, or ringing from your input as a pre-filter step before
         *        encoding. The Advanced input filter removes more types of compression artifacts and is an improvement
         *        when compared to basic Deblock and Denoise filters. To remove video compression artifacts from your
         *        input and improve the video quality: Choose Enabled. Additionally, this filter can help increase the
         *        video quality of your output relative to its bitrate, since noisy inputs are more complex and require
         *        more bits to encode. To help restore loss of detail after applying the filter, you can optionally add
         *        texture or sharpening as an additional step. Jobs that use this feature incur pro-tier pricing. To not
         *        apply advanced input filtering: Choose Disabled. Note that you can still apply basic filtering with
         *        Deblock and Denoise.
         * @see AdvancedInputFilter
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AdvancedInputFilter
         */
        Builder advancedInputFilter(AdvancedInputFilter advancedInputFilter);

        /**
         * Optional settings for Advanced input filter when you set Advanced input filter to Enabled.
         * 
         * @param advancedInputFilterSettings
         *        Optional settings for Advanced input filter when you set Advanced input filter to Enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder advancedInputFilterSettings(AdvancedInputFilterSettings advancedInputFilterSettings);

        /**
         * Optional settings for Advanced input filter when you set Advanced input filter to Enabled. This is a
         * convenience method that creates an instance of the {@link AdvancedInputFilterSettings.Builder} avoiding the
         * need to create one manually via {@link AdvancedInputFilterSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AdvancedInputFilterSettings.Builder#build()} is called
         * immediately and its result is passed to {@link #advancedInputFilterSettings(AdvancedInputFilterSettings)}.
         * 
         * @param advancedInputFilterSettings
         *        a consumer that will call methods on {@link AdvancedInputFilterSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #advancedInputFilterSettings(AdvancedInputFilterSettings)
         */
        default Builder advancedInputFilterSettings(Consumer<AdvancedInputFilterSettings.Builder> advancedInputFilterSettings) {
            return advancedInputFilterSettings(AdvancedInputFilterSettings.builder().applyMutation(advancedInputFilterSettings)
                    .build());
        }

        /**
         * Use audio selector groups to combine multiple sidecar audio inputs so that you can assign them to a single
         * output audio tab (AudioDescription). Note that, if you're working with embedded audio, it's simpler to assign
         * multiple input tracks into a single audio selector rather than use an audio selector group.
         * 
         * @param audioSelectorGroups
         *        Use audio selector groups to combine multiple sidecar audio inputs so that you can assign them to a
         *        single output audio tab (AudioDescription). Note that, if you're working with embedded audio, it's
         *        simpler to assign multiple input tracks into a single audio selector rather than use an audio selector
         *        group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder audioSelectorGroups(Map<String, AudioSelectorGroup> audioSelectorGroups);

        /**
         * Use Audio selectors (AudioSelectors) to specify a track or set of tracks from the input that you will use in
         * your outputs. You can use multiple Audio selectors per input.
         * 
         * @param audioSelectors
         *        Use Audio selectors (AudioSelectors) to specify a track or set of tracks from the input that you will
         *        use in your outputs. You can use multiple Audio selectors per input.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder audioSelectors(Map<String, AudioSelector> audioSelectors);

        /**
         * Use captions selectors to specify the captions data from your input that you use in your outputs. You can use
         * up to 100 captions selectors per input.
         * 
         * @param captionSelectors
         *        Use captions selectors to specify the captions data from your input that you use in your outputs. You
         *        can use up to 100 captions selectors per input.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder captionSelectors(Map<String, CaptionSelector> captionSelectors);

        /**
         * Use Cropping selection (crop) to specify the video area that the service will include in the output video
         * frame. If you specify a value here, it will override any value that you specify in the output setting
         * Cropping selection (crop).
         * 
         * @param crop
         *        Use Cropping selection (crop) to specify the video area that the service will include in the output
         *        video frame. If you specify a value here, it will override any value that you specify in the output
         *        setting Cropping selection (crop).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder crop(Rectangle crop);

        /**
         * Use Cropping selection (crop) to specify the video area that the service will include in the output video
         * frame. If you specify a value here, it will override any value that you specify in the output setting
         * Cropping selection (crop). This is a convenience method that creates an instance of the
         * {@link Rectangle.Builder} avoiding the need to create one manually via {@link Rectangle#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Rectangle.Builder#build()} is called immediately and its result
         * is passed to {@link #crop(Rectangle)}.
         * 
         * @param crop
         *        a consumer that will call methods on {@link Rectangle.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #crop(Rectangle)
         */
        default Builder crop(Consumer<Rectangle.Builder> crop) {
            return crop(Rectangle.builder().applyMutation(crop).build());
        }

        /**
         * Enable Deblock (InputDeblockFilter) to produce smoother motion in the output. Default is disabled. Only
         * manually controllable for MPEG2 and uncompressed video inputs.
         * 
         * @param deblockFilter
         *        Enable Deblock (InputDeblockFilter) to produce smoother motion in the output. Default is disabled.
         *        Only manually controllable for MPEG2 and uncompressed video inputs.
         * @see InputDeblockFilter
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InputDeblockFilter
         */
        Builder deblockFilter(String deblockFilter);

        /**
         * Enable Deblock (InputDeblockFilter) to produce smoother motion in the output. Default is disabled. Only
         * manually controllable for MPEG2 and uncompressed video inputs.
         * 
         * @param deblockFilter
         *        Enable Deblock (InputDeblockFilter) to produce smoother motion in the output. Default is disabled.
         *        Only manually controllable for MPEG2 and uncompressed video inputs.
         * @see InputDeblockFilter
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InputDeblockFilter
         */
        Builder deblockFilter(InputDeblockFilter deblockFilter);

        /**
         * Enable Denoise (InputDenoiseFilter) to filter noise from the input. Default is disabled. Only applicable to
         * MPEG2, H.264, H.265, and uncompressed video inputs.
         * 
         * @param denoiseFilter
         *        Enable Denoise (InputDenoiseFilter) to filter noise from the input. Default is disabled. Only
         *        applicable to MPEG2, H.264, H.265, and uncompressed video inputs.
         * @see InputDenoiseFilter
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InputDenoiseFilter
         */
        Builder denoiseFilter(String denoiseFilter);

        /**
         * Enable Denoise (InputDenoiseFilter) to filter noise from the input. Default is disabled. Only applicable to
         * MPEG2, H.264, H.265, and uncompressed video inputs.
         * 
         * @param denoiseFilter
         *        Enable Denoise (InputDenoiseFilter) to filter noise from the input. Default is disabled. Only
         *        applicable to MPEG2, H.264, H.265, and uncompressed video inputs.
         * @see InputDenoiseFilter
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InputDenoiseFilter
         */
        Builder denoiseFilter(InputDenoiseFilter denoiseFilter);

        /**
         * Use this setting only when your video source has Dolby Vision studio mastering metadata that is carried in a
         * separate XML file. Specify the Amazon S3 location for the metadata XML file. MediaConvert uses this file to
         * provide global and frame-level metadata for Dolby Vision preprocessing. When you specify a file here and your
         * input also has interleaved global and frame level metadata, MediaConvert ignores the interleaved metadata and
         * uses only the the metadata from this external XML file. Note that your IAM service role must grant
         * MediaConvert read permissions to this file. For more information, see
         * https://docs.aws.amazon.com/mediaconvert/latest/ug/iam-role.html.
         * 
         * @param dolbyVisionMetadataXml
         *        Use this setting only when your video source has Dolby Vision studio mastering metadata that is
         *        carried in a separate XML file. Specify the Amazon S3 location for the metadata XML file. MediaConvert
         *        uses this file to provide global and frame-level metadata for Dolby Vision preprocessing. When you
         *        specify a file here and your input also has interleaved global and frame level metadata, MediaConvert
         *        ignores the interleaved metadata and uses only the the metadata from this external XML file. Note that
         *        your IAM service role must grant MediaConvert read permissions to this file. For more information, see
         *        https://docs.aws.amazon.com/mediaconvert/latest/ug/iam-role.html.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dolbyVisionMetadataXml(String dolbyVisionMetadataXml);

        /**
         * Specify whether to apply input filtering to improve the video quality of your input. To apply filtering
         * depending on your input type and quality: Choose Auto. To apply no filtering: Choose Disable. To apply
         * filtering regardless of your input type and quality: Choose Force. When you do, you must also specify a value
         * for Filter strength.
         * 
         * @param filterEnable
         *        Specify whether to apply input filtering to improve the video quality of your input. To apply
         *        filtering depending on your input type and quality: Choose Auto. To apply no filtering: Choose
         *        Disable. To apply filtering regardless of your input type and quality: Choose Force. When you do, you
         *        must also specify a value for Filter strength.
         * @see InputFilterEnable
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InputFilterEnable
         */
        Builder filterEnable(String filterEnable);

        /**
         * Specify whether to apply input filtering to improve the video quality of your input. To apply filtering
         * depending on your input type and quality: Choose Auto. To apply no filtering: Choose Disable. To apply
         * filtering regardless of your input type and quality: Choose Force. When you do, you must also specify a value
         * for Filter strength.
         * 
         * @param filterEnable
         *        Specify whether to apply input filtering to improve the video quality of your input. To apply
         *        filtering depending on your input type and quality: Choose Auto. To apply no filtering: Choose
         *        Disable. To apply filtering regardless of your input type and quality: Choose Force. When you do, you
         *        must also specify a value for Filter strength.
         * @see InputFilterEnable
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InputFilterEnable
         */
        Builder filterEnable(InputFilterEnable filterEnable);

        /**
         * Specify the strength of the input filter. To apply an automatic amount of filtering based the compression
         * artifacts measured in your input: We recommend that you leave Filter strength blank and set Filter enable to
         * Auto. To manually apply filtering: Enter a value from 1 to 5, where 1 is the least amount of filtering and 5
         * is the most. The value that you enter applies to the strength of the Deblock or Denoise filters, or to the
         * strength of the Advanced input filter.
         * 
         * @param filterStrength
         *        Specify the strength of the input filter. To apply an automatic amount of filtering based the
         *        compression artifacts measured in your input: We recommend that you leave Filter strength blank and
         *        set Filter enable to Auto. To manually apply filtering: Enter a value from 1 to 5, where 1 is the
         *        least amount of filtering and 5 is the most. The value that you enter applies to the strength of the
         *        Deblock or Denoise filters, or to the strength of the Advanced input filter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filterStrength(Integer filterStrength);

        /**
         * Enable the image inserter feature to include a graphic overlay on your video. Enable or disable this feature
         * for each input individually. This setting is disabled by default.
         * 
         * @param imageInserter
         *        Enable the image inserter feature to include a graphic overlay on your video. Enable or disable this
         *        feature for each input individually. This setting is disabled by default.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageInserter(ImageInserter imageInserter);

        /**
         * Enable the image inserter feature to include a graphic overlay on your video. Enable or disable this feature
         * for each input individually. This setting is disabled by default. This is a convenience method that creates
         * an instance of the {@link ImageInserter.Builder} avoiding the need to create one manually via
         * {@link ImageInserter#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ImageInserter.Builder#build()} is called immediately and its
         * result is passed to {@link #imageInserter(ImageInserter)}.
         * 
         * @param imageInserter
         *        a consumer that will call methods on {@link ImageInserter.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #imageInserter(ImageInserter)
         */
        default Builder imageInserter(Consumer<ImageInserter.Builder> imageInserter) {
            return imageInserter(ImageInserter.builder().applyMutation(imageInserter).build());
        }

        /**
         * (InputClippings) contains sets of start and end times that together specify a portion of the input to be used
         * in the outputs. If you provide only a start time, the clip will be the entire input from that point to the
         * end. If you provide only an end time, it will be the entire input up to that point. When you specify more
         * than one input clip, the transcoding service creates the job outputs by stringing the clips together in the
         * order you specify them.
         * 
         * @param inputClippings
         *        (InputClippings) contains sets of start and end times that together specify a portion of the input to
         *        be used in the outputs. If you provide only a start time, the clip will be the entire input from that
         *        point to the end. If you provide only an end time, it will be the entire input up to that point. When
         *        you specify more than one input clip, the transcoding service creates the job outputs by stringing the
         *        clips together in the order you specify them.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputClippings(Collection<InputClipping> inputClippings);

        /**
         * (InputClippings) contains sets of start and end times that together specify a portion of the input to be used
         * in the outputs. If you provide only a start time, the clip will be the entire input from that point to the
         * end. If you provide only an end time, it will be the entire input up to that point. When you specify more
         * than one input clip, the transcoding service creates the job outputs by stringing the clips together in the
         * order you specify them.
         * 
         * @param inputClippings
         *        (InputClippings) contains sets of start and end times that together specify a portion of the input to
         *        be used in the outputs. If you provide only a start time, the clip will be the entire input from that
         *        point to the end. If you provide only an end time, it will be the entire input up to that point. When
         *        you specify more than one input clip, the transcoding service creates the job outputs by stringing the
         *        clips together in the order you specify them.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputClippings(InputClipping... inputClippings);

        /**
         * (InputClippings) contains sets of start and end times that together specify a portion of the input to be used
         * in the outputs. If you provide only a start time, the clip will be the entire input from that point to the
         * end. If you provide only an end time, it will be the entire input up to that point. When you specify more
         * than one input clip, the transcoding service creates the job outputs by stringing the clips together in the
         * order you specify them. This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.mediaconvert.model.InputClipping.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.mediaconvert.model.InputClipping#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.mediaconvert.model.InputClipping.Builder#build()} is called
         * immediately and its result is passed to {@link #inputClippings(List<InputClipping>)}.
         * 
         * @param inputClippings
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.mediaconvert.model.InputClipping.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #inputClippings(java.util.Collection<InputClipping>)
         */
        Builder inputClippings(Consumer<InputClipping.Builder>... inputClippings);

        /**
         * When you have a progressive segmented frame (PsF) input, use this setting to flag the input as PsF.
         * MediaConvert doesn't automatically detect PsF. Therefore, flagging your input as PsF results in better
         * preservation of video quality when you do deinterlacing and frame rate conversion. If you don't specify, the
         * default value is Auto (AUTO). Auto is the correct setting for all inputs that are not PsF. Don't set this
         * value to PsF when your input is interlaced. Doing so creates horizontal interlacing artifacts.
         * 
         * @param inputScanType
         *        When you have a progressive segmented frame (PsF) input, use this setting to flag the input as PsF.
         *        MediaConvert doesn't automatically detect PsF. Therefore, flagging your input as PsF results in better
         *        preservation of video quality when you do deinterlacing and frame rate conversion. If you don't
         *        specify, the default value is Auto (AUTO). Auto is the correct setting for all inputs that are not
         *        PsF. Don't set this value to PsF when your input is interlaced. Doing so creates horizontal
         *        interlacing artifacts.
         * @see InputScanType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InputScanType
         */
        Builder inputScanType(String inputScanType);

        /**
         * When you have a progressive segmented frame (PsF) input, use this setting to flag the input as PsF.
         * MediaConvert doesn't automatically detect PsF. Therefore, flagging your input as PsF results in better
         * preservation of video quality when you do deinterlacing and frame rate conversion. If you don't specify, the
         * default value is Auto (AUTO). Auto is the correct setting for all inputs that are not PsF. Don't set this
         * value to PsF when your input is interlaced. Doing so creates horizontal interlacing artifacts.
         * 
         * @param inputScanType
         *        When you have a progressive segmented frame (PsF) input, use this setting to flag the input as PsF.
         *        MediaConvert doesn't automatically detect PsF. Therefore, flagging your input as PsF results in better
         *        preservation of video quality when you do deinterlacing and frame rate conversion. If you don't
         *        specify, the default value is Auto (AUTO). Auto is the correct setting for all inputs that are not
         *        PsF. Don't set this value to PsF when your input is interlaced. Doing so creates horizontal
         *        interlacing artifacts.
         * @see InputScanType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InputScanType
         */
        Builder inputScanType(InputScanType inputScanType);

        /**
         * Use Selection placement (position) to define the video area in your output frame. The area outside of the
         * rectangle that you specify here is black. If you specify a value here, it will override any value that you
         * specify in the output setting Selection placement (position). If you specify a value here, this will override
         * any AFD values in your input, even if you set Respond to AFD (RespondToAfd) to Respond (RESPOND). If you
         * specify a value here, this will ignore anything that you specify for the setting Scaling Behavior
         * (scalingBehavior).
         * 
         * @param position
         *        Use Selection placement (position) to define the video area in your output frame. The area outside of
         *        the rectangle that you specify here is black. If you specify a value here, it will override any value
         *        that you specify in the output setting Selection placement (position). If you specify a value here,
         *        this will override any AFD values in your input, even if you set Respond to AFD (RespondToAfd) to
         *        Respond (RESPOND). If you specify a value here, this will ignore anything that you specify for the
         *        setting Scaling Behavior (scalingBehavior).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder position(Rectangle position);

        /**
         * Use Selection placement (position) to define the video area in your output frame. The area outside of the
         * rectangle that you specify here is black. If you specify a value here, it will override any value that you
         * specify in the output setting Selection placement (position). If you specify a value here, this will override
         * any AFD values in your input, even if you set Respond to AFD (RespondToAfd) to Respond (RESPOND). If you
         * specify a value here, this will ignore anything that you specify for the setting Scaling Behavior
         * (scalingBehavior). This is a convenience method that creates an instance of the {@link Rectangle.Builder}
         * avoiding the need to create one manually via {@link Rectangle#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Rectangle.Builder#build()} is called immediately and its result
         * is passed to {@link #position(Rectangle)}.
         * 
         * @param position
         *        a consumer that will call methods on {@link Rectangle.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #position(Rectangle)
         */
        default Builder position(Consumer<Rectangle.Builder> position) {
            return position(Rectangle.builder().applyMutation(position).build());
        }

        /**
         * Use Program (programNumber) to select a specific program from within a multi-program transport stream. Note
         * that Quad 4K is not currently supported. Default is the first program within the transport stream. If the
         * program you specify doesn't exist, the transcoding service will use this default.
         * 
         * @param programNumber
         *        Use Program (programNumber) to select a specific program from within a multi-program transport stream.
         *        Note that Quad 4K is not currently supported. Default is the first program within the transport
         *        stream. If the program you specify doesn't exist, the transcoding service will use this default.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder programNumber(Integer programNumber);

        /**
         * Set PSI control (InputPsiControl) for transport stream inputs to specify which data the demux process to
         * scans. * Ignore PSI - Scan all PIDs for audio and video. * Use PSI - Scan only PSI data.
         * 
         * @param psiControl
         *        Set PSI control (InputPsiControl) for transport stream inputs to specify which data the demux process
         *        to scans. * Ignore PSI - Scan all PIDs for audio and video. * Use PSI - Scan only PSI data.
         * @see InputPsiControl
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InputPsiControl
         */
        Builder psiControl(String psiControl);

        /**
         * Set PSI control (InputPsiControl) for transport stream inputs to specify which data the demux process to
         * scans. * Ignore PSI - Scan all PIDs for audio and video. * Use PSI - Scan only PSI data.
         * 
         * @param psiControl
         *        Set PSI control (InputPsiControl) for transport stream inputs to specify which data the demux process
         *        to scans. * Ignore PSI - Scan all PIDs for audio and video. * Use PSI - Scan only PSI data.
         * @see InputPsiControl
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InputPsiControl
         */
        Builder psiControl(InputPsiControl psiControl);

        /**
         * Use this Timecode source setting, located under the input settings (InputTimecodeSource), to specify how the
         * service counts input video frames. This input frame count affects only the behavior of features that apply to
         * a single input at a time, such as input clipping and synchronizing some captions formats. Choose Embedded
         * (EMBEDDED) to use the timecodes in your input video. Choose Start at zero (ZEROBASED) to start the first
         * frame at zero. Choose Specified start (SPECIFIEDSTART) to start the first frame at the timecode that you
         * specify in the setting Start timecode (timecodeStart). If you don't specify a value for Timecode source, the
         * service will use Embedded by default. For more information about timecodes, see
         * https://docs.aws.amazon.com/console/mediaconvert/timecode.
         * 
         * @param timecodeSource
         *        Use this Timecode source setting, located under the input settings (InputTimecodeSource), to specify
         *        how the service counts input video frames. This input frame count affects only the behavior of
         *        features that apply to a single input at a time, such as input clipping and synchronizing some
         *        captions formats. Choose Embedded (EMBEDDED) to use the timecodes in your input video. Choose Start at
         *        zero (ZEROBASED) to start the first frame at zero. Choose Specified start (SPECIFIEDSTART) to start
         *        the first frame at the timecode that you specify in the setting Start timecode (timecodeStart). If you
         *        don't specify a value for Timecode source, the service will use Embedded by default. For more
         *        information about timecodes, see https://docs.aws.amazon.com/console/mediaconvert/timecode.
         * @see InputTimecodeSource
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InputTimecodeSource
         */
        Builder timecodeSource(String timecodeSource);

        /**
         * Use this Timecode source setting, located under the input settings (InputTimecodeSource), to specify how the
         * service counts input video frames. This input frame count affects only the behavior of features that apply to
         * a single input at a time, such as input clipping and synchronizing some captions formats. Choose Embedded
         * (EMBEDDED) to use the timecodes in your input video. Choose Start at zero (ZEROBASED) to start the first
         * frame at zero. Choose Specified start (SPECIFIEDSTART) to start the first frame at the timecode that you
         * specify in the setting Start timecode (timecodeStart). If you don't specify a value for Timecode source, the
         * service will use Embedded by default. For more information about timecodes, see
         * https://docs.aws.amazon.com/console/mediaconvert/timecode.
         * 
         * @param timecodeSource
         *        Use this Timecode source setting, located under the input settings (InputTimecodeSource), to specify
         *        how the service counts input video frames. This input frame count affects only the behavior of
         *        features that apply to a single input at a time, such as input clipping and synchronizing some
         *        captions formats. Choose Embedded (EMBEDDED) to use the timecodes in your input video. Choose Start at
         *        zero (ZEROBASED) to start the first frame at zero. Choose Specified start (SPECIFIEDSTART) to start
         *        the first frame at the timecode that you specify in the setting Start timecode (timecodeStart). If you
         *        don't specify a value for Timecode source, the service will use Embedded by default. For more
         *        information about timecodes, see https://docs.aws.amazon.com/console/mediaconvert/timecode.
         * @see InputTimecodeSource
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InputTimecodeSource
         */
        Builder timecodeSource(InputTimecodeSource timecodeSource);

        /**
         * Specify the timecode that you want the service to use for this input's initial frame. To use this setting,
         * you must set the Timecode source setting, located under the input settings (InputTimecodeSource), to
         * Specified start (SPECIFIEDSTART). For more information about timecodes, see
         * https://docs.aws.amazon.com/console/mediaconvert/timecode.
         * 
         * @param timecodeStart
         *        Specify the timecode that you want the service to use for this input's initial frame. To use this
         *        setting, you must set the Timecode source setting, located under the input settings
         *        (InputTimecodeSource), to Specified start (SPECIFIEDSTART). For more information about timecodes, see
         *        https://docs.aws.amazon.com/console/mediaconvert/timecode.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timecodeStart(String timecodeStart);

        /**
         * Input video selectors contain the video settings for the input. Each of your inputs can have up to one video
         * selector.
         * 
         * @param videoSelector
         *        Input video selectors contain the video settings for the input. Each of your inputs can have up to one
         *        video selector.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder videoSelector(VideoSelector videoSelector);

        /**
         * Input video selectors contain the video settings for the input. Each of your inputs can have up to one video
         * selector. This is a convenience method that creates an instance of the {@link VideoSelector.Builder} avoiding
         * the need to create one manually via {@link VideoSelector#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VideoSelector.Builder#build()} is called immediately and its
         * result is passed to {@link #videoSelector(VideoSelector)}.
         * 
         * @param videoSelector
         *        a consumer that will call methods on {@link VideoSelector.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #videoSelector(VideoSelector)
         */
        default Builder videoSelector(Consumer<VideoSelector.Builder> videoSelector) {
            return videoSelector(VideoSelector.builder().applyMutation(videoSelector).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String advancedInputFilter;

        private AdvancedInputFilterSettings advancedInputFilterSettings;

        private Map<String, AudioSelectorGroup> audioSelectorGroups = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, AudioSelector> audioSelectors = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, CaptionSelector> captionSelectors = DefaultSdkAutoConstructMap.getInstance();

        private Rectangle crop;

        private String deblockFilter;

        private String denoiseFilter;

        private String dolbyVisionMetadataXml;

        private String filterEnable;

        private Integer filterStrength;

        private ImageInserter imageInserter;

        private List<InputClipping> inputClippings = DefaultSdkAutoConstructList.getInstance();

        private String inputScanType;

        private Rectangle position;

        private Integer programNumber;

        private String psiControl;

        private String timecodeSource;

        private String timecodeStart;

        private VideoSelector videoSelector;

        private BuilderImpl() {
        }

        private BuilderImpl(InputTemplate model) {
            advancedInputFilter(model.advancedInputFilter);
            advancedInputFilterSettings(model.advancedInputFilterSettings);
            audioSelectorGroups(model.audioSelectorGroups);
            audioSelectors(model.audioSelectors);
            captionSelectors(model.captionSelectors);
            crop(model.crop);
            deblockFilter(model.deblockFilter);
            denoiseFilter(model.denoiseFilter);
            dolbyVisionMetadataXml(model.dolbyVisionMetadataXml);
            filterEnable(model.filterEnable);
            filterStrength(model.filterStrength);
            imageInserter(model.imageInserter);
            inputClippings(model.inputClippings);
            inputScanType(model.inputScanType);
            position(model.position);
            programNumber(model.programNumber);
            psiControl(model.psiControl);
            timecodeSource(model.timecodeSource);
            timecodeStart(model.timecodeStart);
            videoSelector(model.videoSelector);
        }

        public final String getAdvancedInputFilter() {
            return advancedInputFilter;
        }

        public final void setAdvancedInputFilter(String advancedInputFilter) {
            this.advancedInputFilter = advancedInputFilter;
        }

        @Override
        public final Builder advancedInputFilter(String advancedInputFilter) {
            this.advancedInputFilter = advancedInputFilter;
            return this;
        }

        @Override
        public final Builder advancedInputFilter(AdvancedInputFilter advancedInputFilter) {
            this.advancedInputFilter(advancedInputFilter == null ? null : advancedInputFilter.toString());
            return this;
        }

        public final AdvancedInputFilterSettings.Builder getAdvancedInputFilterSettings() {
            return advancedInputFilterSettings != null ? advancedInputFilterSettings.toBuilder() : null;
        }

        public final void setAdvancedInputFilterSettings(AdvancedInputFilterSettings.BuilderImpl advancedInputFilterSettings) {
            this.advancedInputFilterSettings = advancedInputFilterSettings != null ? advancedInputFilterSettings.build() : null;
        }

        @Override
        public final Builder advancedInputFilterSettings(AdvancedInputFilterSettings advancedInputFilterSettings) {
            this.advancedInputFilterSettings = advancedInputFilterSettings;
            return this;
        }

        public final Map<String, AudioSelectorGroup.Builder> getAudioSelectorGroups() {
            Map<String, AudioSelectorGroup.Builder> result = ___mapOfAudioSelectorGroupCopier
                    .copyToBuilder(this.audioSelectorGroups);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setAudioSelectorGroups(Map<String, AudioSelectorGroup.BuilderImpl> audioSelectorGroups) {
            this.audioSelectorGroups = ___mapOfAudioSelectorGroupCopier.copyFromBuilder(audioSelectorGroups);
        }

        @Override
        public final Builder audioSelectorGroups(Map<String, AudioSelectorGroup> audioSelectorGroups) {
            this.audioSelectorGroups = ___mapOfAudioSelectorGroupCopier.copy(audioSelectorGroups);
            return this;
        }

        public final Map<String, AudioSelector.Builder> getAudioSelectors() {
            Map<String, AudioSelector.Builder> result = ___mapOfAudioSelectorCopier.copyToBuilder(this.audioSelectors);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setAudioSelectors(Map<String, AudioSelector.BuilderImpl> audioSelectors) {
            this.audioSelectors = ___mapOfAudioSelectorCopier.copyFromBuilder(audioSelectors);
        }

        @Override
        public final Builder audioSelectors(Map<String, AudioSelector> audioSelectors) {
            this.audioSelectors = ___mapOfAudioSelectorCopier.copy(audioSelectors);
            return this;
        }

        public final Map<String, CaptionSelector.Builder> getCaptionSelectors() {
            Map<String, CaptionSelector.Builder> result = ___mapOfCaptionSelectorCopier.copyToBuilder(this.captionSelectors);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setCaptionSelectors(Map<String, CaptionSelector.BuilderImpl> captionSelectors) {
            this.captionSelectors = ___mapOfCaptionSelectorCopier.copyFromBuilder(captionSelectors);
        }

        @Override
        public final Builder captionSelectors(Map<String, CaptionSelector> captionSelectors) {
            this.captionSelectors = ___mapOfCaptionSelectorCopier.copy(captionSelectors);
            return this;
        }

        public final Rectangle.Builder getCrop() {
            return crop != null ? crop.toBuilder() : null;
        }

        public final void setCrop(Rectangle.BuilderImpl crop) {
            this.crop = crop != null ? crop.build() : null;
        }

        @Override
        public final Builder crop(Rectangle crop) {
            this.crop = crop;
            return this;
        }

        public final String getDeblockFilter() {
            return deblockFilter;
        }

        public final void setDeblockFilter(String deblockFilter) {
            this.deblockFilter = deblockFilter;
        }

        @Override
        public final Builder deblockFilter(String deblockFilter) {
            this.deblockFilter = deblockFilter;
            return this;
        }

        @Override
        public final Builder deblockFilter(InputDeblockFilter deblockFilter) {
            this.deblockFilter(deblockFilter == null ? null : deblockFilter.toString());
            return this;
        }

        public final String getDenoiseFilter() {
            return denoiseFilter;
        }

        public final void setDenoiseFilter(String denoiseFilter) {
            this.denoiseFilter = denoiseFilter;
        }

        @Override
        public final Builder denoiseFilter(String denoiseFilter) {
            this.denoiseFilter = denoiseFilter;
            return this;
        }

        @Override
        public final Builder denoiseFilter(InputDenoiseFilter denoiseFilter) {
            this.denoiseFilter(denoiseFilter == null ? null : denoiseFilter.toString());
            return this;
        }

        public final String getDolbyVisionMetadataXml() {
            return dolbyVisionMetadataXml;
        }

        public final void setDolbyVisionMetadataXml(String dolbyVisionMetadataXml) {
            this.dolbyVisionMetadataXml = dolbyVisionMetadataXml;
        }

        @Override
        public final Builder dolbyVisionMetadataXml(String dolbyVisionMetadataXml) {
            this.dolbyVisionMetadataXml = dolbyVisionMetadataXml;
            return this;
        }

        public final String getFilterEnable() {
            return filterEnable;
        }

        public final void setFilterEnable(String filterEnable) {
            this.filterEnable = filterEnable;
        }

        @Override
        public final Builder filterEnable(String filterEnable) {
            this.filterEnable = filterEnable;
            return this;
        }

        @Override
        public final Builder filterEnable(InputFilterEnable filterEnable) {
            this.filterEnable(filterEnable == null ? null : filterEnable.toString());
            return this;
        }

        public final Integer getFilterStrength() {
            return filterStrength;
        }

        public final void setFilterStrength(Integer filterStrength) {
            this.filterStrength = filterStrength;
        }

        @Override
        public final Builder filterStrength(Integer filterStrength) {
            this.filterStrength = filterStrength;
            return this;
        }

        public final ImageInserter.Builder getImageInserter() {
            return imageInserter != null ? imageInserter.toBuilder() : null;
        }

        public final void setImageInserter(ImageInserter.BuilderImpl imageInserter) {
            this.imageInserter = imageInserter != null ? imageInserter.build() : null;
        }

        @Override
        public final Builder imageInserter(ImageInserter imageInserter) {
            this.imageInserter = imageInserter;
            return this;
        }

        public final List<InputClipping.Builder> getInputClippings() {
            List<InputClipping.Builder> result = ___listOfInputClippingCopier.copyToBuilder(this.inputClippings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setInputClippings(Collection<InputClipping.BuilderImpl> inputClippings) {
            this.inputClippings = ___listOfInputClippingCopier.copyFromBuilder(inputClippings);
        }

        @Override
        public final Builder inputClippings(Collection<InputClipping> inputClippings) {
            this.inputClippings = ___listOfInputClippingCopier.copy(inputClippings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inputClippings(InputClipping... inputClippings) {
            inputClippings(Arrays.asList(inputClippings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inputClippings(Consumer<InputClipping.Builder>... inputClippings) {
            inputClippings(Stream.of(inputClippings).map(c -> InputClipping.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getInputScanType() {
            return inputScanType;
        }

        public final void setInputScanType(String inputScanType) {
            this.inputScanType = inputScanType;
        }

        @Override
        public final Builder inputScanType(String inputScanType) {
            this.inputScanType = inputScanType;
            return this;
        }

        @Override
        public final Builder inputScanType(InputScanType inputScanType) {
            this.inputScanType(inputScanType == null ? null : inputScanType.toString());
            return this;
        }

        public final Rectangle.Builder getPosition() {
            return position != null ? position.toBuilder() : null;
        }

        public final void setPosition(Rectangle.BuilderImpl position) {
            this.position = position != null ? position.build() : null;
        }

        @Override
        public final Builder position(Rectangle position) {
            this.position = position;
            return this;
        }

        public final Integer getProgramNumber() {
            return programNumber;
        }

        public final void setProgramNumber(Integer programNumber) {
            this.programNumber = programNumber;
        }

        @Override
        public final Builder programNumber(Integer programNumber) {
            this.programNumber = programNumber;
            return this;
        }

        public final String getPsiControl() {
            return psiControl;
        }

        public final void setPsiControl(String psiControl) {
            this.psiControl = psiControl;
        }

        @Override
        public final Builder psiControl(String psiControl) {
            this.psiControl = psiControl;
            return this;
        }

        @Override
        public final Builder psiControl(InputPsiControl psiControl) {
            this.psiControl(psiControl == null ? null : psiControl.toString());
            return this;
        }

        public final String getTimecodeSource() {
            return timecodeSource;
        }

        public final void setTimecodeSource(String timecodeSource) {
            this.timecodeSource = timecodeSource;
        }

        @Override
        public final Builder timecodeSource(String timecodeSource) {
            this.timecodeSource = timecodeSource;
            return this;
        }

        @Override
        public final Builder timecodeSource(InputTimecodeSource timecodeSource) {
            this.timecodeSource(timecodeSource == null ? null : timecodeSource.toString());
            return this;
        }

        public final String getTimecodeStart() {
            return timecodeStart;
        }

        public final void setTimecodeStart(String timecodeStart) {
            this.timecodeStart = timecodeStart;
        }

        @Override
        public final Builder timecodeStart(String timecodeStart) {
            this.timecodeStart = timecodeStart;
            return this;
        }

        public final VideoSelector.Builder getVideoSelector() {
            return videoSelector != null ? videoSelector.toBuilder() : null;
        }

        public final void setVideoSelector(VideoSelector.BuilderImpl videoSelector) {
            this.videoSelector = videoSelector != null ? videoSelector.build() : null;
        }

        @Override
        public final Builder videoSelector(VideoSelector videoSelector) {
            this.videoSelector = videoSelector;
            return this;
        }

        @Override
        public InputTemplate build() {
            return new InputTemplate(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
