/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediaconnect.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The settings for the source of the flow.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Source implements SdkPojo, Serializable, ToCopyableBuilder<Source.Builder, Source> {
    private static final SdkField<Integer> DATA_TRANSFER_SUBSCRIBER_FEE_PERCENT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("DataTransferSubscriberFeePercent")
            .getter(getter(Source::dataTransferSubscriberFeePercent))
            .setter(setter(Builder::dataTransferSubscriberFeePercent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dataTransferSubscriberFeePercent")
                    .build()).build();

    private static final SdkField<Encryption> DECRYPTION_FIELD = SdkField.<Encryption> builder(MarshallingType.SDK_POJO)
            .memberName("Decryption").getter(getter(Source::decryption)).setter(setter(Builder::decryption))
            .constructor(Encryption::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("decryption").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(Source::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<String> ENTITLEMENT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EntitlementArn").getter(getter(Source::entitlementArn)).setter(setter(Builder::entitlementArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("entitlementArn").build()).build();

    private static final SdkField<String> INGEST_IP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IngestIp").getter(getter(Source::ingestIp)).setter(setter(Builder::ingestIp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ingestIp").build()).build();

    private static final SdkField<Integer> INGEST_PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("IngestPort").getter(getter(Source::ingestPort)).setter(setter(Builder::ingestPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ingestPort").build()).build();

    private static final SdkField<List<MediaStreamSourceConfiguration>> MEDIA_STREAM_SOURCE_CONFIGURATIONS_FIELD = SdkField
            .<List<MediaStreamSourceConfiguration>> builder(MarshallingType.LIST)
            .memberName("MediaStreamSourceConfigurations")
            .getter(getter(Source::mediaStreamSourceConfigurations))
            .setter(setter(Builder::mediaStreamSourceConfigurations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("mediaStreamSourceConfigurations")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MediaStreamSourceConfiguration> builder(MarshallingType.SDK_POJO)
                                            .constructor(MediaStreamSourceConfiguration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Source::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<Integer> SENDER_CONTROL_PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("SenderControlPort").getter(getter(Source::senderControlPort)).setter(setter(Builder::senderControlPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("senderControlPort").build()).build();

    private static final SdkField<String> SENDER_IP_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SenderIpAddress").getter(getter(Source::senderIpAddress)).setter(setter(Builder::senderIpAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("senderIpAddress").build()).build();

    private static final SdkField<String> SOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceArn").getter(getter(Source::sourceArn)).setter(setter(Builder::sourceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceArn").build()).build();

    private static final SdkField<Transport> TRANSPORT_FIELD = SdkField.<Transport> builder(MarshallingType.SDK_POJO)
            .memberName("Transport").getter(getter(Source::transport)).setter(setter(Builder::transport))
            .constructor(Transport::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("transport").build()).build();

    private static final SdkField<String> VPC_INTERFACE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VpcInterfaceName").getter(getter(Source::vpcInterfaceName)).setter(setter(Builder::vpcInterfaceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("vpcInterfaceName").build()).build();

    private static final SdkField<String> WHITELIST_CIDR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WhitelistCidr").getter(getter(Source::whitelistCidr)).setter(setter(Builder::whitelistCidr))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("whitelistCidr").build()).build();

    private static final SdkField<GatewayBridgeSource> GATEWAY_BRIDGE_SOURCE_FIELD = SdkField
            .<GatewayBridgeSource> builder(MarshallingType.SDK_POJO).memberName("GatewayBridgeSource")
            .getter(getter(Source::gatewayBridgeSource)).setter(setter(Builder::gatewayBridgeSource))
            .constructor(GatewayBridgeSource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("gatewayBridgeSource").build())
            .build();

    private static final SdkField<String> PEER_IP_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PeerIpAddress").getter(getter(Source::peerIpAddress)).setter(setter(Builder::peerIpAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("peerIpAddress").build()).build();

    private static final SdkField<String> ROUTER_INTEGRATION_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RouterIntegrationState").getter(getter(Source::routerIntegrationStateAsString))
            .setter(setter(Builder::routerIntegrationState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("routerIntegrationState").build())
            .build();

    private static final SdkField<FlowTransitEncryption> ROUTER_INTEGRATION_TRANSIT_DECRYPTION_FIELD = SdkField
            .<FlowTransitEncryption> builder(MarshallingType.SDK_POJO)
            .memberName("RouterIntegrationTransitDecryption")
            .getter(getter(Source::routerIntegrationTransitDecryption))
            .setter(setter(Builder::routerIntegrationTransitDecryption))
            .constructor(FlowTransitEncryption::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("routerIntegrationTransitDecryption")
                    .build()).build();

    private static final SdkField<String> CONNECTED_ROUTER_OUTPUT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConnectedRouterOutputArn").getter(getter(Source::connectedRouterOutputArn))
            .setter(setter(Builder::connectedRouterOutputArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("connectedRouterOutputArn").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            DATA_TRANSFER_SUBSCRIBER_FEE_PERCENT_FIELD, DECRYPTION_FIELD, DESCRIPTION_FIELD, ENTITLEMENT_ARN_FIELD,
            INGEST_IP_FIELD, INGEST_PORT_FIELD, MEDIA_STREAM_SOURCE_CONFIGURATIONS_FIELD, NAME_FIELD, SENDER_CONTROL_PORT_FIELD,
            SENDER_IP_ADDRESS_FIELD, SOURCE_ARN_FIELD, TRANSPORT_FIELD, VPC_INTERFACE_NAME_FIELD, WHITELIST_CIDR_FIELD,
            GATEWAY_BRIDGE_SOURCE_FIELD, PEER_IP_ADDRESS_FIELD, ROUTER_INTEGRATION_STATE_FIELD,
            ROUTER_INTEGRATION_TRANSIT_DECRYPTION_FIELD, CONNECTED_ROUTER_OUTPUT_ARN_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Integer dataTransferSubscriberFeePercent;

    private final Encryption decryption;

    private final String description;

    private final String entitlementArn;

    private final String ingestIp;

    private final Integer ingestPort;

    private final List<MediaStreamSourceConfiguration> mediaStreamSourceConfigurations;

    private final String name;

    private final Integer senderControlPort;

    private final String senderIpAddress;

    private final String sourceArn;

    private final Transport transport;

    private final String vpcInterfaceName;

    private final String whitelistCidr;

    private final GatewayBridgeSource gatewayBridgeSource;

    private final String peerIpAddress;

    private final String routerIntegrationState;

    private final FlowTransitEncryption routerIntegrationTransitDecryption;

    private final String connectedRouterOutputArn;

    private Source(BuilderImpl builder) {
        this.dataTransferSubscriberFeePercent = builder.dataTransferSubscriberFeePercent;
        this.decryption = builder.decryption;
        this.description = builder.description;
        this.entitlementArn = builder.entitlementArn;
        this.ingestIp = builder.ingestIp;
        this.ingestPort = builder.ingestPort;
        this.mediaStreamSourceConfigurations = builder.mediaStreamSourceConfigurations;
        this.name = builder.name;
        this.senderControlPort = builder.senderControlPort;
        this.senderIpAddress = builder.senderIpAddress;
        this.sourceArn = builder.sourceArn;
        this.transport = builder.transport;
        this.vpcInterfaceName = builder.vpcInterfaceName;
        this.whitelistCidr = builder.whitelistCidr;
        this.gatewayBridgeSource = builder.gatewayBridgeSource;
        this.peerIpAddress = builder.peerIpAddress;
        this.routerIntegrationState = builder.routerIntegrationState;
        this.routerIntegrationTransitDecryption = builder.routerIntegrationTransitDecryption;
        this.connectedRouterOutputArn = builder.connectedRouterOutputArn;
    }

    /**
     * <p>
     * Percentage from 0-100 of the data transfer cost to be billed to the subscriber.
     * </p>
     * 
     * @return Percentage from 0-100 of the data transfer cost to be billed to the subscriber.
     */
    public final Integer dataTransferSubscriberFeePercent() {
        return dataTransferSubscriberFeePercent;
    }

    /**
     * <p>
     * The type of encryption that is used on the content ingested from this source.
     * </p>
     * 
     * @return The type of encryption that is used on the content ingested from this source.
     */
    public final Encryption decryption() {
        return decryption;
    }

    /**
     * <p>
     * A description for the source. This value is not used or seen outside of the current MediaConnect account.
     * </p>
     * 
     * @return A description for the source. This value is not used or seen outside of the current MediaConnect account.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The ARN of the entitlement that allows you to subscribe to content that comes from another Amazon Web Services
     * account. The entitlement is set by the content originator and the ARN is generated as part of the originator's
     * flow.
     * </p>
     * 
     * @return The ARN of the entitlement that allows you to subscribe to content that comes from another Amazon Web
     *         Services account. The entitlement is set by the content originator and the ARN is generated as part of
     *         the originator's flow.
     */
    public final String entitlementArn() {
        return entitlementArn;
    }

    /**
     * <p>
     * The IP address that the flow will be listening on for incoming content.
     * </p>
     * 
     * @return The IP address that the flow will be listening on for incoming content.
     */
    public final String ingestIp() {
        return ingestIp;
    }

    /**
     * <p>
     * The port that the flow will be listening on for incoming content.
     * </p>
     * 
     * @return The port that the flow will be listening on for incoming content.
     */
    public final Integer ingestPort() {
        return ingestPort;
    }

    /**
     * For responses, this returns true if the service returned a value for the MediaStreamSourceConfigurations
     * property. This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()}
     * method on the property). This is useful because the SDK will never return a null collection or map, but you may
     * need to differentiate between the service returning nothing (or null) and the service returning an empty
     * collection or map. For requests, this returns true if a value for the property was specified in the request
     * builder, and false if a value was not specified.
     */
    public final boolean hasMediaStreamSourceConfigurations() {
        return mediaStreamSourceConfigurations != null && !(mediaStreamSourceConfigurations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The media streams that are associated with the source, and the parameters for those associations.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMediaStreamSourceConfigurations}
     * method.
     * </p>
     * 
     * @return The media streams that are associated with the source, and the parameters for those associations.
     */
    public final List<MediaStreamSourceConfiguration> mediaStreamSourceConfigurations() {
        return mediaStreamSourceConfigurations;
    }

    /**
     * <p>
     * The name of the source.
     * </p>
     * 
     * @return The name of the source.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The IP address that the flow communicates with to initiate connection with the sender.
     * </p>
     * 
     * @return The IP address that the flow communicates with to initiate connection with the sender.
     */
    public final Integer senderControlPort() {
        return senderControlPort;
    }

    /**
     * <p>
     * The port that the flow uses to send outbound requests to initiate connection with the sender.
     * </p>
     * 
     * @return The port that the flow uses to send outbound requests to initiate connection with the sender.
     */
    public final String senderIpAddress() {
        return senderIpAddress;
    }

    /**
     * <p>
     * The ARN of the source.
     * </p>
     * 
     * @return The ARN of the source.
     */
    public final String sourceArn() {
        return sourceArn;
    }

    /**
     * <p>
     * Attributes related to the transport stream that are used in the source.
     * </p>
     * 
     * @return Attributes related to the transport stream that are used in the source.
     */
    public final Transport transport() {
        return transport;
    }

    /**
     * <p>
     * The name of the VPC interface that is used for this source.
     * </p>
     * 
     * @return The name of the VPC interface that is used for this source.
     */
    public final String vpcInterfaceName() {
        return vpcInterfaceName;
    }

    /**
     * <p>
     * The range of IP addresses that should be allowed to contribute content to your source. These IP addresses should
     * be in the form of a Classless Inter-Domain Routing (CIDR) block; for example, 10.0.0.0/16.
     * </p>
     * 
     * @return The range of IP addresses that should be allowed to contribute content to your source. These IP addresses
     *         should be in the form of a Classless Inter-Domain Routing (CIDR) block; for example, 10.0.0.0/16.
     */
    public final String whitelistCidr() {
        return whitelistCidr;
    }

    /**
     * <p>
     * The source configuration for cloud flows receiving a stream from a bridge.
     * </p>
     * 
     * @return The source configuration for cloud flows receiving a stream from a bridge.
     */
    public final GatewayBridgeSource gatewayBridgeSource() {
        return gatewayBridgeSource;
    }

    /**
     * <p>
     * The IP address of the device that is currently sending content to this source.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * For sources that use protocols where you specify the origin (such as SRT Caller), this value matches the
     * configured origin address.
     * </p>
     * </li>
     * <li>
     * <p>
     * For sources that use listener protocols (such as SRT Listener or RTP), this value shows the address of the
     * connected sender.
     * </p>
     * </li>
     * <li>
     * <p>
     * Peer IP addresses aren't available for entitlements and CDI/ST2110 sources.
     * </p>
     * </li>
     * <li>
     * <p>
     * The peer IP address might not be visible for flows that haven't been started yet, or flows that were started
     * before May 2025. In these cases, restart your flow to see the peer IP address.
     * </p>
     * </li>
     * </ul>
     * </note>
     * 
     * @return The IP address of the device that is currently sending content to this source. </p> <note>
     *         <ul>
     *         <li>
     *         <p>
     *         For sources that use protocols where you specify the origin (such as SRT Caller), this value matches the
     *         configured origin address.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For sources that use listener protocols (such as SRT Listener or RTP), this value shows the address of
     *         the connected sender.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Peer IP addresses aren't available for entitlements and CDI/ST2110 sources.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The peer IP address might not be visible for flows that haven't been started yet, or flows that were
     *         started before May 2025. In these cases, restart your flow to see the peer IP address.
     *         </p>
     *         </li>
     *         </ul>
     */
    public final String peerIpAddress() {
        return peerIpAddress;
    }

    /**
     * <p>
     * Indicates if router integration is enabled or disabled on the flow source.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #routerIntegrationState} will return {@link State#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #routerIntegrationStateAsString}.
     * </p>
     * 
     * @return Indicates if router integration is enabled or disabled on the flow source.
     * @see State
     */
    public final State routerIntegrationState() {
        return State.fromValue(routerIntegrationState);
    }

    /**
     * <p>
     * Indicates if router integration is enabled or disabled on the flow source.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #routerIntegrationState} will return {@link State#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #routerIntegrationStateAsString}.
     * </p>
     * 
     * @return Indicates if router integration is enabled or disabled on the flow source.
     * @see State
     */
    public final String routerIntegrationStateAsString() {
        return routerIntegrationState;
    }

    /**
     * <p>
     * The decryption configuration for the flow source when router integration is enabled.
     * </p>
     * 
     * @return The decryption configuration for the flow source when router integration is enabled.
     */
    public final FlowTransitEncryption routerIntegrationTransitDecryption() {
        return routerIntegrationTransitDecryption;
    }

    /**
     * <p>
     * The ARN of the router output that's currently connected to this source.
     * </p>
     * 
     * @return The ARN of the router output that's currently connected to this source.
     */
    public final String connectedRouterOutputArn() {
        return connectedRouterOutputArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(dataTransferSubscriberFeePercent());
        hashCode = 31 * hashCode + Objects.hashCode(decryption());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(entitlementArn());
        hashCode = 31 * hashCode + Objects.hashCode(ingestIp());
        hashCode = 31 * hashCode + Objects.hashCode(ingestPort());
        hashCode = 31 * hashCode
                + Objects.hashCode(hasMediaStreamSourceConfigurations() ? mediaStreamSourceConfigurations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(senderControlPort());
        hashCode = 31 * hashCode + Objects.hashCode(senderIpAddress());
        hashCode = 31 * hashCode + Objects.hashCode(sourceArn());
        hashCode = 31 * hashCode + Objects.hashCode(transport());
        hashCode = 31 * hashCode + Objects.hashCode(vpcInterfaceName());
        hashCode = 31 * hashCode + Objects.hashCode(whitelistCidr());
        hashCode = 31 * hashCode + Objects.hashCode(gatewayBridgeSource());
        hashCode = 31 * hashCode + Objects.hashCode(peerIpAddress());
        hashCode = 31 * hashCode + Objects.hashCode(routerIntegrationStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(routerIntegrationTransitDecryption());
        hashCode = 31 * hashCode + Objects.hashCode(connectedRouterOutputArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Source)) {
            return false;
        }
        Source other = (Source) obj;
        return Objects.equals(dataTransferSubscriberFeePercent(), other.dataTransferSubscriberFeePercent())
                && Objects.equals(decryption(), other.decryption()) && Objects.equals(description(), other.description())
                && Objects.equals(entitlementArn(), other.entitlementArn()) && Objects.equals(ingestIp(), other.ingestIp())
                && Objects.equals(ingestPort(), other.ingestPort())
                && hasMediaStreamSourceConfigurations() == other.hasMediaStreamSourceConfigurations()
                && Objects.equals(mediaStreamSourceConfigurations(), other.mediaStreamSourceConfigurations())
                && Objects.equals(name(), other.name()) && Objects.equals(senderControlPort(), other.senderControlPort())
                && Objects.equals(senderIpAddress(), other.senderIpAddress()) && Objects.equals(sourceArn(), other.sourceArn())
                && Objects.equals(transport(), other.transport()) && Objects.equals(vpcInterfaceName(), other.vpcInterfaceName())
                && Objects.equals(whitelistCidr(), other.whitelistCidr())
                && Objects.equals(gatewayBridgeSource(), other.gatewayBridgeSource())
                && Objects.equals(peerIpAddress(), other.peerIpAddress())
                && Objects.equals(routerIntegrationStateAsString(), other.routerIntegrationStateAsString())
                && Objects.equals(routerIntegrationTransitDecryption(), other.routerIntegrationTransitDecryption())
                && Objects.equals(connectedRouterOutputArn(), other.connectedRouterOutputArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("Source")
                .add("DataTransferSubscriberFeePercent", dataTransferSubscriberFeePercent())
                .add("Decryption", decryption())
                .add("Description", description())
                .add("EntitlementArn", entitlementArn())
                .add("IngestIp", ingestIp())
                .add("IngestPort", ingestPort())
                .add("MediaStreamSourceConfigurations",
                        hasMediaStreamSourceConfigurations() ? mediaStreamSourceConfigurations() : null).add("Name", name())
                .add("SenderControlPort", senderControlPort()).add("SenderIpAddress", senderIpAddress())
                .add("SourceArn", sourceArn()).add("Transport", transport()).add("VpcInterfaceName", vpcInterfaceName())
                .add("WhitelistCidr", whitelistCidr()).add("GatewayBridgeSource", gatewayBridgeSource())
                .add("PeerIpAddress", peerIpAddress()).add("RouterIntegrationState", routerIntegrationStateAsString())
                .add("RouterIntegrationTransitDecryption", routerIntegrationTransitDecryption())
                .add("ConnectedRouterOutputArn", connectedRouterOutputArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DataTransferSubscriberFeePercent":
            return Optional.ofNullable(clazz.cast(dataTransferSubscriberFeePercent()));
        case "Decryption":
            return Optional.ofNullable(clazz.cast(decryption()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "EntitlementArn":
            return Optional.ofNullable(clazz.cast(entitlementArn()));
        case "IngestIp":
            return Optional.ofNullable(clazz.cast(ingestIp()));
        case "IngestPort":
            return Optional.ofNullable(clazz.cast(ingestPort()));
        case "MediaStreamSourceConfigurations":
            return Optional.ofNullable(clazz.cast(mediaStreamSourceConfigurations()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "SenderControlPort":
            return Optional.ofNullable(clazz.cast(senderControlPort()));
        case "SenderIpAddress":
            return Optional.ofNullable(clazz.cast(senderIpAddress()));
        case "SourceArn":
            return Optional.ofNullable(clazz.cast(sourceArn()));
        case "Transport":
            return Optional.ofNullable(clazz.cast(transport()));
        case "VpcInterfaceName":
            return Optional.ofNullable(clazz.cast(vpcInterfaceName()));
        case "WhitelistCidr":
            return Optional.ofNullable(clazz.cast(whitelistCidr()));
        case "GatewayBridgeSource":
            return Optional.ofNullable(clazz.cast(gatewayBridgeSource()));
        case "PeerIpAddress":
            return Optional.ofNullable(clazz.cast(peerIpAddress()));
        case "RouterIntegrationState":
            return Optional.ofNullable(clazz.cast(routerIntegrationStateAsString()));
        case "RouterIntegrationTransitDecryption":
            return Optional.ofNullable(clazz.cast(routerIntegrationTransitDecryption()));
        case "ConnectedRouterOutputArn":
            return Optional.ofNullable(clazz.cast(connectedRouterOutputArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("dataTransferSubscriberFeePercent", DATA_TRANSFER_SUBSCRIBER_FEE_PERCENT_FIELD);
        map.put("decryption", DECRYPTION_FIELD);
        map.put("description", DESCRIPTION_FIELD);
        map.put("entitlementArn", ENTITLEMENT_ARN_FIELD);
        map.put("ingestIp", INGEST_IP_FIELD);
        map.put("ingestPort", INGEST_PORT_FIELD);
        map.put("mediaStreamSourceConfigurations", MEDIA_STREAM_SOURCE_CONFIGURATIONS_FIELD);
        map.put("name", NAME_FIELD);
        map.put("senderControlPort", SENDER_CONTROL_PORT_FIELD);
        map.put("senderIpAddress", SENDER_IP_ADDRESS_FIELD);
        map.put("sourceArn", SOURCE_ARN_FIELD);
        map.put("transport", TRANSPORT_FIELD);
        map.put("vpcInterfaceName", VPC_INTERFACE_NAME_FIELD);
        map.put("whitelistCidr", WHITELIST_CIDR_FIELD);
        map.put("gatewayBridgeSource", GATEWAY_BRIDGE_SOURCE_FIELD);
        map.put("peerIpAddress", PEER_IP_ADDRESS_FIELD);
        map.put("routerIntegrationState", ROUTER_INTEGRATION_STATE_FIELD);
        map.put("routerIntegrationTransitDecryption", ROUTER_INTEGRATION_TRANSIT_DECRYPTION_FIELD);
        map.put("connectedRouterOutputArn", CONNECTED_ROUTER_OUTPUT_ARN_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Source, T> g) {
        return obj -> g.apply((Source) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Source> {
        /**
         * <p>
         * Percentage from 0-100 of the data transfer cost to be billed to the subscriber.
         * </p>
         * 
         * @param dataTransferSubscriberFeePercent
         *        Percentage from 0-100 of the data transfer cost to be billed to the subscriber.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataTransferSubscriberFeePercent(Integer dataTransferSubscriberFeePercent);

        /**
         * <p>
         * The type of encryption that is used on the content ingested from this source.
         * </p>
         * 
         * @param decryption
         *        The type of encryption that is used on the content ingested from this source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder decryption(Encryption decryption);

        /**
         * <p>
         * The type of encryption that is used on the content ingested from this source.
         * </p>
         * This is a convenience method that creates an instance of the {@link Encryption.Builder} avoiding the need to
         * create one manually via {@link Encryption#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Encryption.Builder#build()} is called immediately and its result
         * is passed to {@link #decryption(Encryption)}.
         * 
         * @param decryption
         *        a consumer that will call methods on {@link Encryption.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #decryption(Encryption)
         */
        default Builder decryption(Consumer<Encryption.Builder> decryption) {
            return decryption(Encryption.builder().applyMutation(decryption).build());
        }

        /**
         * <p>
         * A description for the source. This value is not used or seen outside of the current MediaConnect account.
         * </p>
         * 
         * @param description
         *        A description for the source. This value is not used or seen outside of the current MediaConnect
         *        account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The ARN of the entitlement that allows you to subscribe to content that comes from another Amazon Web
         * Services account. The entitlement is set by the content originator and the ARN is generated as part of the
         * originator's flow.
         * </p>
         * 
         * @param entitlementArn
         *        The ARN of the entitlement that allows you to subscribe to content that comes from another Amazon Web
         *        Services account. The entitlement is set by the content originator and the ARN is generated as part of
         *        the originator's flow.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entitlementArn(String entitlementArn);

        /**
         * <p>
         * The IP address that the flow will be listening on for incoming content.
         * </p>
         * 
         * @param ingestIp
         *        The IP address that the flow will be listening on for incoming content.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ingestIp(String ingestIp);

        /**
         * <p>
         * The port that the flow will be listening on for incoming content.
         * </p>
         * 
         * @param ingestPort
         *        The port that the flow will be listening on for incoming content.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ingestPort(Integer ingestPort);

        /**
         * <p>
         * The media streams that are associated with the source, and the parameters for those associations.
         * </p>
         * 
         * @param mediaStreamSourceConfigurations
         *        The media streams that are associated with the source, and the parameters for those associations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mediaStreamSourceConfigurations(Collection<MediaStreamSourceConfiguration> mediaStreamSourceConfigurations);

        /**
         * <p>
         * The media streams that are associated with the source, and the parameters for those associations.
         * </p>
         * 
         * @param mediaStreamSourceConfigurations
         *        The media streams that are associated with the source, and the parameters for those associations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mediaStreamSourceConfigurations(MediaStreamSourceConfiguration... mediaStreamSourceConfigurations);

        /**
         * <p>
         * The media streams that are associated with the source, and the parameters for those associations.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.mediaconnect.model.MediaStreamSourceConfiguration.Builder} avoiding
         * the need to create one manually via
         * {@link software.amazon.awssdk.services.mediaconnect.model.MediaStreamSourceConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.mediaconnect.model.MediaStreamSourceConfiguration.Builder#build()} is
         * called immediately and its result is passed to {@link
         * #mediaStreamSourceConfigurations(List<MediaStreamSourceConfiguration>)}.
         * 
         * @param mediaStreamSourceConfigurations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.mediaconnect.model.MediaStreamSourceConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #mediaStreamSourceConfigurations(java.util.Collection<MediaStreamSourceConfiguration>)
         */
        Builder mediaStreamSourceConfigurations(
                Consumer<MediaStreamSourceConfiguration.Builder>... mediaStreamSourceConfigurations);

        /**
         * <p>
         * The name of the source.
         * </p>
         * 
         * @param name
         *        The name of the source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The IP address that the flow communicates with to initiate connection with the sender.
         * </p>
         * 
         * @param senderControlPort
         *        The IP address that the flow communicates with to initiate connection with the sender.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder senderControlPort(Integer senderControlPort);

        /**
         * <p>
         * The port that the flow uses to send outbound requests to initiate connection with the sender.
         * </p>
         * 
         * @param senderIpAddress
         *        The port that the flow uses to send outbound requests to initiate connection with the sender.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder senderIpAddress(String senderIpAddress);

        /**
         * <p>
         * The ARN of the source.
         * </p>
         * 
         * @param sourceArn
         *        The ARN of the source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceArn(String sourceArn);

        /**
         * <p>
         * Attributes related to the transport stream that are used in the source.
         * </p>
         * 
         * @param transport
         *        Attributes related to the transport stream that are used in the source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transport(Transport transport);

        /**
         * <p>
         * Attributes related to the transport stream that are used in the source.
         * </p>
         * This is a convenience method that creates an instance of the {@link Transport.Builder} avoiding the need to
         * create one manually via {@link Transport#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Transport.Builder#build()} is called immediately and its result
         * is passed to {@link #transport(Transport)}.
         * 
         * @param transport
         *        a consumer that will call methods on {@link Transport.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #transport(Transport)
         */
        default Builder transport(Consumer<Transport.Builder> transport) {
            return transport(Transport.builder().applyMutation(transport).build());
        }

        /**
         * <p>
         * The name of the VPC interface that is used for this source.
         * </p>
         * 
         * @param vpcInterfaceName
         *        The name of the VPC interface that is used for this source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcInterfaceName(String vpcInterfaceName);

        /**
         * <p>
         * The range of IP addresses that should be allowed to contribute content to your source. These IP addresses
         * should be in the form of a Classless Inter-Domain Routing (CIDR) block; for example, 10.0.0.0/16.
         * </p>
         * 
         * @param whitelistCidr
         *        The range of IP addresses that should be allowed to contribute content to your source. These IP
         *        addresses should be in the form of a Classless Inter-Domain Routing (CIDR) block; for example,
         *        10.0.0.0/16.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder whitelistCidr(String whitelistCidr);

        /**
         * <p>
         * The source configuration for cloud flows receiving a stream from a bridge.
         * </p>
         * 
         * @param gatewayBridgeSource
         *        The source configuration for cloud flows receiving a stream from a bridge.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gatewayBridgeSource(GatewayBridgeSource gatewayBridgeSource);

        /**
         * <p>
         * The source configuration for cloud flows receiving a stream from a bridge.
         * </p>
         * This is a convenience method that creates an instance of the {@link GatewayBridgeSource.Builder} avoiding the
         * need to create one manually via {@link GatewayBridgeSource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link GatewayBridgeSource.Builder#build()} is called immediately and
         * its result is passed to {@link #gatewayBridgeSource(GatewayBridgeSource)}.
         * 
         * @param gatewayBridgeSource
         *        a consumer that will call methods on {@link GatewayBridgeSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #gatewayBridgeSource(GatewayBridgeSource)
         */
        default Builder gatewayBridgeSource(Consumer<GatewayBridgeSource.Builder> gatewayBridgeSource) {
            return gatewayBridgeSource(GatewayBridgeSource.builder().applyMutation(gatewayBridgeSource).build());
        }

        /**
         * <p>
         * The IP address of the device that is currently sending content to this source.
         * </p>
         * <note>
         * <ul>
         * <li>
         * <p>
         * For sources that use protocols where you specify the origin (such as SRT Caller), this value matches the
         * configured origin address.
         * </p>
         * </li>
         * <li>
         * <p>
         * For sources that use listener protocols (such as SRT Listener or RTP), this value shows the address of the
         * connected sender.
         * </p>
         * </li>
         * <li>
         * <p>
         * Peer IP addresses aren't available for entitlements and CDI/ST2110 sources.
         * </p>
         * </li>
         * <li>
         * <p>
         * The peer IP address might not be visible for flows that haven't been started yet, or flows that were started
         * before May 2025. In these cases, restart your flow to see the peer IP address.
         * </p>
         * </li>
         * </ul>
         * </note>
         * 
         * @param peerIpAddress
         *        The IP address of the device that is currently sending content to this source. </p> <note>
         *        <ul>
         *        <li>
         *        <p>
         *        For sources that use protocols where you specify the origin (such as SRT Caller), this value matches
         *        the configured origin address.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For sources that use listener protocols (such as SRT Listener or RTP), this value shows the address of
         *        the connected sender.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Peer IP addresses aren't available for entitlements and CDI/ST2110 sources.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The peer IP address might not be visible for flows that haven't been started yet, or flows that were
         *        started before May 2025. In these cases, restart your flow to see the peer IP address.
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder peerIpAddress(String peerIpAddress);

        /**
         * <p>
         * Indicates if router integration is enabled or disabled on the flow source.
         * </p>
         * 
         * @param routerIntegrationState
         *        Indicates if router integration is enabled or disabled on the flow source.
         * @see State
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see State
         */
        Builder routerIntegrationState(String routerIntegrationState);

        /**
         * <p>
         * Indicates if router integration is enabled or disabled on the flow source.
         * </p>
         * 
         * @param routerIntegrationState
         *        Indicates if router integration is enabled or disabled on the flow source.
         * @see State
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see State
         */
        Builder routerIntegrationState(State routerIntegrationState);

        /**
         * <p>
         * The decryption configuration for the flow source when router integration is enabled.
         * </p>
         * 
         * @param routerIntegrationTransitDecryption
         *        The decryption configuration for the flow source when router integration is enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routerIntegrationTransitDecryption(FlowTransitEncryption routerIntegrationTransitDecryption);

        /**
         * <p>
         * The decryption configuration for the flow source when router integration is enabled.
         * </p>
         * This is a convenience method that creates an instance of the {@link FlowTransitEncryption.Builder} avoiding
         * the need to create one manually via {@link FlowTransitEncryption#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FlowTransitEncryption.Builder#build()} is called immediately and
         * its result is passed to {@link #routerIntegrationTransitDecryption(FlowTransitEncryption)}.
         * 
         * @param routerIntegrationTransitDecryption
         *        a consumer that will call methods on {@link FlowTransitEncryption.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #routerIntegrationTransitDecryption(FlowTransitEncryption)
         */
        default Builder routerIntegrationTransitDecryption(
                Consumer<FlowTransitEncryption.Builder> routerIntegrationTransitDecryption) {
            return routerIntegrationTransitDecryption(FlowTransitEncryption.builder()
                    .applyMutation(routerIntegrationTransitDecryption).build());
        }

        /**
         * <p>
         * The ARN of the router output that's currently connected to this source.
         * </p>
         * 
         * @param connectedRouterOutputArn
         *        The ARN of the router output that's currently connected to this source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectedRouterOutputArn(String connectedRouterOutputArn);
    }

    static final class BuilderImpl implements Builder {
        private Integer dataTransferSubscriberFeePercent;

        private Encryption decryption;

        private String description;

        private String entitlementArn;

        private String ingestIp;

        private Integer ingestPort;

        private List<MediaStreamSourceConfiguration> mediaStreamSourceConfigurations = DefaultSdkAutoConstructList.getInstance();

        private String name;

        private Integer senderControlPort;

        private String senderIpAddress;

        private String sourceArn;

        private Transport transport;

        private String vpcInterfaceName;

        private String whitelistCidr;

        private GatewayBridgeSource gatewayBridgeSource;

        private String peerIpAddress;

        private String routerIntegrationState;

        private FlowTransitEncryption routerIntegrationTransitDecryption;

        private String connectedRouterOutputArn;

        private BuilderImpl() {
        }

        private BuilderImpl(Source model) {
            dataTransferSubscriberFeePercent(model.dataTransferSubscriberFeePercent);
            decryption(model.decryption);
            description(model.description);
            entitlementArn(model.entitlementArn);
            ingestIp(model.ingestIp);
            ingestPort(model.ingestPort);
            mediaStreamSourceConfigurations(model.mediaStreamSourceConfigurations);
            name(model.name);
            senderControlPort(model.senderControlPort);
            senderIpAddress(model.senderIpAddress);
            sourceArn(model.sourceArn);
            transport(model.transport);
            vpcInterfaceName(model.vpcInterfaceName);
            whitelistCidr(model.whitelistCidr);
            gatewayBridgeSource(model.gatewayBridgeSource);
            peerIpAddress(model.peerIpAddress);
            routerIntegrationState(model.routerIntegrationState);
            routerIntegrationTransitDecryption(model.routerIntegrationTransitDecryption);
            connectedRouterOutputArn(model.connectedRouterOutputArn);
        }

        public final Integer getDataTransferSubscriberFeePercent() {
            return dataTransferSubscriberFeePercent;
        }

        public final void setDataTransferSubscriberFeePercent(Integer dataTransferSubscriberFeePercent) {
            this.dataTransferSubscriberFeePercent = dataTransferSubscriberFeePercent;
        }

        @Override
        public final Builder dataTransferSubscriberFeePercent(Integer dataTransferSubscriberFeePercent) {
            this.dataTransferSubscriberFeePercent = dataTransferSubscriberFeePercent;
            return this;
        }

        public final Encryption.Builder getDecryption() {
            return decryption != null ? decryption.toBuilder() : null;
        }

        public final void setDecryption(Encryption.BuilderImpl decryption) {
            this.decryption = decryption != null ? decryption.build() : null;
        }

        @Override
        public final Builder decryption(Encryption decryption) {
            this.decryption = decryption;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getEntitlementArn() {
            return entitlementArn;
        }

        public final void setEntitlementArn(String entitlementArn) {
            this.entitlementArn = entitlementArn;
        }

        @Override
        public final Builder entitlementArn(String entitlementArn) {
            this.entitlementArn = entitlementArn;
            return this;
        }

        public final String getIngestIp() {
            return ingestIp;
        }

        public final void setIngestIp(String ingestIp) {
            this.ingestIp = ingestIp;
        }

        @Override
        public final Builder ingestIp(String ingestIp) {
            this.ingestIp = ingestIp;
            return this;
        }

        public final Integer getIngestPort() {
            return ingestPort;
        }

        public final void setIngestPort(Integer ingestPort) {
            this.ingestPort = ingestPort;
        }

        @Override
        public final Builder ingestPort(Integer ingestPort) {
            this.ingestPort = ingestPort;
            return this;
        }

        public final List<MediaStreamSourceConfiguration.Builder> getMediaStreamSourceConfigurations() {
            List<MediaStreamSourceConfiguration.Builder> result = ___listOfMediaStreamSourceConfigurationCopier
                    .copyToBuilder(this.mediaStreamSourceConfigurations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setMediaStreamSourceConfigurations(
                Collection<MediaStreamSourceConfiguration.BuilderImpl> mediaStreamSourceConfigurations) {
            this.mediaStreamSourceConfigurations = ___listOfMediaStreamSourceConfigurationCopier
                    .copyFromBuilder(mediaStreamSourceConfigurations);
        }

        @Override
        public final Builder mediaStreamSourceConfigurations(
                Collection<MediaStreamSourceConfiguration> mediaStreamSourceConfigurations) {
            this.mediaStreamSourceConfigurations = ___listOfMediaStreamSourceConfigurationCopier
                    .copy(mediaStreamSourceConfigurations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder mediaStreamSourceConfigurations(MediaStreamSourceConfiguration... mediaStreamSourceConfigurations) {
            mediaStreamSourceConfigurations(Arrays.asList(mediaStreamSourceConfigurations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder mediaStreamSourceConfigurations(
                Consumer<MediaStreamSourceConfiguration.Builder>... mediaStreamSourceConfigurations) {
            mediaStreamSourceConfigurations(Stream.of(mediaStreamSourceConfigurations)
                    .map(c -> MediaStreamSourceConfiguration.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Integer getSenderControlPort() {
            return senderControlPort;
        }

        public final void setSenderControlPort(Integer senderControlPort) {
            this.senderControlPort = senderControlPort;
        }

        @Override
        public final Builder senderControlPort(Integer senderControlPort) {
            this.senderControlPort = senderControlPort;
            return this;
        }

        public final String getSenderIpAddress() {
            return senderIpAddress;
        }

        public final void setSenderIpAddress(String senderIpAddress) {
            this.senderIpAddress = senderIpAddress;
        }

        @Override
        public final Builder senderIpAddress(String senderIpAddress) {
            this.senderIpAddress = senderIpAddress;
            return this;
        }

        public final String getSourceArn() {
            return sourceArn;
        }

        public final void setSourceArn(String sourceArn) {
            this.sourceArn = sourceArn;
        }

        @Override
        public final Builder sourceArn(String sourceArn) {
            this.sourceArn = sourceArn;
            return this;
        }

        public final Transport.Builder getTransport() {
            return transport != null ? transport.toBuilder() : null;
        }

        public final void setTransport(Transport.BuilderImpl transport) {
            this.transport = transport != null ? transport.build() : null;
        }

        @Override
        public final Builder transport(Transport transport) {
            this.transport = transport;
            return this;
        }

        public final String getVpcInterfaceName() {
            return vpcInterfaceName;
        }

        public final void setVpcInterfaceName(String vpcInterfaceName) {
            this.vpcInterfaceName = vpcInterfaceName;
        }

        @Override
        public final Builder vpcInterfaceName(String vpcInterfaceName) {
            this.vpcInterfaceName = vpcInterfaceName;
            return this;
        }

        public final String getWhitelistCidr() {
            return whitelistCidr;
        }

        public final void setWhitelistCidr(String whitelistCidr) {
            this.whitelistCidr = whitelistCidr;
        }

        @Override
        public final Builder whitelistCidr(String whitelistCidr) {
            this.whitelistCidr = whitelistCidr;
            return this;
        }

        public final GatewayBridgeSource.Builder getGatewayBridgeSource() {
            return gatewayBridgeSource != null ? gatewayBridgeSource.toBuilder() : null;
        }

        public final void setGatewayBridgeSource(GatewayBridgeSource.BuilderImpl gatewayBridgeSource) {
            this.gatewayBridgeSource = gatewayBridgeSource != null ? gatewayBridgeSource.build() : null;
        }

        @Override
        public final Builder gatewayBridgeSource(GatewayBridgeSource gatewayBridgeSource) {
            this.gatewayBridgeSource = gatewayBridgeSource;
            return this;
        }

        public final String getPeerIpAddress() {
            return peerIpAddress;
        }

        public final void setPeerIpAddress(String peerIpAddress) {
            this.peerIpAddress = peerIpAddress;
        }

        @Override
        public final Builder peerIpAddress(String peerIpAddress) {
            this.peerIpAddress = peerIpAddress;
            return this;
        }

        public final String getRouterIntegrationState() {
            return routerIntegrationState;
        }

        public final void setRouterIntegrationState(String routerIntegrationState) {
            this.routerIntegrationState = routerIntegrationState;
        }

        @Override
        public final Builder routerIntegrationState(String routerIntegrationState) {
            this.routerIntegrationState = routerIntegrationState;
            return this;
        }

        @Override
        public final Builder routerIntegrationState(State routerIntegrationState) {
            this.routerIntegrationState(routerIntegrationState == null ? null : routerIntegrationState.toString());
            return this;
        }

        public final FlowTransitEncryption.Builder getRouterIntegrationTransitDecryption() {
            return routerIntegrationTransitDecryption != null ? routerIntegrationTransitDecryption.toBuilder() : null;
        }

        public final void setRouterIntegrationTransitDecryption(
                FlowTransitEncryption.BuilderImpl routerIntegrationTransitDecryption) {
            this.routerIntegrationTransitDecryption = routerIntegrationTransitDecryption != null ? routerIntegrationTransitDecryption
                    .build() : null;
        }

        @Override
        public final Builder routerIntegrationTransitDecryption(FlowTransitEncryption routerIntegrationTransitDecryption) {
            this.routerIntegrationTransitDecryption = routerIntegrationTransitDecryption;
            return this;
        }

        public final String getConnectedRouterOutputArn() {
            return connectedRouterOutputArn;
        }

        public final void setConnectedRouterOutputArn(String connectedRouterOutputArn) {
            this.connectedRouterOutputArn = connectedRouterOutputArn;
        }

        @Override
        public final Builder connectedRouterOutputArn(String connectedRouterOutputArn) {
            this.connectedRouterOutputArn = connectedRouterOutputArn;
            return this;
        }

        @Override
        public Source build() {
            return new Source(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
