/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediaconnect.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A router input in AWS Elemental MediaConnect. A router input is a source of media content that can be routed to one
 * or more router outputs.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RouterInput implements SdkPojo, Serializable, ToCopyableBuilder<RouterInput.Builder, RouterInput> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(RouterInput::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(RouterInput::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(RouterInput::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(RouterInput::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<String> INPUT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InputType").getter(getter(RouterInput::inputTypeAsString)).setter(setter(Builder::inputType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inputType").build()).build();

    private static final SdkField<RouterInputConfiguration> CONFIGURATION_FIELD = SdkField
            .<RouterInputConfiguration> builder(MarshallingType.SDK_POJO).memberName("Configuration")
            .getter(getter(RouterInput::configuration)).setter(setter(Builder::configuration))
            .constructor(RouterInputConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("configuration").build()).build();

    private static final SdkField<Integer> ROUTED_OUTPUTS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("RoutedOutputs").getter(getter(RouterInput::routedOutputs)).setter(setter(Builder::routedOutputs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("routedOutputs").build()).build();

    private static final SdkField<Integer> MAXIMUM_ROUTED_OUTPUTS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaximumRoutedOutputs").getter(getter(RouterInput::maximumRoutedOutputs))
            .setter(setter(Builder::maximumRoutedOutputs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maximumRoutedOutputs").build())
            .build();

    private static final SdkField<String> REGION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RegionName").getter(getter(RouterInput::regionName)).setter(setter(Builder::regionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("regionName").build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AvailabilityZone").getter(getter(RouterInput::availabilityZone))
            .setter(setter(Builder::availabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("availabilityZone").build()).build();

    private static final SdkField<Long> MAXIMUM_BITRATE_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("MaximumBitrate").getter(getter(RouterInput::maximumBitrate)).setter(setter(Builder::maximumBitrate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maximumBitrate").build()).build();

    private static final SdkField<String> TIER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Tier")
            .getter(getter(RouterInput::tierAsString)).setter(setter(Builder::tier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tier").build()).build();

    private static final SdkField<String> ROUTING_SCOPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RoutingScope").getter(getter(RouterInput::routingScopeAsString)).setter(setter(Builder::routingScope))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("routingScope").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedAt")
            .getter(getter(RouterInput::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("UpdatedAt")
            .getter(getter(RouterInput::updatedAt))
            .setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<List<RouterInputMessage>> MESSAGES_FIELD = SdkField
            .<List<RouterInputMessage>> builder(MarshallingType.LIST)
            .memberName("Messages")
            .getter(getter(RouterInput::messages))
            .setter(setter(Builder::messages))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("messages").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RouterInputMessage> builder(MarshallingType.SDK_POJO)
                                            .constructor(RouterInputMessage::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<RouterInputTransitEncryption> TRANSIT_ENCRYPTION_FIELD = SdkField
            .<RouterInputTransitEncryption> builder(MarshallingType.SDK_POJO).memberName("TransitEncryption")
            .getter(getter(RouterInput::transitEncryption)).setter(setter(Builder::transitEncryption))
            .constructor(RouterInputTransitEncryption::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("transitEncryption").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Tags")
            .getter(getter(RouterInput::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<RouterInputStreamDetails> STREAM_DETAILS_FIELD = SdkField
            .<RouterInputStreamDetails> builder(MarshallingType.SDK_POJO).memberName("StreamDetails")
            .getter(getter(RouterInput::streamDetails)).setter(setter(Builder::streamDetails))
            .constructor(RouterInputStreamDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("streamDetails").build()).build();

    private static final SdkField<String> IP_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IpAddress").getter(getter(RouterInput::ipAddress)).setter(setter(Builder::ipAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ipAddress").build()).build();

    private static final SdkField<String> MAINTENANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MaintenanceType").getter(getter(RouterInput::maintenanceTypeAsString))
            .setter(setter(Builder::maintenanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maintenanceType").build()).build();

    private static final SdkField<MaintenanceConfiguration> MAINTENANCE_CONFIGURATION_FIELD = SdkField
            .<MaintenanceConfiguration> builder(MarshallingType.SDK_POJO).memberName("MaintenanceConfiguration")
            .getter(getter(RouterInput::maintenanceConfiguration)).setter(setter(Builder::maintenanceConfiguration))
            .constructor(MaintenanceConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maintenanceConfiguration").build())
            .build();

    private static final SdkField<String> MAINTENANCE_SCHEDULE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MaintenanceScheduleType").getter(getter(RouterInput::maintenanceScheduleTypeAsString))
            .setter(setter(Builder::maintenanceScheduleType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maintenanceScheduleType").build())
            .build();

    private static final SdkField<MaintenanceSchedule> MAINTENANCE_SCHEDULE_FIELD = SdkField
            .<MaintenanceSchedule> builder(MarshallingType.SDK_POJO).memberName("MaintenanceSchedule")
            .getter(getter(RouterInput::maintenanceSchedule)).setter(setter(Builder::maintenanceSchedule))
            .constructor(MaintenanceSchedule::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maintenanceSchedule").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections
            .unmodifiableList(Arrays.asList(NAME_FIELD, ARN_FIELD, ID_FIELD, STATE_FIELD, INPUT_TYPE_FIELD, CONFIGURATION_FIELD,
                    ROUTED_OUTPUTS_FIELD, MAXIMUM_ROUTED_OUTPUTS_FIELD, REGION_NAME_FIELD, AVAILABILITY_ZONE_FIELD,
                    MAXIMUM_BITRATE_FIELD, TIER_FIELD, ROUTING_SCOPE_FIELD, CREATED_AT_FIELD, UPDATED_AT_FIELD, MESSAGES_FIELD,
                    TRANSIT_ENCRYPTION_FIELD, TAGS_FIELD, STREAM_DETAILS_FIELD, IP_ADDRESS_FIELD, MAINTENANCE_TYPE_FIELD,
                    MAINTENANCE_CONFIGURATION_FIELD, MAINTENANCE_SCHEDULE_TYPE_FIELD, MAINTENANCE_SCHEDULE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String arn;

    private final String id;

    private final String state;

    private final String inputType;

    private final RouterInputConfiguration configuration;

    private final Integer routedOutputs;

    private final Integer maximumRoutedOutputs;

    private final String regionName;

    private final String availabilityZone;

    private final Long maximumBitrate;

    private final String tier;

    private final String routingScope;

    private final Instant createdAt;

    private final Instant updatedAt;

    private final List<RouterInputMessage> messages;

    private final RouterInputTransitEncryption transitEncryption;

    private final Map<String, String> tags;

    private final RouterInputStreamDetails streamDetails;

    private final String ipAddress;

    private final String maintenanceType;

    private final MaintenanceConfiguration maintenanceConfiguration;

    private final String maintenanceScheduleType;

    private final MaintenanceSchedule maintenanceSchedule;

    private RouterInput(BuilderImpl builder) {
        this.name = builder.name;
        this.arn = builder.arn;
        this.id = builder.id;
        this.state = builder.state;
        this.inputType = builder.inputType;
        this.configuration = builder.configuration;
        this.routedOutputs = builder.routedOutputs;
        this.maximumRoutedOutputs = builder.maximumRoutedOutputs;
        this.regionName = builder.regionName;
        this.availabilityZone = builder.availabilityZone;
        this.maximumBitrate = builder.maximumBitrate;
        this.tier = builder.tier;
        this.routingScope = builder.routingScope;
        this.createdAt = builder.createdAt;
        this.updatedAt = builder.updatedAt;
        this.messages = builder.messages;
        this.transitEncryption = builder.transitEncryption;
        this.tags = builder.tags;
        this.streamDetails = builder.streamDetails;
        this.ipAddress = builder.ipAddress;
        this.maintenanceType = builder.maintenanceType;
        this.maintenanceConfiguration = builder.maintenanceConfiguration;
        this.maintenanceScheduleType = builder.maintenanceScheduleType;
        this.maintenanceSchedule = builder.maintenanceSchedule;
    }

    /**
     * <p>
     * The name of the router input.
     * </p>
     * 
     * @return The name of the router input.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the router input.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the router input.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The unique identifier of the router input.
     * </p>
     * 
     * @return The unique identifier of the router input.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The current state of the router input.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link RouterInputState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the router input.
     * @see RouterInputState
     */
    public final RouterInputState state() {
        return RouterInputState.fromValue(state);
    }

    /**
     * <p>
     * The current state of the router input.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link RouterInputState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the router input.
     * @see RouterInputState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The type of the router input.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #inputType} will
     * return {@link RouterInputType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #inputTypeAsString}.
     * </p>
     * 
     * @return The type of the router input.
     * @see RouterInputType
     */
    public final RouterInputType inputType() {
        return RouterInputType.fromValue(inputType);
    }

    /**
     * <p>
     * The type of the router input.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #inputType} will
     * return {@link RouterInputType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #inputTypeAsString}.
     * </p>
     * 
     * @return The type of the router input.
     * @see RouterInputType
     */
    public final String inputTypeAsString() {
        return inputType;
    }

    /**
     * Returns the value of the Configuration property for this object.
     * 
     * @return The value of the Configuration property for this object.
     */
    public final RouterInputConfiguration configuration() {
        return configuration;
    }

    /**
     * <p>
     * The number of router outputs associated with the router input.
     * </p>
     * 
     * @return The number of router outputs associated with the router input.
     */
    public final Integer routedOutputs() {
        return routedOutputs;
    }

    /**
     * <p>
     * The maximum number of outputs that can be simultaneously routed to this input.
     * </p>
     * 
     * @return The maximum number of outputs that can be simultaneously routed to this input.
     */
    public final Integer maximumRoutedOutputs() {
        return maximumRoutedOutputs;
    }

    /**
     * <p>
     * The AWS Region where the router input is located.
     * </p>
     * 
     * @return The AWS Region where the router input is located.
     */
    public final String regionName() {
        return regionName;
    }

    /**
     * <p>
     * The Availability Zone of the router input.
     * </p>
     * 
     * @return The Availability Zone of the router input.
     */
    public final String availabilityZone() {
        return availabilityZone;
    }

    /**
     * <p>
     * The maximum bitrate for the router input.
     * </p>
     * 
     * @return The maximum bitrate for the router input.
     */
    public final Long maximumBitrate() {
        return maximumBitrate;
    }

    /**
     * <p>
     * The tier level of the router input.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tier} will return
     * {@link RouterInputTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tierAsString}.
     * </p>
     * 
     * @return The tier level of the router input.
     * @see RouterInputTier
     */
    public final RouterInputTier tier() {
        return RouterInputTier.fromValue(tier);
    }

    /**
     * <p>
     * The tier level of the router input.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tier} will return
     * {@link RouterInputTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tierAsString}.
     * </p>
     * 
     * @return The tier level of the router input.
     * @see RouterInputTier
     */
    public final String tierAsString() {
        return tier;
    }

    /**
     * <p>
     * Indicates whether the router input is configured for Regional or global routing.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #routingScope} will
     * return {@link RoutingScope#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #routingScopeAsString}.
     * </p>
     * 
     * @return Indicates whether the router input is configured for Regional or global routing.
     * @see RoutingScope
     */
    public final RoutingScope routingScope() {
        return RoutingScope.fromValue(routingScope);
    }

    /**
     * <p>
     * Indicates whether the router input is configured for Regional or global routing.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #routingScope} will
     * return {@link RoutingScope#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #routingScopeAsString}.
     * </p>
     * 
     * @return Indicates whether the router input is configured for Regional or global routing.
     * @see RoutingScope
     */
    public final String routingScopeAsString() {
        return routingScope;
    }

    /**
     * <p>
     * The timestamp when the router input was created.
     * </p>
     * 
     * @return The timestamp when the router input was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The timestamp when the router input was last updated.
     * </p>
     * 
     * @return The timestamp when the router input was last updated.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    /**
     * For responses, this returns true if the service returned a value for the Messages property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasMessages() {
        return messages != null && !(messages instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The messages associated with the router input.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMessages} method.
     * </p>
     * 
     * @return The messages associated with the router input.
     */
    public final List<RouterInputMessage> messages() {
        return messages;
    }

    /**
     * Returns the value of the TransitEncryption property for this object.
     * 
     * @return The value of the TransitEncryption property for this object.
     */
    public final RouterInputTransitEncryption transitEncryption() {
        return transitEncryption;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Key-value pairs that can be used to tag and organize this router input.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Key-value pairs that can be used to tag and organize this router input.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * Returns the value of the StreamDetails property for this object.
     * 
     * @return The value of the StreamDetails property for this object.
     */
    public final RouterInputStreamDetails streamDetails() {
        return streamDetails;
    }

    /**
     * <p>
     * The IP address of the router input.
     * </p>
     * 
     * @return The IP address of the router input.
     */
    public final String ipAddress() {
        return ipAddress;
    }

    /**
     * <p>
     * The type of maintenance configuration applied to this router input.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #maintenanceType}
     * will return {@link MaintenanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #maintenanceTypeAsString}.
     * </p>
     * 
     * @return The type of maintenance configuration applied to this router input.
     * @see MaintenanceType
     */
    public final MaintenanceType maintenanceType() {
        return MaintenanceType.fromValue(maintenanceType);
    }

    /**
     * <p>
     * The type of maintenance configuration applied to this router input.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #maintenanceType}
     * will return {@link MaintenanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #maintenanceTypeAsString}.
     * </p>
     * 
     * @return The type of maintenance configuration applied to this router input.
     * @see MaintenanceType
     */
    public final String maintenanceTypeAsString() {
        return maintenanceType;
    }

    /**
     * <p>
     * The maintenance configuration settings applied to this router input.
     * </p>
     * 
     * @return The maintenance configuration settings applied to this router input.
     */
    public final MaintenanceConfiguration maintenanceConfiguration() {
        return maintenanceConfiguration;
    }

    /**
     * <p>
     * The type of maintenance schedule currently in effect for this router input.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #maintenanceScheduleType} will return {@link MaintenanceScheduleType#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #maintenanceScheduleTypeAsString}.
     * </p>
     * 
     * @return The type of maintenance schedule currently in effect for this router input.
     * @see MaintenanceScheduleType
     */
    public final MaintenanceScheduleType maintenanceScheduleType() {
        return MaintenanceScheduleType.fromValue(maintenanceScheduleType);
    }

    /**
     * <p>
     * The type of maintenance schedule currently in effect for this router input.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #maintenanceScheduleType} will return {@link MaintenanceScheduleType#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #maintenanceScheduleTypeAsString}.
     * </p>
     * 
     * @return The type of maintenance schedule currently in effect for this router input.
     * @see MaintenanceScheduleType
     */
    public final String maintenanceScheduleTypeAsString() {
        return maintenanceScheduleType;
    }

    /**
     * <p>
     * The current maintenance schedule details for this router input.
     * </p>
     * 
     * @return The current maintenance schedule details for this router input.
     */
    public final MaintenanceSchedule maintenanceSchedule() {
        return maintenanceSchedule;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(inputTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(configuration());
        hashCode = 31 * hashCode + Objects.hashCode(routedOutputs());
        hashCode = 31 * hashCode + Objects.hashCode(maximumRoutedOutputs());
        hashCode = 31 * hashCode + Objects.hashCode(regionName());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(maximumBitrate());
        hashCode = 31 * hashCode + Objects.hashCode(tierAsString());
        hashCode = 31 * hashCode + Objects.hashCode(routingScopeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(hasMessages() ? messages() : null);
        hashCode = 31 * hashCode + Objects.hashCode(transitEncryption());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(streamDetails());
        hashCode = 31 * hashCode + Objects.hashCode(ipAddress());
        hashCode = 31 * hashCode + Objects.hashCode(maintenanceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(maintenanceConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(maintenanceScheduleTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(maintenanceSchedule());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RouterInput)) {
            return false;
        }
        RouterInput other = (RouterInput) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(arn(), other.arn()) && Objects.equals(id(), other.id())
                && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(inputTypeAsString(), other.inputTypeAsString())
                && Objects.equals(configuration(), other.configuration())
                && Objects.equals(routedOutputs(), other.routedOutputs())
                && Objects.equals(maximumRoutedOutputs(), other.maximumRoutedOutputs())
                && Objects.equals(regionName(), other.regionName())
                && Objects.equals(availabilityZone(), other.availabilityZone())
                && Objects.equals(maximumBitrate(), other.maximumBitrate())
                && Objects.equals(tierAsString(), other.tierAsString())
                && Objects.equals(routingScopeAsString(), other.routingScopeAsString())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(updatedAt(), other.updatedAt())
                && hasMessages() == other.hasMessages() && Objects.equals(messages(), other.messages())
                && Objects.equals(transitEncryption(), other.transitEncryption()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(streamDetails(), other.streamDetails())
                && Objects.equals(ipAddress(), other.ipAddress())
                && Objects.equals(maintenanceTypeAsString(), other.maintenanceTypeAsString())
                && Objects.equals(maintenanceConfiguration(), other.maintenanceConfiguration())
                && Objects.equals(maintenanceScheduleTypeAsString(), other.maintenanceScheduleTypeAsString())
                && Objects.equals(maintenanceSchedule(), other.maintenanceSchedule());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RouterInput").add("Name", name()).add("Arn", arn()).add("Id", id())
                .add("State", stateAsString()).add("InputType", inputTypeAsString()).add("Configuration", configuration())
                .add("RoutedOutputs", routedOutputs()).add("MaximumRoutedOutputs", maximumRoutedOutputs())
                .add("RegionName", regionName()).add("AvailabilityZone", availabilityZone())
                .add("MaximumBitrate", maximumBitrate()).add("Tier", tierAsString()).add("RoutingScope", routingScopeAsString())
                .add("CreatedAt", createdAt()).add("UpdatedAt", updatedAt()).add("Messages", hasMessages() ? messages() : null)
                .add("TransitEncryption", transitEncryption()).add("Tags", hasTags() ? tags() : null)
                .add("StreamDetails", streamDetails()).add("IpAddress", ipAddress())
                .add("MaintenanceType", maintenanceTypeAsString()).add("MaintenanceConfiguration", maintenanceConfiguration())
                .add("MaintenanceScheduleType", maintenanceScheduleTypeAsString())
                .add("MaintenanceSchedule", maintenanceSchedule()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "InputType":
            return Optional.ofNullable(clazz.cast(inputTypeAsString()));
        case "Configuration":
            return Optional.ofNullable(clazz.cast(configuration()));
        case "RoutedOutputs":
            return Optional.ofNullable(clazz.cast(routedOutputs()));
        case "MaximumRoutedOutputs":
            return Optional.ofNullable(clazz.cast(maximumRoutedOutputs()));
        case "RegionName":
            return Optional.ofNullable(clazz.cast(regionName()));
        case "AvailabilityZone":
            return Optional.ofNullable(clazz.cast(availabilityZone()));
        case "MaximumBitrate":
            return Optional.ofNullable(clazz.cast(maximumBitrate()));
        case "Tier":
            return Optional.ofNullable(clazz.cast(tierAsString()));
        case "RoutingScope":
            return Optional.ofNullable(clazz.cast(routingScopeAsString()));
        case "CreatedAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "UpdatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        case "Messages":
            return Optional.ofNullable(clazz.cast(messages()));
        case "TransitEncryption":
            return Optional.ofNullable(clazz.cast(transitEncryption()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "StreamDetails":
            return Optional.ofNullable(clazz.cast(streamDetails()));
        case "IpAddress":
            return Optional.ofNullable(clazz.cast(ipAddress()));
        case "MaintenanceType":
            return Optional.ofNullable(clazz.cast(maintenanceTypeAsString()));
        case "MaintenanceConfiguration":
            return Optional.ofNullable(clazz.cast(maintenanceConfiguration()));
        case "MaintenanceScheduleType":
            return Optional.ofNullable(clazz.cast(maintenanceScheduleTypeAsString()));
        case "MaintenanceSchedule":
            return Optional.ofNullable(clazz.cast(maintenanceSchedule()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("name", NAME_FIELD);
        map.put("arn", ARN_FIELD);
        map.put("id", ID_FIELD);
        map.put("state", STATE_FIELD);
        map.put("inputType", INPUT_TYPE_FIELD);
        map.put("configuration", CONFIGURATION_FIELD);
        map.put("routedOutputs", ROUTED_OUTPUTS_FIELD);
        map.put("maximumRoutedOutputs", MAXIMUM_ROUTED_OUTPUTS_FIELD);
        map.put("regionName", REGION_NAME_FIELD);
        map.put("availabilityZone", AVAILABILITY_ZONE_FIELD);
        map.put("maximumBitrate", MAXIMUM_BITRATE_FIELD);
        map.put("tier", TIER_FIELD);
        map.put("routingScope", ROUTING_SCOPE_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("updatedAt", UPDATED_AT_FIELD);
        map.put("messages", MESSAGES_FIELD);
        map.put("transitEncryption", TRANSIT_ENCRYPTION_FIELD);
        map.put("tags", TAGS_FIELD);
        map.put("streamDetails", STREAM_DETAILS_FIELD);
        map.put("ipAddress", IP_ADDRESS_FIELD);
        map.put("maintenanceType", MAINTENANCE_TYPE_FIELD);
        map.put("maintenanceConfiguration", MAINTENANCE_CONFIGURATION_FIELD);
        map.put("maintenanceScheduleType", MAINTENANCE_SCHEDULE_TYPE_FIELD);
        map.put("maintenanceSchedule", MAINTENANCE_SCHEDULE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RouterInput, T> g) {
        return obj -> g.apply((RouterInput) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RouterInput> {
        /**
         * <p>
         * The name of the router input.
         * </p>
         * 
         * @param name
         *        The name of the router input.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the router input.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the router input.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The unique identifier of the router input.
         * </p>
         * 
         * @param id
         *        The unique identifier of the router input.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The current state of the router input.
         * </p>
         * 
         * @param state
         *        The current state of the router input.
         * @see RouterInputState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouterInputState
         */
        Builder state(String state);

        /**
         * <p>
         * The current state of the router input.
         * </p>
         * 
         * @param state
         *        The current state of the router input.
         * @see RouterInputState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouterInputState
         */
        Builder state(RouterInputState state);

        /**
         * <p>
         * The type of the router input.
         * </p>
         * 
         * @param inputType
         *        The type of the router input.
         * @see RouterInputType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouterInputType
         */
        Builder inputType(String inputType);

        /**
         * <p>
         * The type of the router input.
         * </p>
         * 
         * @param inputType
         *        The type of the router input.
         * @see RouterInputType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouterInputType
         */
        Builder inputType(RouterInputType inputType);

        /**
         * Sets the value of the Configuration property for this object.
         *
         * @param configuration
         *        The new value for the Configuration property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configuration(RouterInputConfiguration configuration);

        /**
         * Sets the value of the Configuration property for this object.
         *
         * This is a convenience method that creates an instance of the {@link RouterInputConfiguration.Builder}
         * avoiding the need to create one manually via {@link RouterInputConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RouterInputConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #configuration(RouterInputConfiguration)}.
         * 
         * @param configuration
         *        a consumer that will call methods on {@link RouterInputConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #configuration(RouterInputConfiguration)
         */
        default Builder configuration(Consumer<RouterInputConfiguration.Builder> configuration) {
            return configuration(RouterInputConfiguration.builder().applyMutation(configuration).build());
        }

        /**
         * <p>
         * The number of router outputs associated with the router input.
         * </p>
         * 
         * @param routedOutputs
         *        The number of router outputs associated with the router input.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routedOutputs(Integer routedOutputs);

        /**
         * <p>
         * The maximum number of outputs that can be simultaneously routed to this input.
         * </p>
         * 
         * @param maximumRoutedOutputs
         *        The maximum number of outputs that can be simultaneously routed to this input.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maximumRoutedOutputs(Integer maximumRoutedOutputs);

        /**
         * <p>
         * The AWS Region where the router input is located.
         * </p>
         * 
         * @param regionName
         *        The AWS Region where the router input is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regionName(String regionName);

        /**
         * <p>
         * The Availability Zone of the router input.
         * </p>
         * 
         * @param availabilityZone
         *        The Availability Zone of the router input.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZone(String availabilityZone);

        /**
         * <p>
         * The maximum bitrate for the router input.
         * </p>
         * 
         * @param maximumBitrate
         *        The maximum bitrate for the router input.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maximumBitrate(Long maximumBitrate);

        /**
         * <p>
         * The tier level of the router input.
         * </p>
         * 
         * @param tier
         *        The tier level of the router input.
         * @see RouterInputTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouterInputTier
         */
        Builder tier(String tier);

        /**
         * <p>
         * The tier level of the router input.
         * </p>
         * 
         * @param tier
         *        The tier level of the router input.
         * @see RouterInputTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouterInputTier
         */
        Builder tier(RouterInputTier tier);

        /**
         * <p>
         * Indicates whether the router input is configured for Regional or global routing.
         * </p>
         * 
         * @param routingScope
         *        Indicates whether the router input is configured for Regional or global routing.
         * @see RoutingScope
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RoutingScope
         */
        Builder routingScope(String routingScope);

        /**
         * <p>
         * Indicates whether the router input is configured for Regional or global routing.
         * </p>
         * 
         * @param routingScope
         *        Indicates whether the router input is configured for Regional or global routing.
         * @see RoutingScope
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RoutingScope
         */
        Builder routingScope(RoutingScope routingScope);

        /**
         * <p>
         * The timestamp when the router input was created.
         * </p>
         * 
         * @param createdAt
         *        The timestamp when the router input was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The timestamp when the router input was last updated.
         * </p>
         * 
         * @param updatedAt
         *        The timestamp when the router input was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);

        /**
         * <p>
         * The messages associated with the router input.
         * </p>
         * 
         * @param messages
         *        The messages associated with the router input.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messages(Collection<RouterInputMessage> messages);

        /**
         * <p>
         * The messages associated with the router input.
         * </p>
         * 
         * @param messages
         *        The messages associated with the router input.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messages(RouterInputMessage... messages);

        /**
         * <p>
         * The messages associated with the router input.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.mediaconnect.model.RouterInputMessage.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.mediaconnect.model.RouterInputMessage#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.mediaconnect.model.RouterInputMessage.Builder#build()} is called
         * immediately and its result is passed to {@link #messages(List<RouterInputMessage>)}.
         * 
         * @param messages
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.mediaconnect.model.RouterInputMessage.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #messages(java.util.Collection<RouterInputMessage>)
         */
        Builder messages(Consumer<RouterInputMessage.Builder>... messages);

        /**
         * Sets the value of the TransitEncryption property for this object.
         *
         * @param transitEncryption
         *        The new value for the TransitEncryption property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transitEncryption(RouterInputTransitEncryption transitEncryption);

        /**
         * Sets the value of the TransitEncryption property for this object.
         *
         * This is a convenience method that creates an instance of the {@link RouterInputTransitEncryption.Builder}
         * avoiding the need to create one manually via {@link RouterInputTransitEncryption#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RouterInputTransitEncryption.Builder#build()} is called
         * immediately and its result is passed to {@link #transitEncryption(RouterInputTransitEncryption)}.
         * 
         * @param transitEncryption
         *        a consumer that will call methods on {@link RouterInputTransitEncryption.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #transitEncryption(RouterInputTransitEncryption)
         */
        default Builder transitEncryption(Consumer<RouterInputTransitEncryption.Builder> transitEncryption) {
            return transitEncryption(RouterInputTransitEncryption.builder().applyMutation(transitEncryption).build());
        }

        /**
         * <p>
         * Key-value pairs that can be used to tag and organize this router input.
         * </p>
         * 
         * @param tags
         *        Key-value pairs that can be used to tag and organize this router input.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * Sets the value of the StreamDetails property for this object.
         *
         * @param streamDetails
         *        The new value for the StreamDetails property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamDetails(RouterInputStreamDetails streamDetails);

        /**
         * Sets the value of the StreamDetails property for this object.
         *
         * This is a convenience method that creates an instance of the {@link RouterInputStreamDetails.Builder}
         * avoiding the need to create one manually via {@link RouterInputStreamDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RouterInputStreamDetails.Builder#build()} is called immediately
         * and its result is passed to {@link #streamDetails(RouterInputStreamDetails)}.
         * 
         * @param streamDetails
         *        a consumer that will call methods on {@link RouterInputStreamDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #streamDetails(RouterInputStreamDetails)
         */
        default Builder streamDetails(Consumer<RouterInputStreamDetails.Builder> streamDetails) {
            return streamDetails(RouterInputStreamDetails.builder().applyMutation(streamDetails).build());
        }

        /**
         * <p>
         * The IP address of the router input.
         * </p>
         * 
         * @param ipAddress
         *        The IP address of the router input.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipAddress(String ipAddress);

        /**
         * <p>
         * The type of maintenance configuration applied to this router input.
         * </p>
         * 
         * @param maintenanceType
         *        The type of maintenance configuration applied to this router input.
         * @see MaintenanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceType
         */
        Builder maintenanceType(String maintenanceType);

        /**
         * <p>
         * The type of maintenance configuration applied to this router input.
         * </p>
         * 
         * @param maintenanceType
         *        The type of maintenance configuration applied to this router input.
         * @see MaintenanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceType
         */
        Builder maintenanceType(MaintenanceType maintenanceType);

        /**
         * <p>
         * The maintenance configuration settings applied to this router input.
         * </p>
         * 
         * @param maintenanceConfiguration
         *        The maintenance configuration settings applied to this router input.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maintenanceConfiguration(MaintenanceConfiguration maintenanceConfiguration);

        /**
         * <p>
         * The maintenance configuration settings applied to this router input.
         * </p>
         * This is a convenience method that creates an instance of the {@link MaintenanceConfiguration.Builder}
         * avoiding the need to create one manually via {@link MaintenanceConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MaintenanceConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #maintenanceConfiguration(MaintenanceConfiguration)}.
         * 
         * @param maintenanceConfiguration
         *        a consumer that will call methods on {@link MaintenanceConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #maintenanceConfiguration(MaintenanceConfiguration)
         */
        default Builder maintenanceConfiguration(Consumer<MaintenanceConfiguration.Builder> maintenanceConfiguration) {
            return maintenanceConfiguration(MaintenanceConfiguration.builder().applyMutation(maintenanceConfiguration).build());
        }

        /**
         * <p>
         * The type of maintenance schedule currently in effect for this router input.
         * </p>
         * 
         * @param maintenanceScheduleType
         *        The type of maintenance schedule currently in effect for this router input.
         * @see MaintenanceScheduleType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceScheduleType
         */
        Builder maintenanceScheduleType(String maintenanceScheduleType);

        /**
         * <p>
         * The type of maintenance schedule currently in effect for this router input.
         * </p>
         * 
         * @param maintenanceScheduleType
         *        The type of maintenance schedule currently in effect for this router input.
         * @see MaintenanceScheduleType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceScheduleType
         */
        Builder maintenanceScheduleType(MaintenanceScheduleType maintenanceScheduleType);

        /**
         * <p>
         * The current maintenance schedule details for this router input.
         * </p>
         * 
         * @param maintenanceSchedule
         *        The current maintenance schedule details for this router input.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maintenanceSchedule(MaintenanceSchedule maintenanceSchedule);

        /**
         * <p>
         * The current maintenance schedule details for this router input.
         * </p>
         * This is a convenience method that creates an instance of the {@link MaintenanceSchedule.Builder} avoiding the
         * need to create one manually via {@link MaintenanceSchedule#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MaintenanceSchedule.Builder#build()} is called immediately and
         * its result is passed to {@link #maintenanceSchedule(MaintenanceSchedule)}.
         * 
         * @param maintenanceSchedule
         *        a consumer that will call methods on {@link MaintenanceSchedule.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #maintenanceSchedule(MaintenanceSchedule)
         */
        default Builder maintenanceSchedule(Consumer<MaintenanceSchedule.Builder> maintenanceSchedule) {
            return maintenanceSchedule(MaintenanceSchedule.builder().applyMutation(maintenanceSchedule).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String arn;

        private String id;

        private String state;

        private String inputType;

        private RouterInputConfiguration configuration;

        private Integer routedOutputs;

        private Integer maximumRoutedOutputs;

        private String regionName;

        private String availabilityZone;

        private Long maximumBitrate;

        private String tier;

        private String routingScope;

        private Instant createdAt;

        private Instant updatedAt;

        private List<RouterInputMessage> messages = DefaultSdkAutoConstructList.getInstance();

        private RouterInputTransitEncryption transitEncryption;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private RouterInputStreamDetails streamDetails;

        private String ipAddress;

        private String maintenanceType;

        private MaintenanceConfiguration maintenanceConfiguration;

        private String maintenanceScheduleType;

        private MaintenanceSchedule maintenanceSchedule;

        private BuilderImpl() {
        }

        private BuilderImpl(RouterInput model) {
            name(model.name);
            arn(model.arn);
            id(model.id);
            state(model.state);
            inputType(model.inputType);
            configuration(model.configuration);
            routedOutputs(model.routedOutputs);
            maximumRoutedOutputs(model.maximumRoutedOutputs);
            regionName(model.regionName);
            availabilityZone(model.availabilityZone);
            maximumBitrate(model.maximumBitrate);
            tier(model.tier);
            routingScope(model.routingScope);
            createdAt(model.createdAt);
            updatedAt(model.updatedAt);
            messages(model.messages);
            transitEncryption(model.transitEncryption);
            tags(model.tags);
            streamDetails(model.streamDetails);
            ipAddress(model.ipAddress);
            maintenanceType(model.maintenanceType);
            maintenanceConfiguration(model.maintenanceConfiguration);
            maintenanceScheduleType(model.maintenanceScheduleType);
            maintenanceSchedule(model.maintenanceSchedule);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(RouterInputState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getInputType() {
            return inputType;
        }

        public final void setInputType(String inputType) {
            this.inputType = inputType;
        }

        @Override
        public final Builder inputType(String inputType) {
            this.inputType = inputType;
            return this;
        }

        @Override
        public final Builder inputType(RouterInputType inputType) {
            this.inputType(inputType == null ? null : inputType.toString());
            return this;
        }

        public final RouterInputConfiguration.Builder getConfiguration() {
            return configuration != null ? configuration.toBuilder() : null;
        }

        public final void setConfiguration(RouterInputConfiguration.BuilderImpl configuration) {
            this.configuration = configuration != null ? configuration.build() : null;
        }

        @Override
        public final Builder configuration(RouterInputConfiguration configuration) {
            this.configuration = configuration;
            return this;
        }

        public final Integer getRoutedOutputs() {
            return routedOutputs;
        }

        public final void setRoutedOutputs(Integer routedOutputs) {
            this.routedOutputs = routedOutputs;
        }

        @Override
        public final Builder routedOutputs(Integer routedOutputs) {
            this.routedOutputs = routedOutputs;
            return this;
        }

        public final Integer getMaximumRoutedOutputs() {
            return maximumRoutedOutputs;
        }

        public final void setMaximumRoutedOutputs(Integer maximumRoutedOutputs) {
            this.maximumRoutedOutputs = maximumRoutedOutputs;
        }

        @Override
        public final Builder maximumRoutedOutputs(Integer maximumRoutedOutputs) {
            this.maximumRoutedOutputs = maximumRoutedOutputs;
            return this;
        }

        public final String getRegionName() {
            return regionName;
        }

        public final void setRegionName(String regionName) {
            this.regionName = regionName;
        }

        @Override
        public final Builder regionName(String regionName) {
            this.regionName = regionName;
            return this;
        }

        public final String getAvailabilityZone() {
            return availabilityZone;
        }

        public final void setAvailabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
        }

        @Override
        public final Builder availabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
            return this;
        }

        public final Long getMaximumBitrate() {
            return maximumBitrate;
        }

        public final void setMaximumBitrate(Long maximumBitrate) {
            this.maximumBitrate = maximumBitrate;
        }

        @Override
        public final Builder maximumBitrate(Long maximumBitrate) {
            this.maximumBitrate = maximumBitrate;
            return this;
        }

        public final String getTier() {
            return tier;
        }

        public final void setTier(String tier) {
            this.tier = tier;
        }

        @Override
        public final Builder tier(String tier) {
            this.tier = tier;
            return this;
        }

        @Override
        public final Builder tier(RouterInputTier tier) {
            this.tier(tier == null ? null : tier.toString());
            return this;
        }

        public final String getRoutingScope() {
            return routingScope;
        }

        public final void setRoutingScope(String routingScope) {
            this.routingScope = routingScope;
        }

        @Override
        public final Builder routingScope(String routingScope) {
            this.routingScope = routingScope;
            return this;
        }

        @Override
        public final Builder routingScope(RoutingScope routingScope) {
            this.routingScope(routingScope == null ? null : routingScope.toString());
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        public final List<RouterInputMessage.Builder> getMessages() {
            List<RouterInputMessage.Builder> result = RouterInputMessagesCopier.copyToBuilder(this.messages);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setMessages(Collection<RouterInputMessage.BuilderImpl> messages) {
            this.messages = RouterInputMessagesCopier.copyFromBuilder(messages);
        }

        @Override
        public final Builder messages(Collection<RouterInputMessage> messages) {
            this.messages = RouterInputMessagesCopier.copy(messages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder messages(RouterInputMessage... messages) {
            messages(Arrays.asList(messages));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder messages(Consumer<RouterInputMessage.Builder>... messages) {
            messages(Stream.of(messages).map(c -> RouterInputMessage.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final RouterInputTransitEncryption.Builder getTransitEncryption() {
            return transitEncryption != null ? transitEncryption.toBuilder() : null;
        }

        public final void setTransitEncryption(RouterInputTransitEncryption.BuilderImpl transitEncryption) {
            this.transitEncryption = transitEncryption != null ? transitEncryption.build() : null;
        }

        @Override
        public final Builder transitEncryption(RouterInputTransitEncryption transitEncryption) {
            this.transitEncryption = transitEncryption;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = ___mapOfStringCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = ___mapOfStringCopier.copy(tags);
            return this;
        }

        public final RouterInputStreamDetails.Builder getStreamDetails() {
            return streamDetails != null ? streamDetails.toBuilder() : null;
        }

        public final void setStreamDetails(RouterInputStreamDetails.BuilderImpl streamDetails) {
            this.streamDetails = streamDetails != null ? streamDetails.build() : null;
        }

        @Override
        public final Builder streamDetails(RouterInputStreamDetails streamDetails) {
            this.streamDetails = streamDetails;
            return this;
        }

        public final String getIpAddress() {
            return ipAddress;
        }

        public final void setIpAddress(String ipAddress) {
            this.ipAddress = ipAddress;
        }

        @Override
        public final Builder ipAddress(String ipAddress) {
            this.ipAddress = ipAddress;
            return this;
        }

        public final String getMaintenanceType() {
            return maintenanceType;
        }

        public final void setMaintenanceType(String maintenanceType) {
            this.maintenanceType = maintenanceType;
        }

        @Override
        public final Builder maintenanceType(String maintenanceType) {
            this.maintenanceType = maintenanceType;
            return this;
        }

        @Override
        public final Builder maintenanceType(MaintenanceType maintenanceType) {
            this.maintenanceType(maintenanceType == null ? null : maintenanceType.toString());
            return this;
        }

        public final MaintenanceConfiguration.Builder getMaintenanceConfiguration() {
            return maintenanceConfiguration != null ? maintenanceConfiguration.toBuilder() : null;
        }

        public final void setMaintenanceConfiguration(MaintenanceConfiguration.BuilderImpl maintenanceConfiguration) {
            this.maintenanceConfiguration = maintenanceConfiguration != null ? maintenanceConfiguration.build() : null;
        }

        @Override
        public final Builder maintenanceConfiguration(MaintenanceConfiguration maintenanceConfiguration) {
            this.maintenanceConfiguration = maintenanceConfiguration;
            return this;
        }

        public final String getMaintenanceScheduleType() {
            return maintenanceScheduleType;
        }

        public final void setMaintenanceScheduleType(String maintenanceScheduleType) {
            this.maintenanceScheduleType = maintenanceScheduleType;
        }

        @Override
        public final Builder maintenanceScheduleType(String maintenanceScheduleType) {
            this.maintenanceScheduleType = maintenanceScheduleType;
            return this;
        }

        @Override
        public final Builder maintenanceScheduleType(MaintenanceScheduleType maintenanceScheduleType) {
            this.maintenanceScheduleType(maintenanceScheduleType == null ? null : maintenanceScheduleType.toString());
            return this;
        }

        public final MaintenanceSchedule.Builder getMaintenanceSchedule() {
            return maintenanceSchedule != null ? maintenanceSchedule.toBuilder() : null;
        }

        public final void setMaintenanceSchedule(MaintenanceSchedule.BuilderImpl maintenanceSchedule) {
            this.maintenanceSchedule = maintenanceSchedule != null ? maintenanceSchedule.build() : null;
        }

        @Override
        public final Builder maintenanceSchedule(MaintenanceSchedule maintenanceSchedule) {
            this.maintenanceSchedule = maintenanceSchedule;
            return this;
        }

        @Override
        public RouterInput build() {
            return new RouterInput(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
