/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.marketplacemetering;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.marketplacemetering.model.BatchMeterUsageRequest;
import software.amazon.awssdk.services.marketplacemetering.model.BatchMeterUsageResponse;
import software.amazon.awssdk.services.marketplacemetering.model.CustomerNotEntitledException;
import software.amazon.awssdk.services.marketplacemetering.model.DisabledApiException;
import software.amazon.awssdk.services.marketplacemetering.model.DuplicateRequestException;
import software.amazon.awssdk.services.marketplacemetering.model.ExpiredTokenException;
import software.amazon.awssdk.services.marketplacemetering.model.InternalServiceErrorException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidCustomerIdentifierException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidEndpointRegionException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidProductCodeException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidPublicKeyVersionException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidRegionException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidTagException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidTokenException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidUsageAllocationsException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidUsageDimensionException;
import software.amazon.awssdk.services.marketplacemetering.model.MarketplaceMeteringException;
import software.amazon.awssdk.services.marketplacemetering.model.MeterUsageRequest;
import software.amazon.awssdk.services.marketplacemetering.model.MeterUsageResponse;
import software.amazon.awssdk.services.marketplacemetering.model.PlatformNotSupportedException;
import software.amazon.awssdk.services.marketplacemetering.model.RegisterUsageRequest;
import software.amazon.awssdk.services.marketplacemetering.model.RegisterUsageResponse;
import software.amazon.awssdk.services.marketplacemetering.model.ResolveCustomerRequest;
import software.amazon.awssdk.services.marketplacemetering.model.ResolveCustomerResponse;
import software.amazon.awssdk.services.marketplacemetering.model.ThrottlingException;
import software.amazon.awssdk.services.marketplacemetering.model.TimestampOutOfBoundsException;
import software.amazon.awssdk.services.marketplacemetering.transform.BatchMeterUsageRequestMarshaller;
import software.amazon.awssdk.services.marketplacemetering.transform.MeterUsageRequestMarshaller;
import software.amazon.awssdk.services.marketplacemetering.transform.RegisterUsageRequestMarshaller;
import software.amazon.awssdk.services.marketplacemetering.transform.ResolveCustomerRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link MarketplaceMeteringAsyncClient}.
 *
 * @see MarketplaceMeteringAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMarketplaceMeteringAsyncClient implements MarketplaceMeteringAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMarketplaceMeteringAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMarketplaceMeteringAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * BatchMeterUsage is called from a SaaS application listed on the AWS Marketplace to post metering records for a
     * set of customers.
     * </p>
     * <p>
     * For identical requests, the API is idempotent; requests can be retried with the same records or a subset of the
     * input records.
     * </p>
     * <p>
     * Every request to BatchMeterUsage is for one product. If you need to meter usage for multiple products, you must
     * make multiple calls to BatchMeterUsage.
     * </p>
     * <p>
     * BatchMeterUsage can process up to 25 UsageRecords at a time.
     * </p>
     * <p>
     * A UsageRecord can optionally include multiple usage allocations, to provide customers with usagedata split into
     * buckets by tags that you define (or allow the customer to define).
     * </p>
     * <p>
     * BatchMeterUsage requests must be less than 1MB in size.
     * </p>
     *
     * @param batchMeterUsageRequest
     *        A BatchMeterUsageRequest contains UsageRecords, which indicate quantities of usage within your
     *        application.
     * @return A Java Future containing the result of the BatchMeterUsage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException An internal error has occurred. Retry your request. If the problem
     *         persists, post a message with details on the AWS forums.</li>
     *         <li>InvalidProductCodeException The product code passed does not match the product code used for
     *         publishing the product.</li>
     *         <li>InvalidUsageDimensionException The usage dimension does not match one of the UsageDimensions
     *         associated with products.</li>
     *         <li>InvalidTagException The tag is invalid, or the number of tags is greater than 5.</li>
     *         <li>InvalidUsageAllocationsException The usage allocation objects are invalid, or the number of
     *         allocations is greater than 500 for a single usage record.</li>
     *         <li>InvalidCustomerIdentifierException You have metered usage for a CustomerIdentifier that does not
     *         exist.</li>
     *         <li>TimestampOutOfBoundsException The timestamp value passed in the meterUsage() is out of allowed range.
     *         </li>
     *         <li>ThrottlingException The calls to the API are throttled.</li>
     *         <li>DisabledApiException The API is disabled in the Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceMeteringException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceMeteringAsyncClient.BatchMeterUsage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/meteringmarketplace-2016-01-14/BatchMeterUsage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchMeterUsageResponse> batchMeterUsage(BatchMeterUsageRequest batchMeterUsageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchMeterUsageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Metering");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchMeterUsage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchMeterUsageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchMeterUsageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchMeterUsageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchMeterUsageRequest, BatchMeterUsageResponse>()
                            .withOperationName("BatchMeterUsage")
                            .withMarshaller(new BatchMeterUsageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchMeterUsageRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = batchMeterUsageRequest.overrideConfiguration().orElse(null);
            CompletableFuture<BatchMeterUsageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * API to emit metering records. For identical requests, the API is idempotent. It simply returns the metering
     * record ID.
     * </p>
     * <p>
     * MeterUsage is authenticated on the buyer's AWS account using credentials from the EC2 instance, ECS task, or EKS
     * pod.
     * </p>
     * <p>
     * MeterUsage can optionally include multiple usage allocations, to provide customers with usage data split into
     * buckets by tags that you define (or allow the customer to define).
     * </p>
     *
     * @param meterUsageRequest
     * @return A Java Future containing the result of the MeterUsage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException An internal error has occurred. Retry your request. If the problem
     *         persists, post a message with details on the AWS forums.</li>
     *         <li>InvalidProductCodeException The product code passed does not match the product code used for
     *         publishing the product.</li>
     *         <li>InvalidUsageDimensionException The usage dimension does not match one of the UsageDimensions
     *         associated with products.</li>
     *         <li>InvalidTagException The tag is invalid, or the number of tags is greater than 5.</li>
     *         <li>InvalidUsageAllocationsException The usage allocation objects are invalid, or the number of
     *         allocations is greater than 500 for a single usage record.</li>
     *         <li>InvalidEndpointRegionException The endpoint being called is in a AWS Region different from your EC2
     *         instance, ECS task, or EKS pod. The Region of the Metering Service endpoint and the AWS Region of the
     *         resource must match.</li>
     *         <li>TimestampOutOfBoundsException The timestamp value passed in the meterUsage() is out of allowed range.
     *         </li>
     *         <li>DuplicateRequestException A metering record has already been emitted by the same EC2 instance, ECS
     *         task, or EKS pod for the given {usageDimension, timestamp} with a different usageQuantity.</li>
     *         <li>ThrottlingException The calls to the API are throttled.</li>
     *         <li>CustomerNotEntitledException Exception thrown when the customer does not have a valid subscription
     *         for the product.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceMeteringException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceMeteringAsyncClient.MeterUsage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/meteringmarketplace-2016-01-14/MeterUsage" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<MeterUsageResponse> meterUsage(MeterUsageRequest meterUsageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, meterUsageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Metering");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "MeterUsage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<MeterUsageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    MeterUsageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<MeterUsageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<MeterUsageRequest, MeterUsageResponse>().withOperationName("MeterUsage")
                            .withMarshaller(new MeterUsageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(meterUsageRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = meterUsageRequest.overrideConfiguration().orElse(null);
            CompletableFuture<MeterUsageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Paid container software products sold through AWS Marketplace must integrate with the AWS Marketplace Metering
     * Service and call the RegisterUsage operation for software entitlement and metering. Free and BYOL products for
     * Amazon ECS or Amazon EKS aren't required to call RegisterUsage, but you may choose to do so if you would like to
     * receive usage data in your seller reports. The sections below explain the behavior of RegisterUsage.
     * RegisterUsage performs two primary functions: metering and entitlement.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>Entitlement</i>: RegisterUsage allows you to verify that the customer running your paid software is subscribed
     * to your product on AWS Marketplace, enabling you to guard against unauthorized use. Your container image that
     * integrates with RegisterUsage is only required to guard against unauthorized use at container startup, as such a
     * CustomerNotSubscribedException/PlatformNotSupportedException will only be thrown on the initial call to
     * RegisterUsage. Subsequent calls from the same Amazon ECS task instance (e.g. task-id) or Amazon EKS pod will not
     * throw a CustomerNotSubscribedException, even if the customer unsubscribes while the Amazon ECS task or Amazon EKS
     * pod is still running.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>Metering</i>: RegisterUsage meters software use per ECS task, per hour, or per pod for Amazon EKS with usage
     * prorated to the second. A minimum of 1 minute of usage applies to tasks that are short lived. For example, if a
     * customer has a 10 node Amazon ECS or Amazon EKS cluster and a service configured as a Daemon Set, then Amazon ECS
     * or Amazon EKS will launch a task on all 10 cluster nodes and the customer will be charged: (10 * hourly_rate).
     * Metering for software use is automatically handled by the AWS Marketplace Metering Control Plane -- your software
     * is not required to perform any metering specific actions, other than call RegisterUsage once for metering of
     * software use to commence. The AWS Marketplace Metering Control Plane will also continue to bill customers for
     * running ECS tasks and Amazon EKS pods, regardless of the customers subscription state, removing the need for your
     * software to perform entitlement checks at runtime.
     * </p>
     * </li>
     * </ul>
     *
     * @param registerUsageRequest
     * @return A Java Future containing the result of the RegisterUsage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidProductCodeException The product code passed does not match the product code used for
     *         publishing the product.</li>
     *         <li>InvalidRegionException RegisterUsage must be called in the same AWS Region the ECS task was launched
     *         in. This prevents a container from hardcoding a Region (e.g. withRegion(“us-east-1”) when calling
     *         RegisterUsage.</li>
     *         <li>InvalidPublicKeyVersionException Public Key version is invalid.</li>
     *         <li>PlatformNotSupportedException AWS Marketplace does not support metering usage from the underlying
     *         platform. Currently, Amazon ECS, Amazon EKS, and AWS Fargate are supported.</li>
     *         <li>CustomerNotEntitledException Exception thrown when the customer does not have a valid subscription
     *         for the product.</li>
     *         <li>ThrottlingException The calls to the API are throttled.</li>
     *         <li>InternalServiceErrorException An internal error has occurred. Retry your request. If the problem
     *         persists, post a message with details on the AWS forums.</li>
     *         <li>DisabledApiException The API is disabled in the Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceMeteringException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceMeteringAsyncClient.RegisterUsage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/meteringmarketplace-2016-01-14/RegisterUsage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterUsageResponse> registerUsage(RegisterUsageRequest registerUsageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerUsageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Metering");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterUsage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterUsageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RegisterUsageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterUsageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterUsageRequest, RegisterUsageResponse>()
                            .withOperationName("RegisterUsage")
                            .withMarshaller(new RegisterUsageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(registerUsageRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = registerUsageRequest.overrideConfiguration().orElse(null);
            CompletableFuture<RegisterUsageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * ResolveCustomer is called by a SaaS application during the registration process. When a buyer visits your website
     * during the registration process, the buyer submits a registration token through their browser. The registration
     * token is resolved through this API to obtain a CustomerIdentifier and product code.
     * </p>
     *
     * @param resolveCustomerRequest
     *        Contains input to the ResolveCustomer operation.
     * @return A Java Future containing the result of the ResolveCustomer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidTokenException Registration token is invalid.</li>
     *         <li>ExpiredTokenException The submitted registration token has expired. This can happen if the buyer's
     *         browser takes too long to redirect to your page, the buyer has resubmitted the registration token, or
     *         your application has held on to the registration token for too long. Your SaaS registration website
     *         should redeem this token as soon as it is submitted by the buyer's browser.</li>
     *         <li>ThrottlingException The calls to the API are throttled.</li>
     *         <li>InternalServiceErrorException An internal error has occurred. Retry your request. If the problem
     *         persists, post a message with details on the AWS forums.</li>
     *         <li>DisabledApiException The API is disabled in the Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceMeteringException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceMeteringAsyncClient.ResolveCustomer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/meteringmarketplace-2016-01-14/ResolveCustomer"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ResolveCustomerResponse> resolveCustomer(ResolveCustomerRequest resolveCustomerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resolveCustomerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Metering");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResolveCustomer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ResolveCustomerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ResolveCustomerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ResolveCustomerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ResolveCustomerRequest, ResolveCustomerResponse>()
                            .withOperationName("ResolveCustomer")
                            .withMarshaller(new ResolveCustomerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(resolveCustomerRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = resolveCustomerRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ResolveCustomerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MarketplaceMeteringException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidUsageAllocationsException")
                                .exceptionBuilderSupplier(InvalidUsageAllocationsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTagException")
                                .exceptionBuilderSupplier(InvalidTagException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTokenException")
                                .exceptionBuilderSupplier(InvalidTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExpiredTokenException")
                                .exceptionBuilderSupplier(ExpiredTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidEndpointRegionException")
                                .exceptionBuilderSupplier(InvalidEndpointRegionException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PlatformNotSupportedException")
                                .exceptionBuilderSupplier(PlatformNotSupportedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TimestampOutOfBoundsException")
                                .exceptionBuilderSupplier(TimestampOutOfBoundsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceErrorException")
                                .exceptionBuilderSupplier(InternalServiceErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidUsageDimensionException")
                                .exceptionBuilderSupplier(InvalidUsageDimensionException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DuplicateRequestException")
                                .exceptionBuilderSupplier(DuplicateRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPublicKeyVersionException")
                                .exceptionBuilderSupplier(InvalidPublicKeyVersionException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRegionException")
                                .exceptionBuilderSupplier(InvalidRegionException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CustomerNotEntitledException")
                                .exceptionBuilderSupplier(CustomerNotEntitledException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DisabledApiException")
                                .exceptionBuilderSupplier(DisabledApiException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidCustomerIdentifierException")
                                .exceptionBuilderSupplier(InvalidCustomerIdentifierException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidProductCodeException")
                                .exceptionBuilderSupplier(InvalidProductCodeException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
