/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lambda.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifications that define the characteristics and constraints for compute instances used by the capacity provider.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceRequirements implements SdkPojo, Serializable,
        ToCopyableBuilder<InstanceRequirements.Builder, InstanceRequirements> {
    private static final SdkField<List<String>> ARCHITECTURES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Architectures")
            .getter(getter(InstanceRequirements::architecturesAsStrings))
            .setter(setter(Builder::architecturesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Architectures").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> ALLOWED_INSTANCE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AllowedInstanceTypes")
            .getter(getter(InstanceRequirements::allowedInstanceTypes))
            .setter(setter(Builder::allowedInstanceTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllowedInstanceTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> EXCLUDED_INSTANCE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ExcludedInstanceTypes")
            .getter(getter(InstanceRequirements::excludedInstanceTypes))
            .setter(setter(Builder::excludedInstanceTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExcludedInstanceTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARCHITECTURES_FIELD,
            ALLOWED_INSTANCE_TYPES_FIELD, EXCLUDED_INSTANCE_TYPES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<String> architectures;

    private final List<String> allowedInstanceTypes;

    private final List<String> excludedInstanceTypes;

    private InstanceRequirements(BuilderImpl builder) {
        this.architectures = builder.architectures;
        this.allowedInstanceTypes = builder.allowedInstanceTypes;
        this.excludedInstanceTypes = builder.excludedInstanceTypes;
    }

    /**
     * <p>
     * A list of supported CPU architectures for compute instances. Valid values include <code>x86_64</code> and
     * <code>arm64</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasArchitectures} method.
     * </p>
     * 
     * @return A list of supported CPU architectures for compute instances. Valid values include <code>x86_64</code> and
     *         <code>arm64</code>.
     */
    public final List<Architecture> architectures() {
        return ArchitecturesListCopier.copyStringToEnum(architectures);
    }

    /**
     * For responses, this returns true if the service returned a value for the Architectures property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasArchitectures() {
        return architectures != null && !(architectures instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of supported CPU architectures for compute instances. Valid values include <code>x86_64</code> and
     * <code>arm64</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasArchitectures} method.
     * </p>
     * 
     * @return A list of supported CPU architectures for compute instances. Valid values include <code>x86_64</code> and
     *         <code>arm64</code>.
     */
    public final List<String> architecturesAsStrings() {
        return architectures;
    }

    /**
     * For responses, this returns true if the service returned a value for the AllowedInstanceTypes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAllowedInstanceTypes() {
        return allowedInstanceTypes != null && !(allowedInstanceTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of EC2 instance types that the capacity provider is allowed to use. If not specified, all compatible
     * instance types are allowed.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAllowedInstanceTypes} method.
     * </p>
     * 
     * @return A list of EC2 instance types that the capacity provider is allowed to use. If not specified, all
     *         compatible instance types are allowed.
     */
    public final List<String> allowedInstanceTypes() {
        return allowedInstanceTypes;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExcludedInstanceTypes property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasExcludedInstanceTypes() {
        return excludedInstanceTypes != null && !(excludedInstanceTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of EC2 instance types that the capacity provider should not use, even if they meet other requirements.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExcludedInstanceTypes} method.
     * </p>
     * 
     * @return A list of EC2 instance types that the capacity provider should not use, even if they meet other
     *         requirements.
     */
    public final List<String> excludedInstanceTypes() {
        return excludedInstanceTypes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasArchitectures() ? architecturesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAllowedInstanceTypes() ? allowedInstanceTypes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasExcludedInstanceTypes() ? excludedInstanceTypes() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceRequirements)) {
            return false;
        }
        InstanceRequirements other = (InstanceRequirements) obj;
        return hasArchitectures() == other.hasArchitectures()
                && Objects.equals(architecturesAsStrings(), other.architecturesAsStrings())
                && hasAllowedInstanceTypes() == other.hasAllowedInstanceTypes()
                && Objects.equals(allowedInstanceTypes(), other.allowedInstanceTypes())
                && hasExcludedInstanceTypes() == other.hasExcludedInstanceTypes()
                && Objects.equals(excludedInstanceTypes(), other.excludedInstanceTypes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InstanceRequirements")
                .add("Architectures", hasArchitectures() ? architecturesAsStrings() : null)
                .add("AllowedInstanceTypes", hasAllowedInstanceTypes() ? allowedInstanceTypes() : null)
                .add("ExcludedInstanceTypes", hasExcludedInstanceTypes() ? excludedInstanceTypes() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Architectures":
            return Optional.ofNullable(clazz.cast(architecturesAsStrings()));
        case "AllowedInstanceTypes":
            return Optional.ofNullable(clazz.cast(allowedInstanceTypes()));
        case "ExcludedInstanceTypes":
            return Optional.ofNullable(clazz.cast(excludedInstanceTypes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Architectures", ARCHITECTURES_FIELD);
        map.put("AllowedInstanceTypes", ALLOWED_INSTANCE_TYPES_FIELD);
        map.put("ExcludedInstanceTypes", EXCLUDED_INSTANCE_TYPES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<InstanceRequirements, T> g) {
        return obj -> g.apply((InstanceRequirements) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceRequirements> {
        /**
         * <p>
         * A list of supported CPU architectures for compute instances. Valid values include <code>x86_64</code> and
         * <code>arm64</code>.
         * </p>
         * 
         * @param architectures
         *        A list of supported CPU architectures for compute instances. Valid values include <code>x86_64</code>
         *        and <code>arm64</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder architecturesWithStrings(Collection<String> architectures);

        /**
         * <p>
         * A list of supported CPU architectures for compute instances. Valid values include <code>x86_64</code> and
         * <code>arm64</code>.
         * </p>
         * 
         * @param architectures
         *        A list of supported CPU architectures for compute instances. Valid values include <code>x86_64</code>
         *        and <code>arm64</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder architecturesWithStrings(String... architectures);

        /**
         * <p>
         * A list of supported CPU architectures for compute instances. Valid values include <code>x86_64</code> and
         * <code>arm64</code>.
         * </p>
         * 
         * @param architectures
         *        A list of supported CPU architectures for compute instances. Valid values include <code>x86_64</code>
         *        and <code>arm64</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder architectures(Collection<Architecture> architectures);

        /**
         * <p>
         * A list of supported CPU architectures for compute instances. Valid values include <code>x86_64</code> and
         * <code>arm64</code>.
         * </p>
         * 
         * @param architectures
         *        A list of supported CPU architectures for compute instances. Valid values include <code>x86_64</code>
         *        and <code>arm64</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder architectures(Architecture... architectures);

        /**
         * <p>
         * A list of EC2 instance types that the capacity provider is allowed to use. If not specified, all compatible
         * instance types are allowed.
         * </p>
         * 
         * @param allowedInstanceTypes
         *        A list of EC2 instance types that the capacity provider is allowed to use. If not specified, all
         *        compatible instance types are allowed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedInstanceTypes(Collection<String> allowedInstanceTypes);

        /**
         * <p>
         * A list of EC2 instance types that the capacity provider is allowed to use. If not specified, all compatible
         * instance types are allowed.
         * </p>
         * 
         * @param allowedInstanceTypes
         *        A list of EC2 instance types that the capacity provider is allowed to use. If not specified, all
         *        compatible instance types are allowed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedInstanceTypes(String... allowedInstanceTypes);

        /**
         * <p>
         * A list of EC2 instance types that the capacity provider should not use, even if they meet other requirements.
         * </p>
         * 
         * @param excludedInstanceTypes
         *        A list of EC2 instance types that the capacity provider should not use, even if they meet other
         *        requirements.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludedInstanceTypes(Collection<String> excludedInstanceTypes);

        /**
         * <p>
         * A list of EC2 instance types that the capacity provider should not use, even if they meet other requirements.
         * </p>
         * 
         * @param excludedInstanceTypes
         *        A list of EC2 instance types that the capacity provider should not use, even if they meet other
         *        requirements.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludedInstanceTypes(String... excludedInstanceTypes);
    }

    static final class BuilderImpl implements Builder {
        private List<String> architectures = DefaultSdkAutoConstructList.getInstance();

        private List<String> allowedInstanceTypes = DefaultSdkAutoConstructList.getInstance();

        private List<String> excludedInstanceTypes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceRequirements model) {
            architecturesWithStrings(model.architectures);
            allowedInstanceTypes(model.allowedInstanceTypes);
            excludedInstanceTypes(model.excludedInstanceTypes);
        }

        public final Collection<String> getArchitectures() {
            if (architectures instanceof SdkAutoConstructList) {
                return null;
            }
            return architectures;
        }

        public final void setArchitectures(Collection<String> architectures) {
            this.architectures = ArchitecturesListCopier.copy(architectures);
        }

        @Override
        public final Builder architecturesWithStrings(Collection<String> architectures) {
            this.architectures = ArchitecturesListCopier.copy(architectures);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder architecturesWithStrings(String... architectures) {
            architecturesWithStrings(Arrays.asList(architectures));
            return this;
        }

        @Override
        public final Builder architectures(Collection<Architecture> architectures) {
            this.architectures = ArchitecturesListCopier.copyEnumToString(architectures);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder architectures(Architecture... architectures) {
            architectures(Arrays.asList(architectures));
            return this;
        }

        public final Collection<String> getAllowedInstanceTypes() {
            if (allowedInstanceTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return allowedInstanceTypes;
        }

        public final void setAllowedInstanceTypes(Collection<String> allowedInstanceTypes) {
            this.allowedInstanceTypes = InstanceTypeSetCopier.copy(allowedInstanceTypes);
        }

        @Override
        public final Builder allowedInstanceTypes(Collection<String> allowedInstanceTypes) {
            this.allowedInstanceTypes = InstanceTypeSetCopier.copy(allowedInstanceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder allowedInstanceTypes(String... allowedInstanceTypes) {
            allowedInstanceTypes(Arrays.asList(allowedInstanceTypes));
            return this;
        }

        public final Collection<String> getExcludedInstanceTypes() {
            if (excludedInstanceTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return excludedInstanceTypes;
        }

        public final void setExcludedInstanceTypes(Collection<String> excludedInstanceTypes) {
            this.excludedInstanceTypes = InstanceTypeSetCopier.copy(excludedInstanceTypes);
        }

        @Override
        public final Builder excludedInstanceTypes(Collection<String> excludedInstanceTypes) {
            this.excludedInstanceTypes = InstanceTypeSetCopier.copy(excludedInstanceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder excludedInstanceTypes(String... excludedInstanceTypes) {
            excludedInstanceTypes(Arrays.asList(excludedInstanceTypes));
            return this;
        }

        @Override
        public InstanceRequirements build() {
            return new InstanceRequirements(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
