/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesis.model;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the input for <code>PutRecord</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PutRecordRequest extends KinesisRequest implements
        ToCopyableBuilder<PutRecordRequest.Builder, PutRecordRequest> {
    private static final SdkField<String> STREAM_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StreamName").getter(getter(PutRecordRequest::streamName)).setter(setter(Builder::streamName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StreamName").build()).build();

    private static final SdkField<SdkBytes> DATA_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("Data").getter(getter(PutRecordRequest::data)).setter(setter(Builder::data))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Data").build()).build();

    private static final SdkField<String> PARTITION_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PartitionKey").getter(getter(PutRecordRequest::partitionKey)).setter(setter(Builder::partitionKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PartitionKey").build()).build();

    private static final SdkField<String> EXPLICIT_HASH_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExplicitHashKey").getter(getter(PutRecordRequest::explicitHashKey))
            .setter(setter(Builder::explicitHashKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExplicitHashKey").build()).build();

    private static final SdkField<String> SEQUENCE_NUMBER_FOR_ORDERING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SequenceNumberForOrdering").getter(getter(PutRecordRequest::sequenceNumberForOrdering))
            .setter(setter(Builder::sequenceNumberForOrdering))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SequenceNumberForOrdering").build())
            .build();

    private static final SdkField<String> STREAM_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StreamARN").getter(getter(PutRecordRequest::streamARN)).setter(setter(Builder::streamARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StreamARN").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STREAM_NAME_FIELD, DATA_FIELD,
            PARTITION_KEY_FIELD, EXPLICIT_HASH_KEY_FIELD, SEQUENCE_NUMBER_FOR_ORDERING_FIELD, STREAM_ARN_FIELD));

    private final String streamName;

    private final SdkBytes data;

    private final String partitionKey;

    private final String explicitHashKey;

    private final String sequenceNumberForOrdering;

    private final String streamARN;

    private PutRecordRequest(BuilderImpl builder) {
        super(builder);
        this.streamName = builder.streamName;
        this.data = builder.data;
        this.partitionKey = builder.partitionKey;
        this.explicitHashKey = builder.explicitHashKey;
        this.sequenceNumberForOrdering = builder.sequenceNumberForOrdering;
        this.streamARN = builder.streamARN;
    }

    /**
     * <p>
     * The name of the stream to put the data record into.
     * </p>
     * 
     * @return The name of the stream to put the data record into.
     */
    public final String streamName() {
        return streamName;
    }

    /**
     * <p>
     * The data blob to put into the record, which is base64-encoded when the blob is serialized. When the data blob
     * (the payload before base64-encoding) is added to the partition key size, the total size must not exceed the
     * maximum record size (1 MiB).
     * </p>
     * 
     * @return The data blob to put into the record, which is base64-encoded when the blob is serialized. When the data
     *         blob (the payload before base64-encoding) is added to the partition key size, the total size must not
     *         exceed the maximum record size (1 MiB).
     */
    public final SdkBytes data() {
        return data;
    }

    /**
     * <p>
     * Determines which shard in the stream the data record is assigned to. Partition keys are Unicode strings with a
     * maximum length limit of 256 characters for each key. Amazon Kinesis Data Streams uses the partition key as input
     * to a hash function that maps the partition key and associated data to a specific shard. Specifically, an MD5 hash
     * function is used to map partition keys to 128-bit integer values and to map associated data records to shards. As
     * a result of this hashing mechanism, all data records with the same partition key map to the same shard within the
     * stream.
     * </p>
     * 
     * @return Determines which shard in the stream the data record is assigned to. Partition keys are Unicode strings
     *         with a maximum length limit of 256 characters for each key. Amazon Kinesis Data Streams uses the
     *         partition key as input to a hash function that maps the partition key and associated data to a specific
     *         shard. Specifically, an MD5 hash function is used to map partition keys to 128-bit integer values and to
     *         map associated data records to shards. As a result of this hashing mechanism, all data records with the
     *         same partition key map to the same shard within the stream.
     */
    public final String partitionKey() {
        return partitionKey;
    }

    /**
     * <p>
     * The hash value used to explicitly determine the shard the data record is assigned to by overriding the partition
     * key hash.
     * </p>
     * 
     * @return The hash value used to explicitly determine the shard the data record is assigned to by overriding the
     *         partition key hash.
     */
    public final String explicitHashKey() {
        return explicitHashKey;
    }

    /**
     * <p>
     * Guarantees strictly increasing sequence numbers, for puts from the same client and to the same partition key.
     * Usage: set the <code>SequenceNumberForOrdering</code> of record <i>n</i> to the sequence number of record
     * <i>n-1</i> (as returned in the result when putting record <i>n-1</i>). If this parameter is not set, records are
     * coarsely ordered based on arrival time.
     * </p>
     * 
     * @return Guarantees strictly increasing sequence numbers, for puts from the same client and to the same partition
     *         key. Usage: set the <code>SequenceNumberForOrdering</code> of record <i>n</i> to the sequence number of
     *         record <i>n-1</i> (as returned in the result when putting record <i>n-1</i>). If this parameter is not
     *         set, records are coarsely ordered based on arrival time.
     */
    public final String sequenceNumberForOrdering() {
        return sequenceNumberForOrdering;
    }

    /**
     * <p>
     * The ARN of the stream.
     * </p>
     * 
     * @return The ARN of the stream.
     */
    public final String streamARN() {
        return streamARN;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(streamName());
        hashCode = 31 * hashCode + Objects.hashCode(data());
        hashCode = 31 * hashCode + Objects.hashCode(partitionKey());
        hashCode = 31 * hashCode + Objects.hashCode(explicitHashKey());
        hashCode = 31 * hashCode + Objects.hashCode(sequenceNumberForOrdering());
        hashCode = 31 * hashCode + Objects.hashCode(streamARN());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PutRecordRequest)) {
            return false;
        }
        PutRecordRequest other = (PutRecordRequest) obj;
        return Objects.equals(streamName(), other.streamName()) && Objects.equals(data(), other.data())
                && Objects.equals(partitionKey(), other.partitionKey())
                && Objects.equals(explicitHashKey(), other.explicitHashKey())
                && Objects.equals(sequenceNumberForOrdering(), other.sequenceNumberForOrdering())
                && Objects.equals(streamARN(), other.streamARN());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PutRecordRequest").add("StreamName", streamName()).add("Data", data())
                .add("PartitionKey", partitionKey()).add("ExplicitHashKey", explicitHashKey())
                .add("SequenceNumberForOrdering", sequenceNumberForOrdering()).add("StreamARN", streamARN()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StreamName":
            return Optional.ofNullable(clazz.cast(streamName()));
        case "Data":
            return Optional.ofNullable(clazz.cast(data()));
        case "PartitionKey":
            return Optional.ofNullable(clazz.cast(partitionKey()));
        case "ExplicitHashKey":
            return Optional.ofNullable(clazz.cast(explicitHashKey()));
        case "SequenceNumberForOrdering":
            return Optional.ofNullable(clazz.cast(sequenceNumberForOrdering()));
        case "StreamARN":
            return Optional.ofNullable(clazz.cast(streamARN()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PutRecordRequest, T> g) {
        return obj -> g.apply((PutRecordRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends KinesisRequest.Builder, SdkPojo, CopyableBuilder<Builder, PutRecordRequest> {
        /**
         * <p>
         * The name of the stream to put the data record into.
         * </p>
         * 
         * @param streamName
         *        The name of the stream to put the data record into.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamName(String streamName);

        /**
         * <p>
         * The data blob to put into the record, which is base64-encoded when the blob is serialized. When the data blob
         * (the payload before base64-encoding) is added to the partition key size, the total size must not exceed the
         * maximum record size (1 MiB).
         * </p>
         * 
         * @param data
         *        The data blob to put into the record, which is base64-encoded when the blob is serialized. When the
         *        data blob (the payload before base64-encoding) is added to the partition key size, the total size must
         *        not exceed the maximum record size (1 MiB).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder data(SdkBytes data);

        /**
         * <p>
         * Determines which shard in the stream the data record is assigned to. Partition keys are Unicode strings with
         * a maximum length limit of 256 characters for each key. Amazon Kinesis Data Streams uses the partition key as
         * input to a hash function that maps the partition key and associated data to a specific shard. Specifically,
         * an MD5 hash function is used to map partition keys to 128-bit integer values and to map associated data
         * records to shards. As a result of this hashing mechanism, all data records with the same partition key map to
         * the same shard within the stream.
         * </p>
         * 
         * @param partitionKey
         *        Determines which shard in the stream the data record is assigned to. Partition keys are Unicode
         *        strings with a maximum length limit of 256 characters for each key. Amazon Kinesis Data Streams uses
         *        the partition key as input to a hash function that maps the partition key and associated data to a
         *        specific shard. Specifically, an MD5 hash function is used to map partition keys to 128-bit integer
         *        values and to map associated data records to shards. As a result of this hashing mechanism, all data
         *        records with the same partition key map to the same shard within the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partitionKey(String partitionKey);

        /**
         * <p>
         * The hash value used to explicitly determine the shard the data record is assigned to by overriding the
         * partition key hash.
         * </p>
         * 
         * @param explicitHashKey
         *        The hash value used to explicitly determine the shard the data record is assigned to by overriding the
         *        partition key hash.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder explicitHashKey(String explicitHashKey);

        /**
         * <p>
         * Guarantees strictly increasing sequence numbers, for puts from the same client and to the same partition key.
         * Usage: set the <code>SequenceNumberForOrdering</code> of record <i>n</i> to the sequence number of record
         * <i>n-1</i> (as returned in the result when putting record <i>n-1</i>). If this parameter is not set, records
         * are coarsely ordered based on arrival time.
         * </p>
         * 
         * @param sequenceNumberForOrdering
         *        Guarantees strictly increasing sequence numbers, for puts from the same client and to the same
         *        partition key. Usage: set the <code>SequenceNumberForOrdering</code> of record <i>n</i> to the
         *        sequence number of record <i>n-1</i> (as returned in the result when putting record <i>n-1</i>). If
         *        this parameter is not set, records are coarsely ordered based on arrival time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sequenceNumberForOrdering(String sequenceNumberForOrdering);

        /**
         * <p>
         * The ARN of the stream.
         * </p>
         * 
         * @param streamARN
         *        The ARN of the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamARN(String streamARN);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends KinesisRequest.BuilderImpl implements Builder {
        private String streamName;

        private SdkBytes data;

        private String partitionKey;

        private String explicitHashKey;

        private String sequenceNumberForOrdering;

        private String streamARN;

        private BuilderImpl() {
        }

        private BuilderImpl(PutRecordRequest model) {
            super(model);
            streamName(model.streamName);
            data(model.data);
            partitionKey(model.partitionKey);
            explicitHashKey(model.explicitHashKey);
            sequenceNumberForOrdering(model.sequenceNumberForOrdering);
            streamARN(model.streamARN);
        }

        public final String getStreamName() {
            return streamName;
        }

        public final void setStreamName(String streamName) {
            this.streamName = streamName;
        }

        @Override
        public final Builder streamName(String streamName) {
            this.streamName = streamName;
            return this;
        }

        public final ByteBuffer getData() {
            return data == null ? null : data.asByteBuffer();
        }

        public final void setData(ByteBuffer data) {
            data(data == null ? null : SdkBytes.fromByteBuffer(data));
        }

        @Override
        public final Builder data(SdkBytes data) {
            this.data = data;
            return this;
        }

        public final String getPartitionKey() {
            return partitionKey;
        }

        public final void setPartitionKey(String partitionKey) {
            this.partitionKey = partitionKey;
        }

        @Override
        public final Builder partitionKey(String partitionKey) {
            this.partitionKey = partitionKey;
            return this;
        }

        public final String getExplicitHashKey() {
            return explicitHashKey;
        }

        public final void setExplicitHashKey(String explicitHashKey) {
            this.explicitHashKey = explicitHashKey;
        }

        @Override
        public final Builder explicitHashKey(String explicitHashKey) {
            this.explicitHashKey = explicitHashKey;
            return this;
        }

        public final String getSequenceNumberForOrdering() {
            return sequenceNumberForOrdering;
        }

        public final void setSequenceNumberForOrdering(String sequenceNumberForOrdering) {
            this.sequenceNumberForOrdering = sequenceNumberForOrdering;
        }

        @Override
        public final Builder sequenceNumberForOrdering(String sequenceNumberForOrdering) {
            this.sequenceNumberForOrdering = sequenceNumberForOrdering;
            return this;
        }

        public final String getStreamARN() {
            return streamARN;
        }

        public final void setStreamARN(String streamARN) {
            this.streamARN = streamARN;
        }

        @Override
        public final Builder streamARN(String streamARN) {
            this.streamARN = streamARN;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public PutRecordRequest build() {
            return new PutRecordRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
