/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotsitewise.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information for an asset property historical value entry that is associated with the <a
 * href="https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_BatchGetAssetPropertyValue.html"
 * >BatchGetAssetPropertyValueHistory</a> API.
 * </p>
 * <p>
 * To identify an asset property, you must specify one of the following:
 * </p>
 * <ul>
 * <li>
 * <p>
 * The <code>assetId</code> and <code>propertyId</code> of an asset property.
 * </p>
 * </li>
 * <li>
 * <p>
 * A <code>propertyAlias</code>, which is a data stream alias (for example,
 * <code>/company/windfarm/3/turbine/7/temperature</code>). To define an asset property's alias, see <a
 * href="https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateAssetProperty.html"
 * >UpdateAssetProperty</a>.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BatchGetAssetPropertyValueHistoryEntry implements SdkPojo, Serializable,
        ToCopyableBuilder<BatchGetAssetPropertyValueHistoryEntry.Builder, BatchGetAssetPropertyValueHistoryEntry> {
    private static final SdkField<String> ENTRY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("entryId").getter(getter(BatchGetAssetPropertyValueHistoryEntry::entryId))
            .setter(setter(Builder::entryId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("entryId").build()).build();

    private static final SdkField<String> ASSET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("assetId").getter(getter(BatchGetAssetPropertyValueHistoryEntry::assetId))
            .setter(setter(Builder::assetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("assetId").build()).build();

    private static final SdkField<String> PROPERTY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("propertyId").getter(getter(BatchGetAssetPropertyValueHistoryEntry::propertyId))
            .setter(setter(Builder::propertyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("propertyId").build()).build();

    private static final SdkField<String> PROPERTY_ALIAS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("propertyAlias").getter(getter(BatchGetAssetPropertyValueHistoryEntry::propertyAlias))
            .setter(setter(Builder::propertyAlias))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("propertyAlias").build()).build();

    private static final SdkField<Instant> START_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("startDate").getter(getter(BatchGetAssetPropertyValueHistoryEntry::startDate))
            .setter(setter(Builder::startDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startDate").build()).build();

    private static final SdkField<Instant> END_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("endDate").getter(getter(BatchGetAssetPropertyValueHistoryEntry::endDate))
            .setter(setter(Builder::endDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endDate").build()).build();

    private static final SdkField<List<String>> QUALITIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("qualities")
            .getter(getter(BatchGetAssetPropertyValueHistoryEntry::qualitiesAsStrings))
            .setter(setter(Builder::qualitiesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("qualities").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TIME_ORDERING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("timeOrdering").getter(getter(BatchGetAssetPropertyValueHistoryEntry::timeOrderingAsString))
            .setter(setter(Builder::timeOrdering))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("timeOrdering").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENTRY_ID_FIELD,
            ASSET_ID_FIELD, PROPERTY_ID_FIELD, PROPERTY_ALIAS_FIELD, START_DATE_FIELD, END_DATE_FIELD, QUALITIES_FIELD,
            TIME_ORDERING_FIELD));

    private static final long serialVersionUID = 1L;

    private final String entryId;

    private final String assetId;

    private final String propertyId;

    private final String propertyAlias;

    private final Instant startDate;

    private final Instant endDate;

    private final List<String> qualities;

    private final String timeOrdering;

    private BatchGetAssetPropertyValueHistoryEntry(BuilderImpl builder) {
        this.entryId = builder.entryId;
        this.assetId = builder.assetId;
        this.propertyId = builder.propertyId;
        this.propertyAlias = builder.propertyAlias;
        this.startDate = builder.startDate;
        this.endDate = builder.endDate;
        this.qualities = builder.qualities;
        this.timeOrdering = builder.timeOrdering;
    }

    /**
     * <p>
     * The ID of the entry.
     * </p>
     * 
     * @return The ID of the entry.
     */
    public final String entryId() {
        return entryId;
    }

    /**
     * <p>
     * The ID of the asset in which the asset property was created.
     * </p>
     * 
     * @return The ID of the asset in which the asset property was created.
     */
    public final String assetId() {
        return assetId;
    }

    /**
     * <p>
     * The ID of the asset property.
     * </p>
     * 
     * @return The ID of the asset property.
     */
    public final String propertyId() {
        return propertyId;
    }

    /**
     * <p>
     * The alias that identifies the property, such as an OPC-UA server data stream path (for example,
     * <code>/company/windfarm/3/turbine/7/temperature</code>). For more information, see <a
     * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/connect-data-streams.html">Mapping industrial
     * data streams to asset properties</a> in the <i>IoT SiteWise User Guide</i>.
     * </p>
     * 
     * @return The alias that identifies the property, such as an OPC-UA server data stream path (for example,
     *         <code>/company/windfarm/3/turbine/7/temperature</code>). For more information, see <a
     *         href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/connect-data-streams.html">Mapping
     *         industrial data streams to asset properties</a> in the <i>IoT SiteWise User Guide</i>.
     */
    public final String propertyAlias() {
        return propertyAlias;
    }

    /**
     * <p>
     * The exclusive start of the range from which to query historical data, expressed in seconds in Unix epoch time.
     * </p>
     * 
     * @return The exclusive start of the range from which to query historical data, expressed in seconds in Unix epoch
     *         time.
     */
    public final Instant startDate() {
        return startDate;
    }

    /**
     * <p>
     * The inclusive end of the range from which to query historical data, expressed in seconds in Unix epoch time.
     * </p>
     * 
     * @return The inclusive end of the range from which to query historical data, expressed in seconds in Unix epoch
     *         time.
     */
    public final Instant endDate() {
        return endDate;
    }

    /**
     * <p>
     * The quality by which to filter asset data.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasQualities} method.
     * </p>
     * 
     * @return The quality by which to filter asset data.
     */
    public final List<Quality> qualities() {
        return QualitiesCopier.copyStringToEnum(qualities);
    }

    /**
     * For responses, this returns true if the service returned a value for the Qualities property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasQualities() {
        return qualities != null && !(qualities instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The quality by which to filter asset data.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasQualities} method.
     * </p>
     * 
     * @return The quality by which to filter asset data.
     */
    public final List<String> qualitiesAsStrings() {
        return qualities;
    }

    /**
     * <p>
     * The chronological sorting order of the requested information.
     * </p>
     * <p>
     * Default: <code>ASCENDING</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #timeOrdering} will
     * return {@link TimeOrdering#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #timeOrderingAsString}.
     * </p>
     * 
     * @return The chronological sorting order of the requested information.</p>
     *         <p>
     *         Default: <code>ASCENDING</code>
     * @see TimeOrdering
     */
    public final TimeOrdering timeOrdering() {
        return TimeOrdering.fromValue(timeOrdering);
    }

    /**
     * <p>
     * The chronological sorting order of the requested information.
     * </p>
     * <p>
     * Default: <code>ASCENDING</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #timeOrdering} will
     * return {@link TimeOrdering#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #timeOrderingAsString}.
     * </p>
     * 
     * @return The chronological sorting order of the requested information.</p>
     *         <p>
     *         Default: <code>ASCENDING</code>
     * @see TimeOrdering
     */
    public final String timeOrderingAsString() {
        return timeOrdering;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(entryId());
        hashCode = 31 * hashCode + Objects.hashCode(assetId());
        hashCode = 31 * hashCode + Objects.hashCode(propertyId());
        hashCode = 31 * hashCode + Objects.hashCode(propertyAlias());
        hashCode = 31 * hashCode + Objects.hashCode(startDate());
        hashCode = 31 * hashCode + Objects.hashCode(endDate());
        hashCode = 31 * hashCode + Objects.hashCode(hasQualities() ? qualitiesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(timeOrderingAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BatchGetAssetPropertyValueHistoryEntry)) {
            return false;
        }
        BatchGetAssetPropertyValueHistoryEntry other = (BatchGetAssetPropertyValueHistoryEntry) obj;
        return Objects.equals(entryId(), other.entryId()) && Objects.equals(assetId(), other.assetId())
                && Objects.equals(propertyId(), other.propertyId()) && Objects.equals(propertyAlias(), other.propertyAlias())
                && Objects.equals(startDate(), other.startDate()) && Objects.equals(endDate(), other.endDate())
                && hasQualities() == other.hasQualities() && Objects.equals(qualitiesAsStrings(), other.qualitiesAsStrings())
                && Objects.equals(timeOrderingAsString(), other.timeOrderingAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("BatchGetAssetPropertyValueHistoryEntry").add("EntryId", entryId()).add("AssetId", assetId())
                .add("PropertyId", propertyId()).add("PropertyAlias", propertyAlias()).add("StartDate", startDate())
                .add("EndDate", endDate()).add("Qualities", hasQualities() ? qualitiesAsStrings() : null)
                .add("TimeOrdering", timeOrderingAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "entryId":
            return Optional.ofNullable(clazz.cast(entryId()));
        case "assetId":
            return Optional.ofNullable(clazz.cast(assetId()));
        case "propertyId":
            return Optional.ofNullable(clazz.cast(propertyId()));
        case "propertyAlias":
            return Optional.ofNullable(clazz.cast(propertyAlias()));
        case "startDate":
            return Optional.ofNullable(clazz.cast(startDate()));
        case "endDate":
            return Optional.ofNullable(clazz.cast(endDate()));
        case "qualities":
            return Optional.ofNullable(clazz.cast(qualitiesAsStrings()));
        case "timeOrdering":
            return Optional.ofNullable(clazz.cast(timeOrderingAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<BatchGetAssetPropertyValueHistoryEntry, T> g) {
        return obj -> g.apply((BatchGetAssetPropertyValueHistoryEntry) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BatchGetAssetPropertyValueHistoryEntry> {
        /**
         * <p>
         * The ID of the entry.
         * </p>
         * 
         * @param entryId
         *        The ID of the entry.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entryId(String entryId);

        /**
         * <p>
         * The ID of the asset in which the asset property was created.
         * </p>
         * 
         * @param assetId
         *        The ID of the asset in which the asset property was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder assetId(String assetId);

        /**
         * <p>
         * The ID of the asset property.
         * </p>
         * 
         * @param propertyId
         *        The ID of the asset property.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder propertyId(String propertyId);

        /**
         * <p>
         * The alias that identifies the property, such as an OPC-UA server data stream path (for example,
         * <code>/company/windfarm/3/turbine/7/temperature</code>). For more information, see <a
         * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/connect-data-streams.html">Mapping industrial
         * data streams to asset properties</a> in the <i>IoT SiteWise User Guide</i>.
         * </p>
         * 
         * @param propertyAlias
         *        The alias that identifies the property, such as an OPC-UA server data stream path (for example,
         *        <code>/company/windfarm/3/turbine/7/temperature</code>). For more information, see <a
         *        href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/connect-data-streams.html">Mapping
         *        industrial data streams to asset properties</a> in the <i>IoT SiteWise User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder propertyAlias(String propertyAlias);

        /**
         * <p>
         * The exclusive start of the range from which to query historical data, expressed in seconds in Unix epoch
         * time.
         * </p>
         * 
         * @param startDate
         *        The exclusive start of the range from which to query historical data, expressed in seconds in Unix
         *        epoch time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startDate(Instant startDate);

        /**
         * <p>
         * The inclusive end of the range from which to query historical data, expressed in seconds in Unix epoch time.
         * </p>
         * 
         * @param endDate
         *        The inclusive end of the range from which to query historical data, expressed in seconds in Unix epoch
         *        time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endDate(Instant endDate);

        /**
         * <p>
         * The quality by which to filter asset data.
         * </p>
         * 
         * @param qualities
         *        The quality by which to filter asset data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder qualitiesWithStrings(Collection<String> qualities);

        /**
         * <p>
         * The quality by which to filter asset data.
         * </p>
         * 
         * @param qualities
         *        The quality by which to filter asset data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder qualitiesWithStrings(String... qualities);

        /**
         * <p>
         * The quality by which to filter asset data.
         * </p>
         * 
         * @param qualities
         *        The quality by which to filter asset data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder qualities(Collection<Quality> qualities);

        /**
         * <p>
         * The quality by which to filter asset data.
         * </p>
         * 
         * @param qualities
         *        The quality by which to filter asset data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder qualities(Quality... qualities);

        /**
         * <p>
         * The chronological sorting order of the requested information.
         * </p>
         * <p>
         * Default: <code>ASCENDING</code>
         * </p>
         * 
         * @param timeOrdering
         *        The chronological sorting order of the requested information.</p>
         *        <p>
         *        Default: <code>ASCENDING</code>
         * @see TimeOrdering
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TimeOrdering
         */
        Builder timeOrdering(String timeOrdering);

        /**
         * <p>
         * The chronological sorting order of the requested information.
         * </p>
         * <p>
         * Default: <code>ASCENDING</code>
         * </p>
         * 
         * @param timeOrdering
         *        The chronological sorting order of the requested information.</p>
         *        <p>
         *        Default: <code>ASCENDING</code>
         * @see TimeOrdering
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TimeOrdering
         */
        Builder timeOrdering(TimeOrdering timeOrdering);
    }

    static final class BuilderImpl implements Builder {
        private String entryId;

        private String assetId;

        private String propertyId;

        private String propertyAlias;

        private Instant startDate;

        private Instant endDate;

        private List<String> qualities = DefaultSdkAutoConstructList.getInstance();

        private String timeOrdering;

        private BuilderImpl() {
        }

        private BuilderImpl(BatchGetAssetPropertyValueHistoryEntry model) {
            entryId(model.entryId);
            assetId(model.assetId);
            propertyId(model.propertyId);
            propertyAlias(model.propertyAlias);
            startDate(model.startDate);
            endDate(model.endDate);
            qualitiesWithStrings(model.qualities);
            timeOrdering(model.timeOrdering);
        }

        public final String getEntryId() {
            return entryId;
        }

        public final void setEntryId(String entryId) {
            this.entryId = entryId;
        }

        @Override
        public final Builder entryId(String entryId) {
            this.entryId = entryId;
            return this;
        }

        public final String getAssetId() {
            return assetId;
        }

        public final void setAssetId(String assetId) {
            this.assetId = assetId;
        }

        @Override
        public final Builder assetId(String assetId) {
            this.assetId = assetId;
            return this;
        }

        public final String getPropertyId() {
            return propertyId;
        }

        public final void setPropertyId(String propertyId) {
            this.propertyId = propertyId;
        }

        @Override
        public final Builder propertyId(String propertyId) {
            this.propertyId = propertyId;
            return this;
        }

        public final String getPropertyAlias() {
            return propertyAlias;
        }

        public final void setPropertyAlias(String propertyAlias) {
            this.propertyAlias = propertyAlias;
        }

        @Override
        public final Builder propertyAlias(String propertyAlias) {
            this.propertyAlias = propertyAlias;
            return this;
        }

        public final Instant getStartDate() {
            return startDate;
        }

        public final void setStartDate(Instant startDate) {
            this.startDate = startDate;
        }

        @Override
        public final Builder startDate(Instant startDate) {
            this.startDate = startDate;
            return this;
        }

        public final Instant getEndDate() {
            return endDate;
        }

        public final void setEndDate(Instant endDate) {
            this.endDate = endDate;
        }

        @Override
        public final Builder endDate(Instant endDate) {
            this.endDate = endDate;
            return this;
        }

        public final Collection<String> getQualities() {
            if (qualities instanceof SdkAutoConstructList) {
                return null;
            }
            return qualities;
        }

        public final void setQualities(Collection<String> qualities) {
            this.qualities = QualitiesCopier.copy(qualities);
        }

        @Override
        public final Builder qualitiesWithStrings(Collection<String> qualities) {
            this.qualities = QualitiesCopier.copy(qualities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder qualitiesWithStrings(String... qualities) {
            qualitiesWithStrings(Arrays.asList(qualities));
            return this;
        }

        @Override
        public final Builder qualities(Collection<Quality> qualities) {
            this.qualities = QualitiesCopier.copyEnumToString(qualities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder qualities(Quality... qualities) {
            qualities(Arrays.asList(qualities));
            return this;
        }

        public final String getTimeOrdering() {
            return timeOrdering;
        }

        public final void setTimeOrdering(String timeOrdering) {
            this.timeOrdering = timeOrdering;
        }

        @Override
        public final Builder timeOrdering(String timeOrdering) {
            this.timeOrdering = timeOrdering;
            return this;
        }

        @Override
        public final Builder timeOrdering(TimeOrdering timeOrdering) {
            this.timeOrdering(timeOrdering == null ? null : timeOrdering.toString());
            return this;
        }

        @Override
        public BatchGetAssetPropertyValueHistoryEntry build() {
            return new BatchGetAssetPropertyValueHistoryEntry(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
