/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotdataplane;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.services.iotdataplane.model.DeleteConnectionRequest;
import software.amazon.awssdk.services.iotdataplane.model.DeleteConnectionResponse;
import software.amazon.awssdk.services.iotdataplane.model.DeleteThingShadowRequest;
import software.amazon.awssdk.services.iotdataplane.model.DeleteThingShadowResponse;
import software.amazon.awssdk.services.iotdataplane.model.GetRetainedMessageRequest;
import software.amazon.awssdk.services.iotdataplane.model.GetRetainedMessageResponse;
import software.amazon.awssdk.services.iotdataplane.model.GetThingShadowRequest;
import software.amazon.awssdk.services.iotdataplane.model.GetThingShadowResponse;
import software.amazon.awssdk.services.iotdataplane.model.ListNamedShadowsForThingRequest;
import software.amazon.awssdk.services.iotdataplane.model.ListNamedShadowsForThingResponse;
import software.amazon.awssdk.services.iotdataplane.model.ListRetainedMessagesRequest;
import software.amazon.awssdk.services.iotdataplane.model.ListRetainedMessagesResponse;
import software.amazon.awssdk.services.iotdataplane.model.PublishRequest;
import software.amazon.awssdk.services.iotdataplane.model.PublishResponse;
import software.amazon.awssdk.services.iotdataplane.model.UpdateThingShadowRequest;
import software.amazon.awssdk.services.iotdataplane.model.UpdateThingShadowResponse;
import software.amazon.awssdk.services.iotdataplane.paginators.ListRetainedMessagesPublisher;

/**
 * Service client for accessing AWS IoT Data Plane asynchronously. This can be created using the static
 * {@link #builder()} method.The asynchronous client performs non-blocking I/O when configured with any
 * {@code SdkAsyncHttpClient} supported in the SDK. However, full non-blocking is not guaranteed as the async client may
 * perform blocking calls in some cases such as credentials retrieval and endpoint discovery as part of the async API
 * call.
 *
 * <fullname>IoT data</fullname>
 * <p>
 * IoT data enables secure, bi-directional communication between Internet-connected things (such as sensors, actuators,
 * embedded devices, or smart appliances) and the Amazon Web Services cloud. It implements a broker for applications and
 * things to publish messages over HTTP (Publish) and retrieve, update, and delete shadows. A shadow is a persistent
 * representation of your things and their state in the Amazon Web Services cloud.
 * </p>
 * <p>
 * Find the endpoint address for actions in IoT data by running this CLI command:
 * </p>
 * <p>
 * <code>aws iot describe-endpoint --endpoint-type iot:Data-ATS</code>
 * </p>
 * <p>
 * The service name used by <a href="https://docs.aws.amazon.com/general/latest/gr/signature-version-4.html">Amazon Web
 * ServicesSignature Version 4</a> to sign requests is: <i>iotdevicegateway</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface IotDataPlaneAsyncClient extends AwsClient {
    String SERVICE_NAME = "iotdata";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "data-ats.iot";

    /**
     * <p>
     * Disconnects a connected MQTT client from Amazon Web Services IoT Core. When you disconnect a client, Amazon Web
     * Services IoT Core closes the client's network connection and optionally cleans the session state.
     * </p>
     *
     * @param deleteConnectionRequest
     * @return A Java Future containing the result of the DeleteConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>InternalFailureException An unexpected error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotDataPlaneAsyncClient.DeleteConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/DeleteConnection" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteConnectionResponse> deleteConnection(DeleteConnectionRequest deleteConnectionRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Disconnects a connected MQTT client from Amazon Web Services IoT Core. When you disconnect a client, Amazon Web
     * Services IoT Core closes the client's network connection and optionally cleans the session state.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteConnectionRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteConnectionRequest#builder()}
     * </p>
     *
     * @param deleteConnectionRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.iotdataplane.model.DeleteConnectionRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the DeleteConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>InternalFailureException An unexpected error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotDataPlaneAsyncClient.DeleteConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/DeleteConnection" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteConnectionResponse> deleteConnection(
            Consumer<DeleteConnectionRequest.Builder> deleteConnectionRequest) {
        return deleteConnection(DeleteConnectionRequest.builder().applyMutation(deleteConnectionRequest).build());
    }

    /**
     * <p>
     * Deletes the shadow for the specified thing.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >DeleteThingShadow</a> action.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/iot/latest/developerguide/API_DeleteThingShadow.html">DeleteThingShadow</a> in
     * the IoT Developer Guide.
     * </p>
     *
     * @param deleteThingShadowRequest
     *        The input for the DeleteThingShadow operation.
     * @return A Java Future containing the result of the DeleteThingShadow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>InternalFailureException An unexpected error has occurred.</li>
     *         <li>MethodNotAllowedException The specified combination of HTTP verb and URI is not supported.</li>
     *         <li>UnsupportedDocumentEncodingException The document encoding is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotDataPlaneAsyncClient.DeleteThingShadow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/DeleteThingShadow" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DeleteThingShadowResponse> deleteThingShadow(DeleteThingShadowRequest deleteThingShadowRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the shadow for the specified thing.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >DeleteThingShadow</a> action.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/iot/latest/developerguide/API_DeleteThingShadow.html">DeleteThingShadow</a> in
     * the IoT Developer Guide.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteThingShadowRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteThingShadowRequest#builder()}
     * </p>
     *
     * @param deleteThingShadowRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.iotdataplane.model.DeleteThingShadowRequest.Builder} to create a
     *        request. The input for the DeleteThingShadow operation.
     * @return A Java Future containing the result of the DeleteThingShadow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>InternalFailureException An unexpected error has occurred.</li>
     *         <li>MethodNotAllowedException The specified combination of HTTP verb and URI is not supported.</li>
     *         <li>UnsupportedDocumentEncodingException The document encoding is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotDataPlaneAsyncClient.DeleteThingShadow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/DeleteThingShadow" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DeleteThingShadowResponse> deleteThingShadow(
            Consumer<DeleteThingShadowRequest.Builder> deleteThingShadowRequest) {
        return deleteThingShadow(DeleteThingShadowRequest.builder().applyMutation(deleteThingShadowRequest).build());
    }

    /**
     * <p>
     * Gets the details of a single retained message for the specified topic.
     * </p>
     * <p>
     * This action returns the message payload of the retained message, which can incur messaging costs. To list only
     * the topic names of the retained messages, call <a
     * href="https://docs.aws.amazon.com/iot/latest/apireference/API_iotdata_ListRetainedMessages.html"
     * >ListRetainedMessages</a>.
     * </p>
     * <p>
     * Requires permission to access the <a
     * href="https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html">GetRetainedMessage</a>
     * action.
     * </p>
     * <p>
     * For more information about messaging costs, see <a
     * href="http://aws.amazon.com/iot-core/pricing/#Messaging">Amazon Web Services IoT Core pricing - Messaging</a>.
     * </p>
     *
     * @param getRetainedMessageRequest
     *        The input for the GetRetainedMessage operation.
     * @return A Java Future containing the result of the GetRetainedMessage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>InternalFailureException An unexpected error has occurred.</li>
     *         <li>MethodNotAllowedException The specified combination of HTTP verb and URI is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotDataPlaneAsyncClient.GetRetainedMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/GetRetainedMessage" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetRetainedMessageResponse> getRetainedMessage(GetRetainedMessageRequest getRetainedMessageRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets the details of a single retained message for the specified topic.
     * </p>
     * <p>
     * This action returns the message payload of the retained message, which can incur messaging costs. To list only
     * the topic names of the retained messages, call <a
     * href="https://docs.aws.amazon.com/iot/latest/apireference/API_iotdata_ListRetainedMessages.html"
     * >ListRetainedMessages</a>.
     * </p>
     * <p>
     * Requires permission to access the <a
     * href="https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html">GetRetainedMessage</a>
     * action.
     * </p>
     * <p>
     * For more information about messaging costs, see <a
     * href="http://aws.amazon.com/iot-core/pricing/#Messaging">Amazon Web Services IoT Core pricing - Messaging</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetRetainedMessageRequest.Builder} avoiding the
     * need to create one manually via {@link GetRetainedMessageRequest#builder()}
     * </p>
     *
     * @param getRetainedMessageRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.iotdataplane.model.GetRetainedMessageRequest.Builder} to create a
     *        request. The input for the GetRetainedMessage operation.
     * @return A Java Future containing the result of the GetRetainedMessage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>InternalFailureException An unexpected error has occurred.</li>
     *         <li>MethodNotAllowedException The specified combination of HTTP verb and URI is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotDataPlaneAsyncClient.GetRetainedMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/GetRetainedMessage" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetRetainedMessageResponse> getRetainedMessage(
            Consumer<GetRetainedMessageRequest.Builder> getRetainedMessageRequest) {
        return getRetainedMessage(GetRetainedMessageRequest.builder().applyMutation(getRetainedMessageRequest).build());
    }

    /**
     * <p>
     * Gets the shadow for the specified thing.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >GetThingShadow</a> action.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/iot/latest/developerguide/API_GetThingShadow.html">GetThingShadow</a> in the IoT
     * Developer Guide.
     * </p>
     *
     * @param getThingShadowRequest
     *        The input for the GetThingShadow operation.
     * @return A Java Future containing the result of the GetThingShadow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>InternalFailureException An unexpected error has occurred.</li>
     *         <li>MethodNotAllowedException The specified combination of HTTP verb and URI is not supported.</li>
     *         <li>UnsupportedDocumentEncodingException The document encoding is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotDataPlaneAsyncClient.GetThingShadow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/GetThingShadow" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetThingShadowResponse> getThingShadow(GetThingShadowRequest getThingShadowRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets the shadow for the specified thing.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >GetThingShadow</a> action.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/iot/latest/developerguide/API_GetThingShadow.html">GetThingShadow</a> in the IoT
     * Developer Guide.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetThingShadowRequest.Builder} avoiding the need to
     * create one manually via {@link GetThingShadowRequest#builder()}
     * </p>
     *
     * @param getThingShadowRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.iotdataplane.model.GetThingShadowRequest.Builder} to create a
     *        request. The input for the GetThingShadow operation.
     * @return A Java Future containing the result of the GetThingShadow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>InternalFailureException An unexpected error has occurred.</li>
     *         <li>MethodNotAllowedException The specified combination of HTTP verb and URI is not supported.</li>
     *         <li>UnsupportedDocumentEncodingException The document encoding is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotDataPlaneAsyncClient.GetThingShadow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/GetThingShadow" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetThingShadowResponse> getThingShadow(Consumer<GetThingShadowRequest.Builder> getThingShadowRequest) {
        return getThingShadow(GetThingShadowRequest.builder().applyMutation(getThingShadowRequest).build());
    }

    /**
     * <p>
     * Lists the shadows for the specified thing.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >ListNamedShadowsForThing</a> action.
     * </p>
     *
     * @param listNamedShadowsForThingRequest
     * @return A Java Future containing the result of the ListNamedShadowsForThing operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>InternalFailureException An unexpected error has occurred.</li>
     *         <li>MethodNotAllowedException The specified combination of HTTP verb and URI is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotDataPlaneAsyncClient.ListNamedShadowsForThing
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/ListNamedShadowsForThing"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListNamedShadowsForThingResponse> listNamedShadowsForThing(
            ListNamedShadowsForThingRequest listNamedShadowsForThingRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the shadows for the specified thing.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >ListNamedShadowsForThing</a> action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListNamedShadowsForThingRequest.Builder} avoiding
     * the need to create one manually via {@link ListNamedShadowsForThingRequest#builder()}
     * </p>
     *
     * @param listNamedShadowsForThingRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.iotdataplane.model.ListNamedShadowsForThingRequest.Builder} to
     *        create a request.
     * @return A Java Future containing the result of the ListNamedShadowsForThing operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>InternalFailureException An unexpected error has occurred.</li>
     *         <li>MethodNotAllowedException The specified combination of HTTP verb and URI is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotDataPlaneAsyncClient.ListNamedShadowsForThing
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/ListNamedShadowsForThing"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListNamedShadowsForThingResponse> listNamedShadowsForThing(
            Consumer<ListNamedShadowsForThingRequest.Builder> listNamedShadowsForThingRequest) {
        return listNamedShadowsForThing(ListNamedShadowsForThingRequest.builder().applyMutation(listNamedShadowsForThingRequest)
                .build());
    }

    /**
     * <p>
     * Lists summary information about the retained messages stored for the account.
     * </p>
     * <p>
     * This action returns only the topic names of the retained messages. It doesn't return any message payloads.
     * Although this action doesn't return a message payload, it can still incur messaging costs.
     * </p>
     * <p>
     * To get the message payload of a retained message, call <a
     * href="https://docs.aws.amazon.com/iot/latest/apireference/API_iotdata_GetRetainedMessage.html"
     * >GetRetainedMessage</a> with the topic name of the retained message.
     * </p>
     * <p>
     * Requires permission to access the <a
     * href="https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html"
     * >ListRetainedMessages</a> action.
     * </p>
     * <p>
     * For more information about messaging costs, see <a
     * href="http://aws.amazon.com/iot-core/pricing/#Messaging">Amazon Web Services IoT Core pricing - Messaging</a>.
     * </p>
     *
     * @param listRetainedMessagesRequest
     * @return A Java Future containing the result of the ListRetainedMessages operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>InternalFailureException An unexpected error has occurred.</li>
     *         <li>MethodNotAllowedException The specified combination of HTTP verb and URI is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotDataPlaneAsyncClient.ListRetainedMessages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/ListRetainedMessages" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListRetainedMessagesResponse> listRetainedMessages(
            ListRetainedMessagesRequest listRetainedMessagesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists summary information about the retained messages stored for the account.
     * </p>
     * <p>
     * This action returns only the topic names of the retained messages. It doesn't return any message payloads.
     * Although this action doesn't return a message payload, it can still incur messaging costs.
     * </p>
     * <p>
     * To get the message payload of a retained message, call <a
     * href="https://docs.aws.amazon.com/iot/latest/apireference/API_iotdata_GetRetainedMessage.html"
     * >GetRetainedMessage</a> with the topic name of the retained message.
     * </p>
     * <p>
     * Requires permission to access the <a
     * href="https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html"
     * >ListRetainedMessages</a> action.
     * </p>
     * <p>
     * For more information about messaging costs, see <a
     * href="http://aws.amazon.com/iot-core/pricing/#Messaging">Amazon Web Services IoT Core pricing - Messaging</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListRetainedMessagesRequest.Builder} avoiding the
     * need to create one manually via {@link ListRetainedMessagesRequest#builder()}
     * </p>
     *
     * @param listRetainedMessagesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.iotdataplane.model.ListRetainedMessagesRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the ListRetainedMessages operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>InternalFailureException An unexpected error has occurred.</li>
     *         <li>MethodNotAllowedException The specified combination of HTTP verb and URI is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotDataPlaneAsyncClient.ListRetainedMessages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/ListRetainedMessages" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListRetainedMessagesResponse> listRetainedMessages(
            Consumer<ListRetainedMessagesRequest.Builder> listRetainedMessagesRequest) {
        return listRetainedMessages(ListRetainedMessagesRequest.builder().applyMutation(listRetainedMessagesRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listRetainedMessages(software.amazon.awssdk.services.iotdataplane.model.ListRetainedMessagesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.iotdataplane.paginators.ListRetainedMessagesPublisher publisher = client.listRetainedMessagesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.iotdataplane.paginators.ListRetainedMessagesPublisher publisher = client.listRetainedMessagesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.iotdataplane.model.ListRetainedMessagesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.iotdataplane.model.ListRetainedMessagesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRetainedMessages(software.amazon.awssdk.services.iotdataplane.model.ListRetainedMessagesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRetainedMessagesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>InternalFailureException An unexpected error has occurred.</li>
     *         <li>MethodNotAllowedException The specified combination of HTTP verb and URI is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotDataPlaneAsyncClient.ListRetainedMessages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/ListRetainedMessages" target="_top">AWS
     *      API Documentation</a>
     */
    default ListRetainedMessagesPublisher listRetainedMessagesPaginator(ListRetainedMessagesRequest listRetainedMessagesRequest) {
        return new ListRetainedMessagesPublisher(this, listRetainedMessagesRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listRetainedMessages(software.amazon.awssdk.services.iotdataplane.model.ListRetainedMessagesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.iotdataplane.paginators.ListRetainedMessagesPublisher publisher = client.listRetainedMessagesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.iotdataplane.paginators.ListRetainedMessagesPublisher publisher = client.listRetainedMessagesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.iotdataplane.model.ListRetainedMessagesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.iotdataplane.model.ListRetainedMessagesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRetainedMessages(software.amazon.awssdk.services.iotdataplane.model.ListRetainedMessagesRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListRetainedMessagesRequest.Builder} avoiding the
     * need to create one manually via {@link ListRetainedMessagesRequest#builder()}
     * </p>
     *
     * @param listRetainedMessagesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.iotdataplane.model.ListRetainedMessagesRequest.Builder} to create a
     *        request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>InternalFailureException An unexpected error has occurred.</li>
     *         <li>MethodNotAllowedException The specified combination of HTTP verb and URI is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotDataPlaneAsyncClient.ListRetainedMessages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/ListRetainedMessages" target="_top">AWS
     *      API Documentation</a>
     */
    default ListRetainedMessagesPublisher listRetainedMessagesPaginator(
            Consumer<ListRetainedMessagesRequest.Builder> listRetainedMessagesRequest) {
        return listRetainedMessagesPaginator(ListRetainedMessagesRequest.builder().applyMutation(listRetainedMessagesRequest)
                .build());
    }

    /**
     * <p>
     * Publishes an MQTT message.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >Publish</a> action.
     * </p>
     * <p>
     * For more information about MQTT messages, see <a
     * href="http://docs.aws.amazon.com/iot/latest/developerguide/mqtt.html">MQTT Protocol</a> in the IoT Developer
     * Guide.
     * </p>
     * <p>
     * For more information about messaging costs, see <a
     * href="http://aws.amazon.com/iot-core/pricing/#Messaging">Amazon Web Services IoT Core pricing - Messaging</a>.
     * </p>
     *
     * @param publishRequest
     *        The input for the Publish operation.
     * @return A Java Future containing the result of the Publish operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalFailureException An unexpected error has occurred.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>MethodNotAllowedException The specified combination of HTTP verb and URI is not supported.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotDataPlaneAsyncClient.Publish
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/Publish" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<PublishResponse> publish(PublishRequest publishRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Publishes an MQTT message.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >Publish</a> action.
     * </p>
     * <p>
     * For more information about MQTT messages, see <a
     * href="http://docs.aws.amazon.com/iot/latest/developerguide/mqtt.html">MQTT Protocol</a> in the IoT Developer
     * Guide.
     * </p>
     * <p>
     * For more information about messaging costs, see <a
     * href="http://aws.amazon.com/iot-core/pricing/#Messaging">Amazon Web Services IoT Core pricing - Messaging</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PublishRequest.Builder} avoiding the need to create
     * one manually via {@link PublishRequest#builder()}
     * </p>
     *
     * @param publishRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.iotdataplane.model.PublishRequest.Builder} to create a request. The
     *        input for the Publish operation.
     * @return A Java Future containing the result of the Publish operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalFailureException An unexpected error has occurred.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>MethodNotAllowedException The specified combination of HTTP verb and URI is not supported.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotDataPlaneAsyncClient.Publish
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/Publish" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<PublishResponse> publish(Consumer<PublishRequest.Builder> publishRequest) {
        return publish(PublishRequest.builder().applyMutation(publishRequest).build());
    }

    /**
     * <p>
     * Updates the shadow for the specified thing.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >UpdateThingShadow</a> action.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/iot/latest/developerguide/API_UpdateThingShadow.html">UpdateThingShadow</a> in
     * the IoT Developer Guide.
     * </p>
     *
     * @param updateThingShadowRequest
     *        The input for the UpdateThingShadow operation.
     * @return A Java Future containing the result of the UpdateThingShadow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException The specified version does not match the version of the document.</li>
     *         <li>RequestEntityTooLargeException The payload exceeds the maximum size allowed.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>InternalFailureException An unexpected error has occurred.</li>
     *         <li>MethodNotAllowedException The specified combination of HTTP verb and URI is not supported.</li>
     *         <li>UnsupportedDocumentEncodingException The document encoding is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotDataPlaneAsyncClient.UpdateThingShadow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/UpdateThingShadow" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<UpdateThingShadowResponse> updateThingShadow(UpdateThingShadowRequest updateThingShadowRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates the shadow for the specified thing.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >UpdateThingShadow</a> action.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/iot/latest/developerguide/API_UpdateThingShadow.html">UpdateThingShadow</a> in
     * the IoT Developer Guide.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateThingShadowRequest.Builder} avoiding the need
     * to create one manually via {@link UpdateThingShadowRequest#builder()}
     * </p>
     *
     * @param updateThingShadowRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.iotdataplane.model.UpdateThingShadowRequest.Builder} to create a
     *        request. The input for the UpdateThingShadow operation.
     * @return A Java Future containing the result of the UpdateThingShadow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException The specified version does not match the version of the document.</li>
     *         <li>RequestEntityTooLargeException The payload exceeds the maximum size allowed.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>InternalFailureException An unexpected error has occurred.</li>
     *         <li>MethodNotAllowedException The specified combination of HTTP verb and URI is not supported.</li>
     *         <li>UnsupportedDocumentEncodingException The document encoding is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotDataPlaneAsyncClient.UpdateThingShadow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/UpdateThingShadow" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<UpdateThingShadowResponse> updateThingShadow(
            Consumer<UpdateThingShadowRequest.Builder> updateThingShadowRequest) {
        return updateThingShadow(UpdateThingShadowRequest.builder().applyMutation(updateThingShadowRequest).build());
    }

    @Override
    default IotDataPlaneServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create a {@link IotDataPlaneAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static IotDataPlaneAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link IotDataPlaneAsyncClient}.
     */
    static IotDataPlaneAsyncClientBuilder builder() {
        return new DefaultIotDataPlaneAsyncClientBuilder();
    }
}
