/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iot.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the date and time that a job will begin the rollout of the job document to all devices in the target group.
 * Additionally, you can specify the end behavior for each job execution when it reaches the scheduled end time.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SchedulingConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<SchedulingConfig.Builder, SchedulingConfig> {
    private static final SdkField<String> START_TIME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("startTime").getter(getter(SchedulingConfig::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build()).build();

    private static final SdkField<String> END_TIME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("endTime").getter(getter(SchedulingConfig::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endTime").build()).build();

    private static final SdkField<String> END_BEHAVIOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("endBehavior").getter(getter(SchedulingConfig::endBehaviorAsString)).setter(setter(Builder::endBehavior))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endBehavior").build()).build();

    private static final SdkField<List<MaintenanceWindow>> MAINTENANCE_WINDOWS_FIELD = SdkField
            .<List<MaintenanceWindow>> builder(MarshallingType.LIST)
            .memberName("maintenanceWindows")
            .getter(getter(SchedulingConfig::maintenanceWindows))
            .setter(setter(Builder::maintenanceWindows))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maintenanceWindows").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MaintenanceWindow> builder(MarshallingType.SDK_POJO)
                                            .constructor(MaintenanceWindow::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(START_TIME_FIELD,
            END_TIME_FIELD, END_BEHAVIOR_FIELD, MAINTENANCE_WINDOWS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String startTime;

    private final String endTime;

    private final String endBehavior;

    private final List<MaintenanceWindow> maintenanceWindows;

    private SchedulingConfig(BuilderImpl builder) {
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.endBehavior = builder.endBehavior;
        this.maintenanceWindows = builder.maintenanceWindows;
    }

    /**
     * <p>
     * The time a job will begin rollout of the job document to all devices in the target group for a job. The
     * <code>startTime</code> can be scheduled up to a year in advance and must be scheduled a minimum of thirty minutes
     * from the current time. The date and time format for the <code>startTime</code> is YYYY-MM-DD for the date and
     * HH:MM for the time.
     * </p>
     * 
     * @return The time a job will begin rollout of the job document to all devices in the target group for a job. The
     *         <code>startTime</code> can be scheduled up to a year in advance and must be scheduled a minimum of thirty
     *         minutes from the current time. The date and time format for the <code>startTime</code> is YYYY-MM-DD for
     *         the date and HH:MM for the time.
     */
    public final String startTime() {
        return startTime;
    }

    /**
     * <p>
     * The time a job will stop rollout of the job document to all devices in the target group for a job. The
     * <code>endTime</code> must take place no later than two years from the current time and be scheduled a minimum of
     * thirty minutes from the current time. The minimum duration between <code>startTime</code> and
     * <code>endTime</code> is thirty minutes. The maximum duration between <code>startTime</code> and
     * <code>endTime</code> is two years. The date and time format for the <code>endTime</code> is YYYY-MM-DD for the
     * date and HH:MM for the time.
     * </p>
     * 
     * @return The time a job will stop rollout of the job document to all devices in the target group for a job. The
     *         <code>endTime</code> must take place no later than two years from the current time and be scheduled a
     *         minimum of thirty minutes from the current time. The minimum duration between <code>startTime</code> and
     *         <code>endTime</code> is thirty minutes. The maximum duration between <code>startTime</code> and
     *         <code>endTime</code> is two years. The date and time format for the <code>endTime</code> is YYYY-MM-DD
     *         for the date and HH:MM for the time.
     */
    public final String endTime() {
        return endTime;
    }

    /**
     * <p>
     * Specifies the end behavior for all job executions after a job reaches the selected <code>endTime</code>. If
     * <code>endTime</code> is not selected when creating the job, then <code>endBehavior</code> does not apply.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #endBehavior} will
     * return {@link JobEndBehavior#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #endBehaviorAsString}.
     * </p>
     * 
     * @return Specifies the end behavior for all job executions after a job reaches the selected <code>endTime</code>.
     *         If <code>endTime</code> is not selected when creating the job, then <code>endBehavior</code> does not
     *         apply.
     * @see JobEndBehavior
     */
    public final JobEndBehavior endBehavior() {
        return JobEndBehavior.fromValue(endBehavior);
    }

    /**
     * <p>
     * Specifies the end behavior for all job executions after a job reaches the selected <code>endTime</code>. If
     * <code>endTime</code> is not selected when creating the job, then <code>endBehavior</code> does not apply.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #endBehavior} will
     * return {@link JobEndBehavior#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #endBehaviorAsString}.
     * </p>
     * 
     * @return Specifies the end behavior for all job executions after a job reaches the selected <code>endTime</code>.
     *         If <code>endTime</code> is not selected when creating the job, then <code>endBehavior</code> does not
     *         apply.
     * @see JobEndBehavior
     */
    public final String endBehaviorAsString() {
        return endBehavior;
    }

    /**
     * For responses, this returns true if the service returned a value for the MaintenanceWindows property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMaintenanceWindows() {
        return maintenanceWindows != null && !(maintenanceWindows instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An optional configuration within the <code>SchedulingConfig</code> to setup a recurring maintenance window with a
     * predetermined start time and duration for the rollout of a job document to all devices in a target group for a
     * job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMaintenanceWindows} method.
     * </p>
     * 
     * @return An optional configuration within the <code>SchedulingConfig</code> to setup a recurring maintenance
     *         window with a predetermined start time and duration for the rollout of a job document to all devices in a
     *         target group for a job.
     */
    public final List<MaintenanceWindow> maintenanceWindows() {
        return maintenanceWindows;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(endBehaviorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasMaintenanceWindows() ? maintenanceWindows() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SchedulingConfig)) {
            return false;
        }
        SchedulingConfig other = (SchedulingConfig) obj;
        return Objects.equals(startTime(), other.startTime()) && Objects.equals(endTime(), other.endTime())
                && Objects.equals(endBehaviorAsString(), other.endBehaviorAsString())
                && hasMaintenanceWindows() == other.hasMaintenanceWindows()
                && Objects.equals(maintenanceWindows(), other.maintenanceWindows());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SchedulingConfig").add("StartTime", startTime()).add("EndTime", endTime())
                .add("EndBehavior", endBehaviorAsString())
                .add("MaintenanceWindows", hasMaintenanceWindows() ? maintenanceWindows() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "endTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "endBehavior":
            return Optional.ofNullable(clazz.cast(endBehaviorAsString()));
        case "maintenanceWindows":
            return Optional.ofNullable(clazz.cast(maintenanceWindows()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SchedulingConfig, T> g) {
        return obj -> g.apply((SchedulingConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SchedulingConfig> {
        /**
         * <p>
         * The time a job will begin rollout of the job document to all devices in the target group for a job. The
         * <code>startTime</code> can be scheduled up to a year in advance and must be scheduled a minimum of thirty
         * minutes from the current time. The date and time format for the <code>startTime</code> is YYYY-MM-DD for the
         * date and HH:MM for the time.
         * </p>
         * 
         * @param startTime
         *        The time a job will begin rollout of the job document to all devices in the target group for a job.
         *        The <code>startTime</code> can be scheduled up to a year in advance and must be scheduled a minimum of
         *        thirty minutes from the current time. The date and time format for the <code>startTime</code> is
         *        YYYY-MM-DD for the date and HH:MM for the time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(String startTime);

        /**
         * <p>
         * The time a job will stop rollout of the job document to all devices in the target group for a job. The
         * <code>endTime</code> must take place no later than two years from the current time and be scheduled a minimum
         * of thirty minutes from the current time. The minimum duration between <code>startTime</code> and
         * <code>endTime</code> is thirty minutes. The maximum duration between <code>startTime</code> and
         * <code>endTime</code> is two years. The date and time format for the <code>endTime</code> is YYYY-MM-DD for
         * the date and HH:MM for the time.
         * </p>
         * 
         * @param endTime
         *        The time a job will stop rollout of the job document to all devices in the target group for a job. The
         *        <code>endTime</code> must take place no later than two years from the current time and be scheduled a
         *        minimum of thirty minutes from the current time. The minimum duration between <code>startTime</code>
         *        and <code>endTime</code> is thirty minutes. The maximum duration between <code>startTime</code> and
         *        <code>endTime</code> is two years. The date and time format for the <code>endTime</code> is YYYY-MM-DD
         *        for the date and HH:MM for the time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(String endTime);

        /**
         * <p>
         * Specifies the end behavior for all job executions after a job reaches the selected <code>endTime</code>. If
         * <code>endTime</code> is not selected when creating the job, then <code>endBehavior</code> does not apply.
         * </p>
         * 
         * @param endBehavior
         *        Specifies the end behavior for all job executions after a job reaches the selected
         *        <code>endTime</code>. If <code>endTime</code> is not selected when creating the job, then
         *        <code>endBehavior</code> does not apply.
         * @see JobEndBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobEndBehavior
         */
        Builder endBehavior(String endBehavior);

        /**
         * <p>
         * Specifies the end behavior for all job executions after a job reaches the selected <code>endTime</code>. If
         * <code>endTime</code> is not selected when creating the job, then <code>endBehavior</code> does not apply.
         * </p>
         * 
         * @param endBehavior
         *        Specifies the end behavior for all job executions after a job reaches the selected
         *        <code>endTime</code>. If <code>endTime</code> is not selected when creating the job, then
         *        <code>endBehavior</code> does not apply.
         * @see JobEndBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobEndBehavior
         */
        Builder endBehavior(JobEndBehavior endBehavior);

        /**
         * <p>
         * An optional configuration within the <code>SchedulingConfig</code> to setup a recurring maintenance window
         * with a predetermined start time and duration for the rollout of a job document to all devices in a target
         * group for a job.
         * </p>
         * 
         * @param maintenanceWindows
         *        An optional configuration within the <code>SchedulingConfig</code> to setup a recurring maintenance
         *        window with a predetermined start time and duration for the rollout of a job document to all devices
         *        in a target group for a job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maintenanceWindows(Collection<MaintenanceWindow> maintenanceWindows);

        /**
         * <p>
         * An optional configuration within the <code>SchedulingConfig</code> to setup a recurring maintenance window
         * with a predetermined start time and duration for the rollout of a job document to all devices in a target
         * group for a job.
         * </p>
         * 
         * @param maintenanceWindows
         *        An optional configuration within the <code>SchedulingConfig</code> to setup a recurring maintenance
         *        window with a predetermined start time and duration for the rollout of a job document to all devices
         *        in a target group for a job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maintenanceWindows(MaintenanceWindow... maintenanceWindows);

        /**
         * <p>
         * An optional configuration within the <code>SchedulingConfig</code> to setup a recurring maintenance window
         * with a predetermined start time and duration for the rollout of a job document to all devices in a target
         * group for a job.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.iot.model.MaintenanceWindow.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.iot.model.MaintenanceWindow#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.iot.model.MaintenanceWindow.Builder#build()} is called immediately and
         * its result is passed to {@link #maintenanceWindows(List<MaintenanceWindow>)}.
         * 
         * @param maintenanceWindows
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.iot.model.MaintenanceWindow.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #maintenanceWindows(java.util.Collection<MaintenanceWindow>)
         */
        Builder maintenanceWindows(Consumer<MaintenanceWindow.Builder>... maintenanceWindows);
    }

    static final class BuilderImpl implements Builder {
        private String startTime;

        private String endTime;

        private String endBehavior;

        private List<MaintenanceWindow> maintenanceWindows = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(SchedulingConfig model) {
            startTime(model.startTime);
            endTime(model.endTime);
            endBehavior(model.endBehavior);
            maintenanceWindows(model.maintenanceWindows);
        }

        public final String getStartTime() {
            return startTime;
        }

        public final void setStartTime(String startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(String startTime) {
            this.startTime = startTime;
            return this;
        }

        public final String getEndTime() {
            return endTime;
        }

        public final void setEndTime(String endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(String endTime) {
            this.endTime = endTime;
            return this;
        }

        public final String getEndBehavior() {
            return endBehavior;
        }

        public final void setEndBehavior(String endBehavior) {
            this.endBehavior = endBehavior;
        }

        @Override
        public final Builder endBehavior(String endBehavior) {
            this.endBehavior = endBehavior;
            return this;
        }

        @Override
        public final Builder endBehavior(JobEndBehavior endBehavior) {
            this.endBehavior(endBehavior == null ? null : endBehavior.toString());
            return this;
        }

        public final List<MaintenanceWindow.Builder> getMaintenanceWindows() {
            List<MaintenanceWindow.Builder> result = MaintenanceWindowsCopier.copyToBuilder(this.maintenanceWindows);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setMaintenanceWindows(Collection<MaintenanceWindow.BuilderImpl> maintenanceWindows) {
            this.maintenanceWindows = MaintenanceWindowsCopier.copyFromBuilder(maintenanceWindows);
        }

        @Override
        public final Builder maintenanceWindows(Collection<MaintenanceWindow> maintenanceWindows) {
            this.maintenanceWindows = MaintenanceWindowsCopier.copy(maintenanceWindows);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder maintenanceWindows(MaintenanceWindow... maintenanceWindows) {
            maintenanceWindows(Arrays.asList(maintenanceWindows));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder maintenanceWindows(Consumer<MaintenanceWindow.Builder>... maintenanceWindows) {
            maintenanceWindows(Stream.of(maintenanceWindows).map(c -> MaintenanceWindow.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public SchedulingConfig build() {
            return new SchedulingConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
