/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.health.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Summary information about an event, returned by the <a
 * href="https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventsForOrganization.html"
 * >DescribeEventsForOrganization</a> operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OrganizationEvent implements SdkPojo, Serializable,
        ToCopyableBuilder<OrganizationEvent.Builder, OrganizationEvent> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(OrganizationEvent::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> SERVICE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("service")
            .getter(getter(OrganizationEvent::service)).setter(setter(Builder::service))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("service").build()).build();

    private static final SdkField<String> EVENT_TYPE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("eventTypeCode").getter(getter(OrganizationEvent::eventTypeCode)).setter(setter(Builder::eventTypeCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventTypeCode").build()).build();

    private static final SdkField<String> EVENT_TYPE_CATEGORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("eventTypeCategory").getter(getter(OrganizationEvent::eventTypeCategoryAsString))
            .setter(setter(Builder::eventTypeCategory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventTypeCategory").build()).build();

    private static final SdkField<String> EVENT_SCOPE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("eventScopeCode").getter(getter(OrganizationEvent::eventScopeCodeAsString))
            .setter(setter(Builder::eventScopeCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventScopeCode").build()).build();

    private static final SdkField<String> REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("region")
            .getter(getter(OrganizationEvent::region)).setter(setter(Builder::region))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("region").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("startTime").getter(getter(OrganizationEvent::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("endTime").getter(getter(OrganizationEvent::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endTime").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastUpdatedTime").getter(getter(OrganizationEvent::lastUpdatedTime))
            .setter(setter(Builder::lastUpdatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUpdatedTime").build()).build();

    private static final SdkField<String> STATUS_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("statusCode").getter(getter(OrganizationEvent::statusCodeAsString)).setter(setter(Builder::statusCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusCode").build()).build();

    private static final SdkField<String> ACTIONABILITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("actionability").getter(getter(OrganizationEvent::actionabilityAsString))
            .setter(setter(Builder::actionability))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("actionability").build()).build();

    private static final SdkField<List<String>> PERSONAS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("personas")
            .getter(getter(OrganizationEvent::personasAsStrings))
            .setter(setter(Builder::personasWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("personas").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD, SERVICE_FIELD,
            EVENT_TYPE_CODE_FIELD, EVENT_TYPE_CATEGORY_FIELD, EVENT_SCOPE_CODE_FIELD, REGION_FIELD, START_TIME_FIELD,
            END_TIME_FIELD, LAST_UPDATED_TIME_FIELD, STATUS_CODE_FIELD, ACTIONABILITY_FIELD, PERSONAS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final String service;

    private final String eventTypeCode;

    private final String eventTypeCategory;

    private final String eventScopeCode;

    private final String region;

    private final Instant startTime;

    private final Instant endTime;

    private final Instant lastUpdatedTime;

    private final String statusCodeValue;

    private final String actionability;

    private final List<String> personas;

    private OrganizationEvent(BuilderImpl builder) {
        this.arn = builder.arn;
        this.service = builder.service;
        this.eventTypeCode = builder.eventTypeCode;
        this.eventTypeCategory = builder.eventTypeCategory;
        this.eventScopeCode = builder.eventScopeCode;
        this.region = builder.region;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.lastUpdatedTime = builder.lastUpdatedTime;
        this.statusCodeValue = builder.statusCodeValue;
        this.actionability = builder.actionability;
        this.personas = builder.personas;
    }

    /**
     * <p>
     * The unique identifier for the event. The event ARN has the
     * <code>arn:aws:health:<i>event-region</i>::event/<i>SERVICE</i>/<i>EVENT_TYPE_CODE</i>/<i>EVENT_TYPE_PLUS_ID</i> </code>
     * format.
     * </p>
     * <p>
     * For example, an event ARN might look like the following:
     * </p>
     * <p>
     * <code>arn:aws:health:us-east-1::event/EC2/EC2_INSTANCE_RETIREMENT_SCHEDULED/EC2_INSTANCE_RETIREMENT_SCHEDULED_ABC123-DEF456</code>
     * </p>
     * 
     * @return The unique identifier for the event. The event ARN has the
     *         <code>arn:aws:health:<i>event-region</i>::event/<i>SERVICE</i>/<i>EVENT_TYPE_CODE</i>/<i>EVENT_TYPE_PLUS_ID</i> </code>
     *         format.</p>
     *         <p>
     *         For example, an event ARN might look like the following:
     *         </p>
     *         <p>
     *         <code>arn:aws:health:us-east-1::event/EC2/EC2_INSTANCE_RETIREMENT_SCHEDULED/EC2_INSTANCE_RETIREMENT_SCHEDULED_ABC123-DEF456</code>
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The Amazon Web Services service that is affected by the event, such as EC2 and RDS.
     * </p>
     * 
     * @return The Amazon Web Services service that is affected by the event, such as EC2 and RDS.
     */
    public final String service() {
        return service;
    }

    /**
     * <p>
     * The unique identifier for the event type. The format is <code>AWS_SERVICE_DESCRIPTION</code>. For example,
     * <code>AWS_EC2_SYSTEM_MAINTENANCE_EVENT</code>.
     * </p>
     * 
     * @return The unique identifier for the event type. The format is <code>AWS_SERVICE_DESCRIPTION</code>. For
     *         example, <code>AWS_EC2_SYSTEM_MAINTENANCE_EVENT</code>.
     */
    public final String eventTypeCode() {
        return eventTypeCode;
    }

    /**
     * <p>
     * A list of event type category codes. Possible values are <code>issue</code>, <code>accountNotification</code>, or
     * <code>scheduledChange</code>. Currently, the <code>investigation</code> value isn't supported at this time.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventTypeCategory}
     * will return {@link EventTypeCategory#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #eventTypeCategoryAsString}.
     * </p>
     * 
     * @return A list of event type category codes. Possible values are <code>issue</code>,
     *         <code>accountNotification</code>, or <code>scheduledChange</code>. Currently, the
     *         <code>investigation</code> value isn't supported at this time.
     * @see EventTypeCategory
     */
    public final EventTypeCategory eventTypeCategory() {
        return EventTypeCategory.fromValue(eventTypeCategory);
    }

    /**
     * <p>
     * A list of event type category codes. Possible values are <code>issue</code>, <code>accountNotification</code>, or
     * <code>scheduledChange</code>. Currently, the <code>investigation</code> value isn't supported at this time.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventTypeCategory}
     * will return {@link EventTypeCategory#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #eventTypeCategoryAsString}.
     * </p>
     * 
     * @return A list of event type category codes. Possible values are <code>issue</code>,
     *         <code>accountNotification</code>, or <code>scheduledChange</code>. Currently, the
     *         <code>investigation</code> value isn't supported at this time.
     * @see EventTypeCategory
     */
    public final String eventTypeCategoryAsString() {
        return eventTypeCategory;
    }

    /**
     * <p>
     * This parameter specifies if the Health event is a public Amazon Web Services service event or an account-specific
     * event.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the <code>eventScopeCode</code> value is <code>PUBLIC</code>, then the <code>affectedAccounts</code> value is
     * always empty.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the <code>eventScopeCode</code> value is <code>ACCOUNT_SPECIFIC</code>, then the <code>affectedAccounts</code>
     * value lists the affected Amazon Web Services accounts in your organization. For example, if an event affects a
     * service such as Amazon Elastic Compute Cloud and you have Amazon Web Services accounts that use that service,
     * those account IDs appear in the response.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the <code>eventScopeCode</code> value is <code>NONE</code>, then the <code>eventArn</code> that you specified
     * in the request is invalid or doesn't exist.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventScopeCode}
     * will return {@link EventScopeCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #eventScopeCodeAsString}.
     * </p>
     * 
     * @return This parameter specifies if the Health event is a public Amazon Web Services service event or an
     *         account-specific event.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If the <code>eventScopeCode</code> value is <code>PUBLIC</code>, then the <code>affectedAccounts</code>
     *         value is always empty.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the <code>eventScopeCode</code> value is <code>ACCOUNT_SPECIFIC</code>, then the
     *         <code>affectedAccounts</code> value lists the affected Amazon Web Services accounts in your organization.
     *         For example, if an event affects a service such as Amazon Elastic Compute Cloud and you have Amazon Web
     *         Services accounts that use that service, those account IDs appear in the response.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the <code>eventScopeCode</code> value is <code>NONE</code>, then the <code>eventArn</code> that you
     *         specified in the request is invalid or doesn't exist.
     *         </p>
     *         </li>
     * @see EventScopeCode
     */
    public final EventScopeCode eventScopeCode() {
        return EventScopeCode.fromValue(eventScopeCode);
    }

    /**
     * <p>
     * This parameter specifies if the Health event is a public Amazon Web Services service event or an account-specific
     * event.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the <code>eventScopeCode</code> value is <code>PUBLIC</code>, then the <code>affectedAccounts</code> value is
     * always empty.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the <code>eventScopeCode</code> value is <code>ACCOUNT_SPECIFIC</code>, then the <code>affectedAccounts</code>
     * value lists the affected Amazon Web Services accounts in your organization. For example, if an event affects a
     * service such as Amazon Elastic Compute Cloud and you have Amazon Web Services accounts that use that service,
     * those account IDs appear in the response.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the <code>eventScopeCode</code> value is <code>NONE</code>, then the <code>eventArn</code> that you specified
     * in the request is invalid or doesn't exist.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventScopeCode}
     * will return {@link EventScopeCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #eventScopeCodeAsString}.
     * </p>
     * 
     * @return This parameter specifies if the Health event is a public Amazon Web Services service event or an
     *         account-specific event.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If the <code>eventScopeCode</code> value is <code>PUBLIC</code>, then the <code>affectedAccounts</code>
     *         value is always empty.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the <code>eventScopeCode</code> value is <code>ACCOUNT_SPECIFIC</code>, then the
     *         <code>affectedAccounts</code> value lists the affected Amazon Web Services accounts in your organization.
     *         For example, if an event affects a service such as Amazon Elastic Compute Cloud and you have Amazon Web
     *         Services accounts that use that service, those account IDs appear in the response.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the <code>eventScopeCode</code> value is <code>NONE</code>, then the <code>eventArn</code> that you
     *         specified in the request is invalid or doesn't exist.
     *         </p>
     *         </li>
     * @see EventScopeCode
     */
    public final String eventScopeCodeAsString() {
        return eventScopeCode;
    }

    /**
     * <p>
     * The Amazon Web Services Region name of the event.
     * </p>
     * 
     * @return The Amazon Web Services Region name of the event.
     */
    public final String region() {
        return region;
    }

    /**
     * <p>
     * The date and time that the event began.
     * </p>
     * 
     * @return The date and time that the event began.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The date and time that the event ended.
     * </p>
     * 
     * @return The date and time that the event ended.
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * The most recent date and time that the event was updated.
     * </p>
     * 
     * @return The most recent date and time that the event was updated.
     */
    public final Instant lastUpdatedTime() {
        return lastUpdatedTime;
    }

    /**
     * <p>
     * The most recent status of the event. Possible values are <code>open</code>, <code>closed</code>, and
     * <code>upcoming</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statusCode} will
     * return {@link EventStatusCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusCodeAsString}.
     * </p>
     * 
     * @return The most recent status of the event. Possible values are <code>open</code>, <code>closed</code>, and
     *         <code>upcoming</code>.
     * @see EventStatusCode
     */
    public final EventStatusCode statusCode() {
        return EventStatusCode.fromValue(statusCodeValue);
    }

    /**
     * <p>
     * The most recent status of the event. Possible values are <code>open</code>, <code>closed</code>, and
     * <code>upcoming</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statusCode} will
     * return {@link EventStatusCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusCodeAsString}.
     * </p>
     * 
     * @return The most recent status of the event. Possible values are <code>open</code>, <code>closed</code>, and
     *         <code>upcoming</code>.
     * @see EventStatusCode
     */
    public final String statusCodeAsString() {
        return statusCodeValue;
    }

    /**
     * <p>
     * The actionability classification of the event. Possible values are <code>ACTION_REQUIRED</code>,
     * <code>ACTION_MAY_BE_REQUIRED</code> and <code>INFORMATIONAL</code>. Events with <code>ACTION_REQUIRED</code>
     * actionability require customer action to resolve or mitigate the event. Events with
     * <code>ACTION_MAY_BE_REQUIRED</code> actionability indicates that the current status is unknown or conditional and
     * inspection is needed to determine if action is required. Events with <code>INFORMATIONAL</code> actionability are
     * provided for awareness and do not require immediate action.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #actionability}
     * will return {@link EventActionability#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #actionabilityAsString}.
     * </p>
     * 
     * @return The actionability classification of the event. Possible values are <code>ACTION_REQUIRED</code>,
     *         <code>ACTION_MAY_BE_REQUIRED</code> and <code>INFORMATIONAL</code>. Events with
     *         <code>ACTION_REQUIRED</code> actionability require customer action to resolve or mitigate the event.
     *         Events with <code>ACTION_MAY_BE_REQUIRED</code> actionability indicates that the current status is
     *         unknown or conditional and inspection is needed to determine if action is required. Events with
     *         <code>INFORMATIONAL</code> actionability are provided for awareness and do not require immediate action.
     * @see EventActionability
     */
    public final EventActionability actionability() {
        return EventActionability.fromValue(actionability);
    }

    /**
     * <p>
     * The actionability classification of the event. Possible values are <code>ACTION_REQUIRED</code>,
     * <code>ACTION_MAY_BE_REQUIRED</code> and <code>INFORMATIONAL</code>. Events with <code>ACTION_REQUIRED</code>
     * actionability require customer action to resolve or mitigate the event. Events with
     * <code>ACTION_MAY_BE_REQUIRED</code> actionability indicates that the current status is unknown or conditional and
     * inspection is needed to determine if action is required. Events with <code>INFORMATIONAL</code> actionability are
     * provided for awareness and do not require immediate action.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #actionability}
     * will return {@link EventActionability#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #actionabilityAsString}.
     * </p>
     * 
     * @return The actionability classification of the event. Possible values are <code>ACTION_REQUIRED</code>,
     *         <code>ACTION_MAY_BE_REQUIRED</code> and <code>INFORMATIONAL</code>. Events with
     *         <code>ACTION_REQUIRED</code> actionability require customer action to resolve or mitigate the event.
     *         Events with <code>ACTION_MAY_BE_REQUIRED</code> actionability indicates that the current status is
     *         unknown or conditional and inspection is needed to determine if action is required. Events with
     *         <code>INFORMATIONAL</code> actionability are provided for awareness and do not require immediate action.
     * @see EventActionability
     */
    public final String actionabilityAsString() {
        return actionability;
    }

    /**
     * <p>
     * A list of persona classifications that indicate the target audience for the event. Possible values are
     * <code>OPERATIONS</code>, <code>SECURITY</code>, and <code>BILLING</code>. Events can be associated with multiple
     * personas to indicate relevance to different teams or roles within an organization.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPersonas} method.
     * </p>
     * 
     * @return A list of persona classifications that indicate the target audience for the event. Possible values are
     *         <code>OPERATIONS</code>, <code>SECURITY</code>, and <code>BILLING</code>. Events can be associated with
     *         multiple personas to indicate relevance to different teams or roles within an organization.
     */
    public final List<EventPersona> personas() {
        return EventPersonaListCopier.copyStringToEnum(personas);
    }

    /**
     * For responses, this returns true if the service returned a value for the Personas property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasPersonas() {
        return personas != null && !(personas instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of persona classifications that indicate the target audience for the event. Possible values are
     * <code>OPERATIONS</code>, <code>SECURITY</code>, and <code>BILLING</code>. Events can be associated with multiple
     * personas to indicate relevance to different teams or roles within an organization.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPersonas} method.
     * </p>
     * 
     * @return A list of persona classifications that indicate the target audience for the event. Possible values are
     *         <code>OPERATIONS</code>, <code>SECURITY</code>, and <code>BILLING</code>. Events can be associated with
     *         multiple personas to indicate relevance to different teams or roles within an organization.
     */
    public final List<String> personasAsStrings() {
        return personas;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(service());
        hashCode = 31 * hashCode + Objects.hashCode(eventTypeCode());
        hashCode = 31 * hashCode + Objects.hashCode(eventTypeCategoryAsString());
        hashCode = 31 * hashCode + Objects.hashCode(eventScopeCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(region());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTime());
        hashCode = 31 * hashCode + Objects.hashCode(statusCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(actionabilityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasPersonas() ? personasAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OrganizationEvent)) {
            return false;
        }
        OrganizationEvent other = (OrganizationEvent) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(service(), other.service())
                && Objects.equals(eventTypeCode(), other.eventTypeCode())
                && Objects.equals(eventTypeCategoryAsString(), other.eventTypeCategoryAsString())
                && Objects.equals(eventScopeCodeAsString(), other.eventScopeCodeAsString())
                && Objects.equals(region(), other.region()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(endTime(), other.endTime()) && Objects.equals(lastUpdatedTime(), other.lastUpdatedTime())
                && Objects.equals(statusCodeAsString(), other.statusCodeAsString())
                && Objects.equals(actionabilityAsString(), other.actionabilityAsString()) && hasPersonas() == other.hasPersonas()
                && Objects.equals(personasAsStrings(), other.personasAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("OrganizationEvent").add("Arn", arn()).add("Service", service())
                .add("EventTypeCode", eventTypeCode()).add("EventTypeCategory", eventTypeCategoryAsString())
                .add("EventScopeCode", eventScopeCodeAsString()).add("Region", region()).add("StartTime", startTime())
                .add("EndTime", endTime()).add("LastUpdatedTime", lastUpdatedTime()).add("StatusCode", statusCodeAsString())
                .add("Actionability", actionabilityAsString()).add("Personas", hasPersonas() ? personasAsStrings() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "service":
            return Optional.ofNullable(clazz.cast(service()));
        case "eventTypeCode":
            return Optional.ofNullable(clazz.cast(eventTypeCode()));
        case "eventTypeCategory":
            return Optional.ofNullable(clazz.cast(eventTypeCategoryAsString()));
        case "eventScopeCode":
            return Optional.ofNullable(clazz.cast(eventScopeCodeAsString()));
        case "region":
            return Optional.ofNullable(clazz.cast(region()));
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "endTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "lastUpdatedTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedTime()));
        case "statusCode":
            return Optional.ofNullable(clazz.cast(statusCodeAsString()));
        case "actionability":
            return Optional.ofNullable(clazz.cast(actionabilityAsString()));
        case "personas":
            return Optional.ofNullable(clazz.cast(personasAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("arn", ARN_FIELD);
        map.put("service", SERVICE_FIELD);
        map.put("eventTypeCode", EVENT_TYPE_CODE_FIELD);
        map.put("eventTypeCategory", EVENT_TYPE_CATEGORY_FIELD);
        map.put("eventScopeCode", EVENT_SCOPE_CODE_FIELD);
        map.put("region", REGION_FIELD);
        map.put("startTime", START_TIME_FIELD);
        map.put("endTime", END_TIME_FIELD);
        map.put("lastUpdatedTime", LAST_UPDATED_TIME_FIELD);
        map.put("statusCode", STATUS_CODE_FIELD);
        map.put("actionability", ACTIONABILITY_FIELD);
        map.put("personas", PERSONAS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<OrganizationEvent, T> g) {
        return obj -> g.apply((OrganizationEvent) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OrganizationEvent> {
        /**
         * <p>
         * The unique identifier for the event. The event ARN has the
         * <code>arn:aws:health:<i>event-region</i>::event/<i>SERVICE</i>/<i>EVENT_TYPE_CODE</i>/<i>EVENT_TYPE_PLUS_ID</i> </code>
         * format.
         * </p>
         * <p>
         * For example, an event ARN might look like the following:
         * </p>
         * <p>
         * <code>arn:aws:health:us-east-1::event/EC2/EC2_INSTANCE_RETIREMENT_SCHEDULED/EC2_INSTANCE_RETIREMENT_SCHEDULED_ABC123-DEF456</code>
         * </p>
         * 
         * @param arn
         *        The unique identifier for the event. The event ARN has the
         *        <code>arn:aws:health:<i>event-region</i>::event/<i>SERVICE</i>/<i>EVENT_TYPE_CODE</i>/<i>EVENT_TYPE_PLUS_ID</i> </code>
         *        format.</p>
         *        <p>
         *        For example, an event ARN might look like the following:
         *        </p>
         *        <p>
         *        <code>arn:aws:health:us-east-1::event/EC2/EC2_INSTANCE_RETIREMENT_SCHEDULED/EC2_INSTANCE_RETIREMENT_SCHEDULED_ABC123-DEF456</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The Amazon Web Services service that is affected by the event, such as EC2 and RDS.
         * </p>
         * 
         * @param service
         *        The Amazon Web Services service that is affected by the event, such as EC2 and RDS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder service(String service);

        /**
         * <p>
         * The unique identifier for the event type. The format is <code>AWS_SERVICE_DESCRIPTION</code>. For example,
         * <code>AWS_EC2_SYSTEM_MAINTENANCE_EVENT</code>.
         * </p>
         * 
         * @param eventTypeCode
         *        The unique identifier for the event type. The format is <code>AWS_SERVICE_DESCRIPTION</code>. For
         *        example, <code>AWS_EC2_SYSTEM_MAINTENANCE_EVENT</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventTypeCode(String eventTypeCode);

        /**
         * <p>
         * A list of event type category codes. Possible values are <code>issue</code>, <code>accountNotification</code>
         * , or <code>scheduledChange</code>. Currently, the <code>investigation</code> value isn't supported at this
         * time.
         * </p>
         * 
         * @param eventTypeCategory
         *        A list of event type category codes. Possible values are <code>issue</code>,
         *        <code>accountNotification</code>, or <code>scheduledChange</code>. Currently, the
         *        <code>investigation</code> value isn't supported at this time.
         * @see EventTypeCategory
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventTypeCategory
         */
        Builder eventTypeCategory(String eventTypeCategory);

        /**
         * <p>
         * A list of event type category codes. Possible values are <code>issue</code>, <code>accountNotification</code>
         * , or <code>scheduledChange</code>. Currently, the <code>investigation</code> value isn't supported at this
         * time.
         * </p>
         * 
         * @param eventTypeCategory
         *        A list of event type category codes. Possible values are <code>issue</code>,
         *        <code>accountNotification</code>, or <code>scheduledChange</code>. Currently, the
         *        <code>investigation</code> value isn't supported at this time.
         * @see EventTypeCategory
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventTypeCategory
         */
        Builder eventTypeCategory(EventTypeCategory eventTypeCategory);

        /**
         * <p>
         * This parameter specifies if the Health event is a public Amazon Web Services service event or an
         * account-specific event.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the <code>eventScopeCode</code> value is <code>PUBLIC</code>, then the <code>affectedAccounts</code> value
         * is always empty.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the <code>eventScopeCode</code> value is <code>ACCOUNT_SPECIFIC</code>, then the
         * <code>affectedAccounts</code> value lists the affected Amazon Web Services accounts in your organization. For
         * example, if an event affects a service such as Amazon Elastic Compute Cloud and you have Amazon Web Services
         * accounts that use that service, those account IDs appear in the response.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the <code>eventScopeCode</code> value is <code>NONE</code>, then the <code>eventArn</code> that you
         * specified in the request is invalid or doesn't exist.
         * </p>
         * </li>
         * </ul>
         * 
         * @param eventScopeCode
         *        This parameter specifies if the Health event is a public Amazon Web Services service event or an
         *        account-specific event.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the <code>eventScopeCode</code> value is <code>PUBLIC</code>, then the
         *        <code>affectedAccounts</code> value is always empty.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the <code>eventScopeCode</code> value is <code>ACCOUNT_SPECIFIC</code>, then the
         *        <code>affectedAccounts</code> value lists the affected Amazon Web Services accounts in your
         *        organization. For example, if an event affects a service such as Amazon Elastic Compute Cloud and you
         *        have Amazon Web Services accounts that use that service, those account IDs appear in the response.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the <code>eventScopeCode</code> value is <code>NONE</code>, then the <code>eventArn</code> that you
         *        specified in the request is invalid or doesn't exist.
         *        </p>
         *        </li>
         * @see EventScopeCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventScopeCode
         */
        Builder eventScopeCode(String eventScopeCode);

        /**
         * <p>
         * This parameter specifies if the Health event is a public Amazon Web Services service event or an
         * account-specific event.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the <code>eventScopeCode</code> value is <code>PUBLIC</code>, then the <code>affectedAccounts</code> value
         * is always empty.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the <code>eventScopeCode</code> value is <code>ACCOUNT_SPECIFIC</code>, then the
         * <code>affectedAccounts</code> value lists the affected Amazon Web Services accounts in your organization. For
         * example, if an event affects a service such as Amazon Elastic Compute Cloud and you have Amazon Web Services
         * accounts that use that service, those account IDs appear in the response.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the <code>eventScopeCode</code> value is <code>NONE</code>, then the <code>eventArn</code> that you
         * specified in the request is invalid or doesn't exist.
         * </p>
         * </li>
         * </ul>
         * 
         * @param eventScopeCode
         *        This parameter specifies if the Health event is a public Amazon Web Services service event or an
         *        account-specific event.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the <code>eventScopeCode</code> value is <code>PUBLIC</code>, then the
         *        <code>affectedAccounts</code> value is always empty.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the <code>eventScopeCode</code> value is <code>ACCOUNT_SPECIFIC</code>, then the
         *        <code>affectedAccounts</code> value lists the affected Amazon Web Services accounts in your
         *        organization. For example, if an event affects a service such as Amazon Elastic Compute Cloud and you
         *        have Amazon Web Services accounts that use that service, those account IDs appear in the response.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the <code>eventScopeCode</code> value is <code>NONE</code>, then the <code>eventArn</code> that you
         *        specified in the request is invalid or doesn't exist.
         *        </p>
         *        </li>
         * @see EventScopeCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventScopeCode
         */
        Builder eventScopeCode(EventScopeCode eventScopeCode);

        /**
         * <p>
         * The Amazon Web Services Region name of the event.
         * </p>
         * 
         * @param region
         *        The Amazon Web Services Region name of the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder region(String region);

        /**
         * <p>
         * The date and time that the event began.
         * </p>
         * 
         * @param startTime
         *        The date and time that the event began.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The date and time that the event ended.
         * </p>
         * 
         * @param endTime
         *        The date and time that the event ended.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * The most recent date and time that the event was updated.
         * </p>
         * 
         * @param lastUpdatedTime
         *        The most recent date and time that the event was updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTime(Instant lastUpdatedTime);

        /**
         * <p>
         * The most recent status of the event. Possible values are <code>open</code>, <code>closed</code>, and
         * <code>upcoming</code>.
         * </p>
         * 
         * @param statusCodeValue
         *        The most recent status of the event. Possible values are <code>open</code>, <code>closed</code>, and
         *        <code>upcoming</code>.
         * @see EventStatusCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventStatusCode
         */
        Builder statusCode(String statusCodeValue);

        /**
         * <p>
         * The most recent status of the event. Possible values are <code>open</code>, <code>closed</code>, and
         * <code>upcoming</code>.
         * </p>
         * 
         * @param statusCodeValue
         *        The most recent status of the event. Possible values are <code>open</code>, <code>closed</code>, and
         *        <code>upcoming</code>.
         * @see EventStatusCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventStatusCode
         */
        Builder statusCode(EventStatusCode statusCodeValue);

        /**
         * <p>
         * The actionability classification of the event. Possible values are <code>ACTION_REQUIRED</code>,
         * <code>ACTION_MAY_BE_REQUIRED</code> and <code>INFORMATIONAL</code>. Events with <code>ACTION_REQUIRED</code>
         * actionability require customer action to resolve or mitigate the event. Events with
         * <code>ACTION_MAY_BE_REQUIRED</code> actionability indicates that the current status is unknown or conditional
         * and inspection is needed to determine if action is required. Events with <code>INFORMATIONAL</code>
         * actionability are provided for awareness and do not require immediate action.
         * </p>
         * 
         * @param actionability
         *        The actionability classification of the event. Possible values are <code>ACTION_REQUIRED</code>,
         *        <code>ACTION_MAY_BE_REQUIRED</code> and <code>INFORMATIONAL</code>. Events with
         *        <code>ACTION_REQUIRED</code> actionability require customer action to resolve or mitigate the event.
         *        Events with <code>ACTION_MAY_BE_REQUIRED</code> actionability indicates that the current status is
         *        unknown or conditional and inspection is needed to determine if action is required. Events with
         *        <code>INFORMATIONAL</code> actionability are provided for awareness and do not require immediate
         *        action.
         * @see EventActionability
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventActionability
         */
        Builder actionability(String actionability);

        /**
         * <p>
         * The actionability classification of the event. Possible values are <code>ACTION_REQUIRED</code>,
         * <code>ACTION_MAY_BE_REQUIRED</code> and <code>INFORMATIONAL</code>. Events with <code>ACTION_REQUIRED</code>
         * actionability require customer action to resolve or mitigate the event. Events with
         * <code>ACTION_MAY_BE_REQUIRED</code> actionability indicates that the current status is unknown or conditional
         * and inspection is needed to determine if action is required. Events with <code>INFORMATIONAL</code>
         * actionability are provided for awareness and do not require immediate action.
         * </p>
         * 
         * @param actionability
         *        The actionability classification of the event. Possible values are <code>ACTION_REQUIRED</code>,
         *        <code>ACTION_MAY_BE_REQUIRED</code> and <code>INFORMATIONAL</code>. Events with
         *        <code>ACTION_REQUIRED</code> actionability require customer action to resolve or mitigate the event.
         *        Events with <code>ACTION_MAY_BE_REQUIRED</code> actionability indicates that the current status is
         *        unknown or conditional and inspection is needed to determine if action is required. Events with
         *        <code>INFORMATIONAL</code> actionability are provided for awareness and do not require immediate
         *        action.
         * @see EventActionability
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventActionability
         */
        Builder actionability(EventActionability actionability);

        /**
         * <p>
         * A list of persona classifications that indicate the target audience for the event. Possible values are
         * <code>OPERATIONS</code>, <code>SECURITY</code>, and <code>BILLING</code>. Events can be associated with
         * multiple personas to indicate relevance to different teams or roles within an organization.
         * </p>
         * 
         * @param personas
         *        A list of persona classifications that indicate the target audience for the event. Possible values are
         *        <code>OPERATIONS</code>, <code>SECURITY</code>, and <code>BILLING</code>. Events can be associated
         *        with multiple personas to indicate relevance to different teams or roles within an organization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder personasWithStrings(Collection<String> personas);

        /**
         * <p>
         * A list of persona classifications that indicate the target audience for the event. Possible values are
         * <code>OPERATIONS</code>, <code>SECURITY</code>, and <code>BILLING</code>. Events can be associated with
         * multiple personas to indicate relevance to different teams or roles within an organization.
         * </p>
         * 
         * @param personas
         *        A list of persona classifications that indicate the target audience for the event. Possible values are
         *        <code>OPERATIONS</code>, <code>SECURITY</code>, and <code>BILLING</code>. Events can be associated
         *        with multiple personas to indicate relevance to different teams or roles within an organization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder personasWithStrings(String... personas);

        /**
         * <p>
         * A list of persona classifications that indicate the target audience for the event. Possible values are
         * <code>OPERATIONS</code>, <code>SECURITY</code>, and <code>BILLING</code>. Events can be associated with
         * multiple personas to indicate relevance to different teams or roles within an organization.
         * </p>
         * 
         * @param personas
         *        A list of persona classifications that indicate the target audience for the event. Possible values are
         *        <code>OPERATIONS</code>, <code>SECURITY</code>, and <code>BILLING</code>. Events can be associated
         *        with multiple personas to indicate relevance to different teams or roles within an organization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder personas(Collection<EventPersona> personas);

        /**
         * <p>
         * A list of persona classifications that indicate the target audience for the event. Possible values are
         * <code>OPERATIONS</code>, <code>SECURITY</code>, and <code>BILLING</code>. Events can be associated with
         * multiple personas to indicate relevance to different teams or roles within an organization.
         * </p>
         * 
         * @param personas
         *        A list of persona classifications that indicate the target audience for the event. Possible values are
         *        <code>OPERATIONS</code>, <code>SECURITY</code>, and <code>BILLING</code>. Events can be associated
         *        with multiple personas to indicate relevance to different teams or roles within an organization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder personas(EventPersona... personas);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private String service;

        private String eventTypeCode;

        private String eventTypeCategory;

        private String eventScopeCode;

        private String region;

        private Instant startTime;

        private Instant endTime;

        private Instant lastUpdatedTime;

        private String statusCodeValue;

        private String actionability;

        private List<String> personas = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(OrganizationEvent model) {
            arn(model.arn);
            service(model.service);
            eventTypeCode(model.eventTypeCode);
            eventTypeCategory(model.eventTypeCategory);
            eventScopeCode(model.eventScopeCode);
            region(model.region);
            startTime(model.startTime);
            endTime(model.endTime);
            lastUpdatedTime(model.lastUpdatedTime);
            statusCode(model.statusCodeValue);
            actionability(model.actionability);
            personasWithStrings(model.personas);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getService() {
            return service;
        }

        public final void setService(String service) {
            this.service = service;
        }

        @Override
        public final Builder service(String service) {
            this.service = service;
            return this;
        }

        public final String getEventTypeCode() {
            return eventTypeCode;
        }

        public final void setEventTypeCode(String eventTypeCode) {
            this.eventTypeCode = eventTypeCode;
        }

        @Override
        public final Builder eventTypeCode(String eventTypeCode) {
            this.eventTypeCode = eventTypeCode;
            return this;
        }

        public final String getEventTypeCategory() {
            return eventTypeCategory;
        }

        public final void setEventTypeCategory(String eventTypeCategory) {
            this.eventTypeCategory = eventTypeCategory;
        }

        @Override
        public final Builder eventTypeCategory(String eventTypeCategory) {
            this.eventTypeCategory = eventTypeCategory;
            return this;
        }

        @Override
        public final Builder eventTypeCategory(EventTypeCategory eventTypeCategory) {
            this.eventTypeCategory(eventTypeCategory == null ? null : eventTypeCategory.toString());
            return this;
        }

        public final String getEventScopeCode() {
            return eventScopeCode;
        }

        public final void setEventScopeCode(String eventScopeCode) {
            this.eventScopeCode = eventScopeCode;
        }

        @Override
        public final Builder eventScopeCode(String eventScopeCode) {
            this.eventScopeCode = eventScopeCode;
            return this;
        }

        @Override
        public final Builder eventScopeCode(EventScopeCode eventScopeCode) {
            this.eventScopeCode(eventScopeCode == null ? null : eventScopeCode.toString());
            return this;
        }

        public final String getRegion() {
            return region;
        }

        public final void setRegion(String region) {
            this.region = region;
        }

        @Override
        public final Builder region(String region) {
            this.region = region;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final Instant getLastUpdatedTime() {
            return lastUpdatedTime;
        }

        public final void setLastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
        }

        @Override
        public final Builder lastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
            return this;
        }

        public final String getStatusCode() {
            return statusCodeValue;
        }

        public final void setStatusCode(String statusCodeValue) {
            this.statusCodeValue = statusCodeValue;
        }

        @Override
        public final Builder statusCode(String statusCodeValue) {
            this.statusCodeValue = statusCodeValue;
            return this;
        }

        @Override
        public final Builder statusCode(EventStatusCode statusCodeValue) {
            this.statusCode(statusCodeValue == null ? null : statusCodeValue.toString());
            return this;
        }

        public final String getActionability() {
            return actionability;
        }

        public final void setActionability(String actionability) {
            this.actionability = actionability;
        }

        @Override
        public final Builder actionability(String actionability) {
            this.actionability = actionability;
            return this;
        }

        @Override
        public final Builder actionability(EventActionability actionability) {
            this.actionability(actionability == null ? null : actionability.toString());
            return this;
        }

        public final Collection<String> getPersonas() {
            if (personas instanceof SdkAutoConstructList) {
                return null;
            }
            return personas;
        }

        public final void setPersonas(Collection<String> personas) {
            this.personas = EventPersonaListCopier.copy(personas);
        }

        @Override
        public final Builder personasWithStrings(Collection<String> personas) {
            this.personas = EventPersonaListCopier.copy(personas);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder personasWithStrings(String... personas) {
            personasWithStrings(Arrays.asList(personas));
            return this;
        }

        @Override
        public final Builder personas(Collection<EventPersona> personas) {
            this.personas = EventPersonaListCopier.copyEnumToString(personas);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder personas(EventPersona... personas) {
            personas(Arrays.asList(personas));
            return this;
        }

        @Override
        public OrganizationEvent build() {
            return new OrganizationEvent(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
