/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.health.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the metadata about a type of event that is reported by Health. The <code>EventType</code> shows the
 * category, service, and the event type code of the event. For example, an <code>issue</code> might be the category,
 * <code>EC2</code> the service, and <code>AWS_EC2_SYSTEM_MAINTENANCE_EVENT</code> the event type code.
 * </p>
 * <p>
 * You can use the <a
 * href="https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventTypes.html">DescribeEventTypes</a> API
 * operation to return this information about an event.
 * </p>
 * <p>
 * You can also use the Amazon CloudWatch Events console to create a rule so that you can get notified or take action
 * when Health delivers a specific event to your Amazon Web Services account. For more information, see <a
 * href="https://docs.aws.amazon.com/health/latest/ug/cloudwatch-events-health.html">Monitor for Health events with
 * Amazon CloudWatch Events</a> in the <i>Health User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EventType implements SdkPojo, Serializable, ToCopyableBuilder<EventType.Builder, EventType> {
    private static final SdkField<String> SERVICE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("service")
            .getter(getter(EventType::service)).setter(setter(Builder::service))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("service").build()).build();

    private static final SdkField<String> CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("code")
            .getter(getter(EventType::code)).setter(setter(Builder::code))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("code").build()).build();

    private static final SdkField<String> CATEGORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("category").getter(getter(EventType::categoryAsString)).setter(setter(Builder::category))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("category").build()).build();

    private static final SdkField<String> ACTIONABILITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("actionability").getter(getter(EventType::actionabilityAsString)).setter(setter(Builder::actionability))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("actionability").build()).build();

    private static final SdkField<List<String>> PERSONAS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("personas")
            .getter(getter(EventType::personasAsStrings))
            .setter(setter(Builder::personasWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("personas").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SERVICE_FIELD, CODE_FIELD,
            CATEGORY_FIELD, ACTIONABILITY_FIELD, PERSONAS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String service;

    private final String code;

    private final String category;

    private final String actionability;

    private final List<String> personas;

    private EventType(BuilderImpl builder) {
        this.service = builder.service;
        this.code = builder.code;
        this.category = builder.category;
        this.actionability = builder.actionability;
        this.personas = builder.personas;
    }

    /**
     * <p>
     * The Amazon Web Services service that is affected by the event. For example, <code>EC2</code>, <code>RDS</code>.
     * </p>
     * 
     * @return The Amazon Web Services service that is affected by the event. For example, <code>EC2</code>,
     *         <code>RDS</code>.
     */
    public final String service() {
        return service;
    }

    /**
     * <p>
     * The unique identifier for the event type. The format is <code>AWS_<i>SERVICE</i>_<i>DESCRIPTION</i> </code>; for
     * example, <code>AWS_EC2_SYSTEM_MAINTENANCE_EVENT</code>.
     * </p>
     * 
     * @return The unique identifier for the event type. The format is
     *         <code>AWS_<i>SERVICE</i>_<i>DESCRIPTION</i> </code>; for example,
     *         <code>AWS_EC2_SYSTEM_MAINTENANCE_EVENT</code>.
     */
    public final String code() {
        return code;
    }

    /**
     * <p>
     * A list of event type category codes. Possible values are <code>issue</code>, <code>accountNotification</code>, or
     * <code>scheduledChange</code>. Currently, the <code>investigation</code> value isn't supported at this time.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #category} will
     * return {@link EventTypeCategory#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #categoryAsString}.
     * </p>
     * 
     * @return A list of event type category codes. Possible values are <code>issue</code>,
     *         <code>accountNotification</code>, or <code>scheduledChange</code>. Currently, the
     *         <code>investigation</code> value isn't supported at this time.
     * @see EventTypeCategory
     */
    public final EventTypeCategory category() {
        return EventTypeCategory.fromValue(category);
    }

    /**
     * <p>
     * A list of event type category codes. Possible values are <code>issue</code>, <code>accountNotification</code>, or
     * <code>scheduledChange</code>. Currently, the <code>investigation</code> value isn't supported at this time.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #category} will
     * return {@link EventTypeCategory#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #categoryAsString}.
     * </p>
     * 
     * @return A list of event type category codes. Possible values are <code>issue</code>,
     *         <code>accountNotification</code>, or <code>scheduledChange</code>. Currently, the
     *         <code>investigation</code> value isn't supported at this time.
     * @see EventTypeCategory
     */
    public final String categoryAsString() {
        return category;
    }

    /**
     * <p>
     * The actionability classification of the event. Possible values are <code>ACTION_REQUIRED</code>,
     * <code>ACTION_MAY_BE_REQUIRED</code> and <code>INFORMATIONAL</code>. Events with <code>ACTION_REQUIRED</code>
     * actionability require customer action to resolve or mitigate the event. Events with
     * <code>ACTION_MAY_BE_REQUIRED</code> actionability indicates that the current status is unknown or conditional and
     * inspection is needed to determine if action is required. Events with <code>INFORMATIONAL</code> actionability are
     * provided for awareness and do not require immediate action.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #actionability}
     * will return {@link EventTypeActionability#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #actionabilityAsString}.
     * </p>
     * 
     * @return The actionability classification of the event. Possible values are <code>ACTION_REQUIRED</code>,
     *         <code>ACTION_MAY_BE_REQUIRED</code> and <code>INFORMATIONAL</code>. Events with
     *         <code>ACTION_REQUIRED</code> actionability require customer action to resolve or mitigate the event.
     *         Events with <code>ACTION_MAY_BE_REQUIRED</code> actionability indicates that the current status is
     *         unknown or conditional and inspection is needed to determine if action is required. Events with
     *         <code>INFORMATIONAL</code> actionability are provided for awareness and do not require immediate action.
     * @see EventTypeActionability
     */
    public final EventTypeActionability actionability() {
        return EventTypeActionability.fromValue(actionability);
    }

    /**
     * <p>
     * The actionability classification of the event. Possible values are <code>ACTION_REQUIRED</code>,
     * <code>ACTION_MAY_BE_REQUIRED</code> and <code>INFORMATIONAL</code>. Events with <code>ACTION_REQUIRED</code>
     * actionability require customer action to resolve or mitigate the event. Events with
     * <code>ACTION_MAY_BE_REQUIRED</code> actionability indicates that the current status is unknown or conditional and
     * inspection is needed to determine if action is required. Events with <code>INFORMATIONAL</code> actionability are
     * provided for awareness and do not require immediate action.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #actionability}
     * will return {@link EventTypeActionability#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #actionabilityAsString}.
     * </p>
     * 
     * @return The actionability classification of the event. Possible values are <code>ACTION_REQUIRED</code>,
     *         <code>ACTION_MAY_BE_REQUIRED</code> and <code>INFORMATIONAL</code>. Events with
     *         <code>ACTION_REQUIRED</code> actionability require customer action to resolve or mitigate the event.
     *         Events with <code>ACTION_MAY_BE_REQUIRED</code> actionability indicates that the current status is
     *         unknown or conditional and inspection is needed to determine if action is required. Events with
     *         <code>INFORMATIONAL</code> actionability are provided for awareness and do not require immediate action.
     * @see EventTypeActionability
     */
    public final String actionabilityAsString() {
        return actionability;
    }

    /**
     * <p>
     * A list of persona classifications that indicate the target audience for the event. Possible values are
     * <code>OPERATIONS</code>, <code>SECURITY</code>, and <code>BILLING</code>. Events can be associated with multiple
     * personas to indicate relevance to different teams or roles within an organization.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPersonas} method.
     * </p>
     * 
     * @return A list of persona classifications that indicate the target audience for the event. Possible values are
     *         <code>OPERATIONS</code>, <code>SECURITY</code>, and <code>BILLING</code>. Events can be associated with
     *         multiple personas to indicate relevance to different teams or roles within an organization.
     */
    public final List<EventTypePersona> personas() {
        return EventTypePersonaListCopier.copyStringToEnum(personas);
    }

    /**
     * For responses, this returns true if the service returned a value for the Personas property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasPersonas() {
        return personas != null && !(personas instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of persona classifications that indicate the target audience for the event. Possible values are
     * <code>OPERATIONS</code>, <code>SECURITY</code>, and <code>BILLING</code>. Events can be associated with multiple
     * personas to indicate relevance to different teams or roles within an organization.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPersonas} method.
     * </p>
     * 
     * @return A list of persona classifications that indicate the target audience for the event. Possible values are
     *         <code>OPERATIONS</code>, <code>SECURITY</code>, and <code>BILLING</code>. Events can be associated with
     *         multiple personas to indicate relevance to different teams or roles within an organization.
     */
    public final List<String> personasAsStrings() {
        return personas;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(service());
        hashCode = 31 * hashCode + Objects.hashCode(code());
        hashCode = 31 * hashCode + Objects.hashCode(categoryAsString());
        hashCode = 31 * hashCode + Objects.hashCode(actionabilityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasPersonas() ? personasAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EventType)) {
            return false;
        }
        EventType other = (EventType) obj;
        return Objects.equals(service(), other.service()) && Objects.equals(code(), other.code())
                && Objects.equals(categoryAsString(), other.categoryAsString())
                && Objects.equals(actionabilityAsString(), other.actionabilityAsString()) && hasPersonas() == other.hasPersonas()
                && Objects.equals(personasAsStrings(), other.personasAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EventType").add("Service", service()).add("Code", code()).add("Category", categoryAsString())
                .add("Actionability", actionabilityAsString()).add("Personas", hasPersonas() ? personasAsStrings() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "service":
            return Optional.ofNullable(clazz.cast(service()));
        case "code":
            return Optional.ofNullable(clazz.cast(code()));
        case "category":
            return Optional.ofNullable(clazz.cast(categoryAsString()));
        case "actionability":
            return Optional.ofNullable(clazz.cast(actionabilityAsString()));
        case "personas":
            return Optional.ofNullable(clazz.cast(personasAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("service", SERVICE_FIELD);
        map.put("code", CODE_FIELD);
        map.put("category", CATEGORY_FIELD);
        map.put("actionability", ACTIONABILITY_FIELD);
        map.put("personas", PERSONAS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<EventType, T> g) {
        return obj -> g.apply((EventType) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EventType> {
        /**
         * <p>
         * The Amazon Web Services service that is affected by the event. For example, <code>EC2</code>,
         * <code>RDS</code>.
         * </p>
         * 
         * @param service
         *        The Amazon Web Services service that is affected by the event. For example, <code>EC2</code>,
         *        <code>RDS</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder service(String service);

        /**
         * <p>
         * The unique identifier for the event type. The format is <code>AWS_<i>SERVICE</i>_<i>DESCRIPTION</i> </code>;
         * for example, <code>AWS_EC2_SYSTEM_MAINTENANCE_EVENT</code>.
         * </p>
         * 
         * @param code
         *        The unique identifier for the event type. The format is
         *        <code>AWS_<i>SERVICE</i>_<i>DESCRIPTION</i> </code>; for example,
         *        <code>AWS_EC2_SYSTEM_MAINTENANCE_EVENT</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder code(String code);

        /**
         * <p>
         * A list of event type category codes. Possible values are <code>issue</code>, <code>accountNotification</code>
         * , or <code>scheduledChange</code>. Currently, the <code>investigation</code> value isn't supported at this
         * time.
         * </p>
         * 
         * @param category
         *        A list of event type category codes. Possible values are <code>issue</code>,
         *        <code>accountNotification</code>, or <code>scheduledChange</code>. Currently, the
         *        <code>investigation</code> value isn't supported at this time.
         * @see EventTypeCategory
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventTypeCategory
         */
        Builder category(String category);

        /**
         * <p>
         * A list of event type category codes. Possible values are <code>issue</code>, <code>accountNotification</code>
         * , or <code>scheduledChange</code>. Currently, the <code>investigation</code> value isn't supported at this
         * time.
         * </p>
         * 
         * @param category
         *        A list of event type category codes. Possible values are <code>issue</code>,
         *        <code>accountNotification</code>, or <code>scheduledChange</code>. Currently, the
         *        <code>investigation</code> value isn't supported at this time.
         * @see EventTypeCategory
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventTypeCategory
         */
        Builder category(EventTypeCategory category);

        /**
         * <p>
         * The actionability classification of the event. Possible values are <code>ACTION_REQUIRED</code>,
         * <code>ACTION_MAY_BE_REQUIRED</code> and <code>INFORMATIONAL</code>. Events with <code>ACTION_REQUIRED</code>
         * actionability require customer action to resolve or mitigate the event. Events with
         * <code>ACTION_MAY_BE_REQUIRED</code> actionability indicates that the current status is unknown or conditional
         * and inspection is needed to determine if action is required. Events with <code>INFORMATIONAL</code>
         * actionability are provided for awareness and do not require immediate action.
         * </p>
         * 
         * @param actionability
         *        The actionability classification of the event. Possible values are <code>ACTION_REQUIRED</code>,
         *        <code>ACTION_MAY_BE_REQUIRED</code> and <code>INFORMATIONAL</code>. Events with
         *        <code>ACTION_REQUIRED</code> actionability require customer action to resolve or mitigate the event.
         *        Events with <code>ACTION_MAY_BE_REQUIRED</code> actionability indicates that the current status is
         *        unknown or conditional and inspection is needed to determine if action is required. Events with
         *        <code>INFORMATIONAL</code> actionability are provided for awareness and do not require immediate
         *        action.
         * @see EventTypeActionability
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventTypeActionability
         */
        Builder actionability(String actionability);

        /**
         * <p>
         * The actionability classification of the event. Possible values are <code>ACTION_REQUIRED</code>,
         * <code>ACTION_MAY_BE_REQUIRED</code> and <code>INFORMATIONAL</code>. Events with <code>ACTION_REQUIRED</code>
         * actionability require customer action to resolve or mitigate the event. Events with
         * <code>ACTION_MAY_BE_REQUIRED</code> actionability indicates that the current status is unknown or conditional
         * and inspection is needed to determine if action is required. Events with <code>INFORMATIONAL</code>
         * actionability are provided for awareness and do not require immediate action.
         * </p>
         * 
         * @param actionability
         *        The actionability classification of the event. Possible values are <code>ACTION_REQUIRED</code>,
         *        <code>ACTION_MAY_BE_REQUIRED</code> and <code>INFORMATIONAL</code>. Events with
         *        <code>ACTION_REQUIRED</code> actionability require customer action to resolve or mitigate the event.
         *        Events with <code>ACTION_MAY_BE_REQUIRED</code> actionability indicates that the current status is
         *        unknown or conditional and inspection is needed to determine if action is required. Events with
         *        <code>INFORMATIONAL</code> actionability are provided for awareness and do not require immediate
         *        action.
         * @see EventTypeActionability
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventTypeActionability
         */
        Builder actionability(EventTypeActionability actionability);

        /**
         * <p>
         * A list of persona classifications that indicate the target audience for the event. Possible values are
         * <code>OPERATIONS</code>, <code>SECURITY</code>, and <code>BILLING</code>. Events can be associated with
         * multiple personas to indicate relevance to different teams or roles within an organization.
         * </p>
         * 
         * @param personas
         *        A list of persona classifications that indicate the target audience for the event. Possible values are
         *        <code>OPERATIONS</code>, <code>SECURITY</code>, and <code>BILLING</code>. Events can be associated
         *        with multiple personas to indicate relevance to different teams or roles within an organization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder personasWithStrings(Collection<String> personas);

        /**
         * <p>
         * A list of persona classifications that indicate the target audience for the event. Possible values are
         * <code>OPERATIONS</code>, <code>SECURITY</code>, and <code>BILLING</code>. Events can be associated with
         * multiple personas to indicate relevance to different teams or roles within an organization.
         * </p>
         * 
         * @param personas
         *        A list of persona classifications that indicate the target audience for the event. Possible values are
         *        <code>OPERATIONS</code>, <code>SECURITY</code>, and <code>BILLING</code>. Events can be associated
         *        with multiple personas to indicate relevance to different teams or roles within an organization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder personasWithStrings(String... personas);

        /**
         * <p>
         * A list of persona classifications that indicate the target audience for the event. Possible values are
         * <code>OPERATIONS</code>, <code>SECURITY</code>, and <code>BILLING</code>. Events can be associated with
         * multiple personas to indicate relevance to different teams or roles within an organization.
         * </p>
         * 
         * @param personas
         *        A list of persona classifications that indicate the target audience for the event. Possible values are
         *        <code>OPERATIONS</code>, <code>SECURITY</code>, and <code>BILLING</code>. Events can be associated
         *        with multiple personas to indicate relevance to different teams or roles within an organization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder personas(Collection<EventTypePersona> personas);

        /**
         * <p>
         * A list of persona classifications that indicate the target audience for the event. Possible values are
         * <code>OPERATIONS</code>, <code>SECURITY</code>, and <code>BILLING</code>. Events can be associated with
         * multiple personas to indicate relevance to different teams or roles within an organization.
         * </p>
         * 
         * @param personas
         *        A list of persona classifications that indicate the target audience for the event. Possible values are
         *        <code>OPERATIONS</code>, <code>SECURITY</code>, and <code>BILLING</code>. Events can be associated
         *        with multiple personas to indicate relevance to different teams or roles within an organization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder personas(EventTypePersona... personas);
    }

    static final class BuilderImpl implements Builder {
        private String service;

        private String code;

        private String category;

        private String actionability;

        private List<String> personas = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(EventType model) {
            service(model.service);
            code(model.code);
            category(model.category);
            actionability(model.actionability);
            personasWithStrings(model.personas);
        }

        public final String getService() {
            return service;
        }

        public final void setService(String service) {
            this.service = service;
        }

        @Override
        public final Builder service(String service) {
            this.service = service;
            return this;
        }

        public final String getCode() {
            return code;
        }

        public final void setCode(String code) {
            this.code = code;
        }

        @Override
        public final Builder code(String code) {
            this.code = code;
            return this;
        }

        public final String getCategory() {
            return category;
        }

        public final void setCategory(String category) {
            this.category = category;
        }

        @Override
        public final Builder category(String category) {
            this.category = category;
            return this;
        }

        @Override
        public final Builder category(EventTypeCategory category) {
            this.category(category == null ? null : category.toString());
            return this;
        }

        public final String getActionability() {
            return actionability;
        }

        public final void setActionability(String actionability) {
            this.actionability = actionability;
        }

        @Override
        public final Builder actionability(String actionability) {
            this.actionability = actionability;
            return this;
        }

        @Override
        public final Builder actionability(EventTypeActionability actionability) {
            this.actionability(actionability == null ? null : actionability.toString());
            return this;
        }

        public final Collection<String> getPersonas() {
            if (personas instanceof SdkAutoConstructList) {
                return null;
            }
            return personas;
        }

        public final void setPersonas(Collection<String> personas) {
            this.personas = EventTypePersonaListCopier.copy(personas);
        }

        @Override
        public final Builder personasWithStrings(Collection<String> personas) {
            this.personas = EventTypePersonaListCopier.copy(personas);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder personasWithStrings(String... personas) {
            personasWithStrings(Arrays.asList(personas));
            return this;
        }

        @Override
        public final Builder personas(Collection<EventTypePersona> personas) {
            this.personas = EventTypePersonaListCopier.copyEnumToString(personas);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder personas(EventTypePersona... personas) {
            personas(Arrays.asList(personas));
            return this;
        }

        @Override
        public EventType build() {
            return new EventType(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
