/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.greengrassv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about a component that is a candidate to deploy to a Greengrass core device.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ComponentCandidate implements SdkPojo, Serializable,
        ToCopyableBuilder<ComponentCandidate.Builder, ComponentCandidate> {
    private static final SdkField<String> COMPONENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("componentName").getter(getter(ComponentCandidate::componentName)).setter(setter(Builder::componentName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("componentName").build()).build();

    private static final SdkField<String> COMPONENT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("componentVersion").getter(getter(ComponentCandidate::componentVersion))
            .setter(setter(Builder::componentVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("componentVersion").build()).build();

    private static final SdkField<Map<String, String>> VERSION_REQUIREMENTS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("versionRequirements")
            .getter(getter(ComponentCandidate::versionRequirements))
            .setter(setter(Builder::versionRequirements))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("versionRequirements").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COMPONENT_NAME_FIELD,
            COMPONENT_VERSION_FIELD, VERSION_REQUIREMENTS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String componentName;

    private final String componentVersion;

    private final Map<String, String> versionRequirements;

    private ComponentCandidate(BuilderImpl builder) {
        this.componentName = builder.componentName;
        this.componentVersion = builder.componentVersion;
        this.versionRequirements = builder.versionRequirements;
    }

    /**
     * <p>
     * The name of the component.
     * </p>
     * 
     * @return The name of the component.
     */
    public final String componentName() {
        return componentName;
    }

    /**
     * <p>
     * The version of the component.
     * </p>
     * 
     * @return The version of the component.
     */
    public final String componentVersion() {
        return componentVersion;
    }

    /**
     * For responses, this returns true if the service returned a value for the VersionRequirements property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasVersionRequirements() {
        return versionRequirements != null && !(versionRequirements instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The version requirements for the component's dependencies. Greengrass core devices get the version requirements
     * from component recipes.
     * </p>
     * <p>
     * IoT Greengrass V2 uses semantic version constraints. For more information, see <a
     * href="https://semver.org/">Semantic Versioning</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVersionRequirements} method.
     * </p>
     * 
     * @return The version requirements for the component's dependencies. Greengrass core devices get the version
     *         requirements from component recipes.</p>
     *         <p>
     *         IoT Greengrass V2 uses semantic version constraints. For more information, see <a
     *         href="https://semver.org/">Semantic Versioning</a>.
     */
    public final Map<String, String> versionRequirements() {
        return versionRequirements;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(componentName());
        hashCode = 31 * hashCode + Objects.hashCode(componentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(hasVersionRequirements() ? versionRequirements() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ComponentCandidate)) {
            return false;
        }
        ComponentCandidate other = (ComponentCandidate) obj;
        return Objects.equals(componentName(), other.componentName())
                && Objects.equals(componentVersion(), other.componentVersion())
                && hasVersionRequirements() == other.hasVersionRequirements()
                && Objects.equals(versionRequirements(), other.versionRequirements());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ComponentCandidate").add("ComponentName", componentName())
                .add("ComponentVersion", componentVersion())
                .add("VersionRequirements", hasVersionRequirements() ? versionRequirements() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "componentName":
            return Optional.ofNullable(clazz.cast(componentName()));
        case "componentVersion":
            return Optional.ofNullable(clazz.cast(componentVersion()));
        case "versionRequirements":
            return Optional.ofNullable(clazz.cast(versionRequirements()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("componentName", COMPONENT_NAME_FIELD);
        map.put("componentVersion", COMPONENT_VERSION_FIELD);
        map.put("versionRequirements", VERSION_REQUIREMENTS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ComponentCandidate, T> g) {
        return obj -> g.apply((ComponentCandidate) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ComponentCandidate> {
        /**
         * <p>
         * The name of the component.
         * </p>
         * 
         * @param componentName
         *        The name of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder componentName(String componentName);

        /**
         * <p>
         * The version of the component.
         * </p>
         * 
         * @param componentVersion
         *        The version of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder componentVersion(String componentVersion);

        /**
         * <p>
         * The version requirements for the component's dependencies. Greengrass core devices get the version
         * requirements from component recipes.
         * </p>
         * <p>
         * IoT Greengrass V2 uses semantic version constraints. For more information, see <a
         * href="https://semver.org/">Semantic Versioning</a>.
         * </p>
         * 
         * @param versionRequirements
         *        The version requirements for the component's dependencies. Greengrass core devices get the version
         *        requirements from component recipes.</p>
         *        <p>
         *        IoT Greengrass V2 uses semantic version constraints. For more information, see <a
         *        href="https://semver.org/">Semantic Versioning</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionRequirements(Map<String, String> versionRequirements);
    }

    static final class BuilderImpl implements Builder {
        private String componentName;

        private String componentVersion;

        private Map<String, String> versionRequirements = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ComponentCandidate model) {
            componentName(model.componentName);
            componentVersion(model.componentVersion);
            versionRequirements(model.versionRequirements);
        }

        public final String getComponentName() {
            return componentName;
        }

        public final void setComponentName(String componentName) {
            this.componentName = componentName;
        }

        @Override
        public final Builder componentName(String componentName) {
            this.componentName = componentName;
            return this;
        }

        public final String getComponentVersion() {
            return componentVersion;
        }

        public final void setComponentVersion(String componentVersion) {
            this.componentVersion = componentVersion;
        }

        @Override
        public final Builder componentVersion(String componentVersion) {
            this.componentVersion = componentVersion;
            return this;
        }

        public final Map<String, String> getVersionRequirements() {
            if (versionRequirements instanceof SdkAutoConstructMap) {
                return null;
            }
            return versionRequirements;
        }

        public final void setVersionRequirements(Map<String, String> versionRequirements) {
            this.versionRequirements = ComponentVersionRequirementMapCopier.copy(versionRequirements);
        }

        @Override
        public final Builder versionRequirements(Map<String, String> versionRequirements) {
            this.versionRequirements = ComponentVersionRequirementMapCopier.copy(versionRequirements);
            return this;
        }

        @Override
        public ComponentCandidate build() {
            return new ComponentCandidate(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
