/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a collection of related data organized in columns and rows.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Table implements SdkPojo, Serializable, ToCopyableBuilder<Table.Builder, Table> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Table::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DATABASE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DatabaseName").getter(getter(Table::databaseName)).setter(setter(Builder::databaseName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DatabaseName").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(Table::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> OWNER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Owner")
            .getter(getter(Table::owner)).setter(setter(Builder::owner))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Owner").build()).build();

    private static final SdkField<Instant> CREATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreateTime").getter(getter(Table::createTime)).setter(setter(Builder::createTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreateTime").build()).build();

    private static final SdkField<Instant> UPDATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("UpdateTime").getter(getter(Table::updateTime)).setter(setter(Builder::updateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdateTime").build()).build();

    private static final SdkField<Instant> LAST_ACCESS_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastAccessTime").getter(getter(Table::lastAccessTime)).setter(setter(Builder::lastAccessTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastAccessTime").build()).build();

    private static final SdkField<Instant> LAST_ANALYZED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastAnalyzedTime").getter(getter(Table::lastAnalyzedTime)).setter(setter(Builder::lastAnalyzedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastAnalyzedTime").build()).build();

    private static final SdkField<Integer> RETENTION_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Retention").getter(getter(Table::retention)).setter(setter(Builder::retention))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Retention").build()).build();

    private static final SdkField<StorageDescriptor> STORAGE_DESCRIPTOR_FIELD = SdkField
            .<StorageDescriptor> builder(MarshallingType.SDK_POJO).memberName("StorageDescriptor")
            .getter(getter(Table::storageDescriptor)).setter(setter(Builder::storageDescriptor))
            .constructor(StorageDescriptor::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageDescriptor").build()).build();

    private static final SdkField<List<Column>> PARTITION_KEYS_FIELD = SdkField
            .<List<Column>> builder(MarshallingType.LIST)
            .memberName("PartitionKeys")
            .getter(getter(Table::partitionKeys))
            .setter(setter(Builder::partitionKeys))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PartitionKeys").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Column> builder(MarshallingType.SDK_POJO)
                                            .constructor(Column::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> VIEW_ORIGINAL_TEXT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ViewOriginalText").getter(getter(Table::viewOriginalText)).setter(setter(Builder::viewOriginalText))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ViewOriginalText").build()).build();

    private static final SdkField<String> VIEW_EXPANDED_TEXT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ViewExpandedText").getter(getter(Table::viewExpandedText)).setter(setter(Builder::viewExpandedText))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ViewExpandedText").build()).build();

    private static final SdkField<String> TABLE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TableType").getter(getter(Table::tableType)).setter(setter(Builder::tableType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TableType").build()).build();

    private static final SdkField<Map<String, String>> PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Parameters")
            .getter(getter(Table::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> CREATED_BY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CreatedBy").getter(getter(Table::createdBy)).setter(setter(Builder::createdBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedBy").build()).build();

    private static final SdkField<Boolean> IS_REGISTERED_WITH_LAKE_FORMATION_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IsRegisteredWithLakeFormation")
            .getter(getter(Table::isRegisteredWithLakeFormation))
            .setter(setter(Builder::isRegisteredWithLakeFormation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsRegisteredWithLakeFormation")
                    .build()).build();

    private static final SdkField<TableIdentifier> TARGET_TABLE_FIELD = SdkField
            .<TableIdentifier> builder(MarshallingType.SDK_POJO).memberName("TargetTable").getter(getter(Table::targetTable))
            .setter(setter(Builder::targetTable)).constructor(TableIdentifier::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetTable").build()).build();

    private static final SdkField<String> CATALOG_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CatalogId").getter(getter(Table::catalogId)).setter(setter(Builder::catalogId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CatalogId").build()).build();

    private static final SdkField<String> VERSION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VersionId").getter(getter(Table::versionId)).setter(setter(Builder::versionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VersionId").build()).build();

    private static final SdkField<FederatedTable> FEDERATED_TABLE_FIELD = SdkField
            .<FederatedTable> builder(MarshallingType.SDK_POJO).memberName("FederatedTable")
            .getter(getter(Table::federatedTable)).setter(setter(Builder::federatedTable)).constructor(FederatedTable::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FederatedTable").build()).build();

    private static final SdkField<ViewDefinition> VIEW_DEFINITION_FIELD = SdkField
            .<ViewDefinition> builder(MarshallingType.SDK_POJO).memberName("ViewDefinition")
            .getter(getter(Table::viewDefinition)).setter(setter(Builder::viewDefinition)).constructor(ViewDefinition::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ViewDefinition").build()).build();

    private static final SdkField<Boolean> IS_MULTI_DIALECT_VIEW_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IsMultiDialectView").getter(getter(Table::isMultiDialectView))
            .setter(setter(Builder::isMultiDialectView))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsMultiDialectView").build())
            .build();

    private static final SdkField<TableStatus> STATUS_FIELD = SdkField.<TableStatus> builder(MarshallingType.SDK_POJO)
            .memberName("Status").getter(getter(Table::status)).setter(setter(Builder::status)).constructor(TableStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            DATABASE_NAME_FIELD, DESCRIPTION_FIELD, OWNER_FIELD, CREATE_TIME_FIELD, UPDATE_TIME_FIELD, LAST_ACCESS_TIME_FIELD,
            LAST_ANALYZED_TIME_FIELD, RETENTION_FIELD, STORAGE_DESCRIPTOR_FIELD, PARTITION_KEYS_FIELD, VIEW_ORIGINAL_TEXT_FIELD,
            VIEW_EXPANDED_TEXT_FIELD, TABLE_TYPE_FIELD, PARAMETERS_FIELD, CREATED_BY_FIELD,
            IS_REGISTERED_WITH_LAKE_FORMATION_FIELD, TARGET_TABLE_FIELD, CATALOG_ID_FIELD, VERSION_ID_FIELD,
            FEDERATED_TABLE_FIELD, VIEW_DEFINITION_FIELD, IS_MULTI_DIALECT_VIEW_FIELD, STATUS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String databaseName;

    private final String description;

    private final String owner;

    private final Instant createTime;

    private final Instant updateTime;

    private final Instant lastAccessTime;

    private final Instant lastAnalyzedTime;

    private final Integer retention;

    private final StorageDescriptor storageDescriptor;

    private final List<Column> partitionKeys;

    private final String viewOriginalText;

    private final String viewExpandedText;

    private final String tableType;

    private final Map<String, String> parameters;

    private final String createdBy;

    private final Boolean isRegisteredWithLakeFormation;

    private final TableIdentifier targetTable;

    private final String catalogId;

    private final String versionId;

    private final FederatedTable federatedTable;

    private final ViewDefinition viewDefinition;

    private final Boolean isMultiDialectView;

    private final TableStatus status;

    private Table(BuilderImpl builder) {
        this.name = builder.name;
        this.databaseName = builder.databaseName;
        this.description = builder.description;
        this.owner = builder.owner;
        this.createTime = builder.createTime;
        this.updateTime = builder.updateTime;
        this.lastAccessTime = builder.lastAccessTime;
        this.lastAnalyzedTime = builder.lastAnalyzedTime;
        this.retention = builder.retention;
        this.storageDescriptor = builder.storageDescriptor;
        this.partitionKeys = builder.partitionKeys;
        this.viewOriginalText = builder.viewOriginalText;
        this.viewExpandedText = builder.viewExpandedText;
        this.tableType = builder.tableType;
        this.parameters = builder.parameters;
        this.createdBy = builder.createdBy;
        this.isRegisteredWithLakeFormation = builder.isRegisteredWithLakeFormation;
        this.targetTable = builder.targetTable;
        this.catalogId = builder.catalogId;
        this.versionId = builder.versionId;
        this.federatedTable = builder.federatedTable;
        this.viewDefinition = builder.viewDefinition;
        this.isMultiDialectView = builder.isMultiDialectView;
        this.status = builder.status;
    }

    /**
     * <p>
     * The table name. For Hive compatibility, this must be entirely lowercase.
     * </p>
     * 
     * @return The table name. For Hive compatibility, this must be entirely lowercase.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The name of the database where the table metadata resides. For Hive compatibility, this must be all lowercase.
     * </p>
     * 
     * @return The name of the database where the table metadata resides. For Hive compatibility, this must be all
     *         lowercase.
     */
    public final String databaseName() {
        return databaseName;
    }

    /**
     * <p>
     * A description of the table.
     * </p>
     * 
     * @return A description of the table.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The owner of the table.
     * </p>
     * 
     * @return The owner of the table.
     */
    public final String owner() {
        return owner;
    }

    /**
     * <p>
     * The time when the table definition was created in the Data Catalog.
     * </p>
     * 
     * @return The time when the table definition was created in the Data Catalog.
     */
    public final Instant createTime() {
        return createTime;
    }

    /**
     * <p>
     * The last time that the table was updated.
     * </p>
     * 
     * @return The last time that the table was updated.
     */
    public final Instant updateTime() {
        return updateTime;
    }

    /**
     * <p>
     * The last time that the table was accessed. This is usually taken from HDFS, and might not be reliable.
     * </p>
     * 
     * @return The last time that the table was accessed. This is usually taken from HDFS, and might not be reliable.
     */
    public final Instant lastAccessTime() {
        return lastAccessTime;
    }

    /**
     * <p>
     * The last time that column statistics were computed for this table.
     * </p>
     * 
     * @return The last time that column statistics were computed for this table.
     */
    public final Instant lastAnalyzedTime() {
        return lastAnalyzedTime;
    }

    /**
     * <p>
     * The retention time for this table.
     * </p>
     * 
     * @return The retention time for this table.
     */
    public final Integer retention() {
        return retention;
    }

    /**
     * <p>
     * A storage descriptor containing information about the physical storage of this table.
     * </p>
     * 
     * @return A storage descriptor containing information about the physical storage of this table.
     */
    public final StorageDescriptor storageDescriptor() {
        return storageDescriptor;
    }

    /**
     * For responses, this returns true if the service returned a value for the PartitionKeys property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPartitionKeys() {
        return partitionKeys != null && !(partitionKeys instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of columns by which the table is partitioned. Only primitive types are supported as partition keys.
     * </p>
     * <p>
     * When you create a table used by Amazon Athena, and you do not specify any <code>partitionKeys</code>, you must at
     * least set the value of <code>partitionKeys</code> to an empty list. For example:
     * </p>
     * <p>
     * <code>"PartitionKeys": []</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPartitionKeys} method.
     * </p>
     * 
     * @return A list of columns by which the table is partitioned. Only primitive types are supported as partition
     *         keys.</p>
     *         <p>
     *         When you create a table used by Amazon Athena, and you do not specify any <code>partitionKeys</code>, you
     *         must at least set the value of <code>partitionKeys</code> to an empty list. For example:
     *         </p>
     *         <p>
     *         <code>"PartitionKeys": []</code>
     */
    public final List<Column> partitionKeys() {
        return partitionKeys;
    }

    /**
     * <p>
     * Included for Apache Hive compatibility. Not used in the normal course of Glue operations. If the table is a
     * <code>VIRTUAL_VIEW</code>, certain Athena configuration encoded in base64.
     * </p>
     * 
     * @return Included for Apache Hive compatibility. Not used in the normal course of Glue operations. If the table is
     *         a <code>VIRTUAL_VIEW</code>, certain Athena configuration encoded in base64.
     */
    public final String viewOriginalText() {
        return viewOriginalText;
    }

    /**
     * <p>
     * Included for Apache Hive compatibility. Not used in the normal course of Glue operations.
     * </p>
     * 
     * @return Included for Apache Hive compatibility. Not used in the normal course of Glue operations.
     */
    public final String viewExpandedText() {
        return viewExpandedText;
    }

    /**
     * <p>
     * The type of this table. Glue will create tables with the <code>EXTERNAL_TABLE</code> type. Other services, such
     * as Athena, may create tables with additional table types.
     * </p>
     * <p>
     * Glue related table types:
     * </p>
     * <dl>
     * <dt>EXTERNAL_TABLE</dt>
     * <dd>
     * <p>
     * Hive compatible attribute - indicates a non-Hive managed table.
     * </p>
     * </dd>
     * <dt>GOVERNED</dt>
     * <dd>
     * <p>
     * Used by Lake Formation. The Glue Data Catalog understands <code>GOVERNED</code>.
     * </p>
     * </dd>
     * </dl>
     * 
     * @return The type of this table. Glue will create tables with the <code>EXTERNAL_TABLE</code> type. Other
     *         services, such as Athena, may create tables with additional table types. </p>
     *         <p>
     *         Glue related table types:
     *         </p>
     *         <dl>
     *         <dt>EXTERNAL_TABLE</dt>
     *         <dd>
     *         <p>
     *         Hive compatible attribute - indicates a non-Hive managed table.
     *         </p>
     *         </dd>
     *         <dt>GOVERNED</dt>
     *         <dd>
     *         <p>
     *         Used by Lake Formation. The Glue Data Catalog understands <code>GOVERNED</code>.
     *         </p>
     *         </dd>
     */
    public final String tableType() {
        return tableType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Parameters property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * These key-value pairs define properties associated with the table.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParameters} method.
     * </p>
     * 
     * @return These key-value pairs define properties associated with the table.
     */
    public final Map<String, String> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The person or entity who created the table.
     * </p>
     * 
     * @return The person or entity who created the table.
     */
    public final String createdBy() {
        return createdBy;
    }

    /**
     * <p>
     * Indicates whether the table has been registered with Lake Formation.
     * </p>
     * 
     * @return Indicates whether the table has been registered with Lake Formation.
     */
    public final Boolean isRegisteredWithLakeFormation() {
        return isRegisteredWithLakeFormation;
    }

    /**
     * <p>
     * A <code>TableIdentifier</code> structure that describes a target table for resource linking.
     * </p>
     * 
     * @return A <code>TableIdentifier</code> structure that describes a target table for resource linking.
     */
    public final TableIdentifier targetTable() {
        return targetTable;
    }

    /**
     * <p>
     * The ID of the Data Catalog in which the table resides.
     * </p>
     * 
     * @return The ID of the Data Catalog in which the table resides.
     */
    public final String catalogId() {
        return catalogId;
    }

    /**
     * <p>
     * The ID of the table version.
     * </p>
     * 
     * @return The ID of the table version.
     */
    public final String versionId() {
        return versionId;
    }

    /**
     * <p>
     * A <code>FederatedTable</code> structure that references an entity outside the Glue Data Catalog.
     * </p>
     * 
     * @return A <code>FederatedTable</code> structure that references an entity outside the Glue Data Catalog.
     */
    public final FederatedTable federatedTable() {
        return federatedTable;
    }

    /**
     * <p>
     * A structure that contains all the information that defines the view, including the dialect or dialects for the
     * view, and the query.
     * </p>
     * 
     * @return A structure that contains all the information that defines the view, including the dialect or dialects
     *         for the view, and the query.
     */
    public final ViewDefinition viewDefinition() {
        return viewDefinition;
    }

    /**
     * <p>
     * Specifies whether the view supports the SQL dialects of one or more different query engines and can therefore be
     * read by those engines.
     * </p>
     * 
     * @return Specifies whether the view supports the SQL dialects of one or more different query engines and can
     *         therefore be read by those engines.
     */
    public final Boolean isMultiDialectView() {
        return isMultiDialectView;
    }

    /**
     * Returns the value of the Status property for this object.
     * 
     * @return The value of the Status property for this object.
     */
    public final TableStatus status() {
        return status;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(databaseName());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(owner());
        hashCode = 31 * hashCode + Objects.hashCode(createTime());
        hashCode = 31 * hashCode + Objects.hashCode(updateTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastAccessTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastAnalyzedTime());
        hashCode = 31 * hashCode + Objects.hashCode(retention());
        hashCode = 31 * hashCode + Objects.hashCode(storageDescriptor());
        hashCode = 31 * hashCode + Objects.hashCode(hasPartitionKeys() ? partitionKeys() : null);
        hashCode = 31 * hashCode + Objects.hashCode(viewOriginalText());
        hashCode = 31 * hashCode + Objects.hashCode(viewExpandedText());
        hashCode = 31 * hashCode + Objects.hashCode(tableType());
        hashCode = 31 * hashCode + Objects.hashCode(hasParameters() ? parameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(createdBy());
        hashCode = 31 * hashCode + Objects.hashCode(isRegisteredWithLakeFormation());
        hashCode = 31 * hashCode + Objects.hashCode(targetTable());
        hashCode = 31 * hashCode + Objects.hashCode(catalogId());
        hashCode = 31 * hashCode + Objects.hashCode(versionId());
        hashCode = 31 * hashCode + Objects.hashCode(federatedTable());
        hashCode = 31 * hashCode + Objects.hashCode(viewDefinition());
        hashCode = 31 * hashCode + Objects.hashCode(isMultiDialectView());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Table)) {
            return false;
        }
        Table other = (Table) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(databaseName(), other.databaseName())
                && Objects.equals(description(), other.description()) && Objects.equals(owner(), other.owner())
                && Objects.equals(createTime(), other.createTime()) && Objects.equals(updateTime(), other.updateTime())
                && Objects.equals(lastAccessTime(), other.lastAccessTime())
                && Objects.equals(lastAnalyzedTime(), other.lastAnalyzedTime()) && Objects.equals(retention(), other.retention())
                && Objects.equals(storageDescriptor(), other.storageDescriptor())
                && hasPartitionKeys() == other.hasPartitionKeys() && Objects.equals(partitionKeys(), other.partitionKeys())
                && Objects.equals(viewOriginalText(), other.viewOriginalText())
                && Objects.equals(viewExpandedText(), other.viewExpandedText()) && Objects.equals(tableType(), other.tableType())
                && hasParameters() == other.hasParameters() && Objects.equals(parameters(), other.parameters())
                && Objects.equals(createdBy(), other.createdBy())
                && Objects.equals(isRegisteredWithLakeFormation(), other.isRegisteredWithLakeFormation())
                && Objects.equals(targetTable(), other.targetTable()) && Objects.equals(catalogId(), other.catalogId())
                && Objects.equals(versionId(), other.versionId()) && Objects.equals(federatedTable(), other.federatedTable())
                && Objects.equals(viewDefinition(), other.viewDefinition())
                && Objects.equals(isMultiDialectView(), other.isMultiDialectView()) && Objects.equals(status(), other.status());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Table").add("Name", name()).add("DatabaseName", databaseName())
                .add("Description", description()).add("Owner", owner()).add("CreateTime", createTime())
                .add("UpdateTime", updateTime()).add("LastAccessTime", lastAccessTime())
                .add("LastAnalyzedTime", lastAnalyzedTime()).add("Retention", retention())
                .add("StorageDescriptor", storageDescriptor()).add("PartitionKeys", hasPartitionKeys() ? partitionKeys() : null)
                .add("ViewOriginalText", viewOriginalText()).add("ViewExpandedText", viewExpandedText())
                .add("TableType", tableType()).add("Parameters", hasParameters() ? parameters() : null)
                .add("CreatedBy", createdBy()).add("IsRegisteredWithLakeFormation", isRegisteredWithLakeFormation())
                .add("TargetTable", targetTable()).add("CatalogId", catalogId()).add("VersionId", versionId())
                .add("FederatedTable", federatedTable()).add("ViewDefinition", viewDefinition())
                .add("IsMultiDialectView", isMultiDialectView()).add("Status", status()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "DatabaseName":
            return Optional.ofNullable(clazz.cast(databaseName()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Owner":
            return Optional.ofNullable(clazz.cast(owner()));
        case "CreateTime":
            return Optional.ofNullable(clazz.cast(createTime()));
        case "UpdateTime":
            return Optional.ofNullable(clazz.cast(updateTime()));
        case "LastAccessTime":
            return Optional.ofNullable(clazz.cast(lastAccessTime()));
        case "LastAnalyzedTime":
            return Optional.ofNullable(clazz.cast(lastAnalyzedTime()));
        case "Retention":
            return Optional.ofNullable(clazz.cast(retention()));
        case "StorageDescriptor":
            return Optional.ofNullable(clazz.cast(storageDescriptor()));
        case "PartitionKeys":
            return Optional.ofNullable(clazz.cast(partitionKeys()));
        case "ViewOriginalText":
            return Optional.ofNullable(clazz.cast(viewOriginalText()));
        case "ViewExpandedText":
            return Optional.ofNullable(clazz.cast(viewExpandedText()));
        case "TableType":
            return Optional.ofNullable(clazz.cast(tableType()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "CreatedBy":
            return Optional.ofNullable(clazz.cast(createdBy()));
        case "IsRegisteredWithLakeFormation":
            return Optional.ofNullable(clazz.cast(isRegisteredWithLakeFormation()));
        case "TargetTable":
            return Optional.ofNullable(clazz.cast(targetTable()));
        case "CatalogId":
            return Optional.ofNullable(clazz.cast(catalogId()));
        case "VersionId":
            return Optional.ofNullable(clazz.cast(versionId()));
        case "FederatedTable":
            return Optional.ofNullable(clazz.cast(federatedTable()));
        case "ViewDefinition":
            return Optional.ofNullable(clazz.cast(viewDefinition()));
        case "IsMultiDialectView":
            return Optional.ofNullable(clazz.cast(isMultiDialectView()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("DatabaseName", DATABASE_NAME_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("Owner", OWNER_FIELD);
        map.put("CreateTime", CREATE_TIME_FIELD);
        map.put("UpdateTime", UPDATE_TIME_FIELD);
        map.put("LastAccessTime", LAST_ACCESS_TIME_FIELD);
        map.put("LastAnalyzedTime", LAST_ANALYZED_TIME_FIELD);
        map.put("Retention", RETENTION_FIELD);
        map.put("StorageDescriptor", STORAGE_DESCRIPTOR_FIELD);
        map.put("PartitionKeys", PARTITION_KEYS_FIELD);
        map.put("ViewOriginalText", VIEW_ORIGINAL_TEXT_FIELD);
        map.put("ViewExpandedText", VIEW_EXPANDED_TEXT_FIELD);
        map.put("TableType", TABLE_TYPE_FIELD);
        map.put("Parameters", PARAMETERS_FIELD);
        map.put("CreatedBy", CREATED_BY_FIELD);
        map.put("IsRegisteredWithLakeFormation", IS_REGISTERED_WITH_LAKE_FORMATION_FIELD);
        map.put("TargetTable", TARGET_TABLE_FIELD);
        map.put("CatalogId", CATALOG_ID_FIELD);
        map.put("VersionId", VERSION_ID_FIELD);
        map.put("FederatedTable", FEDERATED_TABLE_FIELD);
        map.put("ViewDefinition", VIEW_DEFINITION_FIELD);
        map.put("IsMultiDialectView", IS_MULTI_DIALECT_VIEW_FIELD);
        map.put("Status", STATUS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Table, T> g) {
        return obj -> g.apply((Table) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Table> {
        /**
         * <p>
         * The table name. For Hive compatibility, this must be entirely lowercase.
         * </p>
         * 
         * @param name
         *        The table name. For Hive compatibility, this must be entirely lowercase.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The name of the database where the table metadata resides. For Hive compatibility, this must be all
         * lowercase.
         * </p>
         * 
         * @param databaseName
         *        The name of the database where the table metadata resides. For Hive compatibility, this must be all
         *        lowercase.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseName(String databaseName);

        /**
         * <p>
         * A description of the table.
         * </p>
         * 
         * @param description
         *        A description of the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The owner of the table.
         * </p>
         * 
         * @param owner
         *        The owner of the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder owner(String owner);

        /**
         * <p>
         * The time when the table definition was created in the Data Catalog.
         * </p>
         * 
         * @param createTime
         *        The time when the table definition was created in the Data Catalog.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createTime(Instant createTime);

        /**
         * <p>
         * The last time that the table was updated.
         * </p>
         * 
         * @param updateTime
         *        The last time that the table was updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updateTime(Instant updateTime);

        /**
         * <p>
         * The last time that the table was accessed. This is usually taken from HDFS, and might not be reliable.
         * </p>
         * 
         * @param lastAccessTime
         *        The last time that the table was accessed. This is usually taken from HDFS, and might not be reliable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastAccessTime(Instant lastAccessTime);

        /**
         * <p>
         * The last time that column statistics were computed for this table.
         * </p>
         * 
         * @param lastAnalyzedTime
         *        The last time that column statistics were computed for this table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastAnalyzedTime(Instant lastAnalyzedTime);

        /**
         * <p>
         * The retention time for this table.
         * </p>
         * 
         * @param retention
         *        The retention time for this table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retention(Integer retention);

        /**
         * <p>
         * A storage descriptor containing information about the physical storage of this table.
         * </p>
         * 
         * @param storageDescriptor
         *        A storage descriptor containing information about the physical storage of this table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageDescriptor(StorageDescriptor storageDescriptor);

        /**
         * <p>
         * A storage descriptor containing information about the physical storage of this table.
         * </p>
         * This is a convenience method that creates an instance of the {@link StorageDescriptor.Builder} avoiding the
         * need to create one manually via {@link StorageDescriptor#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link StorageDescriptor.Builder#build()} is called immediately and its
         * result is passed to {@link #storageDescriptor(StorageDescriptor)}.
         * 
         * @param storageDescriptor
         *        a consumer that will call methods on {@link StorageDescriptor.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #storageDescriptor(StorageDescriptor)
         */
        default Builder storageDescriptor(Consumer<StorageDescriptor.Builder> storageDescriptor) {
            return storageDescriptor(StorageDescriptor.builder().applyMutation(storageDescriptor).build());
        }

        /**
         * <p>
         * A list of columns by which the table is partitioned. Only primitive types are supported as partition keys.
         * </p>
         * <p>
         * When you create a table used by Amazon Athena, and you do not specify any <code>partitionKeys</code>, you
         * must at least set the value of <code>partitionKeys</code> to an empty list. For example:
         * </p>
         * <p>
         * <code>"PartitionKeys": []</code>
         * </p>
         * 
         * @param partitionKeys
         *        A list of columns by which the table is partitioned. Only primitive types are supported as partition
         *        keys.</p>
         *        <p>
         *        When you create a table used by Amazon Athena, and you do not specify any <code>partitionKeys</code>,
         *        you must at least set the value of <code>partitionKeys</code> to an empty list. For example:
         *        </p>
         *        <p>
         *        <code>"PartitionKeys": []</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partitionKeys(Collection<Column> partitionKeys);

        /**
         * <p>
         * A list of columns by which the table is partitioned. Only primitive types are supported as partition keys.
         * </p>
         * <p>
         * When you create a table used by Amazon Athena, and you do not specify any <code>partitionKeys</code>, you
         * must at least set the value of <code>partitionKeys</code> to an empty list. For example:
         * </p>
         * <p>
         * <code>"PartitionKeys": []</code>
         * </p>
         * 
         * @param partitionKeys
         *        A list of columns by which the table is partitioned. Only primitive types are supported as partition
         *        keys.</p>
         *        <p>
         *        When you create a table used by Amazon Athena, and you do not specify any <code>partitionKeys</code>,
         *        you must at least set the value of <code>partitionKeys</code> to an empty list. For example:
         *        </p>
         *        <p>
         *        <code>"PartitionKeys": []</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partitionKeys(Column... partitionKeys);

        /**
         * <p>
         * A list of columns by which the table is partitioned. Only primitive types are supported as partition keys.
         * </p>
         * <p>
         * When you create a table used by Amazon Athena, and you do not specify any <code>partitionKeys</code>, you
         * must at least set the value of <code>partitionKeys</code> to an empty list. For example:
         * </p>
         * <p>
         * <code>"PartitionKeys": []</code>
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.glue.model.Column.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.glue.model.Column#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.glue.model.Column.Builder#build()} is called immediately and its
         * result is passed to {@link #partitionKeys(List<Column>)}.
         * 
         * @param partitionKeys
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.glue.model.Column.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #partitionKeys(java.util.Collection<Column>)
         */
        Builder partitionKeys(Consumer<Column.Builder>... partitionKeys);

        /**
         * <p>
         * Included for Apache Hive compatibility. Not used in the normal course of Glue operations. If the table is a
         * <code>VIRTUAL_VIEW</code>, certain Athena configuration encoded in base64.
         * </p>
         * 
         * @param viewOriginalText
         *        Included for Apache Hive compatibility. Not used in the normal course of Glue operations. If the table
         *        is a <code>VIRTUAL_VIEW</code>, certain Athena configuration encoded in base64.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder viewOriginalText(String viewOriginalText);

        /**
         * <p>
         * Included for Apache Hive compatibility. Not used in the normal course of Glue operations.
         * </p>
         * 
         * @param viewExpandedText
         *        Included for Apache Hive compatibility. Not used in the normal course of Glue operations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder viewExpandedText(String viewExpandedText);

        /**
         * <p>
         * The type of this table. Glue will create tables with the <code>EXTERNAL_TABLE</code> type. Other services,
         * such as Athena, may create tables with additional table types.
         * </p>
         * <p>
         * Glue related table types:
         * </p>
         * <dl>
         * <dt>EXTERNAL_TABLE</dt>
         * <dd>
         * <p>
         * Hive compatible attribute - indicates a non-Hive managed table.
         * </p>
         * </dd>
         * <dt>GOVERNED</dt>
         * <dd>
         * <p>
         * Used by Lake Formation. The Glue Data Catalog understands <code>GOVERNED</code>.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param tableType
         *        The type of this table. Glue will create tables with the <code>EXTERNAL_TABLE</code> type. Other
         *        services, such as Athena, may create tables with additional table types. </p>
         *        <p>
         *        Glue related table types:
         *        </p>
         *        <dl>
         *        <dt>EXTERNAL_TABLE</dt>
         *        <dd>
         *        <p>
         *        Hive compatible attribute - indicates a non-Hive managed table.
         *        </p>
         *        </dd>
         *        <dt>GOVERNED</dt>
         *        <dd>
         *        <p>
         *        Used by Lake Formation. The Glue Data Catalog understands <code>GOVERNED</code>.
         *        </p>
         *        </dd>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tableType(String tableType);

        /**
         * <p>
         * These key-value pairs define properties associated with the table.
         * </p>
         * 
         * @param parameters
         *        These key-value pairs define properties associated with the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Map<String, String> parameters);

        /**
         * <p>
         * The person or entity who created the table.
         * </p>
         * 
         * @param createdBy
         *        The person or entity who created the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdBy(String createdBy);

        /**
         * <p>
         * Indicates whether the table has been registered with Lake Formation.
         * </p>
         * 
         * @param isRegisteredWithLakeFormation
         *        Indicates whether the table has been registered with Lake Formation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isRegisteredWithLakeFormation(Boolean isRegisteredWithLakeFormation);

        /**
         * <p>
         * A <code>TableIdentifier</code> structure that describes a target table for resource linking.
         * </p>
         * 
         * @param targetTable
         *        A <code>TableIdentifier</code> structure that describes a target table for resource linking.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetTable(TableIdentifier targetTable);

        /**
         * <p>
         * A <code>TableIdentifier</code> structure that describes a target table for resource linking.
         * </p>
         * This is a convenience method that creates an instance of the {@link TableIdentifier.Builder} avoiding the
         * need to create one manually via {@link TableIdentifier#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TableIdentifier.Builder#build()} is called immediately and its
         * result is passed to {@link #targetTable(TableIdentifier)}.
         * 
         * @param targetTable
         *        a consumer that will call methods on {@link TableIdentifier.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targetTable(TableIdentifier)
         */
        default Builder targetTable(Consumer<TableIdentifier.Builder> targetTable) {
            return targetTable(TableIdentifier.builder().applyMutation(targetTable).build());
        }

        /**
         * <p>
         * The ID of the Data Catalog in which the table resides.
         * </p>
         * 
         * @param catalogId
         *        The ID of the Data Catalog in which the table resides.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder catalogId(String catalogId);

        /**
         * <p>
         * The ID of the table version.
         * </p>
         * 
         * @param versionId
         *        The ID of the table version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionId(String versionId);

        /**
         * <p>
         * A <code>FederatedTable</code> structure that references an entity outside the Glue Data Catalog.
         * </p>
         * 
         * @param federatedTable
         *        A <code>FederatedTable</code> structure that references an entity outside the Glue Data Catalog.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder federatedTable(FederatedTable federatedTable);

        /**
         * <p>
         * A <code>FederatedTable</code> structure that references an entity outside the Glue Data Catalog.
         * </p>
         * This is a convenience method that creates an instance of the {@link FederatedTable.Builder} avoiding the need
         * to create one manually via {@link FederatedTable#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FederatedTable.Builder#build()} is called immediately and its
         * result is passed to {@link #federatedTable(FederatedTable)}.
         * 
         * @param federatedTable
         *        a consumer that will call methods on {@link FederatedTable.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #federatedTable(FederatedTable)
         */
        default Builder federatedTable(Consumer<FederatedTable.Builder> federatedTable) {
            return federatedTable(FederatedTable.builder().applyMutation(federatedTable).build());
        }

        /**
         * <p>
         * A structure that contains all the information that defines the view, including the dialect or dialects for
         * the view, and the query.
         * </p>
         * 
         * @param viewDefinition
         *        A structure that contains all the information that defines the view, including the dialect or dialects
         *        for the view, and the query.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder viewDefinition(ViewDefinition viewDefinition);

        /**
         * <p>
         * A structure that contains all the information that defines the view, including the dialect or dialects for
         * the view, and the query.
         * </p>
         * This is a convenience method that creates an instance of the {@link ViewDefinition.Builder} avoiding the need
         * to create one manually via {@link ViewDefinition#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ViewDefinition.Builder#build()} is called immediately and its
         * result is passed to {@link #viewDefinition(ViewDefinition)}.
         * 
         * @param viewDefinition
         *        a consumer that will call methods on {@link ViewDefinition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #viewDefinition(ViewDefinition)
         */
        default Builder viewDefinition(Consumer<ViewDefinition.Builder> viewDefinition) {
            return viewDefinition(ViewDefinition.builder().applyMutation(viewDefinition).build());
        }

        /**
         * <p>
         * Specifies whether the view supports the SQL dialects of one or more different query engines and can therefore
         * be read by those engines.
         * </p>
         * 
         * @param isMultiDialectView
         *        Specifies whether the view supports the SQL dialects of one or more different query engines and can
         *        therefore be read by those engines.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isMultiDialectView(Boolean isMultiDialectView);

        /**
         * Sets the value of the Status property for this object.
         *
         * @param status
         *        The new value for the Status property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(TableStatus status);

        /**
         * Sets the value of the Status property for this object.
         *
         * This is a convenience method that creates an instance of the {@link TableStatus.Builder} avoiding the need to
         * create one manually via {@link TableStatus#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TableStatus.Builder#build()} is called immediately and its result
         * is passed to {@link #status(TableStatus)}.
         * 
         * @param status
         *        a consumer that will call methods on {@link TableStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #status(TableStatus)
         */
        default Builder status(Consumer<TableStatus.Builder> status) {
            return status(TableStatus.builder().applyMutation(status).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String databaseName;

        private String description;

        private String owner;

        private Instant createTime;

        private Instant updateTime;

        private Instant lastAccessTime;

        private Instant lastAnalyzedTime;

        private Integer retention;

        private StorageDescriptor storageDescriptor;

        private List<Column> partitionKeys = DefaultSdkAutoConstructList.getInstance();

        private String viewOriginalText;

        private String viewExpandedText;

        private String tableType;

        private Map<String, String> parameters = DefaultSdkAutoConstructMap.getInstance();

        private String createdBy;

        private Boolean isRegisteredWithLakeFormation;

        private TableIdentifier targetTable;

        private String catalogId;

        private String versionId;

        private FederatedTable federatedTable;

        private ViewDefinition viewDefinition;

        private Boolean isMultiDialectView;

        private TableStatus status;

        private BuilderImpl() {
        }

        private BuilderImpl(Table model) {
            name(model.name);
            databaseName(model.databaseName);
            description(model.description);
            owner(model.owner);
            createTime(model.createTime);
            updateTime(model.updateTime);
            lastAccessTime(model.lastAccessTime);
            lastAnalyzedTime(model.lastAnalyzedTime);
            retention(model.retention);
            storageDescriptor(model.storageDescriptor);
            partitionKeys(model.partitionKeys);
            viewOriginalText(model.viewOriginalText);
            viewExpandedText(model.viewExpandedText);
            tableType(model.tableType);
            parameters(model.parameters);
            createdBy(model.createdBy);
            isRegisteredWithLakeFormation(model.isRegisteredWithLakeFormation);
            targetTable(model.targetTable);
            catalogId(model.catalogId);
            versionId(model.versionId);
            federatedTable(model.federatedTable);
            viewDefinition(model.viewDefinition);
            isMultiDialectView(model.isMultiDialectView);
            status(model.status);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDatabaseName() {
            return databaseName;
        }

        public final void setDatabaseName(String databaseName) {
            this.databaseName = databaseName;
        }

        @Override
        public final Builder databaseName(String databaseName) {
            this.databaseName = databaseName;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getOwner() {
            return owner;
        }

        public final void setOwner(String owner) {
            this.owner = owner;
        }

        @Override
        public final Builder owner(String owner) {
            this.owner = owner;
            return this;
        }

        public final Instant getCreateTime() {
            return createTime;
        }

        public final void setCreateTime(Instant createTime) {
            this.createTime = createTime;
        }

        @Override
        public final Builder createTime(Instant createTime) {
            this.createTime = createTime;
            return this;
        }

        public final Instant getUpdateTime() {
            return updateTime;
        }

        public final void setUpdateTime(Instant updateTime) {
            this.updateTime = updateTime;
        }

        @Override
        public final Builder updateTime(Instant updateTime) {
            this.updateTime = updateTime;
            return this;
        }

        public final Instant getLastAccessTime() {
            return lastAccessTime;
        }

        public final void setLastAccessTime(Instant lastAccessTime) {
            this.lastAccessTime = lastAccessTime;
        }

        @Override
        public final Builder lastAccessTime(Instant lastAccessTime) {
            this.lastAccessTime = lastAccessTime;
            return this;
        }

        public final Instant getLastAnalyzedTime() {
            return lastAnalyzedTime;
        }

        public final void setLastAnalyzedTime(Instant lastAnalyzedTime) {
            this.lastAnalyzedTime = lastAnalyzedTime;
        }

        @Override
        public final Builder lastAnalyzedTime(Instant lastAnalyzedTime) {
            this.lastAnalyzedTime = lastAnalyzedTime;
            return this;
        }

        public final Integer getRetention() {
            return retention;
        }

        public final void setRetention(Integer retention) {
            this.retention = retention;
        }

        @Override
        public final Builder retention(Integer retention) {
            this.retention = retention;
            return this;
        }

        public final StorageDescriptor.Builder getStorageDescriptor() {
            return storageDescriptor != null ? storageDescriptor.toBuilder() : null;
        }

        public final void setStorageDescriptor(StorageDescriptor.BuilderImpl storageDescriptor) {
            this.storageDescriptor = storageDescriptor != null ? storageDescriptor.build() : null;
        }

        @Override
        public final Builder storageDescriptor(StorageDescriptor storageDescriptor) {
            this.storageDescriptor = storageDescriptor;
            return this;
        }

        public final List<Column.Builder> getPartitionKeys() {
            List<Column.Builder> result = ColumnListCopier.copyToBuilder(this.partitionKeys);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPartitionKeys(Collection<Column.BuilderImpl> partitionKeys) {
            this.partitionKeys = ColumnListCopier.copyFromBuilder(partitionKeys);
        }

        @Override
        public final Builder partitionKeys(Collection<Column> partitionKeys) {
            this.partitionKeys = ColumnListCopier.copy(partitionKeys);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder partitionKeys(Column... partitionKeys) {
            partitionKeys(Arrays.asList(partitionKeys));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder partitionKeys(Consumer<Column.Builder>... partitionKeys) {
            partitionKeys(Stream.of(partitionKeys).map(c -> Column.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getViewOriginalText() {
            return viewOriginalText;
        }

        public final void setViewOriginalText(String viewOriginalText) {
            this.viewOriginalText = viewOriginalText;
        }

        @Override
        public final Builder viewOriginalText(String viewOriginalText) {
            this.viewOriginalText = viewOriginalText;
            return this;
        }

        public final String getViewExpandedText() {
            return viewExpandedText;
        }

        public final void setViewExpandedText(String viewExpandedText) {
            this.viewExpandedText = viewExpandedText;
        }

        @Override
        public final Builder viewExpandedText(String viewExpandedText) {
            this.viewExpandedText = viewExpandedText;
            return this;
        }

        public final String getTableType() {
            return tableType;
        }

        public final void setTableType(String tableType) {
            this.tableType = tableType;
        }

        @Override
        public final Builder tableType(String tableType) {
            this.tableType = tableType;
            return this;
        }

        public final Map<String, String> getParameters() {
            if (parameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return parameters;
        }

        public final void setParameters(Map<String, String> parameters) {
            this.parameters = ParametersMapCopier.copy(parameters);
        }

        @Override
        public final Builder parameters(Map<String, String> parameters) {
            this.parameters = ParametersMapCopier.copy(parameters);
            return this;
        }

        public final String getCreatedBy() {
            return createdBy;
        }

        public final void setCreatedBy(String createdBy) {
            this.createdBy = createdBy;
        }

        @Override
        public final Builder createdBy(String createdBy) {
            this.createdBy = createdBy;
            return this;
        }

        public final Boolean getIsRegisteredWithLakeFormation() {
            return isRegisteredWithLakeFormation;
        }

        public final void setIsRegisteredWithLakeFormation(Boolean isRegisteredWithLakeFormation) {
            this.isRegisteredWithLakeFormation = isRegisteredWithLakeFormation;
        }

        @Override
        public final Builder isRegisteredWithLakeFormation(Boolean isRegisteredWithLakeFormation) {
            this.isRegisteredWithLakeFormation = isRegisteredWithLakeFormation;
            return this;
        }

        public final TableIdentifier.Builder getTargetTable() {
            return targetTable != null ? targetTable.toBuilder() : null;
        }

        public final void setTargetTable(TableIdentifier.BuilderImpl targetTable) {
            this.targetTable = targetTable != null ? targetTable.build() : null;
        }

        @Override
        public final Builder targetTable(TableIdentifier targetTable) {
            this.targetTable = targetTable;
            return this;
        }

        public final String getCatalogId() {
            return catalogId;
        }

        public final void setCatalogId(String catalogId) {
            this.catalogId = catalogId;
        }

        @Override
        public final Builder catalogId(String catalogId) {
            this.catalogId = catalogId;
            return this;
        }

        public final String getVersionId() {
            return versionId;
        }

        public final void setVersionId(String versionId) {
            this.versionId = versionId;
        }

        @Override
        public final Builder versionId(String versionId) {
            this.versionId = versionId;
            return this;
        }

        public final FederatedTable.Builder getFederatedTable() {
            return federatedTable != null ? federatedTable.toBuilder() : null;
        }

        public final void setFederatedTable(FederatedTable.BuilderImpl federatedTable) {
            this.federatedTable = federatedTable != null ? federatedTable.build() : null;
        }

        @Override
        public final Builder federatedTable(FederatedTable federatedTable) {
            this.federatedTable = federatedTable;
            return this;
        }

        public final ViewDefinition.Builder getViewDefinition() {
            return viewDefinition != null ? viewDefinition.toBuilder() : null;
        }

        public final void setViewDefinition(ViewDefinition.BuilderImpl viewDefinition) {
            this.viewDefinition = viewDefinition != null ? viewDefinition.build() : null;
        }

        @Override
        public final Builder viewDefinition(ViewDefinition viewDefinition) {
            this.viewDefinition = viewDefinition;
            return this;
        }

        public final Boolean getIsMultiDialectView() {
            return isMultiDialectView;
        }

        public final void setIsMultiDialectView(Boolean isMultiDialectView) {
            this.isMultiDialectView = isMultiDialectView;
        }

        @Override
        public final Builder isMultiDialectView(Boolean isMultiDialectView) {
            this.isMultiDialectView = isMultiDialectView;
            return this;
        }

        public final TableStatus.Builder getStatus() {
            return status != null ? status.toBuilder() : null;
        }

        public final void setStatus(TableStatus.BuilderImpl status) {
            this.status = status != null ? status.build() : null;
        }

        @Override
        public final Builder status(TableStatus status) {
            this.status = status;
            return this;
        }

        @Override
        public Table build() {
            return new Table(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
