/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a zero-ETL integration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Integration implements SdkPojo, Serializable, ToCopyableBuilder<Integration.Builder, Integration> {
    private static final SdkField<String> SOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceArn").getter(getter(Integration::sourceArn)).setter(setter(Builder::sourceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceArn").build()).build();

    private static final SdkField<String> TARGET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TargetArn").getter(getter(Integration::targetArn)).setter(setter(Builder::targetArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetArn").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(Integration::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> INTEGRATION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IntegrationName").getter(getter(Integration::integrationName)).setter(setter(Builder::integrationName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IntegrationName").build()).build();

    private static final SdkField<String> INTEGRATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IntegrationArn").getter(getter(Integration::integrationArn)).setter(setter(Builder::integrationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IntegrationArn").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyId").getter(getter(Integration::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final SdkField<Map<String, String>> ADDITIONAL_ENCRYPTION_CONTEXT_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("AdditionalEncryptionContext")
            .getter(getter(Integration::additionalEncryptionContext))
            .setter(setter(Builder::additionalEncryptionContext))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdditionalEncryptionContext")
                    .build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(Integration::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(Integration::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Instant> CREATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreateTime").getter(getter(Integration::createTime)).setter(setter(Builder::createTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreateTime").build()).build();

    private static final SdkField<IntegrationConfig> INTEGRATION_CONFIG_FIELD = SdkField
            .<IntegrationConfig> builder(MarshallingType.SDK_POJO).memberName("IntegrationConfig")
            .getter(getter(Integration::integrationConfig)).setter(setter(Builder::integrationConfig))
            .constructor(IntegrationConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IntegrationConfig").build()).build();

    private static final SdkField<List<IntegrationError>> ERRORS_FIELD = SdkField
            .<List<IntegrationError>> builder(MarshallingType.LIST)
            .memberName("Errors")
            .getter(getter(Integration::errors))
            .setter(setter(Builder::errors))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Errors").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<IntegrationError> builder(MarshallingType.SDK_POJO)
                                            .constructor(IntegrationError::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DATA_FILTER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataFilter").getter(getter(Integration::dataFilter)).setter(setter(Builder::dataFilter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataFilter").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SOURCE_ARN_FIELD,
            TARGET_ARN_FIELD, DESCRIPTION_FIELD, INTEGRATION_NAME_FIELD, INTEGRATION_ARN_FIELD, KMS_KEY_ID_FIELD,
            ADDITIONAL_ENCRYPTION_CONTEXT_FIELD, TAGS_FIELD, STATUS_FIELD, CREATE_TIME_FIELD, INTEGRATION_CONFIG_FIELD,
            ERRORS_FIELD, DATA_FILTER_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String sourceArn;

    private final String targetArn;

    private final String description;

    private final String integrationName;

    private final String integrationArn;

    private final String kmsKeyId;

    private final Map<String, String> additionalEncryptionContext;

    private final List<Tag> tags;

    private final String status;

    private final Instant createTime;

    private final IntegrationConfig integrationConfig;

    private final List<IntegrationError> errors;

    private final String dataFilter;

    private Integration(BuilderImpl builder) {
        this.sourceArn = builder.sourceArn;
        this.targetArn = builder.targetArn;
        this.description = builder.description;
        this.integrationName = builder.integrationName;
        this.integrationArn = builder.integrationArn;
        this.kmsKeyId = builder.kmsKeyId;
        this.additionalEncryptionContext = builder.additionalEncryptionContext;
        this.tags = builder.tags;
        this.status = builder.status;
        this.createTime = builder.createTime;
        this.integrationConfig = builder.integrationConfig;
        this.errors = builder.errors;
        this.dataFilter = builder.dataFilter;
    }

    /**
     * <p>
     * The ARN for the source of the integration.
     * </p>
     * 
     * @return The ARN for the source of the integration.
     */
    public final String sourceArn() {
        return sourceArn;
    }

    /**
     * <p>
     * The ARN for the target of the integration.
     * </p>
     * 
     * @return The ARN for the target of the integration.
     */
    public final String targetArn() {
        return targetArn;
    }

    /**
     * <p>
     * A description for the integration.
     * </p>
     * 
     * @return A description for the integration.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * A unique name for the integration.
     * </p>
     * 
     * @return A unique name for the integration.
     */
    public final String integrationName() {
        return integrationName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the integration.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the integration.
     */
    public final String integrationArn() {
        return integrationArn;
    }

    /**
     * <p>
     * The ARN of a KMS key used for encrypting the channel.
     * </p>
     * 
     * @return The ARN of a KMS key used for encrypting the channel.
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * For responses, this returns true if the service returned a value for the AdditionalEncryptionContext property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasAdditionalEncryptionContext() {
        return additionalEncryptionContext != null && !(additionalEncryptionContext instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * An optional set of non-secret key–value pairs that contains additional contextual information for encryption.
     * This can only be provided if <code>KMSKeyId</code> is provided.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAdditionalEncryptionContext} method.
     * </p>
     * 
     * @return An optional set of non-secret key–value pairs that contains additional contextual information for
     *         encryption. This can only be provided if <code>KMSKeyId</code> is provided.
     */
    public final Map<String, String> additionalEncryptionContext() {
        return additionalEncryptionContext;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Metadata assigned to the resource consisting of a list of key-value pairs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Metadata assigned to the resource consisting of a list of key-value pairs.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The possible statuses are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * CREATING: The integration is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * ACTIVE: The integration creation succeeds.
     * </p>
     * </li>
     * <li>
     * <p>
     * MODIFYING: The integration is being modified.
     * </p>
     * </li>
     * <li>
     * <p>
     * FAILED: The integration creation fails.
     * </p>
     * </li>
     * <li>
     * <p>
     * DELETING: The integration is deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * SYNCING: The integration is synchronizing.
     * </p>
     * </li>
     * <li>
     * <p>
     * NEEDS_ATTENTION: The integration needs attention, such as synchronization.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link IntegrationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The possible statuses are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         CREATING: The integration is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ACTIVE: The integration creation succeeds.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         MODIFYING: The integration is being modified.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FAILED: The integration creation fails.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DELETING: The integration is deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         SYNCING: The integration is synchronizing.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         NEEDS_ATTENTION: The integration needs attention, such as synchronization.
     *         </p>
     *         </li>
     * @see IntegrationStatus
     */
    public final IntegrationStatus status() {
        return IntegrationStatus.fromValue(status);
    }

    /**
     * <p>
     * The possible statuses are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * CREATING: The integration is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * ACTIVE: The integration creation succeeds.
     * </p>
     * </li>
     * <li>
     * <p>
     * MODIFYING: The integration is being modified.
     * </p>
     * </li>
     * <li>
     * <p>
     * FAILED: The integration creation fails.
     * </p>
     * </li>
     * <li>
     * <p>
     * DELETING: The integration is deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * SYNCING: The integration is synchronizing.
     * </p>
     * </li>
     * <li>
     * <p>
     * NEEDS_ATTENTION: The integration needs attention, such as synchronization.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link IntegrationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The possible statuses are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         CREATING: The integration is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ACTIVE: The integration creation succeeds.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         MODIFYING: The integration is being modified.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FAILED: The integration creation fails.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DELETING: The integration is deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         SYNCING: The integration is synchronizing.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         NEEDS_ATTENTION: The integration needs attention, such as synchronization.
     *         </p>
     *         </li>
     * @see IntegrationStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The time that the integration was created, in UTC.
     * </p>
     * 
     * @return The time that the integration was created, in UTC.
     */
    public final Instant createTime() {
        return createTime;
    }

    /**
     * <p>
     * Properties associated with the integration.
     * </p>
     * 
     * @return Properties associated with the integration.
     */
    public final IntegrationConfig integrationConfig() {
        return integrationConfig;
    }

    /**
     * For responses, this returns true if the service returned a value for the Errors property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasErrors() {
        return errors != null && !(errors instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of errors associated with the integration.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasErrors} method.
     * </p>
     * 
     * @return A list of errors associated with the integration.
     */
    public final List<IntegrationError> errors() {
        return errors;
    }

    /**
     * <p>
     * Selects source tables for the integration using Maxwell filter syntax.
     * </p>
     * 
     * @return Selects source tables for the integration using Maxwell filter syntax.
     */
    public final String dataFilter() {
        return dataFilter;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(sourceArn());
        hashCode = 31 * hashCode + Objects.hashCode(targetArn());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(integrationName());
        hashCode = 31 * hashCode + Objects.hashCode(integrationArn());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(hasAdditionalEncryptionContext() ? additionalEncryptionContext() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createTime());
        hashCode = 31 * hashCode + Objects.hashCode(integrationConfig());
        hashCode = 31 * hashCode + Objects.hashCode(hasErrors() ? errors() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dataFilter());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Integration)) {
            return false;
        }
        Integration other = (Integration) obj;
        return Objects.equals(sourceArn(), other.sourceArn()) && Objects.equals(targetArn(), other.targetArn())
                && Objects.equals(description(), other.description())
                && Objects.equals(integrationName(), other.integrationName())
                && Objects.equals(integrationArn(), other.integrationArn()) && Objects.equals(kmsKeyId(), other.kmsKeyId())
                && hasAdditionalEncryptionContext() == other.hasAdditionalEncryptionContext()
                && Objects.equals(additionalEncryptionContext(), other.additionalEncryptionContext())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(createTime(), other.createTime())
                && Objects.equals(integrationConfig(), other.integrationConfig()) && hasErrors() == other.hasErrors()
                && Objects.equals(errors(), other.errors()) && Objects.equals(dataFilter(), other.dataFilter());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Integration").add("SourceArn", sourceArn()).add("TargetArn", targetArn())
                .add("Description", description()).add("IntegrationName", integrationName())
                .add("IntegrationArn", integrationArn()).add("KmsKeyId", kmsKeyId())
                .add("AdditionalEncryptionContext", hasAdditionalEncryptionContext() ? additionalEncryptionContext() : null)
                .add("Tags", hasTags() ? tags() : null).add("Status", statusAsString()).add("CreateTime", createTime())
                .add("IntegrationConfig", integrationConfig()).add("Errors", hasErrors() ? errors() : null)
                .add("DataFilter", dataFilter()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SourceArn":
            return Optional.ofNullable(clazz.cast(sourceArn()));
        case "TargetArn":
            return Optional.ofNullable(clazz.cast(targetArn()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "IntegrationName":
            return Optional.ofNullable(clazz.cast(integrationName()));
        case "IntegrationArn":
            return Optional.ofNullable(clazz.cast(integrationArn()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "AdditionalEncryptionContext":
            return Optional.ofNullable(clazz.cast(additionalEncryptionContext()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "CreateTime":
            return Optional.ofNullable(clazz.cast(createTime()));
        case "IntegrationConfig":
            return Optional.ofNullable(clazz.cast(integrationConfig()));
        case "Errors":
            return Optional.ofNullable(clazz.cast(errors()));
        case "DataFilter":
            return Optional.ofNullable(clazz.cast(dataFilter()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("SourceArn", SOURCE_ARN_FIELD);
        map.put("TargetArn", TARGET_ARN_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("IntegrationName", INTEGRATION_NAME_FIELD);
        map.put("IntegrationArn", INTEGRATION_ARN_FIELD);
        map.put("KmsKeyId", KMS_KEY_ID_FIELD);
        map.put("AdditionalEncryptionContext", ADDITIONAL_ENCRYPTION_CONTEXT_FIELD);
        map.put("Tags", TAGS_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("CreateTime", CREATE_TIME_FIELD);
        map.put("IntegrationConfig", INTEGRATION_CONFIG_FIELD);
        map.put("Errors", ERRORS_FIELD);
        map.put("DataFilter", DATA_FILTER_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Integration, T> g) {
        return obj -> g.apply((Integration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Integration> {
        /**
         * <p>
         * The ARN for the source of the integration.
         * </p>
         * 
         * @param sourceArn
         *        The ARN for the source of the integration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceArn(String sourceArn);

        /**
         * <p>
         * The ARN for the target of the integration.
         * </p>
         * 
         * @param targetArn
         *        The ARN for the target of the integration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetArn(String targetArn);

        /**
         * <p>
         * A description for the integration.
         * </p>
         * 
         * @param description
         *        A description for the integration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A unique name for the integration.
         * </p>
         * 
         * @param integrationName
         *        A unique name for the integration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder integrationName(String integrationName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for the integration.
         * </p>
         * 
         * @param integrationArn
         *        The Amazon Resource Name (ARN) for the integration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder integrationArn(String integrationArn);

        /**
         * <p>
         * The ARN of a KMS key used for encrypting the channel.
         * </p>
         * 
         * @param kmsKeyId
         *        The ARN of a KMS key used for encrypting the channel.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * An optional set of non-secret key–value pairs that contains additional contextual information for encryption.
         * This can only be provided if <code>KMSKeyId</code> is provided.
         * </p>
         * 
         * @param additionalEncryptionContext
         *        An optional set of non-secret key–value pairs that contains additional contextual information for
         *        encryption. This can only be provided if <code>KMSKeyId</code> is provided.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalEncryptionContext(Map<String, String> additionalEncryptionContext);

        /**
         * <p>
         * Metadata assigned to the resource consisting of a list of key-value pairs.
         * </p>
         * 
         * @param tags
         *        Metadata assigned to the resource consisting of a list of key-value pairs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Metadata assigned to the resource consisting of a list of key-value pairs.
         * </p>
         * 
         * @param tags
         *        Metadata assigned to the resource consisting of a list of key-value pairs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Metadata assigned to the resource consisting of a list of key-value pairs.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.glue.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.glue.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.glue.model.Tag.Builder#build()}
         * is called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.glue.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The possible statuses are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * CREATING: The integration is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * ACTIVE: The integration creation succeeds.
         * </p>
         * </li>
         * <li>
         * <p>
         * MODIFYING: The integration is being modified.
         * </p>
         * </li>
         * <li>
         * <p>
         * FAILED: The integration creation fails.
         * </p>
         * </li>
         * <li>
         * <p>
         * DELETING: The integration is deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * SYNCING: The integration is synchronizing.
         * </p>
         * </li>
         * <li>
         * <p>
         * NEEDS_ATTENTION: The integration needs attention, such as synchronization.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The possible statuses are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        CREATING: The integration is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ACTIVE: The integration creation succeeds.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        MODIFYING: The integration is being modified.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FAILED: The integration creation fails.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DELETING: The integration is deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        SYNCING: The integration is synchronizing.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        NEEDS_ATTENTION: The integration needs attention, such as synchronization.
         *        </p>
         *        </li>
         * @see IntegrationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IntegrationStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The possible statuses are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * CREATING: The integration is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * ACTIVE: The integration creation succeeds.
         * </p>
         * </li>
         * <li>
         * <p>
         * MODIFYING: The integration is being modified.
         * </p>
         * </li>
         * <li>
         * <p>
         * FAILED: The integration creation fails.
         * </p>
         * </li>
         * <li>
         * <p>
         * DELETING: The integration is deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * SYNCING: The integration is synchronizing.
         * </p>
         * </li>
         * <li>
         * <p>
         * NEEDS_ATTENTION: The integration needs attention, such as synchronization.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The possible statuses are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        CREATING: The integration is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ACTIVE: The integration creation succeeds.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        MODIFYING: The integration is being modified.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FAILED: The integration creation fails.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DELETING: The integration is deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        SYNCING: The integration is synchronizing.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        NEEDS_ATTENTION: The integration needs attention, such as synchronization.
         *        </p>
         *        </li>
         * @see IntegrationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IntegrationStatus
         */
        Builder status(IntegrationStatus status);

        /**
         * <p>
         * The time that the integration was created, in UTC.
         * </p>
         * 
         * @param createTime
         *        The time that the integration was created, in UTC.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createTime(Instant createTime);

        /**
         * <p>
         * Properties associated with the integration.
         * </p>
         * 
         * @param integrationConfig
         *        Properties associated with the integration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder integrationConfig(IntegrationConfig integrationConfig);

        /**
         * <p>
         * Properties associated with the integration.
         * </p>
         * This is a convenience method that creates an instance of the {@link IntegrationConfig.Builder} avoiding the
         * need to create one manually via {@link IntegrationConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link IntegrationConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #integrationConfig(IntegrationConfig)}.
         * 
         * @param integrationConfig
         *        a consumer that will call methods on {@link IntegrationConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #integrationConfig(IntegrationConfig)
         */
        default Builder integrationConfig(Consumer<IntegrationConfig.Builder> integrationConfig) {
            return integrationConfig(IntegrationConfig.builder().applyMutation(integrationConfig).build());
        }

        /**
         * <p>
         * A list of errors associated with the integration.
         * </p>
         * 
         * @param errors
         *        A list of errors associated with the integration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errors(Collection<IntegrationError> errors);

        /**
         * <p>
         * A list of errors associated with the integration.
         * </p>
         * 
         * @param errors
         *        A list of errors associated with the integration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errors(IntegrationError... errors);

        /**
         * <p>
         * A list of errors associated with the integration.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.glue.model.IntegrationError.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.glue.model.IntegrationError#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.glue.model.IntegrationError.Builder#build()} is called immediately and
         * its result is passed to {@link #errors(List<IntegrationError>)}.
         * 
         * @param errors
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.glue.model.IntegrationError.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #errors(java.util.Collection<IntegrationError>)
         */
        Builder errors(Consumer<IntegrationError.Builder>... errors);

        /**
         * <p>
         * Selects source tables for the integration using Maxwell filter syntax.
         * </p>
         * 
         * @param dataFilter
         *        Selects source tables for the integration using Maxwell filter syntax.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataFilter(String dataFilter);
    }

    static final class BuilderImpl implements Builder {
        private String sourceArn;

        private String targetArn;

        private String description;

        private String integrationName;

        private String integrationArn;

        private String kmsKeyId;

        private Map<String, String> additionalEncryptionContext = DefaultSdkAutoConstructMap.getInstance();

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String status;

        private Instant createTime;

        private IntegrationConfig integrationConfig;

        private List<IntegrationError> errors = DefaultSdkAutoConstructList.getInstance();

        private String dataFilter;

        private BuilderImpl() {
        }

        private BuilderImpl(Integration model) {
            sourceArn(model.sourceArn);
            targetArn(model.targetArn);
            description(model.description);
            integrationName(model.integrationName);
            integrationArn(model.integrationArn);
            kmsKeyId(model.kmsKeyId);
            additionalEncryptionContext(model.additionalEncryptionContext);
            tags(model.tags);
            status(model.status);
            createTime(model.createTime);
            integrationConfig(model.integrationConfig);
            errors(model.errors);
            dataFilter(model.dataFilter);
        }

        public final String getSourceArn() {
            return sourceArn;
        }

        public final void setSourceArn(String sourceArn) {
            this.sourceArn = sourceArn;
        }

        @Override
        public final Builder sourceArn(String sourceArn) {
            this.sourceArn = sourceArn;
            return this;
        }

        public final String getTargetArn() {
            return targetArn;
        }

        public final void setTargetArn(String targetArn) {
            this.targetArn = targetArn;
        }

        @Override
        public final Builder targetArn(String targetArn) {
            this.targetArn = targetArn;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getIntegrationName() {
            return integrationName;
        }

        public final void setIntegrationName(String integrationName) {
            this.integrationName = integrationName;
        }

        @Override
        public final Builder integrationName(String integrationName) {
            this.integrationName = integrationName;
            return this;
        }

        public final String getIntegrationArn() {
            return integrationArn;
        }

        public final void setIntegrationArn(String integrationArn) {
            this.integrationArn = integrationArn;
        }

        @Override
        public final Builder integrationArn(String integrationArn) {
            this.integrationArn = integrationArn;
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final Map<String, String> getAdditionalEncryptionContext() {
            if (additionalEncryptionContext instanceof SdkAutoConstructMap) {
                return null;
            }
            return additionalEncryptionContext;
        }

        public final void setAdditionalEncryptionContext(Map<String, String> additionalEncryptionContext) {
            this.additionalEncryptionContext = IntegrationAdditionalEncryptionContextMapCopier.copy(additionalEncryptionContext);
        }

        @Override
        public final Builder additionalEncryptionContext(Map<String, String> additionalEncryptionContext) {
            this.additionalEncryptionContext = IntegrationAdditionalEncryptionContextMapCopier.copy(additionalEncryptionContext);
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = IntegrationTagsListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = IntegrationTagsListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = IntegrationTagsListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(IntegrationStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getCreateTime() {
            return createTime;
        }

        public final void setCreateTime(Instant createTime) {
            this.createTime = createTime;
        }

        @Override
        public final Builder createTime(Instant createTime) {
            this.createTime = createTime;
            return this;
        }

        public final IntegrationConfig.Builder getIntegrationConfig() {
            return integrationConfig != null ? integrationConfig.toBuilder() : null;
        }

        public final void setIntegrationConfig(IntegrationConfig.BuilderImpl integrationConfig) {
            this.integrationConfig = integrationConfig != null ? integrationConfig.build() : null;
        }

        @Override
        public final Builder integrationConfig(IntegrationConfig integrationConfig) {
            this.integrationConfig = integrationConfig;
            return this;
        }

        public final List<IntegrationError.Builder> getErrors() {
            List<IntegrationError.Builder> result = IntegrationErrorListCopier.copyToBuilder(this.errors);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setErrors(Collection<IntegrationError.BuilderImpl> errors) {
            this.errors = IntegrationErrorListCopier.copyFromBuilder(errors);
        }

        @Override
        public final Builder errors(Collection<IntegrationError> errors) {
            this.errors = IntegrationErrorListCopier.copy(errors);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder errors(IntegrationError... errors) {
            errors(Arrays.asList(errors));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder errors(Consumer<IntegrationError.Builder>... errors) {
            errors(Stream.of(errors).map(c -> IntegrationError.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getDataFilter() {
            return dataFilter;
        }

        public final void setDataFilter(String dataFilter) {
            this.dataFilter = dataFilter;
        }

        @Override
        public final Builder dataFilter(String dataFilter) {
            this.dataFilter = dataFilter;
            return this;
        }

        @Override
        public Integration build() {
            return new Integration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
