/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration for an Iceberg compaction optimizer. This configuration defines parameters for optimizing the
 * layout of data files in Iceberg tables.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IcebergCompactionConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<IcebergCompactionConfiguration.Builder, IcebergCompactionConfiguration> {
    private static final SdkField<String> STRATEGY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("strategy").getter(getter(IcebergCompactionConfiguration::strategyAsString))
            .setter(setter(Builder::strategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("strategy").build()).build();

    private static final SdkField<Integer> MIN_INPUT_FILES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("minInputFiles").getter(getter(IcebergCompactionConfiguration::minInputFiles))
            .setter(setter(Builder::minInputFiles))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("minInputFiles").build()).build();

    private static final SdkField<Integer> DELETE_FILE_THRESHOLD_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("deleteFileThreshold").getter(getter(IcebergCompactionConfiguration::deleteFileThreshold))
            .setter(setter(Builder::deleteFileThreshold))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deleteFileThreshold").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STRATEGY_FIELD,
            MIN_INPUT_FILES_FIELD, DELETE_FILE_THRESHOLD_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String strategy;

    private final Integer minInputFiles;

    private final Integer deleteFileThreshold;

    private IcebergCompactionConfiguration(BuilderImpl builder) {
        this.strategy = builder.strategy;
        this.minInputFiles = builder.minInputFiles;
        this.deleteFileThreshold = builder.deleteFileThreshold;
    }

    /**
     * <p>
     * The strategy to use for compaction. Valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>binpack</code>: Combines small files into larger files, typically targeting sizes over 100MB, while
     * applying any pending deletes. This is the recommended compaction strategy for most use cases.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sort</code>: Organizes data based on specified columns which are sorted hierarchically during compaction,
     * improving query performance for filtered operations. This strategy is recommended when your queries frequently
     * filter on specific columns. To use this strategy, you must first define a sort order in your Iceberg table
     * properties using the <code>sort_order</code> table property.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>z-order</code>: Optimizes data organization by blending multiple attributes into a single scalar value that
     * can be used for sorting, allowing efficient querying across multiple dimensions. This strategy is recommended
     * when you need to query data across multiple dimensions simultaneously. To use this strategy, you must first
     * define a sort order in your Iceberg table properties using the <code>sort_order</code> table property.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If an input is not provided, the default value 'binpack' will be used.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #strategy} will
     * return {@link CompactionStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #strategyAsString}.
     * </p>
     * 
     * @return The strategy to use for compaction. Valid values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>binpack</code>: Combines small files into larger files, typically targeting sizes over 100MB, while
     *         applying any pending deletes. This is the recommended compaction strategy for most use cases.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>sort</code>: Organizes data based on specified columns which are sorted hierarchically during
     *         compaction, improving query performance for filtered operations. This strategy is recommended when your
     *         queries frequently filter on specific columns. To use this strategy, you must first define a sort order
     *         in your Iceberg table properties using the <code>sort_order</code> table property.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>z-order</code>: Optimizes data organization by blending multiple attributes into a single scalar
     *         value that can be used for sorting, allowing efficient querying across multiple dimensions. This strategy
     *         is recommended when you need to query data across multiple dimensions simultaneously. To use this
     *         strategy, you must first define a sort order in your Iceberg table properties using the
     *         <code>sort_order</code> table property.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If an input is not provided, the default value 'binpack' will be used.
     * @see CompactionStrategy
     */
    public final CompactionStrategy strategy() {
        return CompactionStrategy.fromValue(strategy);
    }

    /**
     * <p>
     * The strategy to use for compaction. Valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>binpack</code>: Combines small files into larger files, typically targeting sizes over 100MB, while
     * applying any pending deletes. This is the recommended compaction strategy for most use cases.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sort</code>: Organizes data based on specified columns which are sorted hierarchically during compaction,
     * improving query performance for filtered operations. This strategy is recommended when your queries frequently
     * filter on specific columns. To use this strategy, you must first define a sort order in your Iceberg table
     * properties using the <code>sort_order</code> table property.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>z-order</code>: Optimizes data organization by blending multiple attributes into a single scalar value that
     * can be used for sorting, allowing efficient querying across multiple dimensions. This strategy is recommended
     * when you need to query data across multiple dimensions simultaneously. To use this strategy, you must first
     * define a sort order in your Iceberg table properties using the <code>sort_order</code> table property.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If an input is not provided, the default value 'binpack' will be used.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #strategy} will
     * return {@link CompactionStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #strategyAsString}.
     * </p>
     * 
     * @return The strategy to use for compaction. Valid values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>binpack</code>: Combines small files into larger files, typically targeting sizes over 100MB, while
     *         applying any pending deletes. This is the recommended compaction strategy for most use cases.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>sort</code>: Organizes data based on specified columns which are sorted hierarchically during
     *         compaction, improving query performance for filtered operations. This strategy is recommended when your
     *         queries frequently filter on specific columns. To use this strategy, you must first define a sort order
     *         in your Iceberg table properties using the <code>sort_order</code> table property.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>z-order</code>: Optimizes data organization by blending multiple attributes into a single scalar
     *         value that can be used for sorting, allowing efficient querying across multiple dimensions. This strategy
     *         is recommended when you need to query data across multiple dimensions simultaneously. To use this
     *         strategy, you must first define a sort order in your Iceberg table properties using the
     *         <code>sort_order</code> table property.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If an input is not provided, the default value 'binpack' will be used.
     * @see CompactionStrategy
     */
    public final String strategyAsString() {
        return strategy;
    }

    /**
     * <p>
     * The minimum number of data files that must be present in a partition before compaction will actually compact
     * files. This parameter helps control when compaction is triggered, preventing unnecessary compaction operations on
     * partitions with few files. If an input is not provided, the default value 100 will be used.
     * </p>
     * 
     * @return The minimum number of data files that must be present in a partition before compaction will actually
     *         compact files. This parameter helps control when compaction is triggered, preventing unnecessary
     *         compaction operations on partitions with few files. If an input is not provided, the default value 100
     *         will be used.
     */
    public final Integer minInputFiles() {
        return minInputFiles;
    }

    /**
     * <p>
     * The minimum number of deletes that must be present in a data file to make it eligible for compaction. This
     * parameter helps optimize compaction by focusing on files that contain a significant number of delete operations,
     * which can improve query performance by removing deleted records. If an input is not provided, the default value 1
     * will be used.
     * </p>
     * 
     * @return The minimum number of deletes that must be present in a data file to make it eligible for compaction.
     *         This parameter helps optimize compaction by focusing on files that contain a significant number of delete
     *         operations, which can improve query performance by removing deleted records. If an input is not provided,
     *         the default value 1 will be used.
     */
    public final Integer deleteFileThreshold() {
        return deleteFileThreshold;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(strategyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(minInputFiles());
        hashCode = 31 * hashCode + Objects.hashCode(deleteFileThreshold());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IcebergCompactionConfiguration)) {
            return false;
        }
        IcebergCompactionConfiguration other = (IcebergCompactionConfiguration) obj;
        return Objects.equals(strategyAsString(), other.strategyAsString())
                && Objects.equals(minInputFiles(), other.minInputFiles())
                && Objects.equals(deleteFileThreshold(), other.deleteFileThreshold());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("IcebergCompactionConfiguration").add("Strategy", strategyAsString())
                .add("MinInputFiles", minInputFiles()).add("DeleteFileThreshold", deleteFileThreshold()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "strategy":
            return Optional.ofNullable(clazz.cast(strategyAsString()));
        case "minInputFiles":
            return Optional.ofNullable(clazz.cast(minInputFiles()));
        case "deleteFileThreshold":
            return Optional.ofNullable(clazz.cast(deleteFileThreshold()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("strategy", STRATEGY_FIELD);
        map.put("minInputFiles", MIN_INPUT_FILES_FIELD);
        map.put("deleteFileThreshold", DELETE_FILE_THRESHOLD_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<IcebergCompactionConfiguration, T> g) {
        return obj -> g.apply((IcebergCompactionConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IcebergCompactionConfiguration> {
        /**
         * <p>
         * The strategy to use for compaction. Valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>binpack</code>: Combines small files into larger files, typically targeting sizes over 100MB, while
         * applying any pending deletes. This is the recommended compaction strategy for most use cases.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>sort</code>: Organizes data based on specified columns which are sorted hierarchically during
         * compaction, improving query performance for filtered operations. This strategy is recommended when your
         * queries frequently filter on specific columns. To use this strategy, you must first define a sort order in
         * your Iceberg table properties using the <code>sort_order</code> table property.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>z-order</code>: Optimizes data organization by blending multiple attributes into a single scalar value
         * that can be used for sorting, allowing efficient querying across multiple dimensions. This strategy is
         * recommended when you need to query data across multiple dimensions simultaneously. To use this strategy, you
         * must first define a sort order in your Iceberg table properties using the <code>sort_order</code> table
         * property.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If an input is not provided, the default value 'binpack' will be used.
         * </p>
         * 
         * @param strategy
         *        The strategy to use for compaction. Valid values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>binpack</code>: Combines small files into larger files, typically targeting sizes over 100MB,
         *        while applying any pending deletes. This is the recommended compaction strategy for most use cases.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>sort</code>: Organizes data based on specified columns which are sorted hierarchically during
         *        compaction, improving query performance for filtered operations. This strategy is recommended when
         *        your queries frequently filter on specific columns. To use this strategy, you must first define a sort
         *        order in your Iceberg table properties using the <code>sort_order</code> table property.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>z-order</code>: Optimizes data organization by blending multiple attributes into a single scalar
         *        value that can be used for sorting, allowing efficient querying across multiple dimensions. This
         *        strategy is recommended when you need to query data across multiple dimensions simultaneously. To use
         *        this strategy, you must first define a sort order in your Iceberg table properties using the
         *        <code>sort_order</code> table property.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If an input is not provided, the default value 'binpack' will be used.
         * @see CompactionStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CompactionStrategy
         */
        Builder strategy(String strategy);

        /**
         * <p>
         * The strategy to use for compaction. Valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>binpack</code>: Combines small files into larger files, typically targeting sizes over 100MB, while
         * applying any pending deletes. This is the recommended compaction strategy for most use cases.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>sort</code>: Organizes data based on specified columns which are sorted hierarchically during
         * compaction, improving query performance for filtered operations. This strategy is recommended when your
         * queries frequently filter on specific columns. To use this strategy, you must first define a sort order in
         * your Iceberg table properties using the <code>sort_order</code> table property.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>z-order</code>: Optimizes data organization by blending multiple attributes into a single scalar value
         * that can be used for sorting, allowing efficient querying across multiple dimensions. This strategy is
         * recommended when you need to query data across multiple dimensions simultaneously. To use this strategy, you
         * must first define a sort order in your Iceberg table properties using the <code>sort_order</code> table
         * property.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If an input is not provided, the default value 'binpack' will be used.
         * </p>
         * 
         * @param strategy
         *        The strategy to use for compaction. Valid values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>binpack</code>: Combines small files into larger files, typically targeting sizes over 100MB,
         *        while applying any pending deletes. This is the recommended compaction strategy for most use cases.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>sort</code>: Organizes data based on specified columns which are sorted hierarchically during
         *        compaction, improving query performance for filtered operations. This strategy is recommended when
         *        your queries frequently filter on specific columns. To use this strategy, you must first define a sort
         *        order in your Iceberg table properties using the <code>sort_order</code> table property.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>z-order</code>: Optimizes data organization by blending multiple attributes into a single scalar
         *        value that can be used for sorting, allowing efficient querying across multiple dimensions. This
         *        strategy is recommended when you need to query data across multiple dimensions simultaneously. To use
         *        this strategy, you must first define a sort order in your Iceberg table properties using the
         *        <code>sort_order</code> table property.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If an input is not provided, the default value 'binpack' will be used.
         * @see CompactionStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CompactionStrategy
         */
        Builder strategy(CompactionStrategy strategy);

        /**
         * <p>
         * The minimum number of data files that must be present in a partition before compaction will actually compact
         * files. This parameter helps control when compaction is triggered, preventing unnecessary compaction
         * operations on partitions with few files. If an input is not provided, the default value 100 will be used.
         * </p>
         * 
         * @param minInputFiles
         *        The minimum number of data files that must be present in a partition before compaction will actually
         *        compact files. This parameter helps control when compaction is triggered, preventing unnecessary
         *        compaction operations on partitions with few files. If an input is not provided, the default value 100
         *        will be used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minInputFiles(Integer minInputFiles);

        /**
         * <p>
         * The minimum number of deletes that must be present in a data file to make it eligible for compaction. This
         * parameter helps optimize compaction by focusing on files that contain a significant number of delete
         * operations, which can improve query performance by removing deleted records. If an input is not provided, the
         * default value 1 will be used.
         * </p>
         * 
         * @param deleteFileThreshold
         *        The minimum number of deletes that must be present in a data file to make it eligible for compaction.
         *        This parameter helps optimize compaction by focusing on files that contain a significant number of
         *        delete operations, which can improve query performance by removing deleted records. If an input is not
         *        provided, the default value 1 will be used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deleteFileThreshold(Integer deleteFileThreshold);
    }

    static final class BuilderImpl implements Builder {
        private String strategy;

        private Integer minInputFiles;

        private Integer deleteFileThreshold;

        private BuilderImpl() {
        }

        private BuilderImpl(IcebergCompactionConfiguration model) {
            strategy(model.strategy);
            minInputFiles(model.minInputFiles);
            deleteFileThreshold(model.deleteFileThreshold);
        }

        public final String getStrategy() {
            return strategy;
        }

        public final void setStrategy(String strategy) {
            this.strategy = strategy;
        }

        @Override
        public final Builder strategy(String strategy) {
            this.strategy = strategy;
            return this;
        }

        @Override
        public final Builder strategy(CompactionStrategy strategy) {
            this.strategy(strategy == null ? null : strategy.toString());
            return this;
        }

        public final Integer getMinInputFiles() {
            return minInputFiles;
        }

        public final void setMinInputFiles(Integer minInputFiles) {
            this.minInputFiles = minInputFiles;
        }

        @Override
        public final Builder minInputFiles(Integer minInputFiles) {
            this.minInputFiles = minInputFiles;
            return this;
        }

        public final Integer getDeleteFileThreshold() {
            return deleteFileThreshold;
        }

        public final void setDeleteFileThreshold(Integer deleteFileThreshold) {
            this.deleteFileThreshold = deleteFileThreshold;
        }

        @Override
        public final Builder deleteFileThreshold(Integer deleteFileThreshold) {
            this.deleteFileThreshold = deleteFileThreshold;
            return this;
        }

        @Override
        public IcebergCompactionConfiguration build() {
            return new IcebergCompactionConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
