/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A set of instructions that define the set of server processes to run on computes in a fleet. Server processes run
 * either an executable in a custom game build or a Amazon GameLift Servers Realtime script. Amazon GameLift Servers
 * launches the processes, manages their life cycle, and replaces them as needed. Computes check regularly for an
 * updated runtime configuration.
 * </p>
 * <p>
 * An Amazon GameLift Servers instance is limited to 50 processes running concurrently. To calculate the total number of
 * processes defined in a runtime configuration, add the values of the <code>ConcurrentExecutions</code> parameter for
 * each server process. Learn more about <a
 * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-multiprocess.html"> Running Multiple
 * Processes on a Fleet</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RuntimeConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<RuntimeConfiguration.Builder, RuntimeConfiguration> {
    private static final SdkField<List<ServerProcess>> SERVER_PROCESSES_FIELD = SdkField
            .<List<ServerProcess>> builder(MarshallingType.LIST)
            .memberName("ServerProcesses")
            .getter(getter(RuntimeConfiguration::serverProcesses))
            .setter(setter(Builder::serverProcesses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerProcesses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ServerProcess> builder(MarshallingType.SDK_POJO)
                                            .constructor(ServerProcess::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> MAX_CONCURRENT_GAME_SESSION_ACTIVATIONS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxConcurrentGameSessionActivations")
            .getter(getter(RuntimeConfiguration::maxConcurrentGameSessionActivations))
            .setter(setter(Builder::maxConcurrentGameSessionActivations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("MaxConcurrentGameSessionActivations").build()).build();

    private static final SdkField<Integer> GAME_SESSION_ACTIVATION_TIMEOUT_SECONDS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("GameSessionActivationTimeoutSeconds")
            .getter(getter(RuntimeConfiguration::gameSessionActivationTimeoutSeconds))
            .setter(setter(Builder::gameSessionActivationTimeoutSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("GameSessionActivationTimeoutSeconds").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SERVER_PROCESSES_FIELD,
            MAX_CONCURRENT_GAME_SESSION_ACTIVATIONS_FIELD, GAME_SESSION_ACTIVATION_TIMEOUT_SECONDS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<ServerProcess> serverProcesses;

    private final Integer maxConcurrentGameSessionActivations;

    private final Integer gameSessionActivationTimeoutSeconds;

    private RuntimeConfiguration(BuilderImpl builder) {
        this.serverProcesses = builder.serverProcesses;
        this.maxConcurrentGameSessionActivations = builder.maxConcurrentGameSessionActivations;
        this.gameSessionActivationTimeoutSeconds = builder.gameSessionActivationTimeoutSeconds;
    }

    /**
     * For responses, this returns true if the service returned a value for the ServerProcesses property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasServerProcesses() {
        return serverProcesses != null && !(serverProcesses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A collection of server process configurations that identify what server processes to run on fleet computes.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasServerProcesses} method.
     * </p>
     * 
     * @return A collection of server process configurations that identify what server processes to run on fleet
     *         computes.
     */
    public final List<ServerProcess> serverProcesses() {
        return serverProcesses;
    }

    /**
     * <p>
     * The number of game sessions in status <code>ACTIVATING</code> to allow on an instance or compute. This setting
     * limits the instance resources that can be used for new game activations at any one time.
     * </p>
     * 
     * @return The number of game sessions in status <code>ACTIVATING</code> to allow on an instance or compute. This
     *         setting limits the instance resources that can be used for new game activations at any one time.
     */
    public final Integer maxConcurrentGameSessionActivations() {
        return maxConcurrentGameSessionActivations;
    }

    /**
     * <p>
     * The maximum amount of time (in seconds) allowed to launch a new game session and have it report ready to host
     * players. During this time, the game session is in status <code>ACTIVATING</code>. If the game session does not
     * become active before the timeout, it is ended and the game session status is changed to <code>TERMINATED</code>.
     * </p>
     * 
     * @return The maximum amount of time (in seconds) allowed to launch a new game session and have it report ready to
     *         host players. During this time, the game session is in status <code>ACTIVATING</code>. If the game
     *         session does not become active before the timeout, it is ended and the game session status is changed to
     *         <code>TERMINATED</code>.
     */
    public final Integer gameSessionActivationTimeoutSeconds() {
        return gameSessionActivationTimeoutSeconds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasServerProcesses() ? serverProcesses() : null);
        hashCode = 31 * hashCode + Objects.hashCode(maxConcurrentGameSessionActivations());
        hashCode = 31 * hashCode + Objects.hashCode(gameSessionActivationTimeoutSeconds());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RuntimeConfiguration)) {
            return false;
        }
        RuntimeConfiguration other = (RuntimeConfiguration) obj;
        return hasServerProcesses() == other.hasServerProcesses() && Objects.equals(serverProcesses(), other.serverProcesses())
                && Objects.equals(maxConcurrentGameSessionActivations(), other.maxConcurrentGameSessionActivations())
                && Objects.equals(gameSessionActivationTimeoutSeconds(), other.gameSessionActivationTimeoutSeconds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RuntimeConfiguration").add("ServerProcesses", hasServerProcesses() ? serverProcesses() : null)
                .add("MaxConcurrentGameSessionActivations", maxConcurrentGameSessionActivations())
                .add("GameSessionActivationTimeoutSeconds", gameSessionActivationTimeoutSeconds()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ServerProcesses":
            return Optional.ofNullable(clazz.cast(serverProcesses()));
        case "MaxConcurrentGameSessionActivations":
            return Optional.ofNullable(clazz.cast(maxConcurrentGameSessionActivations()));
        case "GameSessionActivationTimeoutSeconds":
            return Optional.ofNullable(clazz.cast(gameSessionActivationTimeoutSeconds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ServerProcesses", SERVER_PROCESSES_FIELD);
        map.put("MaxConcurrentGameSessionActivations", MAX_CONCURRENT_GAME_SESSION_ACTIVATIONS_FIELD);
        map.put("GameSessionActivationTimeoutSeconds", GAME_SESSION_ACTIVATION_TIMEOUT_SECONDS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RuntimeConfiguration, T> g) {
        return obj -> g.apply((RuntimeConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RuntimeConfiguration> {
        /**
         * <p>
         * A collection of server process configurations that identify what server processes to run on fleet computes.
         * </p>
         * 
         * @param serverProcesses
         *        A collection of server process configurations that identify what server processes to run on fleet
         *        computes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverProcesses(Collection<ServerProcess> serverProcesses);

        /**
         * <p>
         * A collection of server process configurations that identify what server processes to run on fleet computes.
         * </p>
         * 
         * @param serverProcesses
         *        A collection of server process configurations that identify what server processes to run on fleet
         *        computes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverProcesses(ServerProcess... serverProcesses);

        /**
         * <p>
         * A collection of server process configurations that identify what server processes to run on fleet computes.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.gamelift.model.ServerProcess.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.gamelift.model.ServerProcess#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.gamelift.model.ServerProcess.Builder#build()} is called immediately
         * and its result is passed to {@link #serverProcesses(List<ServerProcess>)}.
         * 
         * @param serverProcesses
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.gamelift.model.ServerProcess.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #serverProcesses(java.util.Collection<ServerProcess>)
         */
        Builder serverProcesses(Consumer<ServerProcess.Builder>... serverProcesses);

        /**
         * <p>
         * The number of game sessions in status <code>ACTIVATING</code> to allow on an instance or compute. This
         * setting limits the instance resources that can be used for new game activations at any one time.
         * </p>
         * 
         * @param maxConcurrentGameSessionActivations
         *        The number of game sessions in status <code>ACTIVATING</code> to allow on an instance or compute. This
         *        setting limits the instance resources that can be used for new game activations at any one time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxConcurrentGameSessionActivations(Integer maxConcurrentGameSessionActivations);

        /**
         * <p>
         * The maximum amount of time (in seconds) allowed to launch a new game session and have it report ready to host
         * players. During this time, the game session is in status <code>ACTIVATING</code>. If the game session does
         * not become active before the timeout, it is ended and the game session status is changed to
         * <code>TERMINATED</code>.
         * </p>
         * 
         * @param gameSessionActivationTimeoutSeconds
         *        The maximum amount of time (in seconds) allowed to launch a new game session and have it report ready
         *        to host players. During this time, the game session is in status <code>ACTIVATING</code>. If the game
         *        session does not become active before the timeout, it is ended and the game session status is changed
         *        to <code>TERMINATED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionActivationTimeoutSeconds(Integer gameSessionActivationTimeoutSeconds);
    }

    static final class BuilderImpl implements Builder {
        private List<ServerProcess> serverProcesses = DefaultSdkAutoConstructList.getInstance();

        private Integer maxConcurrentGameSessionActivations;

        private Integer gameSessionActivationTimeoutSeconds;

        private BuilderImpl() {
        }

        private BuilderImpl(RuntimeConfiguration model) {
            serverProcesses(model.serverProcesses);
            maxConcurrentGameSessionActivations(model.maxConcurrentGameSessionActivations);
            gameSessionActivationTimeoutSeconds(model.gameSessionActivationTimeoutSeconds);
        }

        public final List<ServerProcess.Builder> getServerProcesses() {
            List<ServerProcess.Builder> result = ServerProcessListCopier.copyToBuilder(this.serverProcesses);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setServerProcesses(Collection<ServerProcess.BuilderImpl> serverProcesses) {
            this.serverProcesses = ServerProcessListCopier.copyFromBuilder(serverProcesses);
        }

        @Override
        public final Builder serverProcesses(Collection<ServerProcess> serverProcesses) {
            this.serverProcesses = ServerProcessListCopier.copy(serverProcesses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder serverProcesses(ServerProcess... serverProcesses) {
            serverProcesses(Arrays.asList(serverProcesses));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder serverProcesses(Consumer<ServerProcess.Builder>... serverProcesses) {
            serverProcesses(Stream.of(serverProcesses).map(c -> ServerProcess.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Integer getMaxConcurrentGameSessionActivations() {
            return maxConcurrentGameSessionActivations;
        }

        public final void setMaxConcurrentGameSessionActivations(Integer maxConcurrentGameSessionActivations) {
            this.maxConcurrentGameSessionActivations = maxConcurrentGameSessionActivations;
        }

        @Override
        public final Builder maxConcurrentGameSessionActivations(Integer maxConcurrentGameSessionActivations) {
            this.maxConcurrentGameSessionActivations = maxConcurrentGameSessionActivations;
            return this;
        }

        public final Integer getGameSessionActivationTimeoutSeconds() {
            return gameSessionActivationTimeoutSeconds;
        }

        public final void setGameSessionActivationTimeoutSeconds(Integer gameSessionActivationTimeoutSeconds) {
            this.gameSessionActivationTimeoutSeconds = gameSessionActivationTimeoutSeconds;
        }

        @Override
        public final Builder gameSessionActivationTimeoutSeconds(Integer gameSessionActivationTimeoutSeconds) {
            this.gameSessionActivationTimeoutSeconds = gameSessionActivationTimeoutSeconds;
            return this;
        }

        @Override
        public RuntimeConfiguration build() {
            return new RuntimeConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
