/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration of the self-managed Microsoft Active Directory (AD) directory to which the Windows File Server or
 * ONTAP storage virtual machine (SVM) instance is joined.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SelfManagedActiveDirectoryAttributes implements SdkPojo, Serializable,
        ToCopyableBuilder<SelfManagedActiveDirectoryAttributes.Builder, SelfManagedActiveDirectoryAttributes> {
    private static final SdkField<String> DOMAIN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DomainName").getter(getter(SelfManagedActiveDirectoryAttributes::domainName))
            .setter(setter(Builder::domainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainName").build()).build();

    private static final SdkField<String> ORGANIZATIONAL_UNIT_DISTINGUISHED_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("OrganizationalUnitDistinguishedName")
            .getter(getter(SelfManagedActiveDirectoryAttributes::organizationalUnitDistinguishedName))
            .setter(setter(Builder::organizationalUnitDistinguishedName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("OrganizationalUnitDistinguishedName").build()).build();

    private static final SdkField<String> FILE_SYSTEM_ADMINISTRATORS_GROUP_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("FileSystemAdministratorsGroup")
            .getter(getter(SelfManagedActiveDirectoryAttributes::fileSystemAdministratorsGroup))
            .setter(setter(Builder::fileSystemAdministratorsGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileSystemAdministratorsGroup")
                    .build()).build();

    private static final SdkField<String> USER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UserName").getter(getter(SelfManagedActiveDirectoryAttributes::userName))
            .setter(setter(Builder::userName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserName").build()).build();

    private static final SdkField<List<String>> DNS_IPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DnsIps")
            .getter(getter(SelfManagedActiveDirectoryAttributes::dnsIps))
            .setter(setter(Builder::dnsIps))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsIps").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DOMAIN_JOIN_SERVICE_ACCOUNT_SECRET_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DomainJoinServiceAccountSecret")
            .getter(getter(SelfManagedActiveDirectoryAttributes::domainJoinServiceAccountSecret))
            .setter(setter(Builder::domainJoinServiceAccountSecret))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainJoinServiceAccountSecret")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DOMAIN_NAME_FIELD,
            ORGANIZATIONAL_UNIT_DISTINGUISHED_NAME_FIELD, FILE_SYSTEM_ADMINISTRATORS_GROUP_FIELD, USER_NAME_FIELD, DNS_IPS_FIELD,
            DOMAIN_JOIN_SERVICE_ACCOUNT_SECRET_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String domainName;

    private final String organizationalUnitDistinguishedName;

    private final String fileSystemAdministratorsGroup;

    private final String userName;

    private final List<String> dnsIps;

    private final String domainJoinServiceAccountSecret;

    private SelfManagedActiveDirectoryAttributes(BuilderImpl builder) {
        this.domainName = builder.domainName;
        this.organizationalUnitDistinguishedName = builder.organizationalUnitDistinguishedName;
        this.fileSystemAdministratorsGroup = builder.fileSystemAdministratorsGroup;
        this.userName = builder.userName;
        this.dnsIps = builder.dnsIps;
        this.domainJoinServiceAccountSecret = builder.domainJoinServiceAccountSecret;
    }

    /**
     * <p>
     * The fully qualified domain name of the self-managed AD directory.
     * </p>
     * 
     * @return The fully qualified domain name of the self-managed AD directory.
     */
    public final String domainName() {
        return domainName;
    }

    /**
     * <p>
     * The fully qualified distinguished name of the organizational unit within the self-managed AD directory to which
     * the Windows File Server or ONTAP storage virtual machine (SVM) instance is joined.
     * </p>
     * 
     * @return The fully qualified distinguished name of the organizational unit within the self-managed AD directory to
     *         which the Windows File Server or ONTAP storage virtual machine (SVM) instance is joined.
     */
    public final String organizationalUnitDistinguishedName() {
        return organizationalUnitDistinguishedName;
    }

    /**
     * <p>
     * The name of the domain group whose members have administrative privileges for the FSx file system.
     * </p>
     * 
     * @return The name of the domain group whose members have administrative privileges for the FSx file system.
     */
    public final String fileSystemAdministratorsGroup() {
        return fileSystemAdministratorsGroup;
    }

    /**
     * <p>
     * The user name for the service account on your self-managed AD domain that FSx uses to join to your AD domain.
     * </p>
     * 
     * @return The user name for the service account on your self-managed AD domain that FSx uses to join to your AD
     *         domain.
     */
    public final String userName() {
        return userName;
    }

    /**
     * For responses, this returns true if the service returned a value for the DnsIps property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDnsIps() {
        return dnsIps != null && !(dnsIps instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of up to three IP addresses of DNS servers or domain controllers in the self-managed AD directory.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDnsIps} method.
     * </p>
     * 
     * @return A list of up to three IP addresses of DNS servers or domain controllers in the self-managed AD directory.
     */
    public final List<String> dnsIps() {
        return dnsIps;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the Amazon Web Services Secrets Manager secret containing the service account
     * credentials used to join the file system to your self-managed Active Directory domain.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the Amazon Web Services Secrets Manager secret containing the service
     *         account credentials used to join the file system to your self-managed Active Directory domain.
     */
    public final String domainJoinServiceAccountSecret() {
        return domainJoinServiceAccountSecret;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(domainName());
        hashCode = 31 * hashCode + Objects.hashCode(organizationalUnitDistinguishedName());
        hashCode = 31 * hashCode + Objects.hashCode(fileSystemAdministratorsGroup());
        hashCode = 31 * hashCode + Objects.hashCode(userName());
        hashCode = 31 * hashCode + Objects.hashCode(hasDnsIps() ? dnsIps() : null);
        hashCode = 31 * hashCode + Objects.hashCode(domainJoinServiceAccountSecret());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SelfManagedActiveDirectoryAttributes)) {
            return false;
        }
        SelfManagedActiveDirectoryAttributes other = (SelfManagedActiveDirectoryAttributes) obj;
        return Objects.equals(domainName(), other.domainName())
                && Objects.equals(organizationalUnitDistinguishedName(), other.organizationalUnitDistinguishedName())
                && Objects.equals(fileSystemAdministratorsGroup(), other.fileSystemAdministratorsGroup())
                && Objects.equals(userName(), other.userName()) && hasDnsIps() == other.hasDnsIps()
                && Objects.equals(dnsIps(), other.dnsIps())
                && Objects.equals(domainJoinServiceAccountSecret(), other.domainJoinServiceAccountSecret());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SelfManagedActiveDirectoryAttributes").add("DomainName", domainName())
                .add("OrganizationalUnitDistinguishedName", organizationalUnitDistinguishedName())
                .add("FileSystemAdministratorsGroup", fileSystemAdministratorsGroup()).add("UserName", userName())
                .add("DnsIps", hasDnsIps() ? dnsIps() : null)
                .add("DomainJoinServiceAccountSecret", domainJoinServiceAccountSecret()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DomainName":
            return Optional.ofNullable(clazz.cast(domainName()));
        case "OrganizationalUnitDistinguishedName":
            return Optional.ofNullable(clazz.cast(organizationalUnitDistinguishedName()));
        case "FileSystemAdministratorsGroup":
            return Optional.ofNullable(clazz.cast(fileSystemAdministratorsGroup()));
        case "UserName":
            return Optional.ofNullable(clazz.cast(userName()));
        case "DnsIps":
            return Optional.ofNullable(clazz.cast(dnsIps()));
        case "DomainJoinServiceAccountSecret":
            return Optional.ofNullable(clazz.cast(domainJoinServiceAccountSecret()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("DomainName", DOMAIN_NAME_FIELD);
        map.put("OrganizationalUnitDistinguishedName", ORGANIZATIONAL_UNIT_DISTINGUISHED_NAME_FIELD);
        map.put("FileSystemAdministratorsGroup", FILE_SYSTEM_ADMINISTRATORS_GROUP_FIELD);
        map.put("UserName", USER_NAME_FIELD);
        map.put("DnsIps", DNS_IPS_FIELD);
        map.put("DomainJoinServiceAccountSecret", DOMAIN_JOIN_SERVICE_ACCOUNT_SECRET_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SelfManagedActiveDirectoryAttributes, T> g) {
        return obj -> g.apply((SelfManagedActiveDirectoryAttributes) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SelfManagedActiveDirectoryAttributes> {
        /**
         * <p>
         * The fully qualified domain name of the self-managed AD directory.
         * </p>
         * 
         * @param domainName
         *        The fully qualified domain name of the self-managed AD directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainName(String domainName);

        /**
         * <p>
         * The fully qualified distinguished name of the organizational unit within the self-managed AD directory to
         * which the Windows File Server or ONTAP storage virtual machine (SVM) instance is joined.
         * </p>
         * 
         * @param organizationalUnitDistinguishedName
         *        The fully qualified distinguished name of the organizational unit within the self-managed AD directory
         *        to which the Windows File Server or ONTAP storage virtual machine (SVM) instance is joined.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder organizationalUnitDistinguishedName(String organizationalUnitDistinguishedName);

        /**
         * <p>
         * The name of the domain group whose members have administrative privileges for the FSx file system.
         * </p>
         * 
         * @param fileSystemAdministratorsGroup
         *        The name of the domain group whose members have administrative privileges for the FSx file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileSystemAdministratorsGroup(String fileSystemAdministratorsGroup);

        /**
         * <p>
         * The user name for the service account on your self-managed AD domain that FSx uses to join to your AD domain.
         * </p>
         * 
         * @param userName
         *        The user name for the service account on your self-managed AD domain that FSx uses to join to your AD
         *        domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userName(String userName);

        /**
         * <p>
         * A list of up to three IP addresses of DNS servers or domain controllers in the self-managed AD directory.
         * </p>
         * 
         * @param dnsIps
         *        A list of up to three IP addresses of DNS servers or domain controllers in the self-managed AD
         *        directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsIps(Collection<String> dnsIps);

        /**
         * <p>
         * A list of up to three IP addresses of DNS servers or domain controllers in the self-managed AD directory.
         * </p>
         * 
         * @param dnsIps
         *        A list of up to three IP addresses of DNS servers or domain controllers in the self-managed AD
         *        directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsIps(String... dnsIps);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Amazon Web Services Secrets Manager secret containing the service
         * account credentials used to join the file system to your self-managed Active Directory domain.
         * </p>
         * 
         * @param domainJoinServiceAccountSecret
         *        The Amazon Resource Name (ARN) of the Amazon Web Services Secrets Manager secret containing the
         *        service account credentials used to join the file system to your self-managed Active Directory domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainJoinServiceAccountSecret(String domainJoinServiceAccountSecret);
    }

    static final class BuilderImpl implements Builder {
        private String domainName;

        private String organizationalUnitDistinguishedName;

        private String fileSystemAdministratorsGroup;

        private String userName;

        private List<String> dnsIps = DefaultSdkAutoConstructList.getInstance();

        private String domainJoinServiceAccountSecret;

        private BuilderImpl() {
        }

        private BuilderImpl(SelfManagedActiveDirectoryAttributes model) {
            domainName(model.domainName);
            organizationalUnitDistinguishedName(model.organizationalUnitDistinguishedName);
            fileSystemAdministratorsGroup(model.fileSystemAdministratorsGroup);
            userName(model.userName);
            dnsIps(model.dnsIps);
            domainJoinServiceAccountSecret(model.domainJoinServiceAccountSecret);
        }

        public final String getDomainName() {
            return domainName;
        }

        public final void setDomainName(String domainName) {
            this.domainName = domainName;
        }

        @Override
        public final Builder domainName(String domainName) {
            this.domainName = domainName;
            return this;
        }

        public final String getOrganizationalUnitDistinguishedName() {
            return organizationalUnitDistinguishedName;
        }

        public final void setOrganizationalUnitDistinguishedName(String organizationalUnitDistinguishedName) {
            this.organizationalUnitDistinguishedName = organizationalUnitDistinguishedName;
        }

        @Override
        public final Builder organizationalUnitDistinguishedName(String organizationalUnitDistinguishedName) {
            this.organizationalUnitDistinguishedName = organizationalUnitDistinguishedName;
            return this;
        }

        public final String getFileSystemAdministratorsGroup() {
            return fileSystemAdministratorsGroup;
        }

        public final void setFileSystemAdministratorsGroup(String fileSystemAdministratorsGroup) {
            this.fileSystemAdministratorsGroup = fileSystemAdministratorsGroup;
        }

        @Override
        public final Builder fileSystemAdministratorsGroup(String fileSystemAdministratorsGroup) {
            this.fileSystemAdministratorsGroup = fileSystemAdministratorsGroup;
            return this;
        }

        public final String getUserName() {
            return userName;
        }

        public final void setUserName(String userName) {
            this.userName = userName;
        }

        @Override
        public final Builder userName(String userName) {
            this.userName = userName;
            return this;
        }

        public final Collection<String> getDnsIps() {
            if (dnsIps instanceof SdkAutoConstructList) {
                return null;
            }
            return dnsIps;
        }

        public final void setDnsIps(Collection<String> dnsIps) {
            this.dnsIps = DnsIpsCopier.copy(dnsIps);
        }

        @Override
        public final Builder dnsIps(Collection<String> dnsIps) {
            this.dnsIps = DnsIpsCopier.copy(dnsIps);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsIps(String... dnsIps) {
            dnsIps(Arrays.asList(dnsIps));
            return this;
        }

        public final String getDomainJoinServiceAccountSecret() {
            return domainJoinServiceAccountSecret;
        }

        public final void setDomainJoinServiceAccountSecret(String domainJoinServiceAccountSecret) {
            this.domainJoinServiceAccountSecret = domainJoinServiceAccountSecret;
        }

        @Override
        public final Builder domainJoinServiceAccountSecret(String domainJoinServiceAccountSecret) {
            this.domainJoinServiceAccountSecret = domainJoinServiceAccountSecret;
            return this;
        }

        @Override
        public SelfManagedActiveDirectoryAttributes build() {
            return new SelfManagedActiveDirectoryAttributes(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
