/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration object for the Microsoft Windows file system used in <code>CreateFileSystem</code> and
 * <code>CreateFileSystemFromBackup</code> operations.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateFileSystemWindowsConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<CreateFileSystemWindowsConfiguration.Builder, CreateFileSystemWindowsConfiguration> {
    private static final SdkField<String> ACTIVE_DIRECTORY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ActiveDirectoryId").getter(getter(CreateFileSystemWindowsConfiguration::activeDirectoryId))
            .setter(setter(Builder::activeDirectoryId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActiveDirectoryId").build()).build();

    private static final SdkField<SelfManagedActiveDirectoryConfiguration> SELF_MANAGED_ACTIVE_DIRECTORY_CONFIGURATION_FIELD = SdkField
            .<SelfManagedActiveDirectoryConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("SelfManagedActiveDirectoryConfiguration")
            .getter(getter(CreateFileSystemWindowsConfiguration::selfManagedActiveDirectoryConfiguration))
            .setter(setter(Builder::selfManagedActiveDirectoryConfiguration))
            .constructor(SelfManagedActiveDirectoryConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("SelfManagedActiveDirectoryConfiguration").build()).build();

    private static final SdkField<String> DEPLOYMENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DeploymentType").getter(getter(CreateFileSystemWindowsConfiguration::deploymentTypeAsString))
            .setter(setter(Builder::deploymentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeploymentType").build()).build();

    private static final SdkField<String> PREFERRED_SUBNET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PreferredSubnetId").getter(getter(CreateFileSystemWindowsConfiguration::preferredSubnetId))
            .setter(setter(Builder::preferredSubnetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreferredSubnetId").build()).build();

    private static final SdkField<Integer> THROUGHPUT_CAPACITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ThroughputCapacity").getter(getter(CreateFileSystemWindowsConfiguration::throughputCapacity))
            .setter(setter(Builder::throughputCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ThroughputCapacity").build())
            .build();

    private static final SdkField<String> WEEKLY_MAINTENANCE_START_TIME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("WeeklyMaintenanceStartTime")
            .getter(getter(CreateFileSystemWindowsConfiguration::weeklyMaintenanceStartTime))
            .setter(setter(Builder::weeklyMaintenanceStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WeeklyMaintenanceStartTime").build())
            .build();

    private static final SdkField<String> DAILY_AUTOMATIC_BACKUP_START_TIME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DailyAutomaticBackupStartTime")
            .getter(getter(CreateFileSystemWindowsConfiguration::dailyAutomaticBackupStartTime))
            .setter(setter(Builder::dailyAutomaticBackupStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DailyAutomaticBackupStartTime")
                    .build()).build();

    private static final SdkField<Integer> AUTOMATIC_BACKUP_RETENTION_DAYS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("AutomaticBackupRetentionDays")
            .getter(getter(CreateFileSystemWindowsConfiguration::automaticBackupRetentionDays))
            .setter(setter(Builder::automaticBackupRetentionDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutomaticBackupRetentionDays")
                    .build()).build();

    private static final SdkField<Boolean> COPY_TAGS_TO_BACKUPS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CopyTagsToBackups").getter(getter(CreateFileSystemWindowsConfiguration::copyTagsToBackups))
            .setter(setter(Builder::copyTagsToBackups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CopyTagsToBackups").build()).build();

    private static final SdkField<List<String>> ALIASES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Aliases")
            .getter(getter(CreateFileSystemWindowsConfiguration::aliases))
            .setter(setter(Builder::aliases))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Aliases").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<WindowsAuditLogCreateConfiguration> AUDIT_LOG_CONFIGURATION_FIELD = SdkField
            .<WindowsAuditLogCreateConfiguration> builder(MarshallingType.SDK_POJO).memberName("AuditLogConfiguration")
            .getter(getter(CreateFileSystemWindowsConfiguration::auditLogConfiguration))
            .setter(setter(Builder::auditLogConfiguration)).constructor(WindowsAuditLogCreateConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AuditLogConfiguration").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACTIVE_DIRECTORY_ID_FIELD,
            SELF_MANAGED_ACTIVE_DIRECTORY_CONFIGURATION_FIELD, DEPLOYMENT_TYPE_FIELD, PREFERRED_SUBNET_ID_FIELD,
            THROUGHPUT_CAPACITY_FIELD, WEEKLY_MAINTENANCE_START_TIME_FIELD, DAILY_AUTOMATIC_BACKUP_START_TIME_FIELD,
            AUTOMATIC_BACKUP_RETENTION_DAYS_FIELD, COPY_TAGS_TO_BACKUPS_FIELD, ALIASES_FIELD, AUDIT_LOG_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String activeDirectoryId;

    private final SelfManagedActiveDirectoryConfiguration selfManagedActiveDirectoryConfiguration;

    private final String deploymentType;

    private final String preferredSubnetId;

    private final Integer throughputCapacity;

    private final String weeklyMaintenanceStartTime;

    private final String dailyAutomaticBackupStartTime;

    private final Integer automaticBackupRetentionDays;

    private final Boolean copyTagsToBackups;

    private final List<String> aliases;

    private final WindowsAuditLogCreateConfiguration auditLogConfiguration;

    private CreateFileSystemWindowsConfiguration(BuilderImpl builder) {
        this.activeDirectoryId = builder.activeDirectoryId;
        this.selfManagedActiveDirectoryConfiguration = builder.selfManagedActiveDirectoryConfiguration;
        this.deploymentType = builder.deploymentType;
        this.preferredSubnetId = builder.preferredSubnetId;
        this.throughputCapacity = builder.throughputCapacity;
        this.weeklyMaintenanceStartTime = builder.weeklyMaintenanceStartTime;
        this.dailyAutomaticBackupStartTime = builder.dailyAutomaticBackupStartTime;
        this.automaticBackupRetentionDays = builder.automaticBackupRetentionDays;
        this.copyTagsToBackups = builder.copyTagsToBackups;
        this.aliases = builder.aliases;
        this.auditLogConfiguration = builder.auditLogConfiguration;
    }

    /**
     * <p>
     * The ID for an existing Amazon Web Services Managed Microsoft Active Directory (AD) instance that the file system
     * should join when it's created.
     * </p>
     * 
     * @return The ID for an existing Amazon Web Services Managed Microsoft Active Directory (AD) instance that the file
     *         system should join when it's created.
     */
    public final String activeDirectoryId() {
        return activeDirectoryId;
    }

    /**
     * Returns the value of the SelfManagedActiveDirectoryConfiguration property for this object.
     * 
     * @return The value of the SelfManagedActiveDirectoryConfiguration property for this object.
     */
    public final SelfManagedActiveDirectoryConfiguration selfManagedActiveDirectoryConfiguration() {
        return selfManagedActiveDirectoryConfiguration;
    }

    /**
     * <p>
     * Specifies the file system deployment type, valid values are the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>MULTI_AZ_1</code> - Deploys a high availability file system that is configured for Multi-AZ redundancy to
     * tolerate temporary Availability Zone (AZ) unavailability. You can only deploy a Multi-AZ file system in Amazon
     * Web Services Regions that have a minimum of three Availability Zones. Also supports HDD storage type
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SINGLE_AZ_1</code> - (Default) Choose to deploy a file system that is configured for single AZ redundancy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SINGLE_AZ_2</code> - The latest generation Single AZ file system. Specifies a file system that is
     * configured for single AZ redundancy and supports HDD storage type.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html"> Availability and
     * Durability: Single-AZ and Multi-AZ File Systems</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #deploymentType}
     * will return {@link WindowsDeploymentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #deploymentTypeAsString}.
     * </p>
     * 
     * @return Specifies the file system deployment type, valid values are the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>MULTI_AZ_1</code> - Deploys a high availability file system that is configured for Multi-AZ
     *         redundancy to tolerate temporary Availability Zone (AZ) unavailability. You can only deploy a Multi-AZ
     *         file system in Amazon Web Services Regions that have a minimum of three Availability Zones. Also supports
     *         HDD storage type
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SINGLE_AZ_1</code> - (Default) Choose to deploy a file system that is configured for single AZ
     *         redundancy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SINGLE_AZ_2</code> - The latest generation Single AZ file system. Specifies a file system that is
     *         configured for single AZ redundancy and supports HDD storage type.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html"> Availability
     *         and Durability: Single-AZ and Multi-AZ File Systems</a>.
     * @see WindowsDeploymentType
     */
    public final WindowsDeploymentType deploymentType() {
        return WindowsDeploymentType.fromValue(deploymentType);
    }

    /**
     * <p>
     * Specifies the file system deployment type, valid values are the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>MULTI_AZ_1</code> - Deploys a high availability file system that is configured for Multi-AZ redundancy to
     * tolerate temporary Availability Zone (AZ) unavailability. You can only deploy a Multi-AZ file system in Amazon
     * Web Services Regions that have a minimum of three Availability Zones. Also supports HDD storage type
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SINGLE_AZ_1</code> - (Default) Choose to deploy a file system that is configured for single AZ redundancy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SINGLE_AZ_2</code> - The latest generation Single AZ file system. Specifies a file system that is
     * configured for single AZ redundancy and supports HDD storage type.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html"> Availability and
     * Durability: Single-AZ and Multi-AZ File Systems</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #deploymentType}
     * will return {@link WindowsDeploymentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #deploymentTypeAsString}.
     * </p>
     * 
     * @return Specifies the file system deployment type, valid values are the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>MULTI_AZ_1</code> - Deploys a high availability file system that is configured for Multi-AZ
     *         redundancy to tolerate temporary Availability Zone (AZ) unavailability. You can only deploy a Multi-AZ
     *         file system in Amazon Web Services Regions that have a minimum of three Availability Zones. Also supports
     *         HDD storage type
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SINGLE_AZ_1</code> - (Default) Choose to deploy a file system that is configured for single AZ
     *         redundancy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SINGLE_AZ_2</code> - The latest generation Single AZ file system. Specifies a file system that is
     *         configured for single AZ redundancy and supports HDD storage type.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html"> Availability
     *         and Durability: Single-AZ and Multi-AZ File Systems</a>.
     * @see WindowsDeploymentType
     */
    public final String deploymentTypeAsString() {
        return deploymentType;
    }

    /**
     * <p>
     * Required when <code>DeploymentType</code> is set to <code>MULTI_AZ_1</code>. This specifies the subnet in which
     * you want the preferred file server to be located. For in-Amazon Web Services applications, we recommend that you
     * launch your clients in the same Availability Zone (AZ) as your preferred file server to reduce cross-AZ data
     * transfer costs and minimize latency.
     * </p>
     * 
     * @return Required when <code>DeploymentType</code> is set to <code>MULTI_AZ_1</code>. This specifies the subnet in
     *         which you want the preferred file server to be located. For in-Amazon Web Services applications, we
     *         recommend that you launch your clients in the same Availability Zone (AZ) as your preferred file server
     *         to reduce cross-AZ data transfer costs and minimize latency.
     */
    public final String preferredSubnetId() {
        return preferredSubnetId;
    }

    /**
     * <p>
     * Sets the throughput capacity of an Amazon FSx file system, measured in megabytes per second (MB/s), in 2 to the
     * <i>n</i>th increments, between 2^3 (8) and 2^11 (2048).
     * </p>
     * 
     * @return Sets the throughput capacity of an Amazon FSx file system, measured in megabytes per second (MB/s), in 2
     *         to the <i>n</i>th increments, between 2^3 (8) and 2^11 (2048).
     */
    public final Integer throughputCapacity() {
        return throughputCapacity;
    }

    /**
     * <p>
     * The preferred start time to perform weekly maintenance, formatted d:HH:MM in the UTC time zone, where d is the
     * weekday number, from 1 through 7, beginning with Monday and ending with Sunday.
     * </p>
     * 
     * @return The preferred start time to perform weekly maintenance, formatted d:HH:MM in the UTC time zone, where d
     *         is the weekday number, from 1 through 7, beginning with Monday and ending with Sunday.
     */
    public final String weeklyMaintenanceStartTime() {
        return weeklyMaintenanceStartTime;
    }

    /**
     * <p>
     * The preferred time to take daily automatic backups, formatted HH:MM in the UTC time zone.
     * </p>
     * 
     * @return The preferred time to take daily automatic backups, formatted HH:MM in the UTC time zone.
     */
    public final String dailyAutomaticBackupStartTime() {
        return dailyAutomaticBackupStartTime;
    }

    /**
     * <p>
     * The number of days to retain automatic backups. Setting this property to <code>0</code> disables automatic
     * backups. You can retain automatic backups for a maximum of 90 days. The default is <code>30</code>.
     * </p>
     * 
     * @return The number of days to retain automatic backups. Setting this property to <code>0</code> disables
     *         automatic backups. You can retain automatic backups for a maximum of 90 days. The default is
     *         <code>30</code>.
     */
    public final Integer automaticBackupRetentionDays() {
        return automaticBackupRetentionDays;
    }

    /**
     * <p>
     * A boolean flag indicating whether tags for the file system should be copied to backups. This value defaults to
     * false. If it's set to true, all tags for the file system are copied to all automatic and user-initiated backups
     * where the user doesn't specify tags. If this value is true, and you specify one or more tags, only the specified
     * tags are copied to backups. If you specify one or more tags when creating a user-initiated backup, no tags are
     * copied from the file system, regardless of this value.
     * </p>
     * 
     * @return A boolean flag indicating whether tags for the file system should be copied to backups. This value
     *         defaults to false. If it's set to true, all tags for the file system are copied to all automatic and
     *         user-initiated backups where the user doesn't specify tags. If this value is true, and you specify one or
     *         more tags, only the specified tags are copied to backups. If you specify one or more tags when creating a
     *         user-initiated backup, no tags are copied from the file system, regardless of this value.
     */
    public final Boolean copyTagsToBackups() {
        return copyTagsToBackups;
    }

    /**
     * For responses, this returns true if the service returned a value for the Aliases property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAliases() {
        return aliases != null && !(aliases instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of one or more DNS alias names that you want to associate with the Amazon FSx file system. Aliases allow
     * you to use existing DNS names to access the data in your Amazon FSx file system. You can associate up to 50
     * aliases with a file system at any time. You can associate additional DNS aliases after you create the file system
     * using the AssociateFileSystemAliases operation. You can remove DNS aliases from the file system after it is
     * created using the DisassociateFileSystemAliases operation. You only need to specify the alias name in the request
     * payload.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/managing-dns-aliases.html">Working with DNS Aliases</a>
     * and <a href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/walkthrough05-file-system-custom-CNAME.html">
     * Walkthrough 5: Using DNS aliases to access your file system</a>, including additional steps you must take to be
     * able to access your file system using a DNS alias.
     * </p>
     * <p>
     * An alias name has to meet the following requirements:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Formatted as a fully-qualified domain name (FQDN), <code>hostname.domain</code>, for example,
     * <code>accounting.example.com</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Can contain alphanumeric characters, the underscore (_), and the hyphen (-).
     * </p>
     * </li>
     * <li>
     * <p>
     * Cannot start or end with a hyphen.
     * </p>
     * </li>
     * <li>
     * <p>
     * Can start with a numeric.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For DNS alias names, Amazon FSx stores alphabetic characters as lowercase letters (a-z), regardless of how you
     * specify them: as uppercase letters, lowercase letters, or the corresponding letters in escape codes.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAliases} method.
     * </p>
     * 
     * @return An array of one or more DNS alias names that you want to associate with the Amazon FSx file system.
     *         Aliases allow you to use existing DNS names to access the data in your Amazon FSx file system. You can
     *         associate up to 50 aliases with a file system at any time. You can associate additional DNS aliases after
     *         you create the file system using the AssociateFileSystemAliases operation. You can remove DNS aliases
     *         from the file system after it is created using the DisassociateFileSystemAliases operation. You only need
     *         to specify the alias name in the request payload.</p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/managing-dns-aliases.html">Working with DNS
     *         Aliases</a> and <a
     *         href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/walkthrough05-file-system-custom-CNAME.html"
     *         >Walkthrough 5: Using DNS aliases to access your file system</a>, including additional steps you must
     *         take to be able to access your file system using a DNS alias.
     *         </p>
     *         <p>
     *         An alias name has to meet the following requirements:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Formatted as a fully-qualified domain name (FQDN), <code>hostname.domain</code>, for example,
     *         <code>accounting.example.com</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Can contain alphanumeric characters, the underscore (_), and the hyphen (-).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Cannot start or end with a hyphen.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Can start with a numeric.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For DNS alias names, Amazon FSx stores alphabetic characters as lowercase letters (a-z), regardless of
     *         how you specify them: as uppercase letters, lowercase letters, or the corresponding letters in escape
     *         codes.
     */
    public final List<String> aliases() {
        return aliases;
    }

    /**
     * <p>
     * The configuration that Amazon FSx for Windows File Server uses to audit and log user accesses of files, folders,
     * and file shares on the Amazon FSx for Windows File Server file system.
     * </p>
     * 
     * @return The configuration that Amazon FSx for Windows File Server uses to audit and log user accesses of files,
     *         folders, and file shares on the Amazon FSx for Windows File Server file system.
     */
    public final WindowsAuditLogCreateConfiguration auditLogConfiguration() {
        return auditLogConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(activeDirectoryId());
        hashCode = 31 * hashCode + Objects.hashCode(selfManagedActiveDirectoryConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(deploymentTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(preferredSubnetId());
        hashCode = 31 * hashCode + Objects.hashCode(throughputCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(weeklyMaintenanceStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(dailyAutomaticBackupStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(automaticBackupRetentionDays());
        hashCode = 31 * hashCode + Objects.hashCode(copyTagsToBackups());
        hashCode = 31 * hashCode + Objects.hashCode(hasAliases() ? aliases() : null);
        hashCode = 31 * hashCode + Objects.hashCode(auditLogConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateFileSystemWindowsConfiguration)) {
            return false;
        }
        CreateFileSystemWindowsConfiguration other = (CreateFileSystemWindowsConfiguration) obj;
        return Objects.equals(activeDirectoryId(), other.activeDirectoryId())
                && Objects.equals(selfManagedActiveDirectoryConfiguration(), other.selfManagedActiveDirectoryConfiguration())
                && Objects.equals(deploymentTypeAsString(), other.deploymentTypeAsString())
                && Objects.equals(preferredSubnetId(), other.preferredSubnetId())
                && Objects.equals(throughputCapacity(), other.throughputCapacity())
                && Objects.equals(weeklyMaintenanceStartTime(), other.weeklyMaintenanceStartTime())
                && Objects.equals(dailyAutomaticBackupStartTime(), other.dailyAutomaticBackupStartTime())
                && Objects.equals(automaticBackupRetentionDays(), other.automaticBackupRetentionDays())
                && Objects.equals(copyTagsToBackups(), other.copyTagsToBackups()) && hasAliases() == other.hasAliases()
                && Objects.equals(aliases(), other.aliases())
                && Objects.equals(auditLogConfiguration(), other.auditLogConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateFileSystemWindowsConfiguration").add("ActiveDirectoryId", activeDirectoryId())
                .add("SelfManagedActiveDirectoryConfiguration", selfManagedActiveDirectoryConfiguration())
                .add("DeploymentType", deploymentTypeAsString()).add("PreferredSubnetId", preferredSubnetId())
                .add("ThroughputCapacity", throughputCapacity()).add("WeeklyMaintenanceStartTime", weeklyMaintenanceStartTime())
                .add("DailyAutomaticBackupStartTime", dailyAutomaticBackupStartTime())
                .add("AutomaticBackupRetentionDays", automaticBackupRetentionDays())
                .add("CopyTagsToBackups", copyTagsToBackups()).add("Aliases", hasAliases() ? aliases() : null)
                .add("AuditLogConfiguration", auditLogConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ActiveDirectoryId":
            return Optional.ofNullable(clazz.cast(activeDirectoryId()));
        case "SelfManagedActiveDirectoryConfiguration":
            return Optional.ofNullable(clazz.cast(selfManagedActiveDirectoryConfiguration()));
        case "DeploymentType":
            return Optional.ofNullable(clazz.cast(deploymentTypeAsString()));
        case "PreferredSubnetId":
            return Optional.ofNullable(clazz.cast(preferredSubnetId()));
        case "ThroughputCapacity":
            return Optional.ofNullable(clazz.cast(throughputCapacity()));
        case "WeeklyMaintenanceStartTime":
            return Optional.ofNullable(clazz.cast(weeklyMaintenanceStartTime()));
        case "DailyAutomaticBackupStartTime":
            return Optional.ofNullable(clazz.cast(dailyAutomaticBackupStartTime()));
        case "AutomaticBackupRetentionDays":
            return Optional.ofNullable(clazz.cast(automaticBackupRetentionDays()));
        case "CopyTagsToBackups":
            return Optional.ofNullable(clazz.cast(copyTagsToBackups()));
        case "Aliases":
            return Optional.ofNullable(clazz.cast(aliases()));
        case "AuditLogConfiguration":
            return Optional.ofNullable(clazz.cast(auditLogConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateFileSystemWindowsConfiguration, T> g) {
        return obj -> g.apply((CreateFileSystemWindowsConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CreateFileSystemWindowsConfiguration> {
        /**
         * <p>
         * The ID for an existing Amazon Web Services Managed Microsoft Active Directory (AD) instance that the file
         * system should join when it's created.
         * </p>
         * 
         * @param activeDirectoryId
         *        The ID for an existing Amazon Web Services Managed Microsoft Active Directory (AD) instance that the
         *        file system should join when it's created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder activeDirectoryId(String activeDirectoryId);

        /**
         * Sets the value of the SelfManagedActiveDirectoryConfiguration property for this object.
         *
         * @param selfManagedActiveDirectoryConfiguration
         *        The new value for the SelfManagedActiveDirectoryConfiguration property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder selfManagedActiveDirectoryConfiguration(
                SelfManagedActiveDirectoryConfiguration selfManagedActiveDirectoryConfiguration);

        /**
         * Sets the value of the SelfManagedActiveDirectoryConfiguration property for this object.
         *
         * This is a convenience method that creates an instance of the
         * {@link SelfManagedActiveDirectoryConfiguration.Builder} avoiding the need to create one manually via
         * {@link SelfManagedActiveDirectoryConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SelfManagedActiveDirectoryConfiguration.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #selfManagedActiveDirectoryConfiguration(SelfManagedActiveDirectoryConfiguration)}.
         * 
         * @param selfManagedActiveDirectoryConfiguration
         *        a consumer that will call methods on {@link SelfManagedActiveDirectoryConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #selfManagedActiveDirectoryConfiguration(SelfManagedActiveDirectoryConfiguration)
         */
        default Builder selfManagedActiveDirectoryConfiguration(
                Consumer<SelfManagedActiveDirectoryConfiguration.Builder> selfManagedActiveDirectoryConfiguration) {
            return selfManagedActiveDirectoryConfiguration(SelfManagedActiveDirectoryConfiguration.builder()
                    .applyMutation(selfManagedActiveDirectoryConfiguration).build());
        }

        /**
         * <p>
         * Specifies the file system deployment type, valid values are the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>MULTI_AZ_1</code> - Deploys a high availability file system that is configured for Multi-AZ redundancy
         * to tolerate temporary Availability Zone (AZ) unavailability. You can only deploy a Multi-AZ file system in
         * Amazon Web Services Regions that have a minimum of three Availability Zones. Also supports HDD storage type
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SINGLE_AZ_1</code> - (Default) Choose to deploy a file system that is configured for single AZ
         * redundancy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SINGLE_AZ_2</code> - The latest generation Single AZ file system. Specifies a file system that is
         * configured for single AZ redundancy and supports HDD storage type.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html"> Availability and
         * Durability: Single-AZ and Multi-AZ File Systems</a>.
         * </p>
         * 
         * @param deploymentType
         *        Specifies the file system deployment type, valid values are the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>MULTI_AZ_1</code> - Deploys a high availability file system that is configured for Multi-AZ
         *        redundancy to tolerate temporary Availability Zone (AZ) unavailability. You can only deploy a Multi-AZ
         *        file system in Amazon Web Services Regions that have a minimum of three Availability Zones. Also
         *        supports HDD storage type
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SINGLE_AZ_1</code> - (Default) Choose to deploy a file system that is configured for single AZ
         *        redundancy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SINGLE_AZ_2</code> - The latest generation Single AZ file system. Specifies a file system that
         *        is configured for single AZ redundancy and supports HDD storage type.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html">
         *        Availability and Durability: Single-AZ and Multi-AZ File Systems</a>.
         * @see WindowsDeploymentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WindowsDeploymentType
         */
        Builder deploymentType(String deploymentType);

        /**
         * <p>
         * Specifies the file system deployment type, valid values are the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>MULTI_AZ_1</code> - Deploys a high availability file system that is configured for Multi-AZ redundancy
         * to tolerate temporary Availability Zone (AZ) unavailability. You can only deploy a Multi-AZ file system in
         * Amazon Web Services Regions that have a minimum of three Availability Zones. Also supports HDD storage type
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SINGLE_AZ_1</code> - (Default) Choose to deploy a file system that is configured for single AZ
         * redundancy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SINGLE_AZ_2</code> - The latest generation Single AZ file system. Specifies a file system that is
         * configured for single AZ redundancy and supports HDD storage type.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html"> Availability and
         * Durability: Single-AZ and Multi-AZ File Systems</a>.
         * </p>
         * 
         * @param deploymentType
         *        Specifies the file system deployment type, valid values are the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>MULTI_AZ_1</code> - Deploys a high availability file system that is configured for Multi-AZ
         *        redundancy to tolerate temporary Availability Zone (AZ) unavailability. You can only deploy a Multi-AZ
         *        file system in Amazon Web Services Regions that have a minimum of three Availability Zones. Also
         *        supports HDD storage type
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SINGLE_AZ_1</code> - (Default) Choose to deploy a file system that is configured for single AZ
         *        redundancy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SINGLE_AZ_2</code> - The latest generation Single AZ file system. Specifies a file system that
         *        is configured for single AZ redundancy and supports HDD storage type.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html">
         *        Availability and Durability: Single-AZ and Multi-AZ File Systems</a>.
         * @see WindowsDeploymentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WindowsDeploymentType
         */
        Builder deploymentType(WindowsDeploymentType deploymentType);

        /**
         * <p>
         * Required when <code>DeploymentType</code> is set to <code>MULTI_AZ_1</code>. This specifies the subnet in
         * which you want the preferred file server to be located. For in-Amazon Web Services applications, we recommend
         * that you launch your clients in the same Availability Zone (AZ) as your preferred file server to reduce
         * cross-AZ data transfer costs and minimize latency.
         * </p>
         * 
         * @param preferredSubnetId
         *        Required when <code>DeploymentType</code> is set to <code>MULTI_AZ_1</code>. This specifies the subnet
         *        in which you want the preferred file server to be located. For in-Amazon Web Services applications, we
         *        recommend that you launch your clients in the same Availability Zone (AZ) as your preferred file
         *        server to reduce cross-AZ data transfer costs and minimize latency.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredSubnetId(String preferredSubnetId);

        /**
         * <p>
         * Sets the throughput capacity of an Amazon FSx file system, measured in megabytes per second (MB/s), in 2 to
         * the <i>n</i>th increments, between 2^3 (8) and 2^11 (2048).
         * </p>
         * 
         * @param throughputCapacity
         *        Sets the throughput capacity of an Amazon FSx file system, measured in megabytes per second (MB/s), in
         *        2 to the <i>n</i>th increments, between 2^3 (8) and 2^11 (2048).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder throughputCapacity(Integer throughputCapacity);

        /**
         * <p>
         * The preferred start time to perform weekly maintenance, formatted d:HH:MM in the UTC time zone, where d is
         * the weekday number, from 1 through 7, beginning with Monday and ending with Sunday.
         * </p>
         * 
         * @param weeklyMaintenanceStartTime
         *        The preferred start time to perform weekly maintenance, formatted d:HH:MM in the UTC time zone, where
         *        d is the weekday number, from 1 through 7, beginning with Monday and ending with Sunday.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder weeklyMaintenanceStartTime(String weeklyMaintenanceStartTime);

        /**
         * <p>
         * The preferred time to take daily automatic backups, formatted HH:MM in the UTC time zone.
         * </p>
         * 
         * @param dailyAutomaticBackupStartTime
         *        The preferred time to take daily automatic backups, formatted HH:MM in the UTC time zone.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dailyAutomaticBackupStartTime(String dailyAutomaticBackupStartTime);

        /**
         * <p>
         * The number of days to retain automatic backups. Setting this property to <code>0</code> disables automatic
         * backups. You can retain automatic backups for a maximum of 90 days. The default is <code>30</code>.
         * </p>
         * 
         * @param automaticBackupRetentionDays
         *        The number of days to retain automatic backups. Setting this property to <code>0</code> disables
         *        automatic backups. You can retain automatic backups for a maximum of 90 days. The default is
         *        <code>30</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder automaticBackupRetentionDays(Integer automaticBackupRetentionDays);

        /**
         * <p>
         * A boolean flag indicating whether tags for the file system should be copied to backups. This value defaults
         * to false. If it's set to true, all tags for the file system are copied to all automatic and user-initiated
         * backups where the user doesn't specify tags. If this value is true, and you specify one or more tags, only
         * the specified tags are copied to backups. If you specify one or more tags when creating a user-initiated
         * backup, no tags are copied from the file system, regardless of this value.
         * </p>
         * 
         * @param copyTagsToBackups
         *        A boolean flag indicating whether tags for the file system should be copied to backups. This value
         *        defaults to false. If it's set to true, all tags for the file system are copied to all automatic and
         *        user-initiated backups where the user doesn't specify tags. If this value is true, and you specify one
         *        or more tags, only the specified tags are copied to backups. If you specify one or more tags when
         *        creating a user-initiated backup, no tags are copied from the file system, regardless of this value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder copyTagsToBackups(Boolean copyTagsToBackups);

        /**
         * <p>
         * An array of one or more DNS alias names that you want to associate with the Amazon FSx file system. Aliases
         * allow you to use existing DNS names to access the data in your Amazon FSx file system. You can associate up
         * to 50 aliases with a file system at any time. You can associate additional DNS aliases after you create the
         * file system using the AssociateFileSystemAliases operation. You can remove DNS aliases from the file system
         * after it is created using the DisassociateFileSystemAliases operation. You only need to specify the alias
         * name in the request payload.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/managing-dns-aliases.html">Working with DNS
         * Aliases</a> and <a
         * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/walkthrough05-file-system-custom-CNAME.html"
         * >Walkthrough 5: Using DNS aliases to access your file system</a>, including additional steps you must take to
         * be able to access your file system using a DNS alias.
         * </p>
         * <p>
         * An alias name has to meet the following requirements:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Formatted as a fully-qualified domain name (FQDN), <code>hostname.domain</code>, for example,
         * <code>accounting.example.com</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Can contain alphanumeric characters, the underscore (_), and the hyphen (-).
         * </p>
         * </li>
         * <li>
         * <p>
         * Cannot start or end with a hyphen.
         * </p>
         * </li>
         * <li>
         * <p>
         * Can start with a numeric.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For DNS alias names, Amazon FSx stores alphabetic characters as lowercase letters (a-z), regardless of how
         * you specify them: as uppercase letters, lowercase letters, or the corresponding letters in escape codes.
         * </p>
         * 
         * @param aliases
         *        An array of one or more DNS alias names that you want to associate with the Amazon FSx file system.
         *        Aliases allow you to use existing DNS names to access the data in your Amazon FSx file system. You can
         *        associate up to 50 aliases with a file system at any time. You can associate additional DNS aliases
         *        after you create the file system using the AssociateFileSystemAliases operation. You can remove DNS
         *        aliases from the file system after it is created using the DisassociateFileSystemAliases operation.
         *        You only need to specify the alias name in the request payload.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/managing-dns-aliases.html">Working with DNS
         *        Aliases</a> and <a href=
         *        "https://docs.aws.amazon.com/fsx/latest/WindowsGuide/walkthrough05-file-system-custom-CNAME.html"
         *        >Walkthrough 5: Using DNS aliases to access your file system</a>, including additional steps you must
         *        take to be able to access your file system using a DNS alias.
         *        </p>
         *        <p>
         *        An alias name has to meet the following requirements:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Formatted as a fully-qualified domain name (FQDN), <code>hostname.domain</code>, for example,
         *        <code>accounting.example.com</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Can contain alphanumeric characters, the underscore (_), and the hyphen (-).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Cannot start or end with a hyphen.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Can start with a numeric.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For DNS alias names, Amazon FSx stores alphabetic characters as lowercase letters (a-z), regardless of
         *        how you specify them: as uppercase letters, lowercase letters, or the corresponding letters in escape
         *        codes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder aliases(Collection<String> aliases);

        /**
         * <p>
         * An array of one or more DNS alias names that you want to associate with the Amazon FSx file system. Aliases
         * allow you to use existing DNS names to access the data in your Amazon FSx file system. You can associate up
         * to 50 aliases with a file system at any time. You can associate additional DNS aliases after you create the
         * file system using the AssociateFileSystemAliases operation. You can remove DNS aliases from the file system
         * after it is created using the DisassociateFileSystemAliases operation. You only need to specify the alias
         * name in the request payload.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/managing-dns-aliases.html">Working with DNS
         * Aliases</a> and <a
         * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/walkthrough05-file-system-custom-CNAME.html"
         * >Walkthrough 5: Using DNS aliases to access your file system</a>, including additional steps you must take to
         * be able to access your file system using a DNS alias.
         * </p>
         * <p>
         * An alias name has to meet the following requirements:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Formatted as a fully-qualified domain name (FQDN), <code>hostname.domain</code>, for example,
         * <code>accounting.example.com</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Can contain alphanumeric characters, the underscore (_), and the hyphen (-).
         * </p>
         * </li>
         * <li>
         * <p>
         * Cannot start or end with a hyphen.
         * </p>
         * </li>
         * <li>
         * <p>
         * Can start with a numeric.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For DNS alias names, Amazon FSx stores alphabetic characters as lowercase letters (a-z), regardless of how
         * you specify them: as uppercase letters, lowercase letters, or the corresponding letters in escape codes.
         * </p>
         * 
         * @param aliases
         *        An array of one or more DNS alias names that you want to associate with the Amazon FSx file system.
         *        Aliases allow you to use existing DNS names to access the data in your Amazon FSx file system. You can
         *        associate up to 50 aliases with a file system at any time. You can associate additional DNS aliases
         *        after you create the file system using the AssociateFileSystemAliases operation. You can remove DNS
         *        aliases from the file system after it is created using the DisassociateFileSystemAliases operation.
         *        You only need to specify the alias name in the request payload.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/managing-dns-aliases.html">Working with DNS
         *        Aliases</a> and <a href=
         *        "https://docs.aws.amazon.com/fsx/latest/WindowsGuide/walkthrough05-file-system-custom-CNAME.html"
         *        >Walkthrough 5: Using DNS aliases to access your file system</a>, including additional steps you must
         *        take to be able to access your file system using a DNS alias.
         *        </p>
         *        <p>
         *        An alias name has to meet the following requirements:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Formatted as a fully-qualified domain name (FQDN), <code>hostname.domain</code>, for example,
         *        <code>accounting.example.com</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Can contain alphanumeric characters, the underscore (_), and the hyphen (-).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Cannot start or end with a hyphen.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Can start with a numeric.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For DNS alias names, Amazon FSx stores alphabetic characters as lowercase letters (a-z), regardless of
         *        how you specify them: as uppercase letters, lowercase letters, or the corresponding letters in escape
         *        codes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder aliases(String... aliases);

        /**
         * <p>
         * The configuration that Amazon FSx for Windows File Server uses to audit and log user accesses of files,
         * folders, and file shares on the Amazon FSx for Windows File Server file system.
         * </p>
         * 
         * @param auditLogConfiguration
         *        The configuration that Amazon FSx for Windows File Server uses to audit and log user accesses of
         *        files, folders, and file shares on the Amazon FSx for Windows File Server file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder auditLogConfiguration(WindowsAuditLogCreateConfiguration auditLogConfiguration);

        /**
         * <p>
         * The configuration that Amazon FSx for Windows File Server uses to audit and log user accesses of files,
         * folders, and file shares on the Amazon FSx for Windows File Server file system.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link WindowsAuditLogCreateConfiguration.Builder} avoiding the need to create one manually via
         * {@link WindowsAuditLogCreateConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link WindowsAuditLogCreateConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #auditLogConfiguration(WindowsAuditLogCreateConfiguration)}.
         * 
         * @param auditLogConfiguration
         *        a consumer that will call methods on {@link WindowsAuditLogCreateConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #auditLogConfiguration(WindowsAuditLogCreateConfiguration)
         */
        default Builder auditLogConfiguration(Consumer<WindowsAuditLogCreateConfiguration.Builder> auditLogConfiguration) {
            return auditLogConfiguration(WindowsAuditLogCreateConfiguration.builder().applyMutation(auditLogConfiguration)
                    .build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String activeDirectoryId;

        private SelfManagedActiveDirectoryConfiguration selfManagedActiveDirectoryConfiguration;

        private String deploymentType;

        private String preferredSubnetId;

        private Integer throughputCapacity;

        private String weeklyMaintenanceStartTime;

        private String dailyAutomaticBackupStartTime;

        private Integer automaticBackupRetentionDays;

        private Boolean copyTagsToBackups;

        private List<String> aliases = DefaultSdkAutoConstructList.getInstance();

        private WindowsAuditLogCreateConfiguration auditLogConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateFileSystemWindowsConfiguration model) {
            activeDirectoryId(model.activeDirectoryId);
            selfManagedActiveDirectoryConfiguration(model.selfManagedActiveDirectoryConfiguration);
            deploymentType(model.deploymentType);
            preferredSubnetId(model.preferredSubnetId);
            throughputCapacity(model.throughputCapacity);
            weeklyMaintenanceStartTime(model.weeklyMaintenanceStartTime);
            dailyAutomaticBackupStartTime(model.dailyAutomaticBackupStartTime);
            automaticBackupRetentionDays(model.automaticBackupRetentionDays);
            copyTagsToBackups(model.copyTagsToBackups);
            aliases(model.aliases);
            auditLogConfiguration(model.auditLogConfiguration);
        }

        public final String getActiveDirectoryId() {
            return activeDirectoryId;
        }

        public final void setActiveDirectoryId(String activeDirectoryId) {
            this.activeDirectoryId = activeDirectoryId;
        }

        @Override
        public final Builder activeDirectoryId(String activeDirectoryId) {
            this.activeDirectoryId = activeDirectoryId;
            return this;
        }

        public final SelfManagedActiveDirectoryConfiguration.Builder getSelfManagedActiveDirectoryConfiguration() {
            return selfManagedActiveDirectoryConfiguration != null ? selfManagedActiveDirectoryConfiguration.toBuilder() : null;
        }

        public final void setSelfManagedActiveDirectoryConfiguration(
                SelfManagedActiveDirectoryConfiguration.BuilderImpl selfManagedActiveDirectoryConfiguration) {
            this.selfManagedActiveDirectoryConfiguration = selfManagedActiveDirectoryConfiguration != null ? selfManagedActiveDirectoryConfiguration
                    .build() : null;
        }

        @Override
        public final Builder selfManagedActiveDirectoryConfiguration(
                SelfManagedActiveDirectoryConfiguration selfManagedActiveDirectoryConfiguration) {
            this.selfManagedActiveDirectoryConfiguration = selfManagedActiveDirectoryConfiguration;
            return this;
        }

        public final String getDeploymentType() {
            return deploymentType;
        }

        public final void setDeploymentType(String deploymentType) {
            this.deploymentType = deploymentType;
        }

        @Override
        public final Builder deploymentType(String deploymentType) {
            this.deploymentType = deploymentType;
            return this;
        }

        @Override
        public final Builder deploymentType(WindowsDeploymentType deploymentType) {
            this.deploymentType(deploymentType == null ? null : deploymentType.toString());
            return this;
        }

        public final String getPreferredSubnetId() {
            return preferredSubnetId;
        }

        public final void setPreferredSubnetId(String preferredSubnetId) {
            this.preferredSubnetId = preferredSubnetId;
        }

        @Override
        public final Builder preferredSubnetId(String preferredSubnetId) {
            this.preferredSubnetId = preferredSubnetId;
            return this;
        }

        public final Integer getThroughputCapacity() {
            return throughputCapacity;
        }

        public final void setThroughputCapacity(Integer throughputCapacity) {
            this.throughputCapacity = throughputCapacity;
        }

        @Override
        public final Builder throughputCapacity(Integer throughputCapacity) {
            this.throughputCapacity = throughputCapacity;
            return this;
        }

        public final String getWeeklyMaintenanceStartTime() {
            return weeklyMaintenanceStartTime;
        }

        public final void setWeeklyMaintenanceStartTime(String weeklyMaintenanceStartTime) {
            this.weeklyMaintenanceStartTime = weeklyMaintenanceStartTime;
        }

        @Override
        public final Builder weeklyMaintenanceStartTime(String weeklyMaintenanceStartTime) {
            this.weeklyMaintenanceStartTime = weeklyMaintenanceStartTime;
            return this;
        }

        public final String getDailyAutomaticBackupStartTime() {
            return dailyAutomaticBackupStartTime;
        }

        public final void setDailyAutomaticBackupStartTime(String dailyAutomaticBackupStartTime) {
            this.dailyAutomaticBackupStartTime = dailyAutomaticBackupStartTime;
        }

        @Override
        public final Builder dailyAutomaticBackupStartTime(String dailyAutomaticBackupStartTime) {
            this.dailyAutomaticBackupStartTime = dailyAutomaticBackupStartTime;
            return this;
        }

        public final Integer getAutomaticBackupRetentionDays() {
            return automaticBackupRetentionDays;
        }

        public final void setAutomaticBackupRetentionDays(Integer automaticBackupRetentionDays) {
            this.automaticBackupRetentionDays = automaticBackupRetentionDays;
        }

        @Override
        public final Builder automaticBackupRetentionDays(Integer automaticBackupRetentionDays) {
            this.automaticBackupRetentionDays = automaticBackupRetentionDays;
            return this;
        }

        public final Boolean getCopyTagsToBackups() {
            return copyTagsToBackups;
        }

        public final void setCopyTagsToBackups(Boolean copyTagsToBackups) {
            this.copyTagsToBackups = copyTagsToBackups;
        }

        @Override
        public final Builder copyTagsToBackups(Boolean copyTagsToBackups) {
            this.copyTagsToBackups = copyTagsToBackups;
            return this;
        }

        public final Collection<String> getAliases() {
            if (aliases instanceof SdkAutoConstructList) {
                return null;
            }
            return aliases;
        }

        public final void setAliases(Collection<String> aliases) {
            this.aliases = AlternateDNSNamesCopier.copy(aliases);
        }

        @Override
        public final Builder aliases(Collection<String> aliases) {
            this.aliases = AlternateDNSNamesCopier.copy(aliases);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder aliases(String... aliases) {
            aliases(Arrays.asList(aliases));
            return this;
        }

        public final WindowsAuditLogCreateConfiguration.Builder getAuditLogConfiguration() {
            return auditLogConfiguration != null ? auditLogConfiguration.toBuilder() : null;
        }

        public final void setAuditLogConfiguration(WindowsAuditLogCreateConfiguration.BuilderImpl auditLogConfiguration) {
            this.auditLogConfiguration = auditLogConfiguration != null ? auditLogConfiguration.build() : null;
        }

        @Override
        public final Builder auditLogConfiguration(WindowsAuditLogCreateConfiguration auditLogConfiguration) {
            this.auditLogConfiguration = auditLogConfiguration;
            return this;
        }

        @Override
        public CreateFileSystemWindowsConfiguration build() {
            return new CreateFileSystemWindowsConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
