/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.freetier;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.freetier.model.AccessDeniedException;
import software.amazon.awssdk.services.freetier.model.FreeTierException;
import software.amazon.awssdk.services.freetier.model.GetAccountActivityRequest;
import software.amazon.awssdk.services.freetier.model.GetAccountActivityResponse;
import software.amazon.awssdk.services.freetier.model.GetAccountPlanStateRequest;
import software.amazon.awssdk.services.freetier.model.GetAccountPlanStateResponse;
import software.amazon.awssdk.services.freetier.model.GetFreeTierUsageRequest;
import software.amazon.awssdk.services.freetier.model.GetFreeTierUsageResponse;
import software.amazon.awssdk.services.freetier.model.InternalServerException;
import software.amazon.awssdk.services.freetier.model.ListAccountActivitiesRequest;
import software.amazon.awssdk.services.freetier.model.ListAccountActivitiesResponse;
import software.amazon.awssdk.services.freetier.model.ResourceNotFoundException;
import software.amazon.awssdk.services.freetier.model.ThrottlingException;
import software.amazon.awssdk.services.freetier.model.UpgradeAccountPlanRequest;
import software.amazon.awssdk.services.freetier.model.UpgradeAccountPlanResponse;
import software.amazon.awssdk.services.freetier.model.ValidationException;
import software.amazon.awssdk.services.freetier.paginators.GetFreeTierUsageIterable;
import software.amazon.awssdk.services.freetier.paginators.ListAccountActivitiesIterable;

/**
 * Service client for accessing AWS Free Tier. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * You can use the Amazon Web Services Free Tier API to query programmatically your Free Tier usage data.
 * </p>
 * <p>
 * Free Tier tracks your monthly usage data for all free tier offers that are associated with your Amazon Web Services
 * account. You can use the Free Tier API to filter and show only the data that you want.
 * </p>
 * <p>
 * Service endpoint
 * </p>
 * <p>
 * The Free Tier API provides the following endpoint:
 * </p>
 * <ul/>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-free-tier.html">Using the Amazon Web
 * Services Free Tier</a> in the <i>Billing User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface FreeTierClient extends AwsClient {
    String SERVICE_NAME = "freetier";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "freetier";

    /**
     * <p>
     * Returns a specific activity record that is available to the customer.
     * </p>
     *
     * @param getAccountActivityRequest
     * @return Result of the GetAccountActivity operation returned by the service.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of your request.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         This exception is thrown when the requested resource cannot be found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FreeTierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FreeTierClient.GetAccountActivity
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/freetier-2023-09-07/GetAccountActivity" target="_top">AWS
     *      API Documentation</a>
     */
    default GetAccountActivityResponse getAccountActivity(GetAccountActivityRequest getAccountActivityRequest)
            throws InternalServerException, ValidationException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, FreeTierException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a specific activity record that is available to the customer.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetAccountActivityRequest.Builder} avoiding the
     * need to create one manually via {@link GetAccountActivityRequest#builder()}
     * </p>
     *
     * @param getAccountActivityRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.freetier.model.GetAccountActivityRequest.Builder} to create a
     *        request.
     * @return Result of the GetAccountActivity operation returned by the service.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of your request.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         This exception is thrown when the requested resource cannot be found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FreeTierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FreeTierClient.GetAccountActivity
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/freetier-2023-09-07/GetAccountActivity" target="_top">AWS
     *      API Documentation</a>
     */
    default GetAccountActivityResponse getAccountActivity(Consumer<GetAccountActivityRequest.Builder> getAccountActivityRequest)
            throws InternalServerException, ValidationException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, FreeTierException {
        return getAccountActivity(GetAccountActivityRequest.builder().applyMutation(getAccountActivityRequest).build());
    }

    /**
     * <p>
     * This returns all of the information related to the state of the account plan related to Free Tier.
     * </p>
     *
     * @param getAccountPlanStateRequest
     * @return Result of the GetAccountPlanState operation returned by the service.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of your request.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         This exception is thrown when the requested resource cannot be found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FreeTierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FreeTierClient.GetAccountPlanState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/freetier-2023-09-07/GetAccountPlanState" target="_top">AWS
     *      API Documentation</a>
     */
    default GetAccountPlanStateResponse getAccountPlanState(GetAccountPlanStateRequest getAccountPlanStateRequest)
            throws InternalServerException, ValidationException, AccessDeniedException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, FreeTierException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * This returns all of the information related to the state of the account plan related to Free Tier.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetAccountPlanStateRequest.Builder} avoiding the
     * need to create one manually via {@link GetAccountPlanStateRequest#builder()}
     * </p>
     *
     * @param getAccountPlanStateRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.freetier.model.GetAccountPlanStateRequest.Builder} to create a
     *        request.
     * @return Result of the GetAccountPlanState operation returned by the service.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of your request.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         This exception is thrown when the requested resource cannot be found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FreeTierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FreeTierClient.GetAccountPlanState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/freetier-2023-09-07/GetAccountPlanState" target="_top">AWS
     *      API Documentation</a>
     */
    default GetAccountPlanStateResponse getAccountPlanState(
            Consumer<GetAccountPlanStateRequest.Builder> getAccountPlanStateRequest) throws InternalServerException,
            ValidationException, AccessDeniedException, ResourceNotFoundException, ThrottlingException, AwsServiceException,
            SdkClientException, FreeTierException {
        return getAccountPlanState(GetAccountPlanStateRequest.builder().applyMutation(getAccountPlanStateRequest).build());
    }

    /**
     * <p>
     * Returns a list of all Free Tier usage objects that match your filters.
     * </p>
     *
     * @param getFreeTierUsageRequest
     * @return Result of the GetFreeTierUsage operation returned by the service.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of your request.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FreeTierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FreeTierClient.GetFreeTierUsage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/freetier-2023-09-07/GetFreeTierUsage" target="_top">AWS API
     *      Documentation</a>
     */
    default GetFreeTierUsageResponse getFreeTierUsage(GetFreeTierUsageRequest getFreeTierUsageRequest)
            throws InternalServerException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            FreeTierException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of all Free Tier usage objects that match your filters.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetFreeTierUsageRequest.Builder} avoiding the need
     * to create one manually via {@link GetFreeTierUsageRequest#builder()}
     * </p>
     *
     * @param getFreeTierUsageRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.freetier.model.GetFreeTierUsageRequest.Builder} to create a
     *        request.
     * @return Result of the GetFreeTierUsage operation returned by the service.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of your request.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FreeTierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FreeTierClient.GetFreeTierUsage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/freetier-2023-09-07/GetFreeTierUsage" target="_top">AWS API
     *      Documentation</a>
     */
    default GetFreeTierUsageResponse getFreeTierUsage(Consumer<GetFreeTierUsageRequest.Builder> getFreeTierUsageRequest)
            throws InternalServerException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            FreeTierException {
        return getFreeTierUsage(GetFreeTierUsageRequest.builder().applyMutation(getFreeTierUsageRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #getFreeTierUsage(software.amazon.awssdk.services.freetier.model.GetFreeTierUsageRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.freetier.paginators.GetFreeTierUsageIterable responses = client.getFreeTierUsagePaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.freetier.paginators.GetFreeTierUsageIterable responses = client
     *             .getFreeTierUsagePaginator(request);
     *     for (software.amazon.awssdk.services.freetier.model.GetFreeTierUsageResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.freetier.paginators.GetFreeTierUsageIterable responses = client.getFreeTierUsagePaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getFreeTierUsage(software.amazon.awssdk.services.freetier.model.GetFreeTierUsageRequest)} operation.</b>
     * </p>
     *
     * @param getFreeTierUsageRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of your request.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FreeTierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FreeTierClient.GetFreeTierUsage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/freetier-2023-09-07/GetFreeTierUsage" target="_top">AWS API
     *      Documentation</a>
     */
    default GetFreeTierUsageIterable getFreeTierUsagePaginator(GetFreeTierUsageRequest getFreeTierUsageRequest)
            throws InternalServerException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            FreeTierException {
        return new GetFreeTierUsageIterable(this, getFreeTierUsageRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #getFreeTierUsage(software.amazon.awssdk.services.freetier.model.GetFreeTierUsageRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.freetier.paginators.GetFreeTierUsageIterable responses = client.getFreeTierUsagePaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.freetier.paginators.GetFreeTierUsageIterable responses = client
     *             .getFreeTierUsagePaginator(request);
     *     for (software.amazon.awssdk.services.freetier.model.GetFreeTierUsageResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.freetier.paginators.GetFreeTierUsageIterable responses = client.getFreeTierUsagePaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getFreeTierUsage(software.amazon.awssdk.services.freetier.model.GetFreeTierUsageRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetFreeTierUsageRequest.Builder} avoiding the need
     * to create one manually via {@link GetFreeTierUsageRequest#builder()}
     * </p>
     *
     * @param getFreeTierUsageRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.freetier.model.GetFreeTierUsageRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of your request.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FreeTierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FreeTierClient.GetFreeTierUsage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/freetier-2023-09-07/GetFreeTierUsage" target="_top">AWS API
     *      Documentation</a>
     */
    default GetFreeTierUsageIterable getFreeTierUsagePaginator(Consumer<GetFreeTierUsageRequest.Builder> getFreeTierUsageRequest)
            throws InternalServerException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            FreeTierException {
        return getFreeTierUsagePaginator(GetFreeTierUsageRequest.builder().applyMutation(getFreeTierUsageRequest).build());
    }

    /**
     * <p>
     * Returns a list of activities that are available. This operation supports pagination and filtering by status.
     * </p>
     *
     * @param listAccountActivitiesRequest
     * @return Result of the ListAccountActivities operation returned by the service.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of your request.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FreeTierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FreeTierClient.ListAccountActivities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/freetier-2023-09-07/ListAccountActivities"
     *      target="_top">AWS API Documentation</a>
     */
    default ListAccountActivitiesResponse listAccountActivities(ListAccountActivitiesRequest listAccountActivitiesRequest)
            throws InternalServerException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            FreeTierException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of activities that are available. This operation supports pagination and filtering by status.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListAccountActivitiesRequest.Builder} avoiding the
     * need to create one manually via {@link ListAccountActivitiesRequest#builder()}
     * </p>
     *
     * @param listAccountActivitiesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.freetier.model.ListAccountActivitiesRequest.Builder} to create a
     *        request.
     * @return Result of the ListAccountActivities operation returned by the service.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of your request.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FreeTierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FreeTierClient.ListAccountActivities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/freetier-2023-09-07/ListAccountActivities"
     *      target="_top">AWS API Documentation</a>
     */
    default ListAccountActivitiesResponse listAccountActivities(
            Consumer<ListAccountActivitiesRequest.Builder> listAccountActivitiesRequest) throws InternalServerException,
            ValidationException, ThrottlingException, AwsServiceException, SdkClientException, FreeTierException {
        return listAccountActivities(ListAccountActivitiesRequest.builder().applyMutation(listAccountActivitiesRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listAccountActivities(software.amazon.awssdk.services.freetier.model.ListAccountActivitiesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.freetier.paginators.ListAccountActivitiesIterable responses = client.listAccountActivitiesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.freetier.paginators.ListAccountActivitiesIterable responses = client
     *             .listAccountActivitiesPaginator(request);
     *     for (software.amazon.awssdk.services.freetier.model.ListAccountActivitiesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.freetier.paginators.ListAccountActivitiesIterable responses = client.listAccountActivitiesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAccountActivities(software.amazon.awssdk.services.freetier.model.ListAccountActivitiesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAccountActivitiesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of your request.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FreeTierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FreeTierClient.ListAccountActivities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/freetier-2023-09-07/ListAccountActivities"
     *      target="_top">AWS API Documentation</a>
     */
    default ListAccountActivitiesIterable listAccountActivitiesPaginator(ListAccountActivitiesRequest listAccountActivitiesRequest)
            throws InternalServerException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            FreeTierException {
        return new ListAccountActivitiesIterable(this, listAccountActivitiesRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listAccountActivities(software.amazon.awssdk.services.freetier.model.ListAccountActivitiesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.freetier.paginators.ListAccountActivitiesIterable responses = client.listAccountActivitiesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.freetier.paginators.ListAccountActivitiesIterable responses = client
     *             .listAccountActivitiesPaginator(request);
     *     for (software.amazon.awssdk.services.freetier.model.ListAccountActivitiesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.freetier.paginators.ListAccountActivitiesIterable responses = client.listAccountActivitiesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAccountActivities(software.amazon.awssdk.services.freetier.model.ListAccountActivitiesRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListAccountActivitiesRequest.Builder} avoiding the
     * need to create one manually via {@link ListAccountActivitiesRequest#builder()}
     * </p>
     *
     * @param listAccountActivitiesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.freetier.model.ListAccountActivitiesRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of your request.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FreeTierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FreeTierClient.ListAccountActivities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/freetier-2023-09-07/ListAccountActivities"
     *      target="_top">AWS API Documentation</a>
     */
    default ListAccountActivitiesIterable listAccountActivitiesPaginator(
            Consumer<ListAccountActivitiesRequest.Builder> listAccountActivitiesRequest) throws InternalServerException,
            ValidationException, ThrottlingException, AwsServiceException, SdkClientException, FreeTierException {
        return listAccountActivitiesPaginator(ListAccountActivitiesRequest.builder().applyMutation(listAccountActivitiesRequest)
                .build());
    }

    /**
     * <p>
     * The account plan type for the Amazon Web Services account.
     * </p>
     *
     * @param upgradeAccountPlanRequest
     * @return Result of the UpgradeAccountPlan operation returned by the service.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of your request.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         This exception is thrown when the requested resource cannot be found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FreeTierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FreeTierClient.UpgradeAccountPlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/freetier-2023-09-07/UpgradeAccountPlan" target="_top">AWS
     *      API Documentation</a>
     */
    default UpgradeAccountPlanResponse upgradeAccountPlan(UpgradeAccountPlanRequest upgradeAccountPlanRequest)
            throws InternalServerException, ValidationException, AccessDeniedException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, FreeTierException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * The account plan type for the Amazon Web Services account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpgradeAccountPlanRequest.Builder} avoiding the
     * need to create one manually via {@link UpgradeAccountPlanRequest#builder()}
     * </p>
     *
     * @param upgradeAccountPlanRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.freetier.model.UpgradeAccountPlanRequest.Builder} to create a
     *        request.
     * @return Result of the UpgradeAccountPlan operation returned by the service.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of your request.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         This exception is thrown when the requested resource cannot be found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FreeTierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FreeTierClient.UpgradeAccountPlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/freetier-2023-09-07/UpgradeAccountPlan" target="_top">AWS
     *      API Documentation</a>
     */
    default UpgradeAccountPlanResponse upgradeAccountPlan(Consumer<UpgradeAccountPlanRequest.Builder> upgradeAccountPlanRequest)
            throws InternalServerException, ValidationException, AccessDeniedException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, FreeTierException {
        return upgradeAccountPlan(UpgradeAccountPlanRequest.builder().applyMutation(upgradeAccountPlanRequest).build());
    }

    /**
     * Create a {@link FreeTierClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static FreeTierClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link FreeTierClient}.
     */
    static FreeTierClientBuilder builder() {
        return new DefaultFreeTierClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default FreeTierServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
